/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  CIM Firmware Update
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <poll.h>
#include <sys/ioctl.h>
#include <errno.h>

#include "fpd_ioctl.h"
#include "cim.h"
#include "events.h"
#include "paragon_cim.h"
#include "cim_update.h"


static unsigned char sernum[SERIAL_NUM_LENGTH];
static unsigned char devtype[DEVICE_TYPE_LENGTH+1];
static unsigned char hwver[HW_VERSION_LENGTH+1];
static unsigned char fwver[FW_VERSION_LENGTH+1];
static unsigned char auxval[AUX_VAL_LENGTH+1];
static unsigned char numAV;

static cim_update_upper_mem_rec_t m_Map[NUM_REGIONS];

static int enter_cim_fw_update( int fd, int link_if, int mode_chg )
{
    FPD_cimupdate_t cmd;
    int rc;

    cmd.link_if = link_if;
    cmd.mode_chg_option = mode_chg;
    rc = ioctl(fd, FPD_ENTER_CIM_UPDATE_MODE, &cmd);

    return rc;
}

static int exit_cim_fw_update( int fd, int link_if )
{
    FPD_cimupdate_t cmd;
    int rc;

    cmd.link_if = link_if;
    rc = ioctl(fd, FPD_EXIT_CIM_UPDATE_MODE, &cmd);

    return rc;
}

static unsigned char compute_chksum( unsigned char *pbuf, int len )
{
    unsigned char chksum = 0, i;

    for( i = 0; i < len; i++ ) {
        chksum += pbuf[i];
    }

    return chksum;
}

static int setup_expected_init_rsp( int fd, int link_if, int rsplen )
{
    FPD_cmdprog_t pcmd;
    line_connection_t old_conn, *pconn = NULL;

    /* retrieve current line connection to be used for re-connection */
    if( cim_get_line_connection_info(link_if, &pconn) == 0 ) {
        old_conn = *pconn;
    }

    /* required to disconnect link before re-configuring */
    if( cim_disconnect_link(fd, link_if) != 0 ) {
        printf("\nERROR: Failed to disconnect Link %d\n", link_if);
        return -2;
    }

    /* setup 0xC9 cmd */
    pcmd.link_if  = link_if;
    pcmd.cmd      = PCMD_CIM_UPDATE_DEV_TYPE;
    pcmd.byte_cnt = rsplen;
    pcmd.valid    = 1;
    pcmd.in_ram   = 1;
    pcmd.chksum   = 0;
    pcmd.dest     = FPD_CIM_BUFFER;

    if( ioctl(fd, FPD_WRITE_ENTRY_CMD_TABLE, &pcmd) != 0 ) {
        printf("\nERROR: FPD_WRITE_ENTRY_CMD_TABLE ioctl failed (%s)\n", strerror(errno));
        return -3;
    }

    /* retrieve prior line connection */
    if( cim_get_line_connection_info(link_if, &pconn) == 0 ) {
        /* re-connect link */
        pconn = &old_conn;
        if( cim_switch_link(fd, link_if, pconn->target_port, pconn->protocol, pconn->tx_parity, pconn->rx_parity) != 0 ) {
            printf("\nERROR: Failed to re-connect Link %d\n", link_if);
            return -4;
        }
    }

    return 0;
}

static int send_cim_fw_update_pkt( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    /* compose CMD 0x35 ID 0x0b */
    pdata->link_if = link_if;
    pdata->requested_len = 5;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x35;
    pdata->buf[1] = 0x0b;
    pdata->buf[2] = 0;
    pdata->buf[3] = 0;
    pdata->buf[4] = compute_paragon_chksum(&pdata->buf[0], 4);

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int initiate_cim_fw_update( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0xFA;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int abort_cim_fw_update( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0xFC;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int initiate_cim_serial_num_programming( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0xFB;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int program_cim_serial_num( int fd, int link_if, unsigned char *sernum )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 17;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    memcpy(pdata->buf, sernum, SERIAL_NUM_LENGTH);
    pdata->buf[SERIAL_NUM_LENGTH] = compute_chksum(&pdata->buf[0], SERIAL_NUM_LENGTH);

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

int resend_cim_fw_info( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 1;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0xFE;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

/*

@This function will take str=string, provided
@with delims(string to break upon), and produces
@a pointers array(tokens_ptr) that holds pointers
@to tokens itself..

*/
static int parse( char *str, char *delim, char ***tokens_ptr )
{
    char *str_cp;
    char *snew;
    int num_tokens = 0;
    int i;

    // strspn() function calculates the length of initial segment
    // of str which consists entirely of chracters in delim.
    // in easy words, it looks into string and see if it starts
    // with delims provided .. and provides the number of count
    // of matching delims at start. strcspn do opposite .. it looks
    // for strings that don't start with delims.
    // so in this way we 'r basically skipping spaces coming infront
    // of our command line.

    snew = str + strspn (str, delim);

    // snew will be freed of all leading delimiters now.
    // create some space for snew to reside in t.
    if ((str_cp = (char *) calloc (strlen (snew) + 1, sizeof (char))) == NULL) {
        *tokens_ptr = NULL;
        num_tokens = -1;
        return num_tokens;
    }

    strcpy (str_cp, snew);         // finally copy snew to t.
    
    if (strtok (str_cp, delim) == NULL) {
        // no delim in the string
        free(str_cp);
        num_tokens = 0;
        return num_tokens;
    }

    // strtok(NULL,delim) is useful for subsequent strtok calls on t.
    for (num_tokens = 1; strtok (NULL, delim) != NULL; num_tokens++);

    // create an argument array to contain pointers to tokens.
    if ((*tokens_ptr = (char **) calloc (num_tokens + 1, sizeof (char *))) == NULL) {
        free (str_cp);
        num_tokens = -1;
        return num_tokens;
    }

    // we have already found 1 token atleast remember ?
    // else we 'd not here in this block :)
    strcpy (str_cp, snew);

    // get the first token.
    **tokens_ptr = strtok (str_cp, delim);
    
    for (i = 1; i < num_tokens + 1; i++) {
        *((*tokens_ptr) + i) = strtok (NULL, delim);
    }

    return (num_tokens);
}

/* This function will parse the intel hex file and parse
   data + header and build hex record arrays out of it.
*/
static unsigned char *IntelToHexArray( char *input )
{
    unsigned char *hexArr;
    int len = 0;
    unsigned char c;
    unsigned char d;
    unsigned int val1 = 0;
    unsigned int val2 = 0;
    unsigned int result = 0;
    int i = 0;

    len = strlen (input);
    len = len / 2;
    hexArr = (unsigned char *) malloc(len * sizeof(unsigned char));

    if( hexArr ) {
        while (*input != '\n') {
            c = *input;
            d = *++input;

            if (c >= '0' && c <= '9')
                val1 = (val1 * 16) + c - '0';
            else if (c >= 'A' && c <= 'F')
                val1 = (val1 * 16) + c - 'A' + 10;

            if (d >= '0' && d <= '9')
                val2 = (val2 * 16) + d - '0';
            else if (d >= 'A' && d <= 'F')
                val2 = (val2 * 16) + d - 'A' + 10;

            result = val1 * 16 + val2;
            hexArr[i] = (unsigned char) result;
            i++;
            *++input;
            val1 = 0;
            val2 = 0;
        }
    }

    return (hexArr);
}

static int get_image_version( char *hexfile, cim_image_info_t *info )
{
    FILE *hex;
    char** p;
    int i=0;
    int tokens=0;
    char input[MAX_BUFFER];

    if((hex = fopen(hexfile,"r+b")) == NULL) {
        fprintf(stderr,"%s: %s\n", hexfile, strerror(errno));
        return -1;
    }

    for(;;) {
        if (fgets (input, MAX_BUFFER, hex) == NULL)
            break;
 
        tokens = parse (input, BLANK, &p);

        if (tokens > 3)	{
            if((i = strcmp (p[1], "FirmwareVersion")) == 0) {
                if( strncmp(p[3], "??", 2) != 0 ) {
                    strncpy(info->fwver, p[3], FW_VERSION_LENGTH);
                }
                else if(( tokens >= 5 ) && ( strncmp(p[4], "??", 2) != 0 )) {
                    strncpy(info->fwver, p[4], FW_VERSION_LENGTH);
                }
            }
            else if((i = strncmp (p[1], "HardwareVersion", 15)) == 0) {
                if( strncmp(p[3], "??", 2) != 0 ) {
                    strncpy(info->hwver, p[3], HW_VERSION_LENGTH);
                }
                else if(( tokens >= 5 ) && ( strncmp(p[4], "??", 2) != 0 )) {
                    strncpy(info->hwver, p[4], HW_VERSION_LENGTH);
                }
            }
            else if((i = strcmp (p[1], "DeviceType")) == 0) {
                strncpy(info->devtype, p[3], DEVICE_TYPE_LENGTH);
            }
        }

        if( tokens ) {
            free(*p);
            free(p);
        }

        if (input[0] == '\n')
            continue;
    }

    fclose(hex);

    return 0;
}

static int make_image_map( char *hexfile, int debug )
{
    char input[MAX_BUFFER];
    unsigned char *record;
    FILE *hex;
    FILE *fpdbg = NULL;
    int i=0;
    int j=0;
    int tokens=0;
    char **p;
    unsigned char region=0x00;
    int data_length=0;
    unsigned int last_addr = 0;

    if((hex = fopen(hexfile,"r+b")) == NULL) {
        fprintf(stderr,"%s: %s\n", hexfile, strerror(errno));
        return -1;
    }
 
    if( debug ) {
        if( (fpdbg = fopen("process.txt","w")) == NULL ) {
            perror("process.txt fopen:");
            return -2;
        }
    }

    /* Initialize everything with 0xFF */
    for( i = 0; i < NUM_REGIONS; i++) {
        for( j = 0; j < MEM_REC_BUFFER_SIZE; j++) {
            m_Map[i].m_Memory[j] = 0;
            m_Map[i].end_addr    = 0;
        }
    }

    for(;;) {
        if (fgets(input, MAX_BUFFER, hex) == NULL)
            break;

        if (input[0] == '\n') 
            continue;

        if(input[strlen(input)-2] == 13) {
            input[strlen(input)-2]='\n';
        }
        tokens = parse(input,":", &p);

        if( input[0] == ':' ) {
            record = IntelToHexArray(p[0]);
            if( !record ) {
                /* run out of memory */
                free(*p);
                free(p);
                return -3;
            }
        }
        else if ( tokens > 1 ) {
            /* For screwed up hex file that mixed lines together, i.e.,
             * { Display = 3 "Number of Channels" }:03000000213000AC
             */
            record = IntelToHexArray(p[1]);
            if( !record ) {
                /* run out of memory */
                free(*p);
                free(p);
                return -3;
            }
        }
        else {
            record = NULL;
        }

        if( tokens ) {
            free(*p);
            free(p);
        }

        if( record ) {
            /* Upper memory records */
            if (record[3] == 0x02) {
                region = (unsigned char)(record[4]>>4);
                if( debug ) {
                    fprintf(fpdbg, "Upper Memory Records region %d\n", region);
                }
            }
       
            /* Get data records */
            if (record[3] == 0x00) {
                unsigned int address;
                data_length = record[0];
                address = (((unsigned char)record[1]) << 8) + record[2];

                if( debug ) {
                    fprintf(fpdbg, "addr: %02x, len: %d, data: ", address, data_length);
                }

                for( i=4; i < data_length+4; i++) {
                    m_Map[region].m_Memory[address] = record[i];
                    if( debug ) {
                        fprintf(fpdbg, "%02x ", record[i]);
                    }
                    address++;
                }

                if( debug ) {
                    fprintf(fpdbg, "\n");
                }

                if( last_addr < (address - 1) ) {
                    last_addr = address - 1;
                    m_Map[region].end_addr = last_addr;
                }
            }

            free(record);
        }
    }

    if( debug ) {
        fprintf(fpdbg, "\nLast Address: %x\n", m_Map[region].end_addr);
        fclose(fpdbg);
    }

    fclose(hex);

    return 0;
}

static void print_image_map( void )
{
    int i=0;
    unsigned char region;
    unsigned short checksum = 0;
    int numblk;
    int blksize = 128;
    int last_addr_sent;
    FILE *fp;

    if( (fp = fopen("imagemap.txt","w")) == NULL ) {
        perror("imagemap.txt fopen:");
        return;
    }

    for( region = 0; region < NUM_REGIONS; region++) {
        /* calculate number of blocks required */
        numblk = m_Map[region].end_addr/blksize;
        if( m_Map[region].end_addr % blksize ) {
            ++numblk;
        }
        last_addr_sent = numblk * blksize;

        fprintf(fp,"For region %d",region);
        for( i = 0; i < MEM_REC_BUFFER_SIZE; i++) {
            if( !(i % 16) ) {
                fprintf(fp,"\n[%.4x]", i);
            }
            fprintf(fp,"%.2x",m_Map[region].m_Memory[i]);
            if( i < last_addr_sent ) {
                checksum += m_Map[region].m_Memory[i];
            }
        }
        fprintf(fp,"\n\n");
        fprintf(fp, "Number of Blocks : %d\n", numblk);
        fprintf(fp, "Last Address     : 0x%x\n", m_Map[region].end_addr);
    }
    fprintf(fp, "File Checksum    : 0x%04x\n", checksum);

    fclose(fp);

    return;
}

static int send_upper_address_record( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 7;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x02;
    pdata->buf[1] = 0x00;
    pdata->buf[2] = 0x00;
    pdata->buf[3] = 0x04;
    pdata->buf[4] = 0x00;
    pdata->buf[5] = 0xFF;
    pdata->buf[6] = compute_paragon_chksum(&pdata->buf[0], pdata->requested_len-1);

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int send_end_of_block_record( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 5;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x00;
    pdata->buf[1] = 0x00;
    pdata->buf[2] = 0x00;
    pdata->buf[3] = 0x01;
    pdata->buf[4] = 0xFF;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int send_end_of_file_record( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 5;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x00;
    pdata->buf[1] = 0x00;
    pdata->buf[2] = 0x00;
    pdata->buf[3] = 0x05;
    pdata->buf[4] = 0xFB;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int send_info_request_record( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 5;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x00;
    pdata->buf[1] = 0x00;
    pdata->buf[2] = 0x00;
    pdata->buf[3] = 0x02;
    pdata->buf[4] = 0xFE;

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int send_checksum_record( int fd, int link_if, unsigned short chksum )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;

    pdata->link_if = link_if;
    pdata->requested_len = 5;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = 0x00;
    pdata->buf[1] = (chksum & 0xFF00) >> 8;
    pdata->buf[2] = (chksum & 0x00FF);
    pdata->buf[3] = 0x06;
    pdata->buf[4] = compute_paragon_chksum(&pdata->buf[0], pdata->requested_len-1);

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int send_data_record( int fd, int link_if, unsigned char region, unsigned char size, unsigned short addr, unsigned short *chksum )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;
    int i;

    *chksum = 0;
    pdata->link_if = link_if;
    pdata->requested_len = 4 + size + 1;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;
    pdata->buf[0] = size;
    pdata->buf[1] = (addr & 0xff00) >> 8;
    pdata->buf[2] = (addr & 0x00ff);
    pdata->buf[3] = 0x00;
    for( i = 0; i < size; i++ ) {
        pdata->buf[i+4] = m_Map[region].m_Memory[addr+i];
        *chksum += m_Map[region].m_Memory[addr+i];
    }
    pdata->buf[4+size] = compute_paragon_chksum(&pdata->buf[0], 4+size);

    return( paragon_cim_send_pkt(fd, link_if, pdata) );
}

static int wait_for_device_type( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    unsigned char chksum;
    int correct_rsp = 0;
    int size;
    int i;

    /* wait for device type packet (0xC9) */
    while( !correct_rsp ) {
        if( cim_poll_once(fd, CIM_UPDATE_POLL_TIMEOUT, pevt) != 0 ) {
            return -1;
        }

        memset(&process, 0, sizeof(rx_notify_t));
        process.link_if = link_if;
        handle_events(fd, pevt, &process);

        if( process.rx_cim ) {
            do {
                /* retrieve incoming data */
                pdata->link_if = link_if;
                pdata->requested_len = 256;
                pdata->type = FPD_CIM_BUFFER;
                pdata->buf = protocol_buffer;
                paragon_cim_receive_pkt(fd, link_if, pdata);

                if( pdata->actual_len > 0 ) {
                    if( pdata->buf[0] == PCMD_CIM_UPDATE_DEV_TYPE ) {
                        printf("RxPKT [%d]: ", pdata->actual_len);
                        dump_data(pdata->buf, pdata->actual_len);

                        correct_rsp = 1;

                        memset(sernum, 0, SERIAL_NUM_LENGTH);
                        memset(devtype, 0, DEVICE_TYPE_LENGTH+1);
                        memset(hwver, 0, HW_VERSION_LENGTH+1);
                        memset(fwver, 0, FW_VERSION_LENGTH+1);
                        memset(auxval, 0, AUX_VAL_LENGTH+1);

                        /* parse the packet */
                        for( i = 0; i < SERIAL_NUM_LENGTH; i++ ) {
                            sernum[i] = pdata->buf[SERIAL_NUM_OFFSET+i];
                        }

                        numAV = pdata->buf[NUM_AUX_VAL_OFFSET];

                        for( i = 0; i < DEVICE_TYPE_LENGTH; i++ ) {
                            devtype[i] = pdata->buf[DEVICE_TYPE_OFFSET+i];
                        }

                        for( i = 0; i < HW_VERSION_LENGTH; i++ ) {
                            hwver[i] = pdata->buf[HW_VERSION_OFFSET+i];
                        }

                        for( i = 0; i < FW_VERSION_LENGTH; i++ ) {
                            fwver[i] = pdata->buf[FW_VERSION_OFFSET+i];
                        }

                        if( numAV > 0 ) {
                            for( i = 0; i < FW_VERSION_LENGTH; i++ ) {
                                auxval[i] = pdata->buf[AUX_VAL_OFFSET+i];
                            }
                        }

                        /* verify checksum */
                        chksum = pdata->buf[AUX_VAL_OFFSET + (3*numAV)];
                        size = (1 + SERIAL_NUM_LENGTH + 1 +
                                DEVICE_TYPE_LENGTH + HW_VERSION_LENGTH + 
                                FW_VERSION_LENGTH + (numAV * 3));
                        if( compute_chksum(&pdata->buf[0], size) != chksum ) {
                            printf("\n\tWARNING: Chksum is wrong!\n\n");
                        }

                        printf("Serial Number     : %s\n", sernum);
                        printf("Device Type       : %s\n", devtype);
                        printf("Hardware Version  : %s\n", hwver);
                        printf("Firmware Version  : %s\n", fwver);
                        printf("Auxilliary Value #: %d\n", numAV);
                        if( numAV > 0 ) {
                            printf("Auxilliary Version: %s\n", auxval);
                        }
                        printf("Checksum          : %02x\n", chksum);
                    }
                }
            } while( pdata->actual_len > 0 );
        }
        else if( process.rx_priority ) {
            fprintf(stderr, "\n\tERROR: Expecting pkt from CIM buffer but rcvd from Priority\n\n");
        }
    }

    return 0;
}

static unsigned char wait_for_response( int fd, int link_if )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    rx_notify_t process;
    unsigned char rsp = 0;

    if( cim_poll_once(fd, CIM_UPDATE_POLL_TIMEOUT, pevt) != 0 ) {
        return 0;
    }

    memset(&process, 0, sizeof(rx_notify_t));
    process.link_if = link_if;
    handle_events(fd, pevt, &process);

    if( process.rx_cim ) {
        /* retrieve incoming data */
        pdata->link_if = link_if;
        pdata->requested_len = 256;
        pdata->type = FPD_CIM_BUFFER;
        pdata->buf = protocol_buffer;
        paragon_cim_receive_pkt(fd, link_if, pdata);

        if( pdata->actual_len > 0 ) {
            /*
            printf("RxPKT: ");
            dump_data(pdata->buf, pdata->actual_len);
            */
            rsp = pdata->buf[0];
        }
    }
    else if( process.rx_priority ) {
        fprintf(stderr, "ERROR: Expecting pkt from CIM buffer but rcvd from Priority\n");
    }

    return rsp;
}

static int send_cim_image( int fd, int link_if, unsigned char cim_type )
{
    unsigned short checksum, sum;
    int region, block, rec;
    int blksize = 128;
    int numblk;
    int n_tries, n_blk_tries;
    int process_next_blk;
    unsigned char cim_rsp = 0;

    printf("CIM Type: %02x\n", cim_type);

    for( region = 0; region < NUM_REGIONS; region++) {
        /* calculate number of blocks required */
        numblk = m_Map[region].end_addr/blksize;
        if( m_Map[region].end_addr % blksize ) {
            ++numblk;
        }

        /* According to David Hsieh, P2CIM-PS2DUAL requires to have an even
         * number of blocks for CIM Update to work correctly.
         */
        if( cim_type == CIM_TYPE_P2CIM_PS2DUAL ) {
            if( numblk % 2 ) {
                ++numblk;
            }
        }
        printf("Number of Blocks = %d\n", numblk);

        /* send data records in 128 byte blocks */
        checksum = 0;
        for( block = 0; block < numblk; block++ ) {
            /* send upper address record */
            n_tries = 0;
            do {
                n_tries++;
                printf("Sending Upper Address Record....");
                if( send_upper_address_record(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    if( n_tries < MAX_RESEND ) {
                        continue;
                    }
                    else {
                        return -1;
                    }
                }
                printf("passed.\n");

                cim_rsp = wait_for_response(fd, link_if);
            } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));

            process_next_blk = 0;
            n_blk_tries = 0;

            do {
                ++n_blk_tries;

                /* send 16 bytes at a time */
                for( rec = 0; rec < 8; rec++ ) {
                    n_tries = 0;
                    do {
                        n_tries++;
                        if( n_tries > 1 ) 
                            printf("%d: RESENDING data record %d\n", n_tries, (block * 128) + (rec * 16));
                        if( send_data_record(fd, link_if, region, 16, ((block * 128) + (rec * 16)), &sum) != 0 ) {
                            if( n_tries < MAX_RESEND ) {
                                printf("RESENDING data record %d\n", (block * 128) + (rec * 16));
                                continue;
                            }
                            else {
                                return -2;
                            }
                        }

                        cim_rsp = wait_for_response(fd, link_if);
                    } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));

                    checksum += sum;
                }

                /* send end-of-block record */
                n_tries = 0;
                do {
                    n_tries++;
                    printf("Sending EOB %03d Record..........", block+1);
                    if( send_end_of_block_record(fd, link_if) != 0 ) {
                        printf("FAILED.\n");
                        if( n_tries < MAX_RESEND ) {
                            continue;
                        }
                        else {
                            return -3;
                        }
                    }
                    printf("passed.\n");

                    cim_rsp = wait_for_response(fd, link_if);
                } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));

                /* wait for device error code */
                cim_rsp = wait_for_response(fd, link_if);

                switch( cim_rsp ) {
                    case PCMD_CIM_UPDATE_CONT_SEND:
                        process_next_blk = 1;
                        break;

                    case PCMD_CIM_UPDATE_ERASE_ERR:
                        printf("Erase Error: The flash memory could not be erased properly\n");
                        fflush(stdout);
                        if( n_blk_tries >= MAX_RESEND ) {
                            return -4;
                        }
                        break;

                    case PCMD_CIM_UPDATE_PROG_ERR:
                        printf("Programming Error: The flash memory did not program properly\n");
                        fflush(stdout);
                        if( n_blk_tries >= MAX_RESEND ) {
                            return -5;
                        }
                        break;

                    case PCMD_CIM_UPDATE_BLKCHK_ERR:
                        printf("Block Checking Error: The flash memory was programmed but it failed when checked\n");
                        fflush(stdout);
                        if( n_blk_tries >= MAX_RESEND ) {
                            return -6;
                        }
                        break;

                    default:
                        printf("Unknown Error: Rsp = %02x\n", cim_rsp);
                        fflush(stdout);
                        if( n_blk_tries >= MAX_RESEND ) {
                            return -7;
                        }
                        break;
                }
            } while( !process_next_blk );
        }
    }

    switch( cim_type ) {
        case CIM_TYPE_P2CIM_PS2:
        case CIM_TYPE_P2CIM_APS2:
        case CIM_TYPE_P2CIM_PS2DUAL:
        case CIM_TYPE_P2CIM_USB:
        case CIM_TYPE_P2CIM_SUSB:
        case CIM_TYPE_P2CIM_SUN:
        case CIM_TYPE_DCIM_PS2:
        case CIM_TYPE_DCIM_USB:
        case CIM_TYPE_DCIM_SUSB:
        case CIM_TYPE_DCIM_SUN:
            /* these types of CIM does not wait for EOF record to reboot */
            n_tries = 0;
            do {
                n_tries++;

                /* for P2CIM and DCIM, send the enhanced MCUpdate Protocol ID=0x06 */
                printf("Sending Checksum %04x Record....", checksum);
                if( send_checksum_record(fd, link_if, checksum) != 0 ) {
                    printf("FAILED.\n");
                    if( n_tries < MAX_RESEND ) {
                        continue;
                    }
                    else {
                        return -7;
                    }
                }
                printf("passed.\n");

                cim_rsp = wait_for_response(fd, link_if);
            } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));
            break;

        case CIM_TYPE_P2CIM_USBG2:
        case CIM_TYPE_P2CIM_AUSB:
        default:
            /* for P2CIM and DCIM, send the enhanced MCUpdate Protocol ID=0x06 */
            printf("Sending Checksum %04x Record....", checksum);
            if( send_checksum_record(fd, link_if, checksum) != 0 ) {
                printf("FAILED.\n");
                return -7;
            }
            printf("passed.\n");

            cim_rsp = wait_for_response(fd, link_if);

            if( cim_rsp == PCMD_CIM_UPDATE_ACK ) {
                /* send end-of-file record */
                n_tries = 0;
                do {
                    n_tries++;
                    printf("Sending EOF Record..............");
                    if( send_end_of_file_record(fd, link_if) != 0 ) {
                        printf("FAILED.\n");
                        if( n_tries < MAX_RESEND ) {
                            continue;
                        }
                        else {
                            return -8;
                        }
                    }
                    printf("passed.\n");

                    cim_rsp = wait_for_response(fd, link_if);
                } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));
            }
            else {
                printf("Please REDO CIM Update\n");
                return -9;
            }
            break;

        case CIM_TYPE_P2CIM_PWR:
            /* send end-of-file record */
            n_tries = 0;
            do {
                n_tries++;
                printf("Sending EOF Record..............");
                if( send_end_of_file_record(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    if( n_tries < MAX_RESEND ) {
                        continue;
                    }
                    else {
                        return -8;
                    }
                }
                printf("passed.\n");

                cim_rsp = wait_for_response(fd, link_if);
            } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));
            break;
    }

    return 0;
}

static int ask_device_type( unsigned char *ptype, int *psize )
{
    unsigned char cim_type;
    int size;
    int type;
    int i = 1;

    printf("\n\tTypes of CIM\n");
    printf("\t============\n");
    printf("\t %d. DCIM-PS2\n", i++);
    printf("\t %d. DCIM-USB\n", i++);
    printf("\t %d. DCIM-SUSB\n", i++);
    printf("\t %d. DCIM-SUN\n", i++);
    printf("\t %d. P2CIM-PS2\n", i++);
    printf("\t %d. P2CIM-APS2\n", i++);
    printf("\t %d. P2CIM-PS2DUAL\n", i++);
    printf("\t %d. P2CIM-USB\n", i++);
    printf("\t %d. P2CIM-USBG2\n", i++);
    printf("\t%d. P2CIM-AUSB\n", i++);
    printf("\t%d. P2CIM-SUSB\n", i++);
    printf("\t%d. P2CIM-SUN\n", i++);
    printf("\t%d. P2CIM-PWR\n", i++);
    printf("\t%d. UKVMPD\n", i++);
    printf("\t%d. USKVMPD\n", i++);
    printf("\t%d. UUSBPD\n", i++);
    printf("\t%d. AUATC\n", i++);
    printf("\nPlease Input the correct CIM Type: ");
    scanf("%d", &type);

    switch( type ) {
        case 1:
            cim_type = CIM_TYPE_DCIM_PS2;
            size = DCIM_PS2_ID_PKT_SIZE;
            break;
        case 2:
            cim_type = CIM_TYPE_DCIM_USB;
            size = DCIM_USB_ID_PKT_SIZE;
            break;
        case 3:
            cim_type = CIM_TYPE_DCIM_SUSB;
            size = DCIM_SUSB_ID_PKT_SIZE;
            break;
        case 4:
            cim_type = CIM_TYPE_DCIM_SUN;
            size = DCIM_SUN_ID_PKT_SIZE;
            break;
        case 5:
            cim_type = CIM_TYPE_P2CIM_PS2;
            size = P2CIM_PS2_ID_PKT_SIZE;
            break;
        case 6:
            cim_type = CIM_TYPE_P2CIM_APS2;
            size = P2CIM_APS2_ID_PKT_SIZE;
            break;
        case 7:
            cim_type = CIM_TYPE_P2CIM_PS2DUAL;
            size = P2CIM_PS2DUAL_ID_PKT_SIZE;
            break;
        case 8:
            cim_type = CIM_TYPE_P2CIM_USB;
            size = P2CIM_USB_ID_PKT_SIZE;
            break;
        case 9:
            cim_type = CIM_TYPE_P2CIM_USBG2;
            size = P2CIM_USBG2_ID_PKT_SIZE;
            break;
        case 10:
            cim_type = CIM_TYPE_P2CIM_AUSB;
            size = P2CIM_AUSB_ID_PKT_SIZE;
            break;
        case 11:
            cim_type = CIM_TYPE_P2CIM_SUSB;
            size = P2CIM_SUSB_ID_PKT_SIZE;
            break;
        case 12:
            cim_type = CIM_TYPE_P2CIM_SUN;
            size = P2CIM_SUN_ID_PKT_SIZE;
            break;
        case 13:
            cim_type = CIM_TYPE_P2CIM_PWR;
            size = P2CIM_PWR_ID_PKT_SIZE;
            break;
        case 14:
            cim_type = CIM_TYPE_UKVMPD;
            size = 0;
            break;
        case 15:
            cim_type = CIM_TYPE_USKVMPD;
            size = 0;
            break;
        case 16:
            cim_type = CIM_TYPE_UUSBPD;
            size = 0;
            break;
        case 17:
            cim_type = CIM_TYPE_AUATC;
            size = 0;
            break;
        default:
            return -1;
    }

    if( psize ) {
        *psize = size;
    }
    if( ptype ) {
        *ptype = cim_type;
    }

    return 0;
}

static int detect_device_type( int fd, int link_if, unsigned char *pcim_type )
{
    unsigned char devtype[DEVICE_TYPE_LENGTH+1];
    unsigned char dev_id;
    unsigned char cim_type;
    int size = 0;

    /* check first if this is a Power CIM since this type of CIM did 
     * not implement the Read EEPROM Paragon CMD
     */
    if( paragon_cim_send_echo(fd, link_if, &dev_id) == 0 ) {
        /* Power CIM */
        if( dev_id == DEVICE_ID_POWER_CIM_OLD ) {
            cim_type = CIM_TYPE_P2CIM_PWR;
            if( pcim_type ) {
                *pcim_type = cim_type;
            }
            size = P2CIM_PWR_ID_PKT_SIZE;
            return( setup_expected_init_rsp(fd, link_if, size) );
        }
    }

    /* not a Power CIM */
    memset(devtype, 0, DEVICE_TYPE_LENGTH+1);
    if( paragon_cim_get_device_type(fd, link_if, devtype) == 0 ) {
        /* setup expected length of device type */

        /* P2CIM-USBG2, P2CIM-AUSB */
        if( strncmp(devtype, P2CIM_AUSB, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_P2CIM_AUSB;
            size = P2CIM_AUSB_ID_PKT_SIZE;
        }
        /* P2CIM-PS2, P2CIM-APS2, P2CIM-USB */
        else if( strncmp(devtype, P2CIM_PS2, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_P2CIM_PS2;
            size = P2CIM_PS2_ID_PKT_SIZE;
        }
        /* P2CIM-SUSB */
        else if( strncmp(devtype, P2CIM_SUSB, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_P2CIM_SUSB;
            size = P2CIM_SUSB_ID_PKT_SIZE;
        }
        /* P2CIM-SUN */
        else if( strncmp(devtype, P2CIM_SUN, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_P2CIM_SUN;
            size = P2CIM_SUN_ID_PKT_SIZE;
        }
        /* P2CIM-PS2DUAL */
        else if( strncmp(devtype, P2CIM_PS2DUAL, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_P2CIM_PS2DUAL;
            size = P2CIM_PS2DUAL_ID_PKT_SIZE;
        }
        /* DCIM-PS2, DCIM-USB */
        else if( strncmp(devtype, DCIM_USB, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_DCIM_USB;
            size = DCIM_USB_ID_PKT_SIZE;
        }
        /* DCIM-SUSB */
        else if( strncmp(devtype, DCIM_SUSB, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_DCIM_SUSB;
            size = DCIM_SUSB_ID_PKT_SIZE;
        }
        /* DCIM-SUN */
        else if( strncmp(devtype, DCIM_SUN, DEVICE_TYPE_LENGTH) == 0 ) {
            cim_type = CIM_TYPE_DCIM_SUN;
            size = DCIM_SUN_ID_PKT_SIZE;
        }
        else {
            /* Manual Selection - Device type is not programmed in EEPROM */
            printf("\n\tWARNING: Unable to determine CIM Type\n");
            if( ask_device_type(&cim_type, &size) == 0 ) {
                if( size == 0 ) {
                    printf("\n\tERROR: CIM Type does not support FW update!\n\n");
                    return -1;
                }
            }
            else {
                return -2;
            }
        }

        if( pcim_type ) {
            *pcim_type = cim_type;
        }
    }

    return( setup_expected_init_rsp(fd, link_if, size) );
}

static int get_action_args( cim_update_init_action_t *action )
{
    printf("\nCIM Update Initiation Response\n");
    printf("\t1. Download Firmware\n");
    printf("\t2. Program Serial Number\n");
    printf("\t3. Abort\n");
    printf("\t4. Check File Image\n");
    printf("\nEnter Action to take       : ");
    scanf("%d", (int *)action);

    return 0;
}

static int get_file_args( char *name )
{
    printf("\nEnter CIM Firmware Hex File: ");
    scanf("%s", name);
    return 0;
}

static int get_serial_number_args( char *name )
{
    printf("\nEnter Serial Number        : ");
    scanf("%s", name);
    return 0;
}

int cim_update( int fd, int link_if, int state )
{
    cim_update_init_action_t action;
    unsigned char cim_rsp = 0;
    unsigned char filename[128];
    unsigned char new_sernum[SERIAL_NUM_LENGTH];
    unsigned char cim_type = 0;
    cim_image_info_t image;
    unsigned long cim_hw, image_hw;
    unsigned long cim_fw, image_fw;
    int chg_option;
    int n_tries;
    int size;

    get_action_args(&action);

    switch( action ) {
        case DOWNLOAD:
            memset(filename, 0, 128);
            get_file_args(filename);

            memset(&image, 0, sizeof(cim_image_info_t));
            get_image_version(filename, &image);
            printf("Image Device Type          : %s\n", image.devtype);
            printf("Image Hardware Version     : %s\n", image.hwver);
            printf("Image Firmware Version     : %s\n", image.fwver);

            printf("Creating image map....");
            if( make_image_map(filename, 0) == 0 ) {
                printf("passed.\n");
                print_image_map();
            }
            else {
                printf("FAILED.\n");
                return -1;
            }

            if( state == CIM_UPDATE_RECOVERY_STATE ) {
                if( ask_device_type(&cim_type, &size) == 0 ) {
                    if( size == 0 ) {
                        printf("\n\tERROR: CIM Type does not support FW update!\n\n");
                        return -3;
                    }
                }
                else {
                    return -4;
                }

                if( setup_expected_init_rsp(fd, link_if, size) != 0 ) {
                    return -5;
                }
            }
            break;
        case PROGRAM_SERIAL_NUM:
            if( state == CIM_UPDATE_RECOVERY_STATE ) {
                printf("\n\tERROR: This action is not allowed at this state.\n\n");
                return -2;
            }
            memset(new_sernum, 0, SERIAL_NUM_LENGTH);
            get_serial_number_args(new_sernum);
            break;
        case ABORT:
            if( state == CIM_UPDATE_RECOVERY_STATE ) {
                if( ask_device_type(&cim_type, &size) == 0 ) {
                    if( size == 0 ) {
                        printf("\n\tERROR: CIM Type does not support FW update!\n\n");
                        return -3;
                    }
                }
                else {
                    return -4;
                }

                if( setup_expected_init_rsp(fd, link_if, size) != 0 ) {
                    return -5;
                }
            }
            break;
        case CHECK_IMAGE:
            memset(filename, 0, 128);
            get_file_args(filename);

            memset(&image, 0, sizeof(cim_image_info_t));
            get_image_version(filename, &image);
            printf("Image Device Type          : %s\n", image.devtype);
            printf("Image Hardware Version     : %s\n", image.hwver);
            printf("Image Firmware Version     : %s\n", image.fwver);

            printf("Creating image map....");
            if( make_image_map(filename, 1) == 0 ) {
                printf("passed.\n");
                print_image_map();
                return 0;
            }
            else {
                printf("FAILED.\n");
                return -1;
            }
            break;
        case RESEND_INFO:
             /*
              * Never do this because we have no idea what to expect. Instead,
              * just get out and re-run the CIM update
              */
        default:
            printf("\n\tERROR: Invalid action selected.\n");
            return -2;
    }

    if( state == CIM_UPDATE_INITIAL_STATE ) {
        /* determine device type */
        if( detect_device_type(fd, link_if, &cim_type ) != 0 ) {
            return -1;
        }

        chg_option = FPD_CHANGE_MODE_AFTER_PKT_SENT;
    }
    else {
        chg_option = FPD_CHANGE_MODE_IMMEDIATELY;
    }

    /* place FPGA into CIM Update mode */
    printf("\nPlacing FPGA into CIM Update mode....");
    if( enter_cim_fw_update(fd, link_if, chg_option) != 0 ) {
        printf("FAILED.\n");
        return -3;
    }
    printf("passed.\n");

    if( state == CIM_UPDATE_INITIAL_STATE ) {
        printf("Sending CIM FW UPDATE packet to device....");
        if( send_cim_fw_update_pkt(fd, link_if) != 0 ) {
            printf("FAILED.\n");
            return -4;
        }
        printf("passed.\n");

        /* wait for device type packet (0xC9) */
        if( wait_for_device_type(fd, link_if) != 0 ) {
            return -5;
        }

        if( action == DOWNLOAD ) {
            unsigned char reply;
            int got_answer;

            /* check device type */
            if( strncmp(devtype, image.devtype, DEVICE_TYPE_LENGTH) == 0 ) {
                /* check versions */
                cim_fw = strtoul(fwver, NULL, 16);
                image_fw = strtoul(image.fwver, NULL, 16);
                if( cim_fw > image_fw ) {
                    printf("\n\tWARNING: Image is older than CIM FW.\n\n");
                    got_answer = 0;
                    do {
                        printf("\rDo you want to continue (y/n): ");
                        scanf("%c", &reply);

                        switch( reply ) {
                            case 'y':
                            case 'Y':
                                got_answer = 1;
                                break;
                            case 'n':
                            case 'N':
                                action = ABORT;
                                got_answer = 1;
                                break;
                        }
                    } while( !got_answer );

                    if( action == DOWNLOAD ) {
                        cim_hw = strtoul(hwver, NULL, 16);
                        image_hw = strtoul(image.hwver, NULL, 16);

                        if( cim_hw > image_hw ) {
                            printf("\n\tWARNING: Image is used for an older CIM HW.\n\n");
                            got_answer = 0;
                            do {
                                printf("\rDo you want to continue (y/n): ");
                                scanf("%c", &reply);

                                switch( reply ) {
                                    case 'y':
                                    case 'Y':
                                        got_answer = 1;
                                        break;
                                    case 'n':
                                    case 'N':
                                        action = ABORT;
                                        got_answer = 1;
                                        break;
                                }
                            } while( !got_answer );
                        }
                    }
                }
            }
            else {
                /* abort */
                printf("\n\tERROR: CIM Device Type is not the same as Image!\n\n");
                //action = ABORT;
            }

            /* one last check before upgrading */
            if( action == DOWNLOAD ) {
                int got_answer = 0;
                printf("\n");
                do {
                    printf("\rDo you really want to continue (y/n): ");
                    scanf("%c", &reply);

                    switch( reply ) {
                        case 'y':
                        case 'Y':
                            got_answer = 1;
                            break;
                        case 'n':
                        case 'N':
                            action = ABORT;
                            got_answer = 1;
                            break;
                    }
                } while( !got_answer );
            }
        }
    }

    switch( action ) {
        case DOWNLOAD:
            if( state == CIM_UPDATE_INITIAL_STATE ) {
                /* somehow CIM expects to have some delay to work correctly */
                sleep(1);

                /* initiate download process */
                printf("Initiating Download process...");
                if( initiate_cim_fw_update(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    printf("Aborting CIM FW Update....");
                    if( abort_cim_fw_update(fd, link_if) != 0 ) {
                        printf("FAILED.\n");
                        return -9;
                    }
                    printf("passed.\n");

                    /* take FPGA out of CIM Update mode */
                    printf("Taking FPGA out of CIM Update mode....");
                    if( exit_cim_fw_update(fd, link_if) != 0 ) {
                        printf("FAILED.\n");
                        return -10;
                    }
                    printf("passed.\n");
                    return -6;
                }
                printf("passed.\n");
            }

            /* somehow CIM expects to have some delay to work correctly */
            sleep(1);

            /* start download */
            send_cim_image(fd, link_if, cim_type);
            break;

        case PROGRAM_SERIAL_NUM:
            /* somehow CIM expects to have some delay to work correctly */
            sleep(1);

            /* initiate CIM serial # programming */
            printf("Initiating S/N programming...");
            if( initiate_cim_serial_num_programming(fd, link_if) != 0 ) {
                printf("FAILED.\n");
                printf("Aborting CIM FW Update....");
                if( abort_cim_fw_update(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    return -9;
                }
                printf("passed.\n");

                /* take FPGA out of CIM Update mode */
                printf("Taking FPGA out of CIM Update mode....");
                if( exit_cim_fw_update(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    return -10;
                }
                printf("passed.\n");
                return -6;
            }
            printf("passed.\n");

            /* somehow CIM expects to have some delay to work correctly */
            sleep(1);

            n_tries = 0;
            do {
                ++n_tries;
                printf("Programming CIM Serial Number....");
                if( program_cim_serial_num(fd, link_if, new_sernum) != 0 ) {
                    printf("FAILED.\n");
                    if( n_tries < MAX_RESEND ) {
                        continue;
                    }
                    else {
                        return -7;
                    }
                }
                printf("passed.\n");

                cim_rsp = wait_for_response(fd, link_if);
            } while(( cim_rsp != PCMD_CIM_UPDATE_ACK ) && ( n_tries < MAX_RESEND ));

            /* for now, just flow through */

        case RESEND_INFO:
        case ABORT:
        default:
            if( state == CIM_UPDATE_RECOVERY_STATE ) {
                printf("Sending information request record to device....");
                if( send_info_request_record(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    return -4;
                }
                printf("passed.\n");

                /* wait for device type packet (0xC9) */
                if( wait_for_device_type(fd, link_if) != 0 ) {
                    return -3;
                }
            }
            else {
                printf("Aborting CIM FW Update....");
                if( abort_cim_fw_update(fd, link_if) != 0 ) {
                    printf("FAILED.\n");
                    return -9;
                }
                printf("passed.\n");
            }
            break;
    }

    /* take FPGA out of CIM Update mode */
    printf("Taking FPGA out of CIM Update mode....");
    if( exit_cim_fw_update(fd, link_if) != 0 ) {
        printf("FAILED.\n");
        return -10;
    }
    printf("passed.\n");

    return 0;
}
