/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Paragon CIM Protocol Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <termios.h>

#include "cim.h"
#include "paragon_cim.h"


#define FALSE                                           0
#define TRUE                                            1

#define ENTER_KEY                                       10
#define ESCAPE_KEY                                      27
#define DELETE_KEY                                      127

#ifdef STATIC_KEYMAP
#define MAX_KEY_BUFFER                                  42
#else
#define MAX_KEY_BUFFER                                  128
#endif

#define NUM_TRIES                                       10

typedef struct
{
    unsigned char make;
    unsigned char code;
    unsigned char brk;
} KEYBD_CODE;

static unsigned int  keyset = 0;
static KEYBD_CODE    shift_key;
static KEYBD_CODE    keys[MAX_KEY_BUFFER];
static unsigned char shift[MAX_KEY_BUFFER];

static unsigned char letters[26][3] =
{
    {0x1C, 0xF0, 0x1C}, /* a */
    {0x32, 0xF0, 0x32}, /* b */
    {0x21, 0xF0, 0x21}, /* c */
    {0x23, 0xF0, 0x23}, /* d */
    {0x24, 0xF0, 0x24}, /* e */
    {0x2B, 0xF0, 0x2B}, /* f */
    {0x34, 0xF0, 0x34}, /* g */
    {0x33, 0xF0, 0x33}, /* h */
    {0x43, 0xF0, 0x43}, /* i */
    {0x3B, 0xF0, 0x3B}, /* j */
    {0x42, 0xF0, 0x42}, /* k */
    {0x4B, 0xF0, 0x4B}, /* l */
    {0x3A, 0xF0, 0x3A}, /* m */
    {0x31, 0xF0, 0x31}, /* n */
    {0x44, 0xF0, 0x44}, /* o */
    {0x4D, 0xF0, 0x4D}, /* p */
    {0x15, 0xF0, 0x15}, /* q */
    {0x2D, 0xF0, 0x2D}, /* r */
    {0x1B, 0xF0, 0x1B}, /* s */
    {0x2C, 0xF0, 0x2C}, /* t */
    {0x3C, 0xF0, 0x3C}, /* u */
    {0x2A, 0xF0, 0x2A}, /* v */
    {0x1D, 0xF0, 0x1D}, /* w */
    {0x22, 0xF0, 0x22}, /* x */
    {0x35, 0xF0, 0x35}, /* y */
    {0x1A, 0xF0, 0x1A}  /* z */
};

static unsigned char numbers[10][3] = 
{
    {0x45, 0xF0, 0x45}, /* 0 */
    {0x16, 0xF0, 0x16}, /* 1 */
    {0x1E, 0xF0, 0x1E}, /* 2 */
    {0x26, 0xF0, 0x26}, /* 3 */
    {0x25, 0xF0, 0x25}, /* 4 */
    {0x2E, 0xF0, 0x2E}, /* 5 */
    {0x36, 0xF0, 0x36}, /* 6 */
    {0x3D, 0xF0, 0x3D}, /* 7 */
    {0x3E, 0xF0, 0x3E}, /* 8 */
    {0x46, 0xF0, 0x46}  /* 9 */
};

static unsigned char slash[3] = {0x4A, 0xF0, 0x4A};
static unsigned char space[3] = {0x29, 0xF0, 0x29};

static void InitKeys( void )
{
    int i;

    for (i = 0; i < MAX_KEY_BUFFER; i++)
    {
        keys[i].make = slash[0];
        keys[i].code = slash[1];
        keys[i].brk  = slash[2];
        shift[i]     = FALSE;
    }

    shift_key.make = 0x12;
    shift_key.code = 0xF0;
    shift_key.brk  = 0x12;

    return;
}


static void MapKeys( unsigned char keystroke )
{
    static int kid = 0;
    int idx;

    if ((keystroke >= 'a') && (keystroke <= 'z'))
    {
        idx = keystroke - 'a';
        keys[kid].make = letters[idx][0];
        keys[kid].code = letters[idx][1];
        keys[kid].brk  = letters[idx][2];
        shift[kid]     = FALSE;
    }
    else if ((keystroke >= 'A') && (keystroke <= 'Z'))
    {
        idx = keystroke - 'A';
        keys[kid].make = letters[idx][0];
        keys[kid].code = letters[idx][1];
        keys[kid].brk  = letters[idx][2];
        shift[kid]     = TRUE;
    }
    else if ((keystroke >= '0') && (keystroke <= '9'))
    {
        idx = keystroke - '0';
        keys[kid].make = numbers[idx][0];
        keys[kid].code = numbers[idx][1];
        keys[kid].brk  = numbers[idx][2];
        shift[kid]     = FALSE;
    }
    else
    {
        switch (keystroke)
        {
            case '`':
                keys[kid].make = 0x0E;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x0E;
                shift[kid]     = FALSE;
                break;
            case ';':
                keys[kid].make = 0x4C;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4C;
                shift[kid]     = FALSE;
                break;
            case 0x27: // <'>
                keys[kid].make = 0x52;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x52;
                shift[kid]     = FALSE;
                break;
            case 0x0D: // <CR>
                keys[kid].make = 0x5A;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x5A;
                shift[kid]     = FALSE;
                break;
            case 0x08: // <backspace>
                keys[kid].make = 0x66;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x66;
                shift[kid]     = FALSE;
                break;
            case '-':
                keys[kid].make = 0x4E;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4E;
                shift[kid]     = FALSE;
                break;
            case '=':
                keys[kid].make = 0x55;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x55;
                shift[kid]     = FALSE;
                break;
            case ',':
                keys[kid].make = 0x41;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x41;
                shift[kid]     = FALSE;
                break;
            case '.':
                keys[kid].make = 0x49;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x49;
                shift[kid]     = FALSE;
                break;
            case '/':
                keys[kid].make = 0x4A;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4A;
                shift[kid]     = FALSE;
                break;

		/* shift-key is pressed */
            case '~':
                keys[kid].make = 0x0E;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x0E;
                shift[kid]     = TRUE;
                break;
            case ')':
                keys[kid].make = numbers[0][0];
                keys[kid].code = numbers[0][1];
                keys[kid].brk  = numbers[0][2];
                shift[kid]     = TRUE;
                break;
            case '!':
                keys[kid].make = numbers[1][0];
                keys[kid].code = numbers[1][1];
                keys[kid].brk  = numbers[1][2];
                shift[kid]     = TRUE;
                break;
            case '@':
                keys[kid].make = numbers[2][0];
                keys[kid].code = numbers[2][1];
                keys[kid].brk  = numbers[2][2];
                shift[kid]     = TRUE;
                break;
            case '#':
                keys[kid].make = numbers[3][0];
                keys[kid].code = numbers[3][1];
                keys[kid].brk  = numbers[3][2];
                shift[kid]     = TRUE;
                break;
            case '$':
                keys[kid].make = numbers[4][0];
                keys[kid].code = numbers[4][1];
                keys[kid].brk  = numbers[4][2];
                shift[kid]     = TRUE;
                break;
            case '%':
                keys[kid].make = numbers[5][0];
                keys[kid].code = numbers[5][1];
                keys[kid].brk  = numbers[5][2];
                shift[kid]     = TRUE;
                break;
            case '^':
                keys[kid].make = numbers[6][0];
                keys[kid].code = numbers[6][1];
                keys[kid].brk  = numbers[6][2];
                shift[kid]     = TRUE;
                break;
            case '&':
                keys[kid].make = numbers[7][0];
                keys[kid].code = numbers[7][1];
                keys[kid].brk  = numbers[7][2];
                shift[kid]     = TRUE;
                break;
            case '*':
                keys[kid].make = numbers[8][0];
                keys[kid].code = numbers[8][1];
                keys[kid].brk  = numbers[8][2];
                shift[kid]     = TRUE;
                break;
            case '(':
                keys[kid].make = numbers[9][0];
                keys[kid].code = numbers[9][1];
                keys[kid].brk  = numbers[9][2];
                shift[kid]     = TRUE;
                break;
            case '_':
                keys[kid].make = 0x4E;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4E;
                shift[kid]     = TRUE;
                break;
            case '+':
                keys[kid].make = 0x55;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x55;
                shift[kid]     = TRUE;
                break;
            case ':':
                keys[kid].make = 0x4C;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4C;
                shift[kid]     = TRUE;
                break;
            case '"':
                keys[kid].make = 0x52;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x52;
                shift[kid]     = TRUE;
                break;
            case '<':
                keys[kid].make = 0x41;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x41;
                shift[kid]     = TRUE;
                break;
            case '>':
                keys[kid].make = 0x49;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x49;
                shift[kid]     = TRUE;
                break;
            case '?':
                keys[kid].make = 0x4A;
                keys[kid].code = 0xF0;
                keys[kid].brk  = 0x4A;
                shift[kid]     = TRUE;
                break;

            default:
                /* default is space */
                keys[kid].make = space[0];
                keys[kid].code = space[1];
                keys[kid].brk  = space[2];
                shift[kid]     = FALSE;
        }
    }

    if (++kid >= MAX_KEY_BUFFER)
    {
        kid = 0;
    }

    return;
}

static int PMASendKbdLowerCaseChar( int fd, int link_if, unsigned int keyset )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;
    unsigned char PktBuf[6];
    int           bResult;
    int           i, n_tries = 0;

    /* create KBD packet (1 make and 2 break) */
    PktBuf[0] = keys[keyset].make;
    PktBuf[1] = keys[keyset].code;
    PktBuf[2] = keys[keyset].brk;

    pdata->link_if = link_if;
    pdata->requested_len = 2;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_PS2KBD_DATA1;

    /* send pkt one at a time */
    for (i = 0; i < 3; i++) {
        pdata->buf[1] = PktBuf[i];
        do {
            ++n_tries;
            bResult = paragon_cim_send_pkt(fd, link_if, pdata);
            if( bResult != 0 ) {
                sleep(2);
            }
        } while( bResult != 0 && n_tries <= NUM_TRIES );
    }

    return bResult;
}

static int PMASendKbdUpperCaseChar( int fd, int link_if, unsigned int keyset )
{
    FPD_data_t    protocol_data, *pdata = &protocol_data;
    unsigned char PktBuf[6];
    int           bResult;
    int           i, n_tries = 0;

    /*
     * Create KBD packet (total 6)
     *   - shift key make
     *   - char make
     *   - char code
     *   - char break
     *   - shift key code
     *   - shift key break
     */
    PktBuf[0] = shift_key.make;
    PktBuf[1] = keys[keyset].make;
    PktBuf[2] = keys[keyset].code;
    PktBuf[3] = keys[keyset].brk;
    PktBuf[4] = shift_key.code;
    PktBuf[5] = shift_key.brk;

    pdata->link_if = link_if;
    pdata->requested_len = 2;
    pdata->type = FPD_CIM_BUFFER;
    pdata->buf = protocol_buffer;

    /* set command byte */
    pdata->buf[0] = PCMD_PS2KBD_DATA1;

    /* send pkt one at a time */
    for (i = 0; i < 6; i++) {
        pdata->buf[1] = PktBuf[i];
        do {
            ++n_tries;
            bResult = paragon_cim_send_pkt(fd, link_if, pdata);
            if( bResult != 0 ) {
                sleep(2);
            }
        } while( bResult != 0 && n_tries <= NUM_TRIES );
    }

    return bResult;
}

static int keybd_init( void )
{
    InitKeys();
    keyset = 0;

    return 0;
}

int keybd_send_char( int fd, int link_if )
{
    int bResult = 0;
    unsigned char key;
    int running = 1;

    /*
     * Allocate two structures on the stack.
     * The first is to maintain state long enough 
     * to put the console in "raw" mode.  And the
     * second is used to bring it back to the 
     * mode it was originally in.
     */
    struct termios rawmode = {0}, savemode = {0};

    /*
     * Save current console state
     */
    tcgetattr(fileno(stdin), &savemode);

    /*
     * Get the state information necessary to go 
     * into raw mode
     */
    cfmakeraw(&rawmode);

    /*
     * Use that state information to go in to 
     * raw mode.
     */
    tcsetattr(fileno(stdin), 0, &rawmode);

    keybd_init();

    printf("\nStart typing on keyboard. Press ESCAPE key to exit.\n\n\r");

    do {
        key = getchar();

        /* check for exit event */
        if( key == ESCAPE_KEY ) {
            break;
        }
        else if( key == ENTER_KEY ) {
            printf("\n");
        }
        else if( key == DELETE_KEY ) {
            printf("\b");
        }
        else {
            printf("%c", key);
        }

        /* map key  */
        MapKeys(key);

        if (shift[keyset]) {
            bResult = PMASendKbdUpperCaseChar(fd, link_if, keyset);
        }
        else {
            bResult = PMASendKbdLowerCaseChar(fd, link_if, keyset);
        }

        if( bResult != 0 ) {
            running = 0;
        }

        if (++keyset >= MAX_KEY_BUFFER) {
            keyset = 0;
        }
    } while( running );

    /*
     * Restore original console state
     */
    tcsetattr(fileno(stdin), 0, &savemode);

    printf("\n");

    return bResult;
}

int keybd_loop_test( int fd, int link_if )
{
    int bResult;
    unsigned char key;
    int running = 1;
    int len;
    int nreps;
    int i;

    /*
     * Allocate two structures on the stack.
     * The first is to maintain state long enough 
     * to put the console in "raw" mode.  And the
     * second is used to bring it back to the 
     * mode it was originally in.
     */
    struct termios rawmode = {0}, savemode = {0};

    /*
     * Save current console state
     */
    tcgetattr(fileno(stdin), &savemode);

    /*
     * Get the state information necessary to go 
     * into raw mode
     */
    cfmakeraw(&rawmode);

    /*
     * Use that state information to go in to 
     * raw mode.
     */
    tcsetattr(fileno(stdin), 0, &rawmode);

    keybd_init();

    printf("\nEnter Phrase to loop. Press ESC key to exit.\n\n\r");

    do {
        scanf("%c", &key);

        /* check for exit event */
        if( key == ESCAPE_KEY ) {
            break;
        }
        else if( key == ENTER_KEY ) {
            printf("\n");
        }
        else if( key == DELETE_KEY ) {
            printf("\b");
        }
        else {
            printf("%c", key);
        }

        /* map key  */
        MapKeys(key);

        if (++keyset >= MAX_KEY_BUFFER) {
            running = 0;
        }
    } while( running );

    /*
     * Restore original console state
     */
    tcsetattr(fileno(stdin), 0, &savemode);

    len = keyset;
    keyset = 0;

    printf("\nEnter number of repetitions: ");
    scanf("%d", &nreps);

    for( i = 0; i < nreps; i++ ) {
        running = 1;
        keyset = 0;
        do {
            if (shift[keyset]) {
                bResult = PMASendKbdUpperCaseChar(fd, link_if, keyset);
            }
            else {
                bResult = PMASendKbdLowerCaseChar(fd, link_if, keyset);
            }
            
            if( bResult != 0 ) {
                fflush(stdout);
                fprintf(stderr, "\nERROR: Terminating nreps=%d\n", i+1);
                return bResult;
            }

            if (++keyset >= len) {
                running = 0;
            }
        } while( running );
    }

    return 0;
}
