/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  KX2.0 High-Level Protocol Header
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2006-2007 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#ifndef _VM_PROTOCOL_H_
#define _VM_PROTOCOL_H_

#define VM_SOP                                  0xFE
#define VM_EOH                                  0xFF

/* Packet Source mask */
#define PKT_SRC_MASK                            0x80

/* Packet Source definition */
#define PKT_SRC_SWITCH                          0x00
#define PKT_SRC_CIM                             0x80

/* packet request result code */
#define RESULT_NO_ERROR                         0x00
#define RESULT_CIM_RESET_FAILED                 0x01
#define RESULT_WRONG_CIM_SETTINGS               0x04
#define RESULT_WRONG_KBD_SETTINGS               0x08
#define RESULT_WRONG_VERSION_INFO               0x10
#define RESULT_INVALID_VERSION_COMPONENT        0x11
#define RESULT_INVALID_VERSION_TYPE             0x12
#define RESULT_INVALID_COMPONENT_INDEX          0x13
#define RESULT_HEARTBEAT_ERROR                  0x18
#define RESULT_COLORCAL_CNTL_FAILED             0x20
#define RESULT_CIM_TX_TIMEOUT_FAILED            0x21
#define RESULT_CIM_SET_LED_FAILED               0x22
#define RESULT_CIM_FPGA_READ_FAILED             0x23
#define RESULT_CIM_FPGA_WRITE_FAILED            0x24
#define RESULT_NVRAM_WRITE_ERROR                0x30
#define RESULT_NVRAM_REQ_LEN_OVER_LIMIT         0x31
#define RESULT_NVRAM_REQ_OFFSET_OVER_LIMIT      0x32
#define RESULT_NVRAM_READ_ERROR                 0x38
#define RESULT_KBD_MODEL_SETTING_ERROR          0x40
#define RESULT_INVALID_KBD_MODEL                0x41
#define RESULT_KBD_LANGUAGE_SETTING_ERROR       0x42
#define RESULT_INVALID_KBD_LANGUAGE             0x43
#define RESULT_INVALID_KBD_LED_SETTING          0x44
#define RESULT_MOUSE_MODE_ERROR                 0x48
#define RESULT_INVALID_MOUSE_MODE               0x49
#define RESULT_USB_CIM_DISC_ERROR               0x50
#define RESULT_USB_CIM_CONN_ERROR               0x52
#define RESULT_USB_MS_ENUM_ERROR                0x54
#define RESULT_USB_STALL_BULKEP_ERROR           0x56
#define RESULT_USB_MS_RESET_ERROR               0x58
#define RESULT_USB_MS_CLEAR_FEAT_HALT_ERROR     0x5A
#define RESULT_SERIAL_PARAM_INIT_ERROR          0x60
#define RESULT_UNSUPPORTED_SERIAL_BIT_RATE      0x61
#define RESULT_INVALID_PKT_SOP                  0x70
#define RESULT_INVALID_PKT_EOH                  0x71
#define RESULT_INVALID_PKT_SOURCE               0x72
#define RESULT_INVALID_PKT_CATEGORY             0x73
#define RESULT_INVALID_PKT_COMMAND              0x74
#define RESULT_INVALID_TEST                     0x80
#define RESULT_INVALID_TEST_OPTION_CNT          0x81
#define RESULT_INVALID_TEST_OPTION              0x82
#define RESULT_NOT_READY_TO_UPDATE_CODE         0xF0
#define RESULT_CODE_BLK_NOT_SAVED               0xF1
#define RESULT_CODE_CHKSUM_ERROR                0xF2

/* CIM Type */
#define CIM_TYPE_POWER                          0x00
#define CIM_TYPE_SERIAL                         0x01
#define CIM_TYPE_PS2                            0x02
#define CIM_TYPE_SUN                            0x03
#define CIM_TYPE_SUN_USB                        0x08
#define CIM_TYPE_USB_FULL_SPEED                 0x09
#define CIM_TYPE_USB_FULL_AND_HIGH_SPEED        0x0A
#define CIM_TYPE_SWITCH                         0x80
#define CIM_TYPE_ZCIM                           0xA0

/* Enumerated Target USB Speed */
#define USB_SPEED_NA                            0x00
#define USB_SPEED_FULL                          0x01
#define USB_SPEED_HIGH                          0x02

/* Supported Modes */
#define SUPPORTED_MODE_NONE                     0x00
#define SUPPORTED_MODE_VM                       0x01
#define SUPPORTED_MODE_AUDIO                    0x02
#define SUPPORTED_MODE_VM_AND_AUDIO             0x03

/* Keyboard LED Status */
#define KBD_LED_NUM_LOCK                        0x01
#define KBD_LED_CAPS_LOCK                       0x02
#define KBD_LED_SCROLL_LOCK                     0x04
#define KBD_LED_COMPOSE_LOCK                    0x08
#define KBD_LED_KANA_LOCK                       0x10

/* Version Component */
#define VERSION_COMPONENT_MAINBOARD             0x00
#define VERSION_COMPONENT_CLIPR                 0x01
#define VERSION_COMPONENT_UST                   0x02
#define VERSION_COMPONENT_SWITCH                0x03
#define VERSION_COMPONENT_CIM                   0x04
#define VERSION_COMPONENT_PMA                   0x05
#define VERSION_COMPONENT_LUPMA                 0x06
#define VERSION_COMPONENT_FG                    0x07
#define VERSION_COMPONENT_RDM                   0x08
#define VERSION_COMPONENT_COMPUSWITCH           0x09
#define VERSION_COMPONENT_SERIAL_CARD           0x0A

/* Version Type */
#define VERSION_TYPE_SW                         0x00
#define VERSION_TYPE_FW                         0x01
#define VERSION_TYPE_HW                         0x02
#define VERSION_TYPE_FPGA                       0x03
#define VERSION_TYPE_PROTOCOL                   0x04
#define VERSION_TYPE_BOOTLOADER                 0x05
#define VERSION_TYPE_ALL                        0xFF

/* Color Calibration Voltage Reference */
#define COLOR_VOLTAGE_REFERENCE_OFF             0x00
#define COLOR_VOLTAGE_REFERENCE_ON              0x01

/* Keyboard Data Format Maximum Number of Depressed Key */
#define KBD_DATA_MAX_KEY                        6

/* Keyboard Data Format Byte Count */
#define KBD_INPUT_REPORT_BYTE_CNT               8

/* Mouse mode Positioning Bit Mask */
#define MOUSE_POSITIONING_BIT                   0x01

/* Mouse Data Format Byte Count */
#define MOUSE_INPUT_REPORT_BYTE_CNT             4

/* Update Image Types
 *   BIT0 - BTL
 *   BIT1 - Configuration Data for KX20SW 0x3000~0x37FF
 *   BIT2 - Configuration Data for CIM 0x3800~0x3FFF
 *   BIT3 - APP0
 *   BIT4 - FPGA0
 *   BIT5 - APP1
 *   BIT6 - FPGA1 
 *   BIT7 - Echo On/Off
 */
#define UPDATE_FINAL_BOOTLOADER                     0x81
#define UPDATE_VM_APPLICATION_BACKUP                0x88
#define UPDATE_FPGA_BACKUP                          0x90
#define UPDATE_VM_APPLICATION_PRIMARY               0xA0
#define UPDATE_FPGA_PRIMARY                         0xC0
#define UPDATE_ALL_IMAGE                            0xFC

/* Stall Bulk Endpoint settings */
#define STALL_BULK_ENDPOINT_IN                      0x01
#define UNSTALL_BULK_ENDPOINT_IN                    0x02
#define STALL_BULK_ENDPOINT_OUT                     0x04
#define UNSTALL_BULK_ENDPOINT_OUT                   0x08
#define BULK_ENDPOINT_SETTING_PERMANENT             0x10

/* Clear Feature Halt Bulk Endpoint settings */
#define BULK_ENDPOINT_IN                            0x01
#define BULK_ENDPOINT_OUT                           0x02

/* loopback test mode */
#define LPBK_TEST_DISABLED                          0x00
#define LPBK_TEST_CIM_BUFFER                        0x01
#define LPBK_TEST_PRIORITY_BUFFER                   0x02
#define LPBK_TEST_HOST0_BUFFER                      0x04
#define LPBK_TEST_HOST1_BUFFER                      0x08

//-----------------------------------------------------------------------------
// Datatypes
//-----------------------------------------------------------------------------
#pragma pack(1)
typedef struct {
    unsigned char  sop; /* start of packet */
    unsigned char  protocol_version;
    unsigned char  sequence_number;
    unsigned char  category;
    unsigned char  command;
#ifdef BIG_ENDIAN_PKT
    unsigned short payload_length;
#else
    unsigned char  payload_lenlo;
    unsigned char  payload_lenhi;
#endif
    unsigned char  eoh; /* end of header */
} vm_pkt_hdr_t;
#pragma pack()

enum vm_category {
    INITIALIZATION            = 0x00,
    INFORMATION               = 0x02,
    CONTROL                   = 0x04,
    MEMORY                    = 0x10,
    KEYBOARD                  = 0x20,
    MOUSE                     = 0x30,
    USB                       = 0x40,
    SERIAL                    = 0x50,
    POWER_STRIP               = 0xA0,
    SWITCH_TO_SWITCH          = 0xB0,
    Z_CHAINS                  = 0xC0,
    MAINTENANCE               = 0xD0,
    UPDATE                    = 0xF0
};

enum vm_initialization {
    RESET                     = 0x00,
    INIT                      = 0x02,
};

enum vm_information {
    VERSION                   = 0x00,
    CIM_HEARTBEAT             = 0x02,
    SWITCH_HEARTBEAT          = 0x03,
    LINK_STATS                = 0x10,
    CIM_FPGA_ERROR            = 0x11,
    CIM_FPGA_READ             = 0x12,
};

enum vm_control {
    CIM_SET_COLOR_CAL_VOLTAGE = 0x00,
    CIM_SET_TX_TIMEOUT_VALUE  = 0x01,
    CIM_SET_LED               = 0x02,
    CIM_SET_TEST_MODE         = 0x03,
    CIM_FPGA_WRITE            = 0x04,
};

enum vm_memory {
    CIM_EEPROM_WRITE          = 0x00,
    CIM_EEPROM_READ           = 0x01,
};

enum vm_keyboard {
    KBD_MODEL_SET             = 0x00,
    KBD_LANGUAGE_SET          = 0x01,
    TARGET_KBD_LED_SETTINGS   = 0x04,
    TARGET_KBD_LED_SETTINGS_CHG   = 0x05,
    KBD_DATA                  = 0x40,
};

enum vm_mouse {
    MOUSE_MODE_SET            = 0x00,
    MOUSE_DATA                = 0x01,
};

enum vm_usbctl {
    USB_CIM_DISCONNECT        = 0x00,
    USB_CIM_CONNECT           = 0x01,
    VM_USB_DEVICE_CONFIG      = 0x08,
    STALL_BULK_EP             = 0x09,
    MASS_STORAGE_RESET        = 0x40,
    CLR_FEATURE_HALT          = 0x41,
};

enum vm_maintenance {
    RX_ERROR_PKT              = 0x00,
};

enum vm_update {
    UPDATE_REQUEST            = 0x00,
    CODE_BLOCK_TRANSFER       = 0x01,
    LAST_CODE_BLOCK_TRANSFER  = 0x02,
};

enum version_component {
    MAINBOARD,
    CLIPR,
    UST,
    SWITCH,
    CIM,
    PMA,
    LUPMA,
    FG,
    RDM,
    CS,
    SC,
};

enum version_type {
    SW,
    FW,
    HW,
    FPGA,
    PR,
    BL,
};

enum kbd_model {
    STD_101_KEY               = 0x00,
    PINNACLE_DEKO             = 0x10,
    BELLA_DV                  = 0x11,
};

enum kbd_language {
    US_ENGLISH                = 0x00,
    BRITISH_ENGLISH           = 0x02,
    FRENCH                    = 0x10,
    GERMAN                    = 0x11,
    ITALIAN                   = 0x12,
    SPANISH                   = 0x13,
    NORWAY                    = 0x14,
    JAPANESE                  = 0x20,
    KOREAN                    = 0x28,
    CHINESE_TRADITIONAL       = 0x30,
    CHINESE_SIMPLIFIED        = 0x31,
};

enum mouse_position {
    ABSOLUTE                  = 0x00,
    RELATIVE                  = 0x01,
};

enum led_settings {
    LED_OFF                   = 0x00,
    LED_ON                    = 0x01,
    LED_TOGGLE                = 0x02,
};

enum test_mode {
    TEST_EXIT                 = 0x00,
    TEST_LOOPBACK             = 0x01,
};

/* Keyboard Data Format */
typedef struct {
    unsigned char             flag;
    unsigned char             reserved;
    unsigned char             key[KBD_DATA_MAX_KEY];
} kbd_data_format_t;

#pragma pack(1)
/**************************/
/* Initialization Packets */
/**************************/

/* CIM Reset Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} cim_reset_req_t;

/* CIM Ready Indication */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_ready_indication_t;

/* CIM Initialization Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} initialization_req_t;

/* CIM Initialization Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             cim_setting_cnt;
    unsigned char             kbd_setting_cnt;
    unsigned char             cim_type;
    unsigned char             ep_cnt;
    unsigned char             cim_setting_1;
    unsigned char             cim_setting_2;
    unsigned char             kbd_setting_1;
    unsigned char             kbd_setting_2;
} initialization_rsp_t;

/* union of all Initialization Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    cim_reset_req_t           reset_req;
    cim_ready_indication_t    ready_ind;
    initialization_req_t      init_req;
    initialization_rsp_t      init_rsp;
} initialization_pkt_t;

/***********************/
/* Information Packets */
/***********************/

/* Version Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             component;
    unsigned char             version_type;
    unsigned char             component_index;
} version_req_t;

/* Version Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             version[4];
    unsigned char             len[4];
    unsigned char             checksum;
} version_rsp_t;

/* Rejected Version Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} reject_version_rsp_t;

/* CIM Status Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} cim_status_req_t;

/* CIM Status Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_status_rsp_t;

/* Switch Heartbeat Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} switch_heartbeat_req_t;

/* Switch Heartbeat Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} switch_heartbeat_rsp_t;

/* Link Statistics Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} link_stats_req_t;

/* Link Statistics Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             rx_inv_pkt_cnt;
    unsigned char             rx_crc_err_cnt;
    unsigned char             rx_prot_err_cnt;
    unsigned char             rx_buf_full_cnt;
    unsigned char             noise_cnt;
    unsigned char             tx_prot_err_cnt;
    unsigned char             tx_buf_full_cnt;
    unsigned char             tx_crc_cnt;
    unsigned char             max_retry;
    unsigned char             rx_pkt_cnt_lo;
    unsigned char             rx_pkt_cnt_hi;
    unsigned char             tx_pkt_cnt_lo;
    unsigned char             tx_pkt_cnt_hi;
} link_stats_rsp_t;

/* CIM FPGA Errors Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} cim_fpga_error_req_t;

/* CIM FPGA Errors Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             rx_pri_rd_empty_buf;
    unsigned char             tx_pri_sync_err;
    unsigned char             tx_pri_no_eop;
    unsigned char             tx_pri_wr_full_buf;
    unsigned char             rx_cim_rd_empty_buf;
    unsigned char             tx_cim_sync_err;
    unsigned char             tx_cim_no_eop;
    unsigned char             tx_cim_wr_full_buf;
    unsigned char             tx_host1_sync_err;
    unsigned char             tx_host1_no_eop;
    unsigned char             tx_host1_wr_full_buf;
    unsigned char             reserved;
    unsigned char             tx_pri_timeout;
    unsigned char             tx_cim_timeout;
    unsigned char             tx_host_timeout1;
    unsigned char             tx_host_timeout0;
} cim_fpga_error_rsp_t;

/* CIM FPGA Read Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             reg;
} cim_fpga_read_req_t;

/* CIM FPGA Read Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             reg;
    unsigned char             value;
} cim_fpga_read_rsp_t;

/* union of all Information Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    version_req_t             version_req;
    version_rsp_t             version_rsp;
    reject_version_rsp_t      reject_version_rsp;
    cim_status_req_t          cim_status_req;
    cim_status_rsp_t          cim_status_rsp;
    switch_heartbeat_req_t    switch_heartbeat_req;
    switch_heartbeat_rsp_t    switch_heartbeat_rsp;
    link_stats_req_t          link_stats_req;
    link_stats_rsp_t          link_stats_rsp;
    cim_fpga_error_req_t      cim_fpga_error_req;
    cim_fpga_error_rsp_t      cim_fpga_error_rsp;
    cim_fpga_read_req_t       cim_fpga_read_req;
    cim_fpga_read_rsp_t       cim_fpga_read_rsp;
} information_pkt_t;

/*******************/
/* Control Packets */
/*******************/

/* CIM Set Color Calibration Voltage Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             voltage_ref;
} cim_set_cc_voltage_req_t;

/* CIM Set Color Calibration Voltage Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             voltage_ref;
} cim_set_cc_voltage_rsp_t;

/* CIM Set Transmit Timeout Value Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             timeout;
} cim_set_tx_timeout_req_t;

/* CIM Set Transmit Timeout Value Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_set_tx_timeout_rsp_t;

/* CIM Set LED Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             cmd;
} cim_set_led_req_t;

/* CIM Set LED Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_set_led_rsp_t;

/* CIM Set Test Mode Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             test_mode;
    unsigned char             option_cnt;
    unsigned char             options[20];
} cim_set_test_mode_req_t;

/* CIM Set Test Mode Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_set_test_mode_rsp_t;

/* CIM FPGA Write Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             reg;
    unsigned char             value;
} cim_fpga_write_req_t;

/* CIM FPGA Write Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             reg;
    unsigned char             value;
} cim_fpga_write_rsp_t;

/* union of all Control Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    cim_set_cc_voltage_req_t  cc_voltage_req;
    cim_set_cc_voltage_rsp_t  cc_voltage_rsp;
    cim_set_tx_timeout_req_t  tx_timeout_req;
    cim_set_tx_timeout_rsp_t  tx_timeout_rsp;
    cim_set_led_req_t         led_req;
    cim_set_led_rsp_t         led_rsp;
    cim_set_test_mode_req_t   test_mode_req;
    cim_set_test_mode_rsp_t   test_mode_rsp;
    cim_fpga_write_req_t      cim_fpga_write_req;
    cim_fpga_write_rsp_t      cim_fpga_write_rsp;
} control_pkt_t;

/******************/
/* Memory Packets */
/******************/

/* CIM EEPROM Write Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             offset_lo;
    unsigned char             offset_hi;
    unsigned char             len_lo;
    unsigned char             len_hi;
    unsigned char             buf[128];
} cim_eeprom_write_req_t;

/* CIM EEPROM Write Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} cim_eeprom_write_rsp_t;

/* CIM EEPROM Read Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             offset_lo;
    unsigned char             offset_hi;
    unsigned char             len_lo;
    unsigned char             len_hi;
} cim_eeprom_read_req_t;

/* CIM EEPROM Read Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             buf[128];
} cim_eeprom_read_rsp_t;

/* union of all Memory Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    cim_eeprom_write_req_t    eeprom_write_req;
    cim_eeprom_write_rsp_t    eeprom_write_rsp;
    cim_eeprom_read_req_t     eeprom_read_req;
    cim_eeprom_read_rsp_t     eeprom_read_rsp;
} memory_pkt_t;

/********************/
/* Keyboard Packets */
/********************/

/* Keyboard Model Set Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             kbd_model;
} kbd_model_set_req_t;

/* Keyboard Model Set Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             kbd_model;
} kbd_model_set_rsp_t;

/* Keyboard Language Set Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             language;
} kbd_language_set_req_t;

/* Keyboard Language Set Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             language;
} kbd_language_set_rsp_t;

/* Target Keyboard LED Settings Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} kbd_led_setting_req_t;

/* Target Keyboard LED Settings Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             led_setting;
} kbd_led_setting_rsp_t;

/* Target Keyboard LED Settings Changed */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             led_setting;
} kbd_led_setting_chg_t;

/* Keyboard Data Packet */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             kbd_data[KBD_INPUT_REPORT_BYTE_CNT];
} kbd_data_pkt_t;

/* union of all Keyboard Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    kbd_model_set_req_t       kbd_model_set_req;
    kbd_model_set_rsp_t       kbd_model_set_rsp;
    kbd_language_set_req_t    kbd_language_set_req;
    kbd_language_set_rsp_t    kbd_language_set_rsp;
    kbd_led_setting_req_t     kbd_led_setting_req;
    kbd_led_setting_rsp_t     kbd_led_setting_rsp;
    kbd_led_setting_chg_t     kbd_led_setting_chg;
    kbd_data_pkt_t            kbd_data_pkt;
} keyboard_pkt_t;

/*****************/
/* Mouse Packets */
/****************/

/* Mouse Mode Set Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             mouse_mode;
} mouse_mode_set_req_t;

/* Mouse Mode Set Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} mouse_mode_set_rsp_t;

/* Mouse Data Packet */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             mouse_data[MOUSE_INPUT_REPORT_BYTE_CNT];
} mouse_data_pkt_t;

/* union of all Mouse Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    mouse_mode_set_req_t      mouse_mode_set_req;
    mouse_mode_set_rsp_t      mouse_mode_set_rsp;
    mouse_data_pkt_t          mouse_data_pkt;
} mouse_pkt_t;

/***********************/
/* USB Control Packets */
/***********************/

/* USB CIM Disconnect Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} usb_cim_disc_req_t;

/* USB CIM Disconnect Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} usb_cim_disc_rsp_t;

/* USB CIM Connect Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
} usb_cim_conn_req_t;

/* USB CIM Connect Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} usb_cim_conn_rsp_t;

/* Virtual Media USB Device Configuration Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             settings;
} usb_dev_cfg_req_t;

/* Virtual Media USB Device Configuration Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             settings;
} usb_dev_cfg_rsp_t;

/* Stall Bulk Endpoint Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             usbdevID;
    unsigned char             settings;
} stall_bulkep_req_t;

/* Stall Bulk Endpoint Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} stall_bulkep_rsp_t;

/* Mass Storage Reset Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             usbdevID;
} mass_storage_reset_req_t;

/* Mass Storage Reset Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} mass_storage_reset_rsp_t;

/* Clear Feature HALT USB Standard Device Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             usbdevID;
    unsigned char             settings;
} clear_feat_halt_req_t;

/* Clear Feature HALT USB Standard Device Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} clear_feat_halt_rsp_t;

/* union of all USB Control Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    usb_cim_disc_req_t        usb_cim_disc_req;
    usb_cim_disc_rsp_t        usb_cim_disc_rsp;
    usb_cim_conn_req_t        usb_cim_conn_req;
    usb_cim_conn_rsp_t        usb_cim_conn_rsp;
    usb_dev_cfg_req_t         usb_dev_cfg_req;
    usb_dev_cfg_rsp_t         usb_dev_cfg_rsp;
    stall_bulkep_req_t        stall_bulkep_req;
    stall_bulkep_rsp_t        stall_bulkep_rsp;
    mass_storage_reset_req_t  mass_storage_reset_req;
    mass_storage_reset_rsp_t  mass_storage_reset_rsp;
    clear_feat_halt_req_t     clear_feat_halt_req;
    clear_feat_halt_rsp_t     clear_feat_halt_rsp;
} usb_control_pkt_t;

/***********************/
/* Maintenance Packets */
/***********************/

/* Received Error Packet Notification */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
} rx_error_pkt_notify_t;

/* union of all Maintenance Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    rx_error_pkt_notify_t     rx_error_pkt_notify;
} maintenance_pkt_t;

/*****************************/
/* Code/Image Update Packets */
/*****************************/

/* Update Request */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             component;
    unsigned char             version_type;
    unsigned char             component_index;
    unsigned char             version[4];
    unsigned char             len[4];
    unsigned char             checksum;
} update_req_t;

/* Update Response */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             xfer_size[2];
} update_rsp_t;

/* Code Block Transfer */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             start_addr[4];
    unsigned char             size[2];
    unsigned char             code_blk[256];
} code_blk_xfer_t;

/* Code Block Acknowledgement */
typedef struct {
    vm_pkt_hdr_t              hdr;
    unsigned char             result;
    unsigned char             start_addr[4];
    unsigned char             size[2];
} code_blk_ack_t;

/* union of all Update Packets */
typedef union {
    vm_pkt_hdr_t              hdr;
    update_req_t              update_req;
    update_rsp_t              update_rsp;
    code_blk_xfer_t           code_blk_xfer;
    code_blk_ack_t            code_blk_ack;
} update_pkt_t;

#pragma pack()

#endif /* _VM_PROTOCOL_H_ */
