/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Device Diagnostic Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <time.h>

#include "fpd_ioctl.h"
#include "fpd-diagnostics.h"


static int one_pass_only = 0
;

static void dump_data( unsigned char *bufaddr, int pktlen )
{
    int i;

    fprintf(stderr, "Data:\n");
    for( i = 0; i < pktlen; i++ ) {
        fprintf(stderr, "%02x ", bufaddr[i]);
    }
    fprintf(stderr, "\n");

    return;
}

static int verify_data( int type, unsigned char *pbuf, int len, pkt_pattern_t pattern, int round )
{
    int rc, i;

    /* compare contents against expected data */
    switch( pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            rc = memcmp((const void *)pbuf, (const void *)&ascending[0], len);
            if( rc != 0 ) {
                fprintf(stderr, "FAILED\n");
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != ascending[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, ascending[i], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            rc = memcmp((const void *)pbuf, (const void *)&descending[0], len);
            if( rc != 0 ) {
                fprintf(stderr, "FAILED\n");
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != descending[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, descending[i], pbuf[i]);
                    }
                }
            }
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            rc = memcmp((const void *)pbuf, (const void *)&repeating[0], len);
            if( rc != 0 ) {
                fprintf(stderr, "FAILED\n");
                for( i = 0; i < len; i++ ) {
                    if( pbuf[i] != repeating[i] ) {
                        fprintf(stderr, "Offset %d: Expected %02x, Rcvd %02x\n",
                                i, repeating[i], pbuf[i]);
                    }
                }
            }
            break;

        default:
            fprintf(stderr, "FAILED\n");
            fprintf(stderr, "ERROR: Unknown Packet Pattern!\n");
            return -1;
    }

    if( rc != 0 ) {
        if( !debug_on ) {
            fprintf(stderr, "[%d]Comparing contents with expected data (%d bytes) from %s buffer....FAILED\n",
                    round, len, type?"Priority":"CIM");
        }
        fprintf(stderr, "Expected %d bytes of ", len);
        switch( pattern ) {
            case PKT_PATTERN_8BIT_ASCENDING:
                fprintf(stderr, "8-bit ascending ");
                break;
            case PKT_PATTERN_8BIT_DESCENDING:
                fprintf(stderr, "8-bit descending ");
                break;
            case PKT_PATTERN_8BIT_REPEATING:
                fprintf(stderr, "8-bit repeating ");
                break;
            default:
                break;
        }
        fprintf(stderr, "pattern but received different data\n");

        /* dump data */
        if( len > 0 ) {
            dump_data(pbuf, len);
        }
    }

    return rc;
}

static int get_rxpdata_params( FPD_data_t *pdata, unsigned int link_if, int btype )
{
    printf("\nEnter Parameters\n");

    if( link_if <= 4 ) {
        printf("Link IF                     : %d\n", link_if);
    }
    else if( link_if == FPD_LINK_IF_ID_BGND ) {
        printf("Link IF                     : Background\n");
    }
    else {
        return -1;
    }
    pdata->link_if = link_if;
    printf("Buffer Type                 : ");
    if( btype == FPD_CIM_BUFFER ) {
        printf("CIM\n");
    }
    else {
        printf("Priority\n");
    }
    pdata->type = btype;

    printf("Data Length                 : ");
    scanf("%d", &pdata->requested_len);

    if( pdata->requested_len > FPD_MAX_PACKET_SIZE )
        return -1;

    pdata->buf = protocol_buffer;

    return 0;
}

static int get_tx_params( FPD_data_t *pdata, unsigned int link_if, int btype )
{
    int i;

    printf("\nEnter Parameters\n");

    if( link_if <= 4 ) {
        printf("Link IF                     : %d\n", link_if);
    }
    else if( link_if == FPD_LINK_IF_ID_BGND ) {
        printf("Link IF                     : Background\n");
    }
    else {
        return -1;
    }
    pdata->link_if = link_if;
    printf("Buffer Type                 : ");
    if( btype == FPD_CIM_BUFFER ) {
        printf("CIM\n");
    }
    else {
        printf("Priority\n");
    }
    pdata->type = btype;

    printf("Data Length                 : ");
    scanf("%d", &pdata->requested_len);

    if( pdata->requested_len > FPD_MAX_PACKET_SIZE )
        return -1;

    pdata->buf = protocol_buffer;

    for( i = 0; i < pdata->requested_len; i++ ) {
        printf("Data byte %03d               : 0x", i+1);
        scanf("%hhx", &pdata->buf[i]);
    }

    return 0;
}

int write_protocol_data_manually( int fd, unsigned int link_if, int btype )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;

    if( get_tx_params(pdata, link_if, btype) == 0 ) {
        printf("\nSending %d bytes of data to %s buffer....",
               pdata->requested_len, pdata->type?"Priority":"CIM");
        rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
        if( rc == 0 ) {
            if( pdata->actual_len == pdata->requested_len )
                printf("passed.\n");
            else {
                printf("FAILED.\n");
                printf("Actual bytes written %d\n", pdata->actual_len);
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }
    else {
        rc = -1;
    }

    return rc;
}

int read_protocol_data_manually( int fd, unsigned int link_if, int btype )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;

    if( get_rxpdata_params(pdata, link_if, btype) == 0 ) {
        printf("\nReading %d bytes of data from %s buffer....",
               pdata->requested_len, pdata->type?"Priority":"CIM");
        rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
        if( rc == 0 ) {
            int i;

            if( pdata->actual_len == pdata->requested_len ) {
                printf("passed.\n");
                printf("Packet: ");
            }
            else {
                printf("FAILED.\n");
                printf("Requested %d, Got %d\n",
                       pdata->requested_len, pdata->actual_len);
            }

            for( i = 0; i < pdata->actual_len; i++ ) {
                printf("%02x ", pdata->buf[i]);
            }
            printf("\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }

    return 0;
}

int send_protocol_data( int fd, unsigned int link_if, int btype, int pkt_size )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;

    pdata->link_if = link_if;
    pdata->requested_len = pkt_size;
    pdata->type = btype;
    pdata->buf = protocol_buffer;
    switch( pkt_pattern ) {
        case PKT_PATTERN_8BIT_ASCENDING:
            memcpy(pdata->buf, &ascending[0], pkt_size);
            break;

        case PKT_PATTERN_8BIT_DESCENDING:
            memcpy(pdata->buf, &descending[0], pkt_size);
            break;

        case PKT_PATTERN_8BIT_REPEATING:
            memcpy(pdata->buf, &repeating[0], pkt_size);
            break;

        default:
            printf("ERROR: Unknown Packet Pattern!\n");
            return -1;
    }

    printf("\nSending %d bytes of data to %s buffer....",
           pdata->requested_len, pdata->type?"Priority":"CIM");
    rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
    if( rc == 0 ) {
        if( pdata->actual_len == pdata->requested_len )
            printf("passed.\n");
        else {
            printf("FAILED.\n");
            printf("Actual bytes written %d\n", pdata->actual_len);
        }
    }
    else {
        printf("FAILED.\n");
        if(errno) {
            printf( "ioctl failed : %s\n", strerror(errno) );
        }
    }

    return 0;
}

int receive_protocol_data( int fd, unsigned int link_if, int btype, int pkt_size )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    int rx_pdata = 0;
    int rc;

    if( fpd_poll_once(fd, pevt) == 0 ) {
        /* PCI error detected */
        if( pevt->pci_error ) {
            fprintf(stderr, "\nDetected PCI errors");
            fpd_get_pci_errors( fd );
        }

        /* check if FPGA receive protocol data */
        if( link_if == FPD_LINK_IF_ID_BGND ) {
            if( btype == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->bgnd_link_if.rx_cim ) {
                    rx_pdata = 1;
                }
            }

            /* Link error detected */
            if( pevt->bgnd_link_if.error ) {
                fprintf(stderr, "\nBackground Link IF receive errors");
                fpd_get_link_errors( fd );
            }
        }
        else {
            if( btype == FPD_CIM_BUFFER ) {
                /* CIM Buffer */
                if( pevt->link_if[link_if].rx_cim ) {
                    rx_pdata = 1;
                }
            }
            else {
                /* Priority buffer */
                if( pevt->link_if[link_if].rx_priority ) {
                    rx_pdata = 1;
                }
            }

            /* error detected */
            if( pevt->link_if[link_if].error ) {
                fprintf(stderr, "\nLink IF %d receive errors", link_if);
                fpd_get_link_errors( fd );
            }
        }
    }

    if( rx_pdata ) {
        pdata->link_if = link_if;
        pdata->requested_len = pkt_size;
        pdata->type = btype;
        pdata->buf = protocol_buffer;
        printf("Reading %d bytes of data from %s buffer....",
               pdata->requested_len, pdata->type?"Priority":"CIM");
        rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
        if( rc == 0 ) {
            if( pdata->actual_len == pdata->requested_len ) {
                printf("passed.\n");
            }
            else {
                printf("FAILED.\n");
                printf("Requested %d, Got %d\n",
                       pdata->requested_len, pdata->actual_len);
            }

            /* compare contents against expected data */
            printf("Comparing contents with expected data......");
            if( verify_data(pdata->type, pdata->buf, pdata->actual_len, pkt_pattern, 1) == 0 ) {
                printf("passed.\n");
            }
            else {
                return -1;
            }
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf( "ioctl failed : %s\n", strerror(errno) );
            }
        }
    }

    return 0;
}

int send_protocol_data_continuously( int fd, unsigned int link_if, int btype, int option, int dopt )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    int rc;
    int pktlen;
    int result = 0;
    time_t start_time, current_time;
    unsigned int loop = 0;

    if( option ) {
        one_pass_only = 1;
    }
    else {
        one_pass_only = 0;
    }

    if( dopt ) {
        printf("\n");
        system("date");
    }

    pdata->link_if = link_if;
    pdata->type = btype;

    while( 1 ) {
        loop++;

        if( !debug_on ) {
            printf("\n[%d]Sending %d packets to %s buffer......",
                   loop, FPD_MAX_PACKET_SIZE, pdata->type?"PRI":"CIM");
            fflush(stdout);
        }

        for( pktlen = 1; pktlen <= FPD_MAX_PACKET_SIZE; pktlen++ ) {
            pdata->requested_len = pktlen;

            /* determine test pattern to use */
            switch( pktlen % 3 ) {
                case 0:
                    pdata->buf = &ascending[0];
                    break;
                case 1:
                    pdata->buf = &descending[0];
                    break;
                default:
                    pdata->buf = &repeating[0];
                    break;
            }

            if( debug_on ) {
                printf("\n[%d]Sending %d bytes to %s buffer....",
                       loop, pdata->requested_len, pdata->type?"Priority":"CIM");
            }
resend:
            rc = ioctl(fd, FPD_SEND_PROTOCOL_DATA, pdata);
            if( rc == 0 ) {
                if( pdata->actual_len == pdata->requested_len ) {
                    if( debug_on ) {
                        printf("passed.");
                    }

                    /* reset time of last good send */
                    time(&start_time);
                }
                else {
                    time(&current_time);
                    if(( pdata->actual_len == 0 ) && ( difftime(current_time, start_time) < RETRY_TIMEOUT )) {
                        goto resend;
                    }
                    else {
                        printf("FAILED.\n");
                        if( !debug_on ) {
                            printf("\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                   loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        printf("Actual bytes written %d\n", pdata->actual_len);
                        result = -1;
                        goto send_finish;
                    }
                }
            }
            else {
                if( errno ) {
                    if( errno == ENOSPC ) {
                        time(&current_time);
                        if( difftime(current_time, start_time) < RETRY_TIMEOUT ) {
                            goto resend;
                        }
                        else {
                            printf("FAILED.\n");
                            if( !debug_on ) {
                                printf("\n[%d]Sending %d bytes to %s buffer....FAILED\n",
                                       loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                            }
                            printf("ERROR: %s\n", strerror(errno));
                            result = -1;
                            goto send_finish;
                        }
                    }
                }
            }
        }

        if( !debug_on ) {
            printf("passed.");
            fflush(stdout);
        }

        if( one_pass_only ) {
            break;
        }
    }

send_finish:
    if( dopt ) {
        printf("\n\n");
        system("date");
    }

    return result;
}

int receive_protocol_data_continuously( int fd, unsigned int link_if, int btype, int option, int dopt )
{
    FPD_data_t protocol_data, *pdata = &protocol_data;
    FPD_event_t event, *pevt = &event;
    unsigned int timeout = POLL_TIMEOUT;
    struct pollfd fds;
    pkt_pattern_t test_pattern;
    unsigned int loop = 1;
    int pktlen;
    int rx_pdata;
    int more_data_available;
    int rc;

    if( option ) {
        one_pass_only = 1;
    }
    else {
        one_pass_only = 0;
    }

    if( dopt ) {
        printf("\n");
        system("date");
    }

    fds.fd = fd;
    fds.events = POLLIN;

    pktlen = 1;

    if( !debug_on ) {
        printf("\n[%d]Reading %d packets from %s buffer....",
               loop, FPD_MAX_PACKET_SIZE, btype?"PRI":"CIM");
        fflush(stdout);
    }
    else {
        printf("\n");
    }

    while(1) {
        more_data_available = 1;
        rx_pdata = 0;
        if( debug_on ) {
            printf("waiting for events ... ");
        }

        rc = poll(&fds, 1, timeout);
        if ((rc == 0) || (rc < 0 && errno == EINTR)) {
            if( !debug_on ) {
                printf("FAILED.\n");
                printf("[%d] (%d bytes) waiting for events ... timedout\n", loop, pktlen);
            }
            else {
                printf("timedout\n");
            }
            return -1;
        }
        else if( rc < 0 ) {
            if( !debug_on ) {
                printf("FAILED.\n");
                printf("waiting for events ... poll() returned ERROR %d\n", rc);
            }
            else {
                printf("poll() returned %d\n", rc);
            }
            return -2;
        }
        
        if( debug_on ) {
            printf("receive notifications\n");
            printf("getting events ... ");
        }

        rc = ioctl(fd, FPD_CHECK_EVENTS, pevt);

        if( debug_on ) {
            printf("done\n");
        }

        if (rc == 0) {
            /* PCI error detected */
            if( pevt->pci_error ) {
                printf("\n[%d] (%d bytes) Detected PCI errors", loop, pktlen);
                fpd_get_pci_errors( fd );
            }

            /* check if FPGA receive protocol data */
            if( link_if == FPD_LINK_IF_ID_BGND ) {
                if( btype == FPD_CIM_BUFFER ) {
                    /* CIM Buffer */
                    if( pevt->bgnd_link_if.rx_cim ) {
                        rx_pdata = 1;
                    }
                }

                /* error detected */
                if( pevt->bgnd_link_if.error ) {
                    printf("\nBackground Link IF receive errors");
                    fpd_get_link_errors( fd );
                }
            }
            else {
                if( btype == FPD_CIM_BUFFER ) {
                    /* CIM Buffer */
                    if( pevt->link_if[link_if].rx_cim ) {
                        rx_pdata = 1;
                    }
                }
                else {
                    /* Priority buffer */
                    if( pevt->link_if[link_if].rx_priority ) {
                        rx_pdata = 1;
                    }
                }

                /* error detected */
                if( pevt->link_if[link_if].error ) {
                    printf("\n[%d] (%d bytes) Link IF %d receive errors", loop, pktlen, link_if);
                    fpd_get_link_errors( fd );
                }
            }

            if( rx_pdata ) {
                pdata->link_if = link_if;
                pdata->type = btype;
                do {
                    pdata->requested_len = pktlen;
                    pdata->buf = protocol_buffer;
                    if( debug_on ) {
                        printf("[%d]Reading %d bytes of data from %s buffer....",
                               loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                    }
                    rc = ioctl(fd, FPD_RECEIVE_PROTOCOL_DATA, pdata);
                    if( rc == 0 ) {
                        if( pdata->actual_len == pdata->requested_len ) {
                            if( debug_on ) {
                                printf("passed.\n");
                            }
                        }
                        else if( pdata->actual_len == 0 ) {
                            if( debug_on ) {
                                printf("No data.\n");
                            }
                            more_data_available = 0;
                        }
                        else {
                            printf("FAILED.\n");
                            if( !debug_on ) {
                                printf("[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                                       loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                            }
                            printf("Requested %d, Got %d\n",
                                   pdata->requested_len, pdata->actual_len);
                            return -3;
                        }

                        if( pdata->actual_len > 0 ) {
                            if( debug_on ) {
                                printf("Comparing contents with expected data......");
                            }

                            /* determine test pattern used */
                            switch( pktlen % 3 ) {
                                case 0:
                                    test_pattern = PKT_PATTERN_8BIT_ASCENDING;
                                    break;
                                case 1:
                                    test_pattern = PKT_PATTERN_8BIT_DESCENDING;
                                    break;
                                default:
                                    test_pattern = PKT_PATTERN_8BIT_REPEATING;
                                    break;
                            }

                            /* compare contents against expected data */
                            if( verify_data(pdata->type, pdata->buf, pdata->actual_len, test_pattern, loop) == 0 ) {
                                if( debug_on ) {
                                    printf("passed.\n");
                                }
                            }
                            else {
                                return -4;
                            }
                        }

                        /* increment packet length */
                        if( pdata->actual_len == pdata->requested_len ) {
                            if( pktlen == FPD_MAX_PACKET_SIZE ) {
                                pktlen = 1;
                                loop++;
                                if( !debug_on ) {
                                    printf("passed.");
                                    if( !one_pass_only ) {
                                        printf("\n[%d]Reading %d packets from %s buffer....",
                                               loop, FPD_MAX_PACKET_SIZE, pdata->type?"Priority":"CIM");
                                    }
                                    fflush(stdout);
                                }

                                if( one_pass_only ) {
                                    goto receive_finish;
                                } 
                           }
                            else {
                                ++pktlen;
                            }
                        }
                    }
                    else {
                        printf("FAILED.\n");
                        if( !debug_on ) {
                            printf("[%d]Reading %d bytes of data from %s buffer....FAILED\n",
                                   loop, pdata->requested_len, pdata->type?"Priority":"CIM");
                        }
                        if(errno) {
                            printf("ioctl failed : %s\n", strerror(errno));
                        }
                        return -5;
                    }
                } while( more_data_available );
            }
        }
    }

receive_finish:
    if( dopt ) {
        printf("\n\n");
        system("date");
    }

    return 0;
}
