/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  Application to dump the Host Buffer Memory of any of the Link Interfaces
 *  of the FPGA Protocol Device Driver.
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"

#define FPD_MEMDUMP_REV        "$Revision: 1.1 $"
#define FPD_APPS_PASS       0
#define FPD_APPS_FAIL      -1

/* commands */
typedef enum
{
    CMD_DUMP_RXBUF = 1,
    CMD_DUMP_TXBUF,
} FPD_MEMDUMP_CMDS;

static int fpd_verbose = 0;
static void *host_buffer;
static int host_buffer_mmap_len = 0;

static void usage( void );
static void log( char *fmt, ... );
void msg( char *fmt, ... );
void err( char *fmt, ... );


static int init_host_module( int fd, int overall_buf_len )
{
    host_buffer_mmap_len = overall_buf_len;

    host_buffer = (void *) mmap(NULL,
                                host_buffer_mmap_len,
                                PROT_READ | PROT_WRITE,
                                MAP_SHARED,
                                fd,
                                0 );

    if( host_buffer == (void *) -1 ) {
        return -1;
    }

    return 0;
}

static void cleanup_host_module( void )
{
    munmap(host_buffer, host_buffer_mmap_len);
    return;
}

static void dump_data32( unsigned char *bufaddr, int len )
{
    unsigned int *buf = (unsigned int *)bufaddr;
    int remainder;
    int size;
    int i;

    remainder = len & 0x3;
    size = len - remainder;
    size >>= 2;

    for( i = 0; i < size; i++ ) {
        if((i % 4) == 0) {
            printf("\n%08x: ", i*4);
        }
        printf("%08x ", buf[i]);
    }

    if( remainder ) {
        size = len - remainder;
        for( i = size; i < len; i++ ) {
            printf("%02x ", bufaddr[i]);
        }
    }
    printf("\n");

    return;
}

int main( int argc, char * const argv[] )
{
    int ch;
    int fd;
    char dev[128] = "/dev/fpd";
    unsigned int io = 0;
    int link_id = -1, host_chan = -1, bufid = -1;
    unsigned int dma_page_size = 0;
    unsigned char *bufaddr;
    FPD_msg_t gmsg;
    int rc;

    while( (ch = getopt(argc, argv, "?vi:l:h:b:")) != EOF ) {
        switch( ch ) {
            case 'v':
                fpd_verbose = 1;
                break;

            case 'i':
                io = atoi(optarg);
                break;
 
            case 'l':
                link_id = atoi(optarg);
                break;

            case 'h':
                host_chan = atoi(optarg);
                break;

            case 'b':
                bufid = atoi(optarg);
                break;

            case '?':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
        log( "using %s as device node ...\n", dev );
    }

    if( link_id == -1 || host_chan == -1 || bufid == -1 ) {
        err("Please specify Link IF, Host Channel, and Buffer ID\n");
        return -1;
    }

    if( (fd = open(dev, O_RDWR)) < 0 ) {
        err("Cannot open %s : %s\n", dev, strerror(errno));
        return -2;
    }
    log( "File Descriptor %d (%s)\n", fd, dev);

    log( "testing cntl %d...\n", io);

    /* determine Link and Line Interface count */
    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        dma_page_size = gmsg.info.host_buffer_pagesize;

        /* initialize host module */
        if( init_host_module(fd, gmsg.info.host_buffer_memsize) < 0 ) {
            fprintf(stderr, "Host Module init failed\n");
            return -3;
        }
    }

    switch(io) {
        case CMD_DUMP_RXBUF:
            bufaddr = (unsigned char *)host_buffer + bufid * dma_page_size;
            dump_data32(bufaddr, dma_page_size);
            break;
        case CMD_DUMP_TXBUF:
            bufaddr = (unsigned char *)host_buffer + bufid * dma_page_size;
            dump_data32(bufaddr, dma_page_size);
            break;
        default:
            rc = -1;
    }

    if (rc < 0) {
        if(errno) {
            err( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            err( "ioctl failed : Invalid argument\n" );
        }
    }

    cleanup_host_module();
    close(fd);
	
    return 0;
}

static void usage( void )
{
    printf( "Use (" FPD_MEMDUMP_REV "):\n"
            "fpd-memdump [options] [devname]\n\n"
            "    -?         help\n"
            "    -i <cmd>   cmd\n"
            "    -l <arg>   Link Interface ID\n"
            "    -h <arg>   Host Buffer Channel\n"
            "    -b <arg>   Buffer ID\n"
            "    -v         verbose\n"
            " [devname] is /dev/fpd\n\n");
    printf( " Cmd  Arguments     Definitions\n");
    printf( " ------------------------------------------------------------\n");
    printf( " 1    link[0-4]         Dump RX Host buffer\n");
    printf( "      host[0-1]\n");
    printf( "      BufID[0-7]\n");
    printf( " 2    link[0-4]         Dump TX Host buffer\n");
    printf( "      host[0-1]\n");
    printf( "      BufID[0-7]\n\n");
}

static void log( char *fmt, ... )
{
    va_list ap;
    if( fpd_verbose ) {
        printf( "[fpd-memdump] " );
        va_start(ap,fmt);
        vprintf( fmt, ap );
        va_end(ap);
    }	
}

void msg( char *fmt, ... )
{
   va_list ap;
   printf( "[fpd-memdump] " );
   va_start(ap,fmt);
   vprintf( fmt, ap );
   va_end(ap);
}

void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-memdump] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}
