/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Test Board diagnostics
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"

#define FPD_TESTBD_REV      "$Revision: 1.2 $"
#define FPD_APPS_PASS       0
#define FPD_APPS_FAIL      -1

/* test registers */
#define REG_MIRROR          0x80
#define REG_LED             0x84
#define REG_TEST_PIN        0x88
#define REG_DIP_STATUS      0x8c

/* commands */
typedef enum
{
    CMD_GET_INFO = 1,
    CMD_GET_PCI_CFG,
    CMD_READ_REG,
    CMD_WRITE_REG,
    CMD_TEST_ACCESS,
} FPD_TESTBD_CMDS;

int fpd_verbose = 0;

static void display_pci_cfg( FPD_pcicfg_t *pci );
static int test_register_access( int fd );
void usage( void );
static void log( char *fmt, ... );
void msg( char *fmt, ... );
void err( char *fmt, ... );


int main( int argc, char * const argv[] )
{
    int ch;
    int fd;
    char dev[128] = "/dev/fpd";
    unsigned int io = 0;
    int param = -1, param2 = -1;
    FPD_msg_t gmsg;
    int rc;

    while( (ch = getopt(argc, argv, "hvi:a:b:")) != EOF ) {
        switch( ch ) {
            case 'v':
                fpd_verbose = 1;
                break;

            case 'i':
                io = atoi(optarg);
                break;

            case 'a':
                param = strtoul(optarg, NULL, 16);
                break;

            case 'b':
                param2 = strtoul(optarg, NULL, 16);
                break;

            case 'h':
            default:
                usage();
                return 1;
        }
    }

    if( optind == argc-1 ) {
        strcpy(dev, argv[optind++]);
        log( "using %s as device node ...\n", dev );
    }

    if( (fd = open(dev, O_RDWR)) < 0 ) {
        err( "Cannot open %s : %s\n", dev, strerror(errno) );
        return 1;
    }
    log( "File Descriptor %d (%s)\n", fd, dev);

    log( "testing cntl %d...\n", io);

    switch(io) {
        case CMD_GET_INFO:
            rc = ioctl(fd, FPD_GET_INFO, &gmsg);
            if (rc == 0) {
                printf("FPGA Protocol Info: Protocol Version %d.%d.%d\n"
                       "                    FPGA Version %d.%d.%d\n"
                       "                    Driver Version %d.%d.%d\n",
                       gmsg.info.protocol_version.major,
                       gmsg.info.protocol_version.minor,
                       gmsg.info.protocol_version.patch,
                       gmsg.info.fpga_version.major,
                       gmsg.info.fpga_version.minor,
                       gmsg.info.fpga_version.patch,
                       gmsg.info.driver_version.major,
                       gmsg.info.driver_version.minor,
                       gmsg.info.driver_version.patch);
            }
            break;
        case CMD_GET_PCI_CFG:
            rc = ioctl(fd, FPD_GET_PCI_CFG, &gmsg);
            if (rc == 0) {
                /* display info */
                display_pci_cfg(&gmsg.pcicfg);
            }
            break;
        case CMD_READ_REG:
            if(param >= 0) {
                gmsg.rwreg.reg = param;
                rc = ioctl(fd, FPD_READ_REG, &gmsg);
                if (rc == 0) {
                    printf("Reg %x: %08x\n", param, gmsg.rwreg.value);
                }
            }
            else {
                printf("specify register offset in -a option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_WRITE_REG:
            if(param >= 0) {
                gmsg.rwreg.reg = param;
                gmsg.rwreg.value = param2;
                rc = ioctl(fd, FPD_WRITE_REG, &gmsg);
            }
            else {
                printf("specify register offset in -a and value in -b option\n");
                rc = -EINVAL;
            }
            break;
        case CMD_TEST_ACCESS:
            rc = test_register_access(fd);
            break;

        /* DEBUGGING PURPOSES */
        case 50:
            rc = ioctl(fd, FPD_TRACE_ON);
            break;
        case 51:
            rc = ioctl(fd, FPD_TRACE_OFF);
            break;
        case 52:
            if(param >= 0) {
                gmsg.set_loglvl.level = param;
                printf("setting log level to %d\n", gmsg.set_loglvl.level);
                rc = ioctl(fd, FPD_SET_LOG_LVL, &gmsg);
            }
            else {
                printf("specify log level in -a option\n");
                rc = -EINVAL;
            }
            break;
        case 53:
            rc = ioctl(fd, FPD_GET_LOG_LVL, &gmsg);
            if (rc == 0) {
                printf("Current LOG level: %d\n", gmsg.get_loglvl.level);
            }
            break;
        case 255:
            rc = ioctl(fd, FPD_DRIVER_RESET);
            break;
        default:
            rc = -1;
    }

    if (rc < 0) {
        if(errno) {
            err( "ioctl failed : %s\n", strerror(errno) );
        }
        else {
            err( "ioctl failed : Invalid argument\n" );
        }
    }

    close(fd);
	
    return 0;
}

static void display_pci_cfg( FPD_pcicfg_t *pci )
{
    printf("PCI_VENDOR_ID          : %04x\n", pci->vendor_id);
    printf("PCI_DEVICE_ID          : %04x\n", pci->device_id);
    printf("PCI_COMMAND            : %04x\n", pci->command);
    printf("PCI_STATUS             : %04x\n", pci->status);
    printf("PCI_REVISION_ID        : %02x\n", pci->revision_id);
    printf("PCI_CLASS_PROG         : %02x\n", pci->prog_if);
    printf("PCI_CLASS_DEVICE       : %02x\n", pci->sub_class_code);
    printf("PCI_CLASS_CODE         : %02x\n", pci->base_class_code);
    printf("PCI_CACHE_LINE_SIZE    : %02x\n", pci->cache_line_size);
    printf("PCI_LATENCY_TIMER      : %02x\n", pci->latency_timer);
    printf("PCI_HEADER_TYPE        : %02x\n", pci->header_type);
    printf("PCI_BIST               : %02x\n", pci->bist);
    printf("PCI_BASE_ADDRESS_0     : %08x\n", pci->bar0);
    printf("PCI_BASE_ADDRESS_1     : %08x\n", pci->bar1);
    printf("PCI_BASE_ADDRESS_2     : %08x\n", pci->bar2);
    printf("PCI_BASE_ADDRESS_3     : %08x\n", pci->bar3);
    printf("PCI_BASE_ADDRESS_4     : %08x\n", pci->bar4);
    printf("PCI_BASE_ADDRESS_5     : %08x\n", pci->bar5);
    printf("PCI_CARDBUS_CIS        : %08x\n", pci->cardbus_cis_ptr);
    printf("PCI_SUBSYSTEM_VENDOR_ID: %04x\n", pci->subsystem_vendor_id);
    printf("PCI_SUBSYSTEM_ID       : %04x\n", pci->subsystem_id);
    printf("PCI_ROM_ADDRESS        : %08x\n", pci->rom_bar);
    printf("PCI_CAPABILITY_LIST    : %02x\n", pci->cap_ptr);
    printf("PCI_INTERRUPT_LINE     : %02x\n", pci->interrupt_line);
    printf("PCI_INTERRUPT_PIN      : %02x\n", pci->interrupt_pin);
    printf("PCI_MIN_GNT            : %02x\n", pci->min_grant);
    printf("PCI_MAX_LAT            : %02x\n", pci->max_latency);

    return;
}

static int test_register_access( int fd )
{
    FPD_rw_reg_t readmsg, writemsg;
    int idx, rc;
    unsigned int test_pattern[10] = {
        0xaaaaaaaa,
        0x55555555,
        0xcccccccc,
        0x33333333,
        0xf0f0f0f0,
        0x0f0f0f0f,
        0x00000000,
        0xffffffff,
        0xa5a5a5a5,
        0x5a5a5a5a
    };

    readmsg.reg = writemsg.reg = REG_MIRROR;

    printf("Testing register access.......");
    for( idx = 0; idx < 10; idx++ ) {
        /* write pattern */
        writemsg.value = test_pattern[idx];
        rc = ioctl(fd, FPD_WRITE_REG, &writemsg);

        /* read back pattern */
        rc = ioctl(fd, FPD_READ_REG, &readmsg);

        /* compare */
        if( rc == 0 ) {
            if( readmsg.value != test_pattern[idx] ) {
                printf("failed.\n");
                printf("ERROR: Wrote 0x%08x  Read 0x%08x",
                       test_pattern[idx], readmsg.value);
                return -2;
            }
        }
        else {
            return -1;
        }
    }
    printf("passed.\n");

    return 0;
}

void usage( void )
{
    printf( "Use (" FPD_TESTBD_REV "):\n"
            "fpd-testbd [options] [devname]\n\n"
            "    -h         help\n"
            "    -i <cmd>   ioctl cmd\n"
            "    -a <arg>   argument used by ioctl\n"
            "    -b <arg>   2nd argument used by ioctl\n"
            "    -v         verbose\n"
            " [devname] is /dev/dummy\n\n");
    printf( " Cmd  Arguments     Definitions\n");
    printf( " ------------------------------------------------------------\n");
    printf( " 1    none              get general info\n");
    printf( " 2    none              get PCI Configuration Space\n");
    printf( " 3    reg               read FPGA Protocol register\n");
    printf( " 4    reg  val          write to FPGA Protocol register\n");
    printf( " 5    none              test register access\n");
    printf( " 50   none              turn on trace\n");
    printf( " 51   none              turn off trace\n");
    printf( " 52   level [0-5]       set debug level\n");
    printf( " 53   none              get debug level\n\n");
}

static void log( char *fmt, ... )
{
    va_list ap;
    if( fpd_verbose ) {
        printf( "[fpd-testbd] " );
        va_start(ap,fmt);
        vprintf( fmt, ap );
        va_end(ap);
    }	
}

void msg( char *fmt, ... )
{
   va_list ap;
   printf( "[fpd-testbd] " );
   va_start(ap,fmt);
   vprintf( fmt, ap );
   va_end(ap);
}

void err( char *fmt, ... )
{
   va_list ap;
   fprintf( stderr, "[fpd-testbd] " );
   va_start(ap,fmt);
   vfprintf( stderr, fmt, ap );
   va_end(ap);	
}
