/*-----------------------------------------------------------------------------

    DmaMem.c

    Copyright (c) 2002, Raritan Computer, Inc.

    Allocates large blocks of contiguous memory for use as dma buffers.

    The frame grabber driver needs MEGA BYTES of contiguous memory
    Example, for 8 frame grabbers, 2 frames each at 2.5 meg per buffer = 40meg!
	
    The linux kernel kmalloc() will frequently failto allocate even 128k of
    contiguous memory.
	
    There are a couple of alternative ways to allocate the memory.
    1) To allocate at boot time. To this, the driver has to be linked to the
       kernel.
    2) Use the MEM=xx feature of the LILO boot loader to reserve memory at the 
       top of RAM. (The "allocator" module is a convinient was to do this)
    3) Various kernel patches for allocating big DMA buffers.

    The functions in here are designed to abstract any of these methods from 
    the rest of the driver.

    The current implementation is to use the MEM=xx LILO feature and the 
    allocator module to manage the reserved memory.

-----------------------------------------------------------------------------*/

#include <linux/config.h>
#include <linux/module.h>

#include <linux/kernel.h>   /* printk() */
#include <linux/slab.h>     /* kmalloc() */
#include <linux/fs.h>       /* everything... */
#include <linux/errno.h>    /* error codes */
#include <linux/types.h>    /* size_t */
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/ioctl.h>
#include <asm/io.h>
#include <asm/page.h>

#include "dmabuf.h"
#include "debug.h"

#ifdef LINUX_20
#  error "This module can't run with Linux-2.0"
#endif

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

extern unsigned long allocator_allocate_dma (unsigned long bytes, int prio);
extern int allocator_free_dma (unsigned long address);

//----------------------------------------
//				Static Data
//----------------------------------------

static void *dmaMemBase = NULL; // Base of the originally aquired memory
static int   dmaMemSize;        // Total size of the aquired memory
static DMA_MEM dmaMemFree;      // Keeps track of free DMA memory

//----------------------------------------
//				Functions
//----------------------------------------

/******************************************************************************
 *  ROUTINE:        dma_buffer_alloc()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Allocates contiguous DMA memory from reserved memory.
 *
 *  PARAMETERS:
 *
 *      bytes   -  Number of bytes to allocate
 *      pDmaMem -  Ptr to a user-allocated buffer to store the results
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
dma_buffer_alloc( int bytes, DMA_MEM *pDmaMem )
{
    pDmaMem->size = 0;
	
    if (bytes > dmaMemFree.size)
        return -ENOMEM;
	
    *pDmaMem = dmaMemFree;
    pDmaMem->size = bytes;

    dmaMemFree.physicalAddress = (void *) ((int) dmaMemFree.physicalAddress + bytes );
    dmaMemFree.busAddress = (void *) ((int) dmaMemFree.busAddress + bytes );
    dmaMemFree.virtualAddress = (void *) ((int) dmaMemFree.virtualAddress + bytes );
    dmaMemFree.size -= bytes;

    FPD_DEBUG(3, "AllocDmaMemory %08X %08X %08X %08X\n",
              (int) pDmaMem->physicalAddress, (int) pDmaMem->busAddress,
              (int) pDmaMem->virtualAddress, (int) bytes );

    return 0;
}

/******************************************************************************
 *  ROUTINE:        dma_buffer_free()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Frees contiguous DMA memory back to reserved memory.
 *
 *  PARAMETERS:
 *
 *      pDmaMem -  Ptr to a user-allocated buffer containing the memory to be
 *                 freed
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
dma_buffer_free( DMA_MEM *pDmaMem )
{
    // yeah, yeah...not implemented or needed...just looks nice
    return 0;
}

/******************************************************************************
 *  ROUTINE:        dma_buffer_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Acquires the memory from the system by any means possible.
 *
 *  PARAMETERS:
 *
 *      totalRequired -  Number of bytes to acquire for DMA buffer
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
dma_buffer_init( int totalRequired )
{
    unsigned long x;

    dmaMemBase = NULL;

    x = allocator_allocate_dma( totalRequired, GFP_KERNEL );

    if ( x == 0 )
        return -ENOMEM;

    dmaMemBase = (void *) x;
    dmaMemSize = totalRequired;

    dmaMemFree.size = totalRequired;
    dmaMemFree.physicalAddress = (void *) x;
    dmaMemFree.busAddress = dmaMemFree.physicalAddress;
    dmaMemFree.virtualAddress = ioremap( (unsigned long) dmaMemFree.physicalAddress, totalRequired);

    return (dmaMemFree.virtualAddress != NULL) ? 0 : -ENOMEM;
}

/******************************************************************************
 *  ROUTINE:        dma_buffer_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Releases all aquired memory.
 *
 *  PARAMETERS:
 *
 *      None.
 *
 *  RETURNS:
 *
 *     	None.
 *
 *****************************************************************************/
void
dma_buffer_cleanup( void )
{
    if (dmaMemBase != NULL)
        allocator_free_dma( (unsigned long) dmaMemBase );
    return;
}
