/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Diagnostics
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_ioctl.h"
#include "fpd_reg.h"
#include "fpd_intr.h"
#include "debug.h"

#define DIAG_DEBUG_LEVEL                                4

static int dbglvl = DIAG_DEBUG_LEVEL;


/******************************************************************************
 *  ROUTINE:        fpd_diag_write_data()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Writes Protocol Data to the FPGA Protocol.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      plink -  Link Interface info.
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
static int
fpd_diag_write_data( fpd_device_t *pfpd, fpd_linkif_t *plink, FPD_data_t *pdata )
{
    u32 reg;
    u32 buf_addr, status_addr, eop_addr;
    u32 avail_bytes;
    u8 *txbuf, *txbuf_wrptr;
    int cnt, len, revised_len;
    int result = FPD_DIAG_ERROR_NONE;

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            buf_addr = plink->base + LIF_TX_CIM_BUFFER_DATA;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_CIM_BUFFER_EOP;
            break;

        case FPD_PRIORITY_BUFFER:
            buf_addr = plink->base + LIF_TX_PRIORITY_BUFFER_DATA;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_PRIORITY_BUFFER_EOP;
            break;

        default:
            return FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
    }

    txbuf = (u8 *)kmalloc(FPD_PDATA_BUFFER_SIZE, GFP_KERNEL);
    if( txbuf == NULL ) {
        result = FPD_DIAG_ERROR_MEM_ALLOC_FAILED;
        goto fpd_diag_write_data_end;
    }
    txbuf_wrptr = txbuf;

    /* get available bytes for the transmitter */
    reg = FPD_READ(pfpd->remap, status_addr);
    FPD_DDEBUG(pfpd->id, dbglvl, "status ADDR = 0x%08x, reg = 0x%08x\n",
               status_addr, reg);
    avail_bytes = (reg & LIFCBSR_TX_BYTE_AVAIL) >> 13;

    /* check if there is space for the pkt + its length */
    len = pdata->requested_len;
    if( avail_bytes >= (len + sizeof(u32)) ) {
        /* deduct the last packet word from the length */
        if( len & 0x3 ) {
            revised_len = len & ~0x3;
        }
        else {
            revised_len = len - sizeof(u32);
        }

        FPD_DDEBUG(pfpd->id, dbglvl, "CIM TX Len %d Rev Len %d\n",
                   len, revised_len);

        /* write pkt length first */
        FPD_WRITE(pfpd->remap, buf_addr, len);

        if( copy_from_user((void *)txbuf_wrptr, (const void *)pdata->buf, len) != 0 )
        {
            FPD_ERROR("CIM Data Write copy_from_user failed\n");
            result = FPD_DIAG_ERROR_USER_MEM_FAULT;
            goto fpd_diag_write_data_end;
        }

        for(cnt = 0; cnt < revised_len; cnt+=sizeof(u32)) {
            FPD_DDEBUG(pfpd->id, dbglvl, "CIM TX Buffer Data %08x\n",
                       swap32(*(u32 *)txbuf_wrptr));

            /* write data pkt 32-bit at a time */
            FPD_WRITE(pfpd->remap, buf_addr, swap32(*(u32 *)txbuf_wrptr));
            txbuf_wrptr += sizeof(u32);
        }

        FPD_DDEBUG(pfpd->id, dbglvl, "CIM TX EOP Data %08x\n", *(u32 *)txbuf_wrptr);

        /* write last 32-bit data to TX EOP Register */
        FPD_WRITE(pfpd->remap, eop_addr, swap32(*(u32 *)txbuf_wrptr));
        txbuf_wrptr += sizeof(u32);
    }
    else {
        FPD_INFO("No space in FPGA for TX CIM Data");
        result = FPD_DIAG_ERROR_FPGA_BUFFER_NO_SPACE;
    }

 fpd_diag_write_data_end:
    kfree(txbuf);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_diag_test_crc_error()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Verifies the behavior of the FPGA Protocol device once a CRC error
 *      is detected.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_diag_test_crc_error( fpd_device_t *pfpd, FPD_data_t *pdata )
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 tx_timeout_bit;
    u32 tx_buf_bit;
    u32 status_addr;
    u32 avail_bytes;
    u32 len_mask;
    u8  max_retry;
    u8  tx_crc_cnt;
    int error_on = 0;
    int fpga_pkt_len;
    int result = FPD_DIAG_ERROR_NONE;

    /* disable interrupts */
    fpd_disable_interrupt(pfpd);

    /* set FORCE_CRC_ERR in Link Control register */
    if( pdata->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pdata->link_if];
    }

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            tx_timeout_bit = LIFISR_TX_CIM_TIMEOUT;
            tx_buf_bit = LIFISR_TX_CIM_BUF;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;

        case FPD_PRIORITY_BUFFER:
            tx_timeout_bit = LIFISR_TX_PRI_TIMEOUT;
            tx_buf_bit = LIFISR_TX_PRI_BUF;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;

        default:
            result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
            goto test_crc_error_cleanup;
    }

    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg |= LIFCR_FORCE_CRC_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 1;

    /* write packet */
    result = fpd_diag_write_data(pfpd, plink, pdata);
    if( result != 0 ) {
        goto test_crc_error_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_TIMEOUT bit is set */
    if(!( reg & tx_timeout_bit )) {
        result = FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET;
        goto test_crc_error_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is (512 - (packet length + 4)).
     */
    len_mask = pdata->requested_len & 0x03;
    switch( len_mask ) {
        case 1:
            fpga_pkt_len = pdata->requested_len + 3;
            break;
        case 2:
            fpga_pkt_len = pdata->requested_len + 2;
            break;
        case 3:
            fpga_pkt_len = pdata->requested_len + 1;
            break;
        default:
            fpga_pkt_len = pdata->requested_len;
            break;
    }
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != (512 - (fpga_pkt_len + sizeof(u32))) ) {
        result = FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL;
        goto test_crc_error_cleanup;
    }

    /* clear FORCE_CRC_ERR in Link Control register */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_FORCE_CRC_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 0;

    /* put a 1ms delay */
    mdelay(1);

    /*
     * read TX STATISTICS register, TX_CRC_CNT should be 8 and 
     * MAX_RETRY should be 8
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_crc_cnt = (reg & 0x000000ff);
    if( tx_crc_cnt != 255 ) {
        result = FPD_DIAG_ERROR_WRONG_TX_CRC_CNT;
        goto test_crc_error_cleanup;
    }

    if( max_retry != 8 ) {
        result = FPD_DIAG_ERROR_WRONG_MAX_RETRY;
        goto test_crc_error_cleanup;
    }

    /* read STATISTICS 3 register, TX_CRC_CNT and MAX_RETRY should be 0 now */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_crc_cnt = (reg & 0x000000ff);
    if( tx_crc_cnt != 0 ) {
        result = FPD_DIAG_ERROR_TX_CRC_CNT_NOT_CLEARED;
        goto test_crc_error_cleanup;
    }

    if( max_retry != 0 ) {
        result = FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED;
        goto test_crc_error_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_BUF bit is set */
    if(!( reg & tx_buf_bit )) {
        result = FPD_DIAG_ERROR_TX_BUF_NOT_SET;
        goto test_crc_error_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is back to 512.
     */
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != 512 ) {
        result = FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512;
        goto test_crc_error_cleanup;
    }

 test_crc_error_cleanup:
    if( error_on ) {
        /* clear FORCE_CRC_ERR in Link Control register */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        reg &= ~LIFCR_FORCE_CRC_ERR;
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);
    }

    /* enable interrupts */
    fpd_enable_interrupt(pfpd);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_diag_test_protocol_error()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Verifies the behavior of the FPGA Protocol device once a Protocol error
 *      is detected.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_diag_test_protocol_error( fpd_device_t *pfpd, FPD_data_t *pdata )
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 tx_timeout_bit;
    u32 tx_buf_bit;
    u32 status_addr;
    u32 avail_bytes;
    u32 len_mask;
    u8  max_retry;
    u8  tx_prot_err_cnt;
    int error_on = 0;
    int fpga_pkt_len;
    int result = FPD_DIAG_ERROR_NONE;

    /* disable interrupts */
    fpd_disable_interrupt(pfpd);

    /* set FORCE_PROT_ERR in Link Control register */
    if( pdata->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pdata->link_if];
    }

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            tx_timeout_bit = LIFISR_TX_CIM_TIMEOUT;
            tx_buf_bit = LIFISR_TX_CIM_BUF;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;

        case FPD_PRIORITY_BUFFER:
            tx_timeout_bit = LIFISR_TX_PRI_TIMEOUT;
            tx_buf_bit = LIFISR_TX_PRI_BUF;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;

        default:
            result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
            goto test_protocol_error_cleanup;
    }

    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg |= LIFCR_FORCE_PROT_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 1;

    /* write packet */
    result = fpd_diag_write_data(pfpd, plink, pdata);
    if( result != 0 ) {
        goto test_protocol_error_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_TIMEOUT bit is set */
    if(!( reg & tx_timeout_bit )) {
        result = FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET;
        goto test_protocol_error_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is (512 - packet length).
     */
    len_mask = pdata->requested_len & 0x03;
    switch( len_mask ) {
        case 1:
            fpga_pkt_len = pdata->requested_len + 3;
            break;
        case 2:
            fpga_pkt_len = pdata->requested_len + 2;
            break;
        case 3:
            fpga_pkt_len = pdata->requested_len + 1;
            break;
        default:
            fpga_pkt_len = pdata->requested_len;
            break;
    }
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != (512 - (fpga_pkt_len + sizeof(u32))) ) {
        result = FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL;
        goto test_protocol_error_cleanup;
    }

    /* clear FORCE_PROT_ERR in Link Control register */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_FORCE_PROT_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 0;

    /* put a 1ms delay */
    mdelay(1);

    /*
     * read TX STATISTICS register, TX_PROT_ERR_CNT should be 8 and 
     * MAX_RETRY should be 8
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_prot_err_cnt = (reg & 0x0000ff00) >> 8;
    if( tx_prot_err_cnt != 255 ) {
        result = FPD_DIAG_ERROR_WRONG_TX_PROT_ERR_CNT;
        goto test_protocol_error_cleanup;
    }

    if( max_retry != 8 ) {
        result = FPD_DIAG_ERROR_WRONG_MAX_RETRY;
        goto test_protocol_error_cleanup;
    }

    /* read STATISTICS 3 register, TX_PROT_ERR_CNT and MAX_RETRY should be 0 now */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_prot_err_cnt = (reg & 0x0000ff00) >> 8;
    if( tx_prot_err_cnt != 0 ) {
        result = FPD_DIAG_ERROR_TX_PROT_ERR_CNT_NOT_CLEARED;
        goto test_protocol_error_cleanup;
    }

    if( max_retry != 0 ) {
        result = FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED;
        goto test_protocol_error_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_BUF bit is set */
    if(!( reg & tx_buf_bit )) {
        result = FPD_DIAG_ERROR_TX_BUF_NOT_SET;
        goto test_protocol_error_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is back to 512.
     */
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != 512 ) {
        result = FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512;
        goto test_protocol_error_cleanup;
    }

 test_protocol_error_cleanup:
    if( error_on ) {
        /* clear FORCE_PROT_ERR in Link Control register */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        reg &= ~LIFCR_FORCE_PROT_ERR;
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);
    }

    /* enable interrupts */
    fpd_enable_interrupt(pfpd);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_diag_test_invalid_pkt()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Verifies the behavior of the FPGA Protocol device once an invalid
 *      packet is detected.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_diag_test_invalid_pkt( fpd_device_t *pfpd, FPD_data_t *pdata )
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 tx_timeout_bit;
    u32 tx_buf_bit;
    u32 status_addr;
    u32 avail_bytes;
    u32 len_mask;
    u8  max_retry;
    u8  rx_timeout_cnt;
    int error_on = 0;
    int fpga_pkt_len;
    int result = FPD_DIAG_ERROR_NONE;

    /* disable interrupts */
    fpd_disable_interrupt(pfpd);

    /* set FORCE_INV_PKT_ERR in Link Control register */
    if( pdata->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pdata->link_if];
    }

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            tx_timeout_bit = LIFISR_TX_CIM_TIMEOUT;
            tx_buf_bit = LIFISR_TX_CIM_BUF;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;

        case FPD_PRIORITY_BUFFER:
            tx_timeout_bit = LIFISR_TX_PRI_TIMEOUT;
            tx_buf_bit = LIFISR_TX_PRI_BUF;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;

        default:
            result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
            goto test_invalid_pkt_cleanup;
    }

    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg |= LIFCR_FORCE_INV_PKT_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 1;

    /* write packet */
    result = fpd_diag_write_data(pfpd, plink, pdata);
    if( result != 0 ) {
        goto test_invalid_pkt_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_TIMEOUT bit is set */
    if(!( reg & tx_timeout_bit )) {
        result = FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET;
        goto test_invalid_pkt_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is (512 - packet length).
     */
    len_mask = pdata->requested_len & 0x03;
    switch( len_mask ) {
        case 1:
            fpga_pkt_len = pdata->requested_len + 3;
            break;
        case 2:
            fpga_pkt_len = pdata->requested_len + 2;
            break;
        case 3:
            fpga_pkt_len = pdata->requested_len + 1;
            break;
        default:
            fpga_pkt_len = pdata->requested_len;
            break;
    }
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != (512 - (fpga_pkt_len + sizeof(u32))) ) {
        result = FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL;
        goto test_invalid_pkt_cleanup;
    }

    /* clear FORCE_INV_PKT_ERR in Link Control register */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_FORCE_INV_PKT_ERR;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    error_on = 0;

    /* put a 1ms delay */
    mdelay(1);

    /*
     * read RX STATISTICS2 register, RX_TIMEOUT_CNT should be 255
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_RX_STATISTICS2);
    FPD_DDEBUG(pfpd->id, dbglvl, "RXSTATS2 = 0x%08x\n", reg);
    rx_timeout_cnt = (reg & 0x0000ff00) >> 8;
    if( rx_timeout_cnt != 255 ) {
        result = FPD_DIAG_ERROR_WRONG_RX_TIMEOUT_CNT;
        goto test_invalid_pkt_cleanup;
    }

    /*
     * read RX STATISTICS2 register, RX_TIMEOUT_CNT should now be 0
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_RX_STATISTICS2);
    FPD_DDEBUG(pfpd->id, dbglvl, "RXSTATS2 = 0x%08x\n", reg);
    rx_timeout_cnt = (reg & 0x0000ff00) >> 8;
    if( rx_timeout_cnt != 0 ) {
        result = FPD_DIAG_ERROR_RX_TIMEOUT_CNT_NOT_CLEARED;
        goto test_invalid_pkt_cleanup;
    }

    /*
     * read TX STATISTICS register, MAX_RETRY should be 8
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    if( max_retry != 8 ) {
        result = FPD_DIAG_ERROR_WRONG_MAX_RETRY;
        goto test_invalid_pkt_cleanup;
    }

    /* read TX STATISTICS register, MAX_RETRY should be 0 now */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    if( max_retry != 0 ) {
        result = FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED;
        goto test_invalid_pkt_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_BUF bit is set */
    if(!( reg & tx_buf_bit )) {
        result = FPD_DIAG_ERROR_TX_BUF_NOT_SET;
        goto test_invalid_pkt_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is back to 512.
     */
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != 512 ) {
        result = FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512;
        goto test_invalid_pkt_cleanup;
    }

 test_invalid_pkt_cleanup:
    if( error_on ) {
        /* clear FORCE_INV_PKT_ERR in Link Control register */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        reg &= ~LIFCR_FORCE_INV_PKT_ERR;
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);
    }

    /* enable interrupts */
    fpd_enable_interrupt(pfpd);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_diag_test_buffer_full_txside()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Verifies the behavior of the FPGA Protocol device once a CRC error
 *      is detected.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_diag_test_buffer_full_txside( fpd_device_t *pfpd, FPD_data_t *pdata )
{
    fpd_linkif_t *plink;
    u32 reg;
    u32 tx_timeout_bit;
    u32 tx_buf_bit;
    u32 status_addr;
    u32 avail_bytes;
    u32 len_mask;
    u8  max_retry;
    u8  tx_buf_full_cnt;
    int fpga_pkt_len;
    int result = FPD_DIAG_ERROR_NONE;

    /* disable interrupts */
    fpd_disable_interrupt(pfpd);

    /* set FORCE_CRC_ERR in Link Control register */
    if( pdata->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pdata->link_if];
    }

    switch( pdata->type ) {
        case FPD_CIM_BUFFER:
            tx_timeout_bit = LIFISR_TX_CIM_TIMEOUT;
            tx_buf_bit = LIFISR_TX_CIM_BUF;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;

        case FPD_PRIORITY_BUFFER:
            tx_timeout_bit = LIFISR_TX_PRI_TIMEOUT;
            tx_buf_bit = LIFISR_TX_PRI_BUF;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;

        default:
            result = FPD_DIAG_ERROR_INVALID_BUFFER_TYPE;
            goto test_buffer_full_cleanup;
    }

    /* write packet */
    result = fpd_diag_write_data(pfpd, plink, pdata);
    if( result != 0 ) {
        goto test_buffer_full_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_BUF bit is set */
    if(!( reg & tx_buf_bit )) {
        result = FPD_DIAG_ERROR_TX_BUF_NOT_SET;
        goto test_buffer_full_cleanup;
    }

    /* write the same packet again */
    result = fpd_diag_write_data(pfpd, plink, pdata);
    if( result != 0 ) {
        goto test_buffer_full_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        schedule();
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
    } while( reg == 0 );
    FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

    /* clear the interrupt */
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);

    /* check that TX_xxx_TIMEOUT bit is set */
    if(!( reg & tx_timeout_bit )) {
        result = FPD_DIAG_ERROR_TX_TIMEOUT_NOT_SET;
        goto test_buffer_full_cleanup;
    }

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is (512 - (packet length + 4)).
     */
    len_mask = pdata->requested_len & 0x03;
    switch( len_mask ) {
        case 1:
            fpga_pkt_len = pdata->requested_len + 3;
            break;
        case 2:
            fpga_pkt_len = pdata->requested_len + 2;
            break;
        case 3:
            fpga_pkt_len = pdata->requested_len + 1;
            break;
        default:
            fpga_pkt_len = pdata->requested_len;
            break;
    }
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != (512 - (fpga_pkt_len + sizeof(u32))) ) {
        result = FPD_DIAG_ERROR_INCORRECT_TX_BYTE_AVAIL;
        goto test_buffer_full_cleanup;
    }

    /*
     * read TX STATISTICS register, TX_BUF_FULL_CNT should be 255 and 
     * MAX_RETRY should be 8
     */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_buf_full_cnt = (reg & 0x00ff0000) >> 16;
    if( tx_buf_full_cnt != 255 ) {
        result = FPD_DIAG_ERROR_WRONG_TX_BUF_FULL_CNT;
        goto test_buffer_full_cleanup;
    }

    if( max_retry != 8 ) {
        result = FPD_DIAG_ERROR_WRONG_MAX_RETRY;
        goto test_buffer_full_cleanup;
    }

    /* wait for non-zero value of Link Interrupt Status register */
    do {
        do {
            schedule();
            reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS);
        } while( reg == 0 );
        FPD_DDEBUG(pfpd->id, dbglvl, "LIFISR = 0x%08x\n", reg);

        /* clear the interrupt */
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_INTERRUPT_STATUS, reg);
    } while(!( reg & tx_buf_bit ));

    /*
     * read the TX CIM/Priority Buffer Status register and check that
     * TX_BYTE_AVAIL is back to 512.
     */
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_TX_BYTE_AVAIL) >> 13);
    FPD_DDEBUG(pfpd->id, dbglvl, "avail_bytes = %d\n", avail_bytes);
    if( avail_bytes != 512 ) {
        result = FPD_DIAG_ERROR_TX_BYTE_AVAIL_NOT_512;
        goto test_buffer_full_cleanup;
    }

    /* read TX STATISTICS register to clear it */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);

    /* read TX STATISTICS register, TX_CRC_CNT and MAX_RETRY should be 0 now */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_TX_STATISTICS);
    FPD_DDEBUG(pfpd->id, dbglvl, "TXSTATS = 0x%08x\n", reg);
    max_retry = (reg & 0x0f000000) >> 24;
    tx_buf_full_cnt = (reg & 0x00ff0000) >> 16;
    if( tx_buf_full_cnt != 0 ) {
        result = FPD_DIAG_ERROR_TX_BUF_FULL_CNT_NOT_CLEARED;
        goto test_buffer_full_cleanup;
    }

    if( max_retry != 0 ) {
        result = FPD_DIAG_ERROR_MAX_RETRY_NOT_CLEARED;
        goto test_buffer_full_cleanup;
    }

 test_buffer_full_cleanup:
    /* enable interrupts */
    fpd_enable_interrupt(pfpd);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_diag_test_buffer_full_rxside()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Verifies the behavior of the FPGA Protocol device once an invalid
 *      packet is detected.
 *
 *  PARAMETERS:
 *
 *      pfpd  -  FPD device info
 *      pdata -  Ptr to a user-allocated buffer containing the protocol data
 *               that will be written to the CIM or Priority buffer.
 *
 *  RETURNS:
 *
 *     	0 in case of success, negative errno value if fails.
 *
 *****************************************************************************/
int
fpd_diag_test_buffer_full_rxside( fpd_device_t *pfpd, int enable )
{
    int result = FPD_DIAG_ERROR_NONE;

    if( enable ) {
        /* disable interrupts */
        fpd_disable_interrupt(pfpd);
        FPD_INFO("DISABLING interrupts\n");
    }
    else {
        /* enable interrupts */
        fpd_enable_interrupt(pfpd);
        FPD_INFO("Enabling interrupts\n");
    }

    return result;
}
