/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Host Buffer Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>
#include <asm/io.h>

#include "fpd.h"
#include "fpd_reg.h"
#include "fpd_host.h"
#include "debug.h"
#include "dmabuf.h"

static u32 fpd_host_cnt = 0;


static int
insert_buf_into_process_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_LIMBO:
            break;
        case FPD_HOST_BUFFER_STATE_PROCESSING:
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
        case FPD_HOST_BUFFER_STATE_FREE:
        default:
            return -1;
    }

    list_add_tail( &phostbuf->list, &pdir->process_buffer.list );
    ++pdir->process_buffer.count;
    phostbuf->state = FPD_HOST_BUFFER_STATE_PROCESSING;

    return 0;
}

static int
insert_buf_into_user_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_LIMBO:
            break;
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
        case FPD_HOST_BUFFER_STATE_PROCESSING:
        case FPD_HOST_BUFFER_STATE_FREE:
        default:
            return -1;
    }

    list_add_tail( &phostbuf->list, &pdir->user_buffer.list );
    ++pdir->user_buffer.count;
    phostbuf->state = FPD_HOST_BUFFER_STATE_USER_BUSY;

    return 0;
}

static int
insert_buf_into_free_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_LIMBO:
            break;
        case FPD_HOST_BUFFER_STATE_PROCESSING:
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
        case FPD_HOST_BUFFER_STATE_FREE:
        default:
            return -1;
    }

    list_add_tail( &phostbuf->list, &pdir->free_buffer.list );
    ++pdir->free_buffer.count;
    phostbuf->state = FPD_HOST_BUFFER_STATE_FREE;

    return 0;
}

static int
extract_buf_from_process_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_PROCESSING:
            break;
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
        case FPD_HOST_BUFFER_STATE_FREE:
        case FPD_HOST_BUFFER_STATE_LIMBO:
        default:
            return -1;
    }
   
    list_del_init( &phostbuf->list );
    phostbuf->state = FPD_HOST_BUFFER_STATE_LIMBO;
    if( pdir->process_buffer.count > 0 ) {
        --pdir->process_buffer.count;
    }
    else {
        FPD_ERROR("Decrementing process list count of ZERO!\n");
    }

    return 0;
}

static int
extract_buf_from_user_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
            break;
        case FPD_HOST_BUFFER_STATE_PROCESSING:
        case FPD_HOST_BUFFER_STATE_FREE:
        case FPD_HOST_BUFFER_STATE_LIMBO:
        default:
            return -1;
    }
   
    list_del_init( &phostbuf->list );
    phostbuf->state = FPD_HOST_BUFFER_STATE_LIMBO;
    if( pdir->user_buffer.count > 0 ) {
        --pdir->user_buffer.count;
    }
    else {
        FPD_ERROR("Decrementing user list count of ZERO!\n");
    }

    return 0;
}

static int
extract_buf_from_free_list( fpd_host_txrx_t *pdir, fpd_host_buf_t *phostbuf )
{
    switch( phostbuf->state ) {
        case FPD_HOST_BUFFER_STATE_FREE:
            break;
        case FPD_HOST_BUFFER_STATE_PROCESSING:
        case FPD_HOST_BUFFER_STATE_USER_BUSY:
        case FPD_HOST_BUFFER_STATE_LIMBO:
        default:
            return -1;
    }
   
    list_del_init( &phostbuf->list );
    phostbuf->state = FPD_HOST_BUFFER_STATE_LIMBO;
    if( pdir->free_buffer.count > 0 ) {
        --pdir->free_buffer.count;
    }
    else {
        FPD_ERROR("Decrementing free list count of ZERO!\n");
    }

    return 0;
}


/******************************************************************************
 *  ROUTINE:        fpd_host_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the Host Buffer Interface
 *
 *  PARAMETERS:
 *
 *      phost     Pointer to the Host Interface structure.
 *      link_id   Link Interface ID.
 *      parg      Pointer to data to be stored in private member for later
 *                access 
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_host_init( fpd_host_t *phost, u32 link_if, u32 host_cnt, void *parg )
{
    fpd_host_txrx_t *txhost, *rxhost;
    fpd_host_chan_t *pchan;
    int i;
    fpd_host_buf_t *phostbuf;
    int j;
    int rc;

    phost->channel = kmalloc(host_cnt * sizeof(fpd_host_chan_t *), GFP_KERNEL);
    if( phost->channel == NULL ) {
        FPD_ERROR("Failed to allocate Link IF Host Channel\n");
        return -ENOMEM;
    }

    for( i = 0; i < host_cnt; i++ ) {
        phost->channel[i] = kmalloc(sizeof(fpd_host_chan_t), GFP_KERNEL);
        if( phost->channel[i] == NULL ) {
            FPD_ERROR("Failed to allocate Link IF Host Channel %d\n", i);
            return -ENOMEM;
        }
        pchan = phost->channel[i];
        txhost = &pchan->tx;
        rxhost = &pchan->rx;

        /* allocate tx and rx buffers  */
        rc = dma_buffer_alloc(FPD_HOST_DMA_BUFFER_SIZE, &txhost->dmabuf);
        if( rc < 0 ) {
            FPD_ERROR("Failed to allocate Link IF Host TXbuf\n");
            kfree(phost->channel[i]);
            return -ENOMEM;
        }

        rc = dma_buffer_alloc(FPD_HOST_DMA_BUFFER_SIZE, &rxhost->dmabuf);
        if( rc < 0 ) {
            FPD_ERROR("Failed to allocate Link IF Host RXbuf\n");
            dma_buffer_free(&txhost->dmabuf);
            kfree(phost->channel[i]);
            return -ENOMEM;
        }

        memset(txhost->dmabuf.virtualAddress, 0x00, FPD_HOST_DMA_BUFFER_SIZE);
        memset(rxhost->dmabuf.virtualAddress, 0x00, FPD_HOST_DMA_BUFFER_SIZE);

        pchan->function = FPD_HOST_FUNCTION_NONE;
        pchan->dma_size = 512;
        pchan->rx_pkt_size_notification = 512;
        pchan->rx_bdnum = 0;

        INIT_LIST_HEAD(&txhost->process_buffer.list);
        txhost->process_buffer.count = 0;
        INIT_LIST_HEAD(&txhost->user_buffer.list);
        txhost->user_buffer.count = 0;
        INIT_LIST_HEAD(&txhost->free_buffer.list);
        txhost->free_buffer.count  = 0;
        fpd_lock_init(&txhost->lock, 0);

        INIT_LIST_HEAD(&rxhost->process_buffer.list);
        rxhost->process_buffer.count = 0;
        INIT_LIST_HEAD(&rxhost->user_buffer.list);
        rxhost->user_buffer.count = 0;
        INIT_LIST_HEAD(&rxhost->free_buffer.list);
        rxhost->free_buffer.count  = 0;
        fpd_lock_init(&rxhost->lock, 0);

        for( j = 0; j < FPD_HOST_BUFFER_CNT; j++ ) {
            phostbuf = &txhost->hostbuf[j];
            phostbuf->id = j + (FPD_HOST_BUFFER_CNT * 2 * i) + (link_if * 2 * host_cnt * FPD_HOST_BUFFER_CNT);
            phostbuf->state = FPD_HOST_BUFFER_STATE_LIMBO;
            phostbuf->phys_addr = txhost->dmabuf.physicalAddress + (j * FPD_HOST_BUFFER_SIZE);
            phostbuf->virtual_addr = txhost->dmabuf.virtualAddress + (j * FPD_HOST_BUFFER_SIZE);
            INIT_LIST_HEAD(&phostbuf->list);
            if( insert_buf_into_free_list(txhost, &txhost->hostbuf[j]) != 0 ) {
                FPD_ERROR("Failed to insert TXBuffer %d to free list\n", phostbuf->id);
            }

#ifndef NONCACHEABLE_RXBDTBL
            /* not use in TX */
            phostbuf->rx_bdtbl = NULL;
#endif
            phostbuf->startbd = phostbuf->endbd = 0;

            /*
            FPD_INFO("Link%d Host%d TX Buf%d ID=%d PA=%p VA=%p\n",
                     link_if, i, j, phostbuf->id, phostbuf->phys_addr,
                     phostbuf->virtual_addr);
            */

            phostbuf = &rxhost->hostbuf[j];
            phostbuf->id = j + (FPD_HOST_BUFFER_CNT * ((2 * i) + 1)) + (link_if * 2 * host_cnt * FPD_HOST_BUFFER_CNT);
            phostbuf->state = FPD_HOST_BUFFER_STATE_LIMBO;
            phostbuf->phys_addr = rxhost->dmabuf.physicalAddress + (j * FPD_HOST_BUFFER_SIZE);
            phostbuf->virtual_addr = rxhost->dmabuf.virtualAddress + (j * FPD_HOST_BUFFER_SIZE);
#ifndef NONCACHEABLE_RXBDTBL
            phostbuf->rx_bdtbl = (u32 *)kmalloc(FPD_HOST_MAX_RX_BD_TABLE_SIZE, GFP_KERNEL);
            if( !phostbuf->rx_bdtbl ) {
                FPD_ERROR("LinkIF Rx Host buffer pointer table allocation failed\n");
                dma_buffer_free(&txhost->dmabuf);
                dma_buffer_free(&rxhost->dmabuf);
                return -ENOMEM;
            }
            memset(phostbuf->rx_bdtbl, 0, FPD_HOST_MAX_RX_BD_TABLE_SIZE);
            phostbuf->startbd = phostbuf->endbd = 0;
#endif /* NONCACHEABLE_RXBDTBL */
            INIT_LIST_HEAD(&phostbuf->list);
            if( insert_buf_into_free_list(rxhost, &rxhost->hostbuf[j]) != 0 ) {
                FPD_ERROR("Failed to insert RXBuffer %d to free list\n", phostbuf->id);
            }

            /*
            FPD_INFO("Link%d Host%d RX Buf%d ID=%d PA=%p VA=%p\n",
                     link_if, i, j, phostbuf->id, phostbuf->phys_addr,
                     phostbuf->virtual_addr);
            */
        }

        txhost->current_rxbuf = NULL;
        txhost->len = 0;
        txhost->buffer_empty = 0;
        rxhost->current_rxbuf = NULL;
        rxhost->buffer_empty = 0;

        /*
        FPD_INFO("Link %d, HOST TXBuf=%p, RXBuf=%p\n", link_if,
                 txhost->dmabuf.virtualAddress, rxhost->dmabuf.virtualAddress);
        */
    }

    phost->link_if = link_if;
    phost->private = parg;
    fpd_host_cnt = host_cnt;

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_init_rxbdtbl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the Host Buffer's Receive Buffer Descriptor Table.
 *
 *  PARAMETERS:
 *
 *      phost      Pointer to the Host Interface structure.
 *      host_chan  Host Buffer channel.
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
fpd_host_init_rxbdtbl( fpd_host_t *phost, u32 host_chan )
{
#ifdef NONCACHEABLE_RXBDTBL
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    int rc;
    int j;

    for( j = 0; j < FPD_HOST_BUFFER_CNT; j++ ) {
        /* RX */
        phostbuf = &rxhost->hostbuf[j];
        /* allocate tx and rx buffers  */
        rc = dma_buffer_alloc(FPD_HOST_MAX_RX_BD_TABLE_SIZE, &phostbuf->rx_bdtbl);
        if( rc < 0 ) {
            FPD_ERROR("LinkIF Rx Host buffer (%d) pointer table allocation failed\n", j);
            return -ENOMEM;
        }
        FPD_DEBUG(2, "host%d - %d PA=%p VA=%p\n", host_chan, j,
                  phostbuf->rx_bdtbl.physicalAddress,
                  phostbuf->rx_bdtbl.virtualAddress);
        memset(phostbuf->rx_bdtbl.virtualAddress, 0, FPD_HOST_MAX_RX_BD_TABLE_SIZE);
        phostbuf->startbd = phostbuf->endbd = 0;
    }
#endif /* NONCACHEABLE_RXBDTBL */
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the Host Buffer interface module.
 *
 *  PARAMETERS:
 *
 *      phost    Pointer to the host buffer structure.
 *
 *  RETURNS:
 *
 *     	0       Cleanup is successful.
 *     -1       Cleanup failed.
 *
 *****************************************************************************/
int
fpd_host_cleanup( fpd_host_t *phost )
{
    fpd_host_chan_t *pchan;
    fpd_host_txrx_t *txhost, *rxhost;
    fpd_host_buf_t *phostbuf;
    int i, j;

    for( i = 0; i < fpd_host_cnt; i++ ) {
        pchan = phost->channel[i];
        txhost = &pchan->tx;
        rxhost = &pchan->rx;

        dma_buffer_free(&txhost->dmabuf);
        dma_buffer_free(&rxhost->dmabuf);
        for( j = 0; j < FPD_HOST_BUFFER_CNT; j++ ) {
            phostbuf = &rxhost->hostbuf[j];
#ifndef NONCACHEABLE_RXBDTBL
            kfree(phostbuf->rx_bdtbl);
#endif
        }

        kfree(phost->channel[i]);
    }

    kfree(phost->channel);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_cleanup_rxbdtbl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the Host Buffer's Receive Buffer Descriptor Table.
 *
 *  PARAMETERS:
 *
 *      phost      Pointer to the Host Interface structure.
 *      host_chan  Host Buffer channel.
 *
 *  RETURNS:
 *
 *     	0       Cleanup is successful.
 *     -1       Cleanup failed.
 *
 *****************************************************************************/
int
fpd_host_cleanup_rxbdtbl( fpd_host_t *phost, u32 host_chan )
{
#ifdef NONCACHEABLE_RXBDTBL
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    int j;

    for( j = 0; j < FPD_HOST_BUFFER_CNT; j++ ) {
        /* RX */
        phostbuf = &rxhost->hostbuf[j];
        dma_buffer_free(&phostbuf->rx_bdtbl);
    }
#endif /* NONCACHEABLE_RXBDTBL */
    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_reset()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Resets the Host Data interface module.
 *
 *  PARAMETERS:
 *
 *      phost       Pointer to the host data structure.
 *      host_chan   Host Buffer channel.
 *
 *  RETURNS:
 *
 *     	0       Reset is successful.
 *     -errno   Reset failed.
 *
 *****************************************************************************/
int
fpd_host_reset( fpd_host_t *phost, int host_chan )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_linkif_t *plink = pfpd->link[phost->link_if];
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_txrx_t *txhost = &pchan->tx;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    u32 tx_ctrl, tx_addr;
    u32 rxbd_addr, rxctl_addr;
    u32 host_reset_bit;
    u32 reg;

    switch( host_chan ) {
        case 0:
            host_reset_bit = LIFCR_HOST_BUF0_RST;
            tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV0;
            tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV0;
            rxbd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV0;
            rxctl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV0;
            break;
        case 1:
            host_reset_bit = LIFCR_HOST_BUF1_RST;
            tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV1;
            tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV1;
            rxbd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV1;
            rxctl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV1;
            break;
        default:
            return -EINVAL;
    }

    /* reset the Link IF */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg |= host_reset_bit;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    /* wait for any pending DMA PCI xactions */
    while(FPD_READ(pfpd->remap, tx_ctrl) & LIFDTCR_DMA_START);
    while(FPD_READ(pfpd->remap, rxctl_addr) & LIFDRCR_DMA_START);

    /* reset TX and RX registers */
    FPD_WRITE(pfpd->remap, tx_addr, 0);
    FPD_WRITE(pfpd->remap, tx_ctrl, 0);
    FPD_WRITE(pfpd->remap, rxbd_addr, 0);
    FPD_WRITE(pfpd->remap, rxctl_addr, 0);

    /* re-initialize host data */
    pchan->function = FPD_HOST_FUNCTION_NONE;
    pchan->dma_size = 512;
    pchan->rx_pkt_size_notification = 512;
    pchan->rx_bdnum = 0;

    fpd_lock_acquire(&txhost->lock);
    txhost->current_rxbuf = NULL;
    txhost->len = 0;
    txhost->buffer_empty = 0;
    while( !list_empty(&txhost->process_buffer.list) ) {
        buf_entry = txhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        phostbuf->len = 0;
        phostbuf->startbd = 0;
        phostbuf->endbd = 0;
#ifndef NONCACHEABLE_RXBDTBL
        phostbuf->rx_bdtbl = NULL;
#endif

        if( extract_buf_from_process_list(txhost, phostbuf) == 0 ) {
            if( insert_buf_into_free_list(txhost, phostbuf) != 0 ) {
                FPD_ERROR("Failed to insert TX Buffer %d to free list\n", phostbuf->id);
                break;
            }
        }
        else {
            FPD_ERROR("Failed to extract TX Buffer %d from process list\n", phostbuf->id);
            break;
        }
    }

    while( !list_empty(&txhost->user_buffer.list) ) {
        buf_entry = txhost->user_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        phostbuf->len = 0;
        phostbuf->startbd = 0;
        phostbuf->endbd = 0;
#ifndef NONCACHEABLE_RXBDTBL
        phostbuf->rx_bdtbl = NULL;
#endif

        if( extract_buf_from_user_list(txhost, phostbuf) == 0 ) {
            if( insert_buf_into_free_list(txhost, phostbuf) != 0 ) {
                FPD_ERROR("Failed to insert TX Buffer %d to free list\n", phostbuf->id);
                break;
            }
        }
        else {
            FPD_ERROR("Failed to extract TX Buffer %d from user list\n", phostbuf->id);
            break;
        }
    }
    fpd_lock_release(&txhost->lock);

    fpd_lock_acquire(&rxhost->lock);
    rxhost->current_rxbuf = NULL;
    rxhost->len = 0;
    rxhost->buffer_empty = 0;
    while( !list_empty(&rxhost->process_buffer.list) ) {
        buf_entry = rxhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        phostbuf->len = 0;
        phostbuf->startbd = 0;
        phostbuf->endbd = 0;
#ifdef NONCACHEABLE_RXBDTBL
        memset(phostbuf->rx_bdtbl.virtualAddress, 0, FPD_HOST_MAX_RX_BD_TABLE_SIZE);
#else
        memset(phostbuf->rx_bdtbl, 0, FPD_HOST_MAX_RX_BD_TABLE_SIZE);
#endif

        if( extract_buf_from_process_list(rxhost, phostbuf) == 0 ) {
            if( insert_buf_into_free_list(rxhost, phostbuf) != 0 ) {
                FPD_ERROR("Failed to insert RX Buffer %d to free list\n", phostbuf->id);
                break;
            }
        }
        else {
            FPD_ERROR("Failed to extract RX Buffer %d from process list\n", phostbuf->id);
            break;
        }
    }
    fpd_lock_release(&rxhost->lock);

    /* take host out of reset */
    reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
    reg &= ~host_reset_bit;
    FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_config()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Configures the Host Buffer interface module.
 *
 *  PARAMETERS:
 *
 *      phost     Pointer to the host buffer structure.
 *      host_chan Host Buffer channel to use.
 *      dma_size  DMA packet size that is either 512 or 64 byte link layer pkt
 *      rxlen     Receive packet length to be filled in before notifying user
 *
 *  RETURNS:
 *
 *     	0       Configuration is successful.
 *     -errno   Configuration failed.
 *
 *****************************************************************************/
int
fpd_host_config( fpd_host_t *phost, int host_chan, u8 function, int dma_size, int rxlen )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_linkif_t *plink = pfpd->link[phost->link_if];
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    u32 dmabd_addr, dmactl_addr;
    u32 regval;
    u32 *ptr;
    int bdnum;
    int i;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    int size;

    switch( host_chan ) {
        case 0:
            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV0;
            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV0;
            break;
        case 1:
            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV1;
            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV1;
            break;
        default:
            return -EINVAL;
    }

    switch( dma_size ) {
        case 64:
            bdnum = FPD_HOST_BUFFER_SIZE >> 6;
            break;
        case 512:
            bdnum = FPD_HOST_BUFFER_SIZE >> 9;
            break;
        default:
            return -EINVAL;
    }

    if( bdnum > FPD_HOST_MAX_RX_BD ) {
        bdnum = FPD_HOST_MAX_RX_BD;
    }

    /* reset Host channel */
    fpd_host_reset(phost, host_chan);

    pchan->function = function;
    pchan->dma_size = dma_size;
    pchan->rx_pkt_size_notification = rxlen;

    switch( function ) {
        case FPD_HOST_FUNCTION_NONE:
            break;

        case FPD_HOST_FUNCTION_TRANSPARENT:
        case FPD_HOST_FUNCTION_VIRTUAL_MEDIA:
            pchan->rx_bdnum = bdnum;
            FPD_DEBUG(3, "LinkIF%d Host%d RXBD=%d\n", phost->link_if, host_chan, pchan->rx_bdnum);

            /* configure DMA RX Buffers */
            fpd_lock_acquire(&rxhost->lock);

            if( !list_empty(&rxhost->free_buffer.list) ) {
                buf_entry = rxhost->free_buffer.list.next;
                phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

                if( extract_buf_from_free_list(rxhost, phostbuf) == 0 ) {
                    if( insert_buf_into_process_list(rxhost, phostbuf) == 0 ) {
                        rxhost->current_rxbuf = phostbuf;
                        size = dma_size >> 2;

                        /* setup RX buffer descriptor table */
#ifdef NONCACHEABLE_RXBDTBL
                        ptr = (u32 *)phostbuf->rx_bdtbl.virtualAddress;
#else
                        ptr = phostbuf->rx_bdtbl;
#endif
                        for( i = 0; i < pchan->rx_bdnum; i++ ) {
                            writel((u32)(phostbuf->phys_addr + (i * dma_size)), ptr++);
                            writel(FPD_HOST_DMA_RXBD_INT, ptr++);
                        }

                        /* set base address of buffer descriptor table */
#ifdef NONCACHEABLE_RXBDTBL
                        FPD_WRITE(pfpd->remap, dmabd_addr, (u32)phostbuf->rx_bdtbl.physicalAddress);
#else
                        FPD_WRITE(pfpd->remap, dmabd_addr, __pa(phostbuf->rx_bdtbl));
#endif

                        /* start DMA RX */
                        regval = (LIFDRCR_DMA_START | (pchan->rx_bdnum & LIFDRCR_BD_TABLE_SIZE));
                        FPD_WRITE(pfpd->remap, dmactl_addr, regval);

                        /* now go through the free list and initialize each
                         * RX buffer descriptor table
                         */
                        list_for_each( buf_entry, &rxhost->free_buffer.list ) {
                            phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

#ifdef NONCACHEABLE_RXBDTBL
                            ptr = (u32 *)phostbuf->rx_bdtbl.virtualAddress;
#else
                            ptr = phostbuf->rx_bdtbl;
#endif
                            for( i = 0; i < pchan->rx_bdnum; i++ ) {
                                writel((u32)(phostbuf->phys_addr + (i * dma_size)), ptr++);
                                writel(FPD_HOST_DMA_RXBD_INT, ptr++);
                            }
                        }
                    }
                    else {
                        FPD_ERROR("Failed to insert Buffer %d to process list\n", phostbuf->id);
                        fpd_lock_release(&rxhost->lock);
                        return -EIO;
                    }
                }
                else {
                    FPD_ERROR("Failed to extract Buffer %d from free list\n", phostbuf->id);
                    fpd_lock_release(&rxhost->lock);
                    return -EIO;
                }
            }
            else {
                FPD_DEBUG(1, "Run out of DMA RX buffer\n");
                fpd_lock_release(&rxhost->lock);
                return -ENOSPC;
            }

            fpd_lock_release(&rxhost->lock);
            break;

        default:
            return -EINVAL;
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_check_events()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Checks if the Host Buffer has some more data to be process. It may
 *      have stopped due to short packets.
 *
 *  PARAMETERS:
 *
 *      phost      Pointer to the host buffer structure.
 *      host_chan  Host Buffer channel to use.
 *
 *  RETURNS:
 *
 *     	0       No event.
 *      1       An event requires processing.
 *
 *  NOTE:
 *
 *      The event semaphore should already be acquired at this point.
 *
 *****************************************************************************/
int
fpd_host_check_events( fpd_host_t *phost, int host_chan )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost;
    FPD_event_t *pevent;
    int has_event = 0;

    if( pchan->function == FPD_HOST_FUNCTION_TRANSPARENT ) {
        rxhost = &pchan->rx;
        pevent = &pfpd->event;

        fpd_lock_acquire(&rxhost->lock);

        if( rxhost->len > 0 ) {
            FPD_DEBUG(3, "Link%d Host%d RXLen=%d\n", phost->link_if, host_chan, rxhost->len);
            pevent->link_if[phost->link_if].host[host_chan].function = pchan->function;
            pevent->link_if[phost->link_if].host[host_chan].has_data = 1;
            pevent->event_mask |= (FPD_EVENT_MASK_LINK0 << phost->link_if);
            fpd_lock_write_resource(&pfpd->event_lock, pevent->event_mask);
            has_event = 1;
        }

        fpd_lock_release(&rxhost->lock);
    }

    return has_event;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_get_rxbdtbl()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the Host Buffer's Receive Buffer Descriptor Table Pointer.
 *
 *  PARAMETERS:
 *
 *      phost      Pointer to the host buffer structure.
 *      host_chan  Host Buffer channel to use.
 *      pbdtbl     Pointer to a user-allocated buffer for storing the info.
 *
 *  RETURNS:
 *
 *     	0       Configuration is successful.
 *     -errno   Configuration failed.
 *
 *****************************************************************************/
int
fpd_host_get_rxbdtbl( fpd_host_t *phost, int host_chan, FPD_hostrxbdtbl_t *pbdtbl )
{
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    int result = 0;

    if( pbdtbl->buf_index >= FPD_HOST_BUFFER_CNT ) {
        return -EINVAL;
    }

    fpd_lock_acquire(&rxhost->lock);

    phostbuf = &rxhost->hostbuf[pbdtbl->buf_index];
    pbdtbl->overall_len = rxhost->len;
    pbdtbl->state = phostbuf->state;
    pbdtbl->bdnum = pchan->rx_bdnum;
    pbdtbl->startbd = phostbuf->startbd;
    pbdtbl->endbd = phostbuf->endbd;
#ifdef NONCACHEABLE_RXBDTBL
    pbdtbl->phys_addr = (u32)phostbuf->rx_bdtbl.physicalAddress;
    memcpy(&pbdtbl->bdtbl[0], phostbuf->rx_bdtbl.virtualAddress, pchan->rx_bdnum * 8);
#else
    pbdtbl->phys_addr = __pa(phostbuf->rx_bdtbl);
    memcpy(&pbdtbl->bdtbl[0], phostbuf->rx_bdtbl, pchan->rx_bdnum * 8);
#endif
    if( rxhost->current_rxbuf ) {
        if( rxhost->current_rxbuf->id == phostbuf->id ) {
            pbdtbl->is_current_buffer = 1;
        }
        else {
            pbdtbl->is_current_buffer = 0;
        }
    }
    else {
        pbdtbl->is_current_buffer = 0;
    }

    fpd_lock_release(&rxhost->lock);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_get_txbuf_info()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Retrieves the Host Buffer's Transmit Buffer Records.
 *
 *  PARAMETERS:
 *
 *      phost      Pointer to the host buffer structure.
 *      host_chan  Host Buffer channel to use.
 *      pinfo      Pointer to a user-allocated buffer for storing the info.
 *
 *  RETURNS:
 *
 *     	0       Configuration is successful.
 *     -errno   Configuration failed.
 *
 *****************************************************************************/
int
fpd_host_get_txbuf_info( fpd_host_t *phost, int host_chan, FPD_host_txbuf_info_t *pinfo )
{
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *txhost = &pchan->tx;
    fpd_host_buf_t *phostbuf;
    int result = 0;
    int i;

    fpd_lock_acquire(&txhost->lock);

    pinfo->buf_cnt = FPD_HOST_BUFFER_CNT;
    for( i = 0; i < pinfo->buf_cnt; i++ ) {
        phostbuf = &txhost->hostbuf[i];
        pinfo->info[i].state = phostbuf->state;
        pinfo->info[i].len = phostbuf->len;
    }

    fpd_lock_release(&txhost->lock);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_get_txbuffer()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Requests for a free DMA Transmit buffer.
 *
 *  PARAMETERS:
 *
 *      phost        Pointer to the host buffer structure.
 *      host_chan    Host Buffer channel.
 *
 *  RETURNS:
 *
 *     	0       Request is successful.
 *     -errno   Request failed.
 *
 *****************************************************************************/
int
fpd_host_get_txbuffer( fpd_host_t *phost, FPD_hostbuf_t *pbuf )
{
    fpd_host_chan_t *pchan = phost->channel[pbuf->host_chan];
    fpd_host_txrx_t *txhost = &pchan->tx;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    int rc = 0;

    fpd_lock_acquire(&txhost->lock);

    if( !list_empty(&txhost->free_buffer.list) ) {
        buf_entry = txhost->free_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        if( extract_buf_from_free_list(txhost, phostbuf) == 0 ) {
            if( insert_buf_into_user_list(txhost, phostbuf) == 0 ) {
                pbuf->buf_id = phostbuf->id;
                pbuf->buf_size = FPD_HOST_BUFFER_SIZE;
            }
            else {
                FPD_ERROR("Failed to insert Buffer %d to user list\n", phostbuf->id);
                rc = -EIO;
            }
        }
        else {
            FPD_ERROR("Failed to extract Buffer %d from free list\n", phostbuf->id);
            rc = -EIO;
        }
    }
    else {
        FPD_DEBUG(2, "Run out of DMA TX buffer\n");
        pbuf->buf_id = -1;
        pbuf->buf_size = 0;
        rc = -ENOSPC;
    }

    fpd_lock_release(&txhost->lock);

    return rc;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_write()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Writes a data packet to the FPGA Protocol's Host Buffer
 *
 *  PARAMETERS:
 *
 *      phost     Pointer to the Host Buffer data structure.
 *      host_chan Host Buffer channel to use.
 *      len       Number of bytes requested.
 *      buf_id    DMA buffer id.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully written to the TX buffer.
 *     	0       No data is written.
 *     -ENOSPC  Run out of space in driver's TX Buffer.
 *     -EFAULT  Failed to write.
 *
 *****************************************************************************/
int
fpd_host_write( fpd_host_t *phost, int host_chan, int len, int buf_id )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_linkif_t *plink = pfpd->link[phost->link_if];
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *txhost = &pchan->tx;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    u32 tx_addr, tx_ctrl;
    u32 regval;
    u16 id;

    if( buf_id < txhost->hostbuf[0].id || buf_id > txhost->hostbuf[FPD_HOST_BUFFER_CNT - 1].id ) {
        FPD_ERROR("Inconsistent Buffer ID %d\n", buf_id);
        return -EINVAL;
    }

    id = buf_id - txhost->hostbuf[0].id;
    phostbuf = &txhost->hostbuf[id];
    if( buf_id != phostbuf->id ) {
        FPD_ERROR("Inconsistent Buffer ID %d\n", buf_id);
        return -EINVAL;
    }

    /* get lock */
    fpd_lock_acquire(&txhost->lock);

    phostbuf->len = len;

    if( extract_buf_from_user_list(txhost, phostbuf) < 0 ) {
        FPD_ERROR("Failed to extract Buffer %d from user list\n", buf_id);
        fpd_lock_release(&txhost->lock);
        return -EINVAL;
    }

    if( insert_buf_into_process_list(txhost, phostbuf) < 0 ) {
        FPD_ERROR("Failed to insert Buffer %d to process list\n", buf_id);
        fpd_lock_release(&txhost->lock);
        return -EINVAL;
    }

    /* check if a TX DMA xaction is incomplete */
    if( txhost->len != 0 ) {
        fpd_lock_release(&txhost->lock);
        return len;
    }

    if( host_chan == 0 ) {
        tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV0;
        tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV0;
    }
    else {
        tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV1;
        tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV1;
    }

    /* write the DMA TX Address register for the host data buffer, with the 
     * starting address where the data resides in processor memory
     */
    if( txhost->process_buffer.count > 1 ) {
        /* In case there are other buffers that requires processing prior to
         * this buffer, point phostbuf to the first entry of the list.
         */
        FPD_ERROR("OW SHOOT! HERE'S A BUG! TX process %d\n",
                  txhost->process_buffer.count);
        buf_entry = txhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );
    }
    FPD_WRITE(pfpd->remap, tx_addr, (u32)phostbuf->phys_addr);

    /* write the DMA TX Control register setting the PKT_SIZE and BYTE_CNT
     * fields, and setting the DMA_START bit
     */
    regval = (LIFDTCR_DMA_START | (len & LIFDTCR_BYTE_CNT));
    if( pchan->dma_size == 64 ) {
        regval |= LIFDTCR_PKT_SIZE;
    }
    FPD_WRITE(pfpd->remap, tx_ctrl, regval);

    txhost->len = len;

    /* release lock */
    fpd_lock_release(&txhost->lock);

    return len;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_read()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Reads a data packet from the FPGA Protocol Driver's DMA buffer.
 *
 *  PARAMETERS:
 *
 *      phost     Pointer to the Host Buffer data structure.
 *      host_chan Host Buffer channel to use.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully written to the TX buffer.
 *     	0       No data is written.
 *     -ENOSPC  Run out of space in driver's TX Buffer.
 *     -EFAULT  Failed to write.
 *
 *****************************************************************************/
int
fpd_host_read( fpd_host_t *phost, FPD_hostdata_t *pbuf )
{
    fpd_host_chan_t *pchan = phost->channel[pbuf->host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    dma_rxbd_table_t *pbdtbl;
    FPD_host_rxparam_t *prxinfo = &pbuf->info.rx;
    int len = 0, req_len, entry_len;
    int result = 0;
    int completed = 0;
    int i;

    /* get lock */
    fpd_lock_acquire(&rxhost->lock);

    /* check if there is data available */
    if( rxhost->len == 0 ) {
        fpd_lock_release(&rxhost->lock);
        prxinfo->buf_id = -1;
        prxinfo->buf_size = 0;
        prxinfo->offset = 0;
        prxinfo->short_pkt = 0;
        prxinfo->reached_page_end = 0;
        pbuf->actual_len = 0;
        return 0;
    }

    if( !list_empty(&rxhost->process_buffer.list) ) {
        buf_entry = rxhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        if( pbuf->requested_len > rxhost->len ) {
            req_len = rxhost->len;
        }
        else {
            req_len = pbuf->requested_len;
        }

        /* check if there is a short pkt within the requested length or
         * actual length whichever is less
         */
#ifdef NONCACHEABLE_RXBDTBL
        pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl.virtualAddress;
#else
        pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl;
#endif
        prxinfo->short_pkt = 0;
        for( i = phostbuf->startbd; i < phostbuf->endbd; i++ ) {
            entry_len = readl(&pbdtbl->bd[i].status) & FPD_HOST_DMA_RXBD_BYTE_CNT;
            len += entry_len;
            if( entry_len != pchan->dma_size ) {
                /* short pkt */
                prxinfo->short_pkt = 1;
                completed = 1;
                break;
            }

            if( len >= req_len ) {
                /* reached the requested length */
                completed = 1;
                break;
            }
        }

        if( completed ) {
            prxinfo->endbd = i;
        }
        else {
            prxinfo->endbd = i - 1;
        }

        /* reached end of page buffer */
        if( prxinfo->endbd < (pchan->rx_bdnum - 1) ) {
            prxinfo->reached_page_end = 0;
        }
        else {
            prxinfo->reached_page_end = 1;
        }
        prxinfo->buf_id = phostbuf->id;
        prxinfo->buf_size = FPD_HOST_BUFFER_SIZE;
        prxinfo->offset = phostbuf->startbd * pchan->dma_size;
        prxinfo->startbd = phostbuf->startbd;
        prxinfo->len = len;
        if( len > req_len ) {
            pbuf->actual_len = req_len;
        }
        else {
            pbuf->actual_len = len;
        }
    }
    else {
        /* this is impossible */
        FPD_ERROR("CRAP! No RX buffer in process list!\n");
        prxinfo->buf_id = -1;
        prxinfo->buf_size = 0;
        prxinfo->offset = 0;
        prxinfo->short_pkt = 0;
        prxinfo->reached_page_end = 0;
        pbuf->actual_len = 0;
        result = -EIO;
    }

    /* release lock */
    fpd_lock_release(&rxhost->lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_read_complete()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Notifies the driver that the user-application has completed reading
 *      the data packet from the FPGA Protocol Driver's DMA buffer.  This 
 *      releases the portion of DMA buffer memory that is busy and updates
 *      the offset where the next read will take place.
 *
 *  PARAMETERS:
 *
 *      phost     Pointer to the Host Buffer data structure.
 *      host_chan Host Buffer channel to use.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully written to the TX buffer.
 *     	0       No data is written.
 *     -ENOSPC  Run out of space in driver's TX Buffer.
 *     -EFAULT  Failed to write.
 *
 *****************************************************************************/
int
fpd_host_read_complete( fpd_host_t *phost, int host_chan, FPD_host_rxparam_t *prxinfo )
{
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;
    int result = 0;

    /* get lock */
    fpd_lock_acquire(&rxhost->lock);

    if( !list_empty(&rxhost->process_buffer.list) ) {
        buf_entry = rxhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        if( prxinfo->buf_id == phostbuf->id ) {
            if( prxinfo->startbd != phostbuf->startbd ) {
                FPD_ERROR("User sent inconsistent RX DMA buffer info!\n");
            }

            /* move the offset */
            phostbuf->startbd = prxinfo->endbd + 1;
            if( phostbuf->startbd >= pchan->rx_bdnum ) {
                FPD_DEBUG(2, "Free RX Buffer %d\n", phostbuf->id);
                /* buffer has been read completely, put to free list */
                if( extract_buf_from_process_list(rxhost, phostbuf) == 0 ) {
                    if( insert_buf_into_free_list(rxhost, phostbuf) == 0 ) {
                        /*
                         * Re-initialize buffer descriptors after the buffers
                         * are swapped. Otherwise, there is a race condition
                         * between clearing the current RX BD Table and 
                         * hitting a RX_DMA_NO_BUF event. As a consequence
                         * the driver may switch the RX BD Table but FPGA
                         * will not re-read the register unless the DMA RX
                         * stop.
                         */
                        phostbuf->startbd = phostbuf->endbd = 0;

                        /* check if an empty buffer occurred */
                        if( rxhost->buffer_empty ) {
                            if( extract_buf_from_free_list(rxhost, phostbuf) == 0 ) {
                                if( insert_buf_into_process_list(rxhost, phostbuf) == 0 ) {
                                    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
                                    fpd_linkif_t *plink = pfpd->link[phost->link_if];
                                    dma_rxbd_table_t *pbdtbl;
                                    u32 dmabd_addr, dmactl_addr;
                                    u32 regval;
                                    int i;

                                    switch( host_chan ) {
                                        case 0:
                                            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV0;
                                            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV0;
                                            break;
                                        case 1:
                                            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV1;
                                            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV1;
                                            break;
                                        default:
                                            return -EINVAL;
                                    }

                                    FPD_DEBUG(2, "Switching RX Buffer to %d after buffer empty condition\n", phostbuf->id);
                                    rxhost->buffer_empty = 0;
                                    rxhost->current_rxbuf = phostbuf;

                                    /* re-initialize buffer descriptors */
#ifdef NONCACHEABLE_RXBDTBL
                                    pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl.virtualAddress;
#else
                                    pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl;
#endif
                                    for( i = 0; i < pchan->rx_bdnum; i++ ) {
                                        writel(FPD_HOST_DMA_RXBD_INT, &pbdtbl->bd[i].status);
                                    }

                                    /* set base address of buffer descriptor table */
#ifdef NONCACHEABLE_RXBDTBL
                                    FPD_WRITE(pfpd->remap, dmabd_addr, (u32)phostbuf->rx_bdtbl.physicalAddress);
#else
                                    FPD_WRITE(pfpd->remap, dmabd_addr, __pa(phostbuf->rx_bdtbl));
#endif

                                    /* restart DMA RX */
                                    regval = (LIFDRCR_DMA_START | (pchan->rx_bdnum & LIFDRCR_BD_TABLE_SIZE));
                                    FPD_WRITE(pfpd->remap, dmactl_addr, regval);
                                    regval = (LIFDRCR_DMA_START | 0x40000000 | (pchan->rx_bdnum & LIFDRCR_BD_TABLE_SIZE));
                                    FPD_WRITE(pfpd->remap, dmactl_addr, regval);
                                }
                                else {
                                    FPD_ERROR("Failed to insert Buffer %d to process list\n", phostbuf->id);
                                    result = -EIO;
                                }
                            }
                            else {
                                FPD_ERROR("Failed to extract Buffer %d from free list\n", phostbuf->id);
                                result = -EIO;
                            }
                        }
                    }
                    else {
                        FPD_ERROR("Failed to insert RX Buffer %d to free list\n", phostbuf->id);
                        result = -EIO;
                    }
                }
                else {
                    FPD_ERROR("Failed to extract RX Buffer %d from process list\n", phostbuf->id);
                    result = -EIO;
                }
            }

            /* update size */
            rxhost->len -= prxinfo->len;
            if( rxhost->len < 0 ) {
                FPD_ERROR("CRAP! Fix this bug (rxlen=%d)\n", rxhost->len);
            }
            FPD_DEBUG(2, "len=%d startbd=%d endbd=%d\n", rxhost->len, phostbuf->startbd, phostbuf->endbd);
        }
        else {
            FPD_ERROR("Inconsistent RX DMA buffer! (hostbufID=%d, userID=%d)\n", phostbuf->id, prxinfo->buf_id);
            result = -EIO;
        }
    }

    /* release lock */
    fpd_lock_release(&rxhost->lock);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_handle_txdone()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Completes the TX DMA transaction.
 *
 *  PARAMETERS:
 *
 *      phost        Pointer to the Host Buffer data structure.
 *      host_chan    Host Buffer channel to be used.
 *
 *  RETURNS:
 *
 *     	0       Successful
 *
 *****************************************************************************/
int
fpd_host_handle_txdone( fpd_host_t *phost, int host_chan )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_linkif_t *plink = pfpd->link[phost->link_if];
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *txhost = &pchan->tx;
    u32 tx_status;
    u32 tx_bytes;
    fpd_host_buf_t *phostbuf;
    struct list_head *buf_entry;

    if( host_chan == 0 ) {
        tx_status = plink->base + LIF_DMA_TX_STATUS_DEV0;
    }
    else {
        tx_status = plink->base + LIF_DMA_TX_STATUS_DEV1;
    }

    /* get lock */
    fpd_lock_acquire(&txhost->lock);

    /* verify that the byte count in the DMA TX Status register is equal to
     * the requested number of bytes in the DMA TX Control register
     */
    tx_bytes = FPD_READ(pfpd->remap, tx_status) & LIFDTCR_BYTE_CNT;
    if( tx_bytes != txhost->len ) {
        FPD_ERROR("Host TX only %d bytes but requested %d\n", tx_bytes, txhost->len);
    }

    if( !list_empty(&txhost->process_buffer.list) ) {
        buf_entry = txhost->process_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        if( extract_buf_from_process_list(txhost, phostbuf) < 0 ) {
            FPD_ERROR("Failed to extract Buffer %d from process list\n", phostbuf->id);
            return -EINVAL;
        }

        if( insert_buf_into_free_list(txhost, phostbuf) < 0 ) {
            FPD_ERROR("Failed to insert Buffer %d to free list\n", phostbuf->id);
            return -EINVAL;
        }

        /* check if there are more to process */
        if( !list_empty(&txhost->process_buffer.list) ) {
            u32 regval;
            u32 tx_addr, tx_ctrl;

            if( host_chan == 0 ) {
                tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV0;
                tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV0;
            }
            else {
                tx_addr = plink->base + LIF_DMA_TX_ADDRESS_DEV1;
                tx_ctrl = plink->base + LIF_DMA_TX_CONTROL_DEV1;
            }

            /* write the DMA TX Address register for the host data buffer, with the 
             * starting address where the data resides in processor memory
             */
            buf_entry = txhost->process_buffer.list.next;
            phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );
            FPD_WRITE(pfpd->remap, tx_addr, (u32)phostbuf->phys_addr);

            /* write the DMA TX Control register setting the PKT_SIZE and BYTE_CNT
             * fields, and setting the DMA_START bit
             */
            FPD_DEBUG(2, "Restart DMA TX hostbuf=%d len=%d\n", phostbuf->id, phostbuf->len);
            regval = (LIFDTCR_DMA_START | (phostbuf->len & LIFDTCR_BYTE_CNT));
            if( pchan->dma_size == 64 ) {
                regval |= LIFDTCR_PKT_SIZE;
            }
            FPD_WRITE(pfpd->remap, tx_ctrl, regval);

            txhost->len = phostbuf->len;
        }
        else {
            txhost->len = 0;
        }
    }
    else {
        FPD_ERROR("Uh oh! DMA TX done but empty process list\n");
        txhost->len = 0;
    }

    /* release lock */
    fpd_lock_release(&txhost->lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_handle_rx()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Handles the RX DMA transaction.
 *
 *  PARAMETERS:
 *
 *      phost        Pointer to the Host Buffer data structure.
 *      host_chan    Host Buffer channel to be used.
 *
 *  RETURNS:
 *
 *     	0       Everything went smooth and there is more space available for
 *              new packets to come in.
 *     	1       No more space in current RX Buffer Descriptor Table.  This
 *              requires switching to a new RX Buffer.
 *
 *****************************************************************************/
int
fpd_host_handle_rx( fpd_host_t *phost, int host_chan )
{
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf = rxhost->current_rxbuf;
    dma_rxbd_table_t *pbdtbl;
    int do_swap_buffer = 0;
    int i, start, end;
    u32 status;

    /* get lock */
    fpd_lock_acquire(&rxhost->lock);

    start = phostbuf->endbd;
    if( phostbuf->endbd == phostbuf->startbd ) {
        end = pchan->rx_bdnum;
    }
    else if( phostbuf->endbd > phostbuf->startbd ) {
        end = pchan->rx_bdnum;
    }
    else {
        end = phostbuf->startbd;
    }
    FPD_DEBUG(2, "ID=%d startbd=%d endbd=%d start=%d end=%d\n",
              phostbuf->id, phostbuf->startbd, phostbuf->endbd, start, end);

#ifdef NONCACHEABLE_RXBDTBL
    pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl.virtualAddress;
#else
    pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl;
#endif
    for( i = start; i < end; i++ ) {
        status = readl(&pbdtbl->bd[i].status);
        FPD_DEBUG(3, "(%d) addr=%p status=%08x\n", i, (u32 *)(&pbdtbl->bd[i].status), status);
        if( status & FPD_HOST_DMA_RXBD_BUSY ) {
            rxhost->len += status & FPD_HOST_DMA_RXBD_BYTE_CNT;
        }
        else {
            /* empty */
            FPD_DEBUG(2, "(%d) addr=%p status=%08x\n", i, (u32 *)(&pbdtbl->bd[i].status), status);
            break;
        }
    }

    phostbuf->endbd = i;
    if( phostbuf->endbd >= pchan->rx_bdnum ) {
        do_swap_buffer = 1;
    }

    FPD_DEBUG(2, "len=%d ID=%d startbd=%d endbd=%d\n",
              rxhost->len, phostbuf->id, phostbuf->startbd, phostbuf->endbd);

    /* release lock */
    fpd_lock_release(&rxhost->lock);

    return do_swap_buffer;
}

/******************************************************************************
 *  ROUTINE:        fpd_host_handle_txdone()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Handles situation where RX DMA has no buffer left to use.
 *
 *  PARAMETERS:
 *
 *      phost        Pointer to the Host Buffer data structure.
 *      host_chan    Host Buffer channel to be used.
 *
 *  RETURNS:
 *
 *     	0       Successful
 *
 *****************************************************************************/
int
fpd_host_handle_rx_nobuf( fpd_host_t *phost, int host_chan )
{
    fpd_device_t *pfpd = (fpd_device_t *)phost->private;
    fpd_linkif_t *plink = pfpd->link[phost->link_if];
    fpd_host_chan_t *pchan = phost->channel[host_chan];
    fpd_host_txrx_t *rxhost = &pchan->rx;
    fpd_host_buf_t *phostbuf;
    dma_rxbd_table_t *pbdtbl;
    struct list_head *buf_entry;
    u32 dmabd_addr, dmactl_addr;
    u32 regval;
    int i;

    switch( host_chan ) {
        case 0:
            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV0;
            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV0;
            break;
        case 1:
            dmabd_addr = plink->base + LIF_DMA_RX_BD_TABLE_BASE_DEV1;
            dmactl_addr = plink->base + LIF_DMA_RX_CONTROL_DEV1;
            break;
        default:
            return -EINVAL;
    }

    fpd_lock_acquire(&rxhost->lock);

    /* swap to an empty buffer if available */
    if( !list_empty(&rxhost->free_buffer.list) ) {
        buf_entry = rxhost->free_buffer.list.next;
        phostbuf = list_entry( buf_entry, fpd_host_buf_t, list );

        if( extract_buf_from_free_list(rxhost, phostbuf) == 0 ) {
            if( insert_buf_into_process_list(rxhost, phostbuf) == 0 ) {
                FPD_DEBUG(2, "Switching RX Buffer to %d\n", phostbuf->id);
                rxhost->current_rxbuf = phostbuf;

                /* re-initialize buffer descriptors */
#ifdef NONCACHEABLE_RXBDTBL
                pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl.virtualAddress;
#else
                pbdtbl = (dma_rxbd_table_t *)phostbuf->rx_bdtbl;
#endif
                for( i = 0; i < pchan->rx_bdnum; i++ ) {
                    writel(FPD_HOST_DMA_RXBD_INT, &pbdtbl->bd[i].status);
                }

                /* set base address of buffer descriptor table */
#ifdef NONCACHEABLE_RXBDTBL
                FPD_WRITE(pfpd->remap, dmabd_addr, (u32)phostbuf->rx_bdtbl.physicalAddress);
#else
                FPD_WRITE(pfpd->remap, dmabd_addr, __pa(phostbuf->rx_bdtbl));
#endif

                /* restart DMA RX */
                regval = (LIFDRCR_DMA_START | (pchan->rx_bdnum & LIFDRCR_BD_TABLE_SIZE));
                FPD_WRITE(pfpd->remap, dmactl_addr, regval);
                regval = (LIFDRCR_DMA_START | 0x40000000 | (pchan->rx_bdnum & LIFDRCR_BD_TABLE_SIZE));
                FPD_WRITE(pfpd->remap, dmactl_addr, regval);
            }
            else {
                FPD_ERROR("Failed to insert Buffer %d to process list\n", phostbuf->id);
                fpd_lock_release(&rxhost->lock);
                return -EIO;
            }
        }
        else {
            FPD_ERROR("Failed to extract Buffer %d from free list\n", phostbuf->id);
            fpd_lock_release(&rxhost->lock);
            return -EIO;
        }
    }
    else {
        /* No RX buffers left so notify the process that frees the buffer to 
         * re-initialize a new one and restart the DMA RX.
         */
        FPD_DEBUG(2, "Empty Link%d DMA%d RX buffer\n", phost->link_if, host_chan);
        rxhost->buffer_empty = 1;
        fpd_lock_release(&rxhost->lock);
        return -ENOSPC;
    }

    fpd_lock_release(&rxhost->lock);

    return 0;
}
