/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver Paragon (Legacy CIM) Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_ioctl.h"
#include "fpd_reg.h"
#include "debug.h"


/******************************************************************************
 *  ROUTINE:        fpd_paragon_invalidate_table()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Invalidates all contents in the Command Lookup Table
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device data structure.
 *      link_id   Link Interface ID
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_invalidate_table( fpd_device_t *pfpd, int link_id )
{
    fpd_linkif_t *plink;
    int table_addr;
    u32 cmd_table_addr;
    u32 val;

    if( link_id == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[link_id];
    }

    /* requires that Link IF must be disabled first */
    if( plink->enabled ) {
        return -EACCES;
    }

    cmd_table_addr = plink->base + LIF_PARAGON_LOOKUP_TABLE;

    for( table_addr = 0; table_addr < 256; table_addr++ ) {
        val = (((table_addr << 16) & LIFLTR_PCMD_BYTE) |
               LIFLTR_WRITE_ENTRY);
        FPD_WRITE(pfpd->remap, cmd_table_addr, val);
    }

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_write_cmd()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Programs a single command entry to the Command Lookup Table.
 *
 *  PARAMETERS:
 *
 *      pfpd   Pointer to the FPGA Protocol device data structure.
 *      pcmd   Pointer to a user-allocated buffer containing the Paragon 
 *             Command to be programmed.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_write_cmd( fpd_device_t *pfpd, FPD_cmdprog_t *pcmd )
{
    fpd_linkif_t *plink;
    u32 cmd_table_addr;
    u32 val = LIFLTR_WRITE_ENTRY;

    if( pcmd->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcmd->link_if];
    }

    /* requires that Link IF must be disabled first */
    if( plink->enabled ) {
        return -EACCES;
    }

    cmd_table_addr = plink->base + LIF_PARAGON_LOOKUP_TABLE;
    val |= (pcmd->cmd << 16) & LIFLTR_PCMD_BYTE;
    val |= (pcmd->byte_cnt & LIFLTR_PCMD_BYTE_CNT);
    if( pcmd->in_ram ) {
        val |= LIFLTR_PCMD_CNT_IN_RAM;
    }
    if( pcmd->chksum ) {
        val |= LIFLTR_PCMD_W_CHKSUM;
    }
    if( pcmd->dest == FPD_PRIORITY_BUFFER ) {
        val |= LIFLTR_PCMD_DEST;
    }
    if( pcmd->valid ) {
        val |= LIFLTR_PCMD_VAL;
    }
    FPD_WRITE(pfpd->remap, cmd_table_addr, val);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_read_cmd()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Reads a single command entry from the Command Lookup Table.
 *
 *  PARAMETERS:
 *
 *      pfpd   Pointer to the FPGA Protocol device data structure.
 *      pcmd   Pointer to a user-allocated buffer to store the properties of
 *             the command entry.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_read_cmd( fpd_device_t *pfpd, FPD_cmdprog_t *pcmd )
{
    fpd_linkif_t *plink;
    u32 cmd_table_addr;
    u32 val;
    u32 reg;

    if( pcmd->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcmd->link_if];
    }

    /* To read an entry from the lookup table,
     *  1. Write to the lookup table register with WRITE_ENTRY set to zero,
     *     and PCMD_BYTE set to the command whose entry shall be read
     *  2. Read the lookup table register.
     */
    cmd_table_addr = plink->base + LIF_PARAGON_LOOKUP_TABLE;
    val = (pcmd->cmd << 16) & LIFLTR_PCMD_BYTE;
    FPD_WRITE(pfpd->remap, cmd_table_addr, val);
    reg = FPD_READ(pfpd->remap, cmd_table_addr);

    if( reg & LIFLTR_PCMD_VAL ) {
        pcmd->valid = 1;
    }
    else {
        pcmd->valid = 0;
    }

    if( reg & LIFLTR_PCMD_DEST ) {
        pcmd->dest = FPD_PRIORITY_BUFFER;
    }
    else {
        pcmd->dest = FPD_CIM_BUFFER;
    }

    if( reg & LIFLTR_PCMD_CNT_IN_RAM ) {
        pcmd->in_ram = 1;
    }
    else {
        pcmd->in_ram = 0;
    }

    if( reg & LIFLTR_PCMD_W_CHKSUM ) {
        pcmd->chksum = 1;
    }
    else {
        pcmd->chksum = 0;
    }

    pcmd->byte_cnt = (reg & LIFLTR_PCMD_BYTE_CNT);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_handle_invalid_cmd()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Handles the reception of a Paragon command that is not programmed in
 *      the Command Lookup Table.
 *
 *  PARAMETERS:
 *
 *      plink   Pointer to the Link Interface data structure.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_handle_invalid_cmd( fpd_linkif_t *plink )
{
    fpd_device_t *pfpd = (fpd_device_t *)plink->private;
    u8 invCmd;
    u8 id;
    u8 store;
    u32 val;

    fpd_lock_acquire(&plink->invcmd_lock);

    /* read the last invalid command received to clear the interrupt 
     * and get the last invalid command
     */
    val = FPD_READ(pfpd->remap, plink->base + LIF_RX_INVALID_COMMAND);
    invCmd = (val & LIFRICR_RX_LAST_INV_CMD) >> 8;
    FPD_DEBUG(2, "Link%d RX Invalid Paragon Cmd %02x\n", plink->link_id, invCmd);

    /* verify that there is space to store this invalid cmd */
    if( plink->inv_cmd_index < FPD_MAX_INVALID_PARAGON_CMD ) {
        store = 1;

        /* verify that this is not stored already */
        for(id = 0; id < plink->inv_cmd_index; id++) {
            if( invCmd == plink->inv_cmd[id]) {
                store = 0;
                break;
            }
        }

        if( store ) {
            plink->inv_cmd[plink->inv_cmd_index++] = invCmd;
        }
    }

    fpd_lock_release(&plink->invcmd_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_get_invalid_cmd()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Gets all the received Paragon command that is not programmed in
 *      the Command Lookup Table.
 *
 *  PARAMETERS:
 *
 *      pfpd    Pointer to the FPGA Protocol device data structure.
 *      pcmd    Pointer to a user-allocated buffer that will be used to store
 *              the RX invalid commands.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_get_invalid_cmd( fpd_device_t *pfpd, FPD_rxinvcmd_t *pcmd )
{
    fpd_linkif_t *plink;

    if( pcmd->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcmd->link_if];
    }

    fpd_lock_acquire(&plink->invcmd_lock);

    /* store */
    memcpy(pcmd->inv_cmd, plink->inv_cmd, FPD_MAX_INVALID_PARAGON_CMD);
    pcmd->n_entries = plink->inv_cmd_index;

    /* clear */
    memset(plink->inv_cmd, 0, FPD_MAX_INVALID_PARAGON_CMD);
    plink->inv_cmd_index = 0;

    fpd_lock_release(&plink->invcmd_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_handle_echo_rsp()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Handles the response of an Echo Request.
 *
 *  PARAMETERS:
 *
 *      plink   Pointer to the Link Interface data structure.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_handle_echo_rsp( fpd_linkif_t *plink )
{
    fpd_device_t *pfpd = (fpd_device_t *)plink->private;
    u32 val;

    fpd_lock_acquire(&plink->echo_rsp);

    /* read the echo response received, which will clear the interrupt */
    val = FPD_READ(pfpd->remap, plink->base + LIF_ECHO_REQ_RESPONSE) & 0xFF;
    fpd_lock_write_resource(&plink->echo_rsp, val);
    FPD_DDEBUG(pfpd->id, 3, "Echo Response %02x\n", val);

    fpd_lock_release(&plink->echo_rsp);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_get_echo_rsp()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Gets the received response for an Echo Request.
 *
 *  PARAMETERS:
 *
 *      pfpd    Pointer to the FPGA Protocol device data structure.
 *      pcmd    Pointer to a user-allocated buffer that will be used to store
 *              the RX invalid commands.
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_get_echo_rsp( fpd_device_t *pfpd, FPD_echorsp_t *pcmd )
{
    fpd_linkif_t *plink;

    if( pcmd->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcmd->link_if];
    }

    fpd_lock_acquire(&plink->echo_rsp);

    pcmd->echo_rsp = fpd_lock_read_resource(&plink->echo_rsp);

    fpd_lock_release(&plink->echo_rsp);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_enter_cim_update()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Put the FPGA Link Interface into CIM FW Update mode.
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device data structure.
 *      link_id   Link Interface ID
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_enter_cim_update( fpd_device_t *pfpd, int link_id, u32 chg_mode )
{
    fpd_linkif_t *plink;
    int result = 0;
    u32 link_base;
    u32 reg;

    switch( chg_mode ) {
        case FPD_CHANGE_MODE_AFTER_PKT_SENT:
        case FPD_CHANGE_MODE_IMMEDIATELY:
            break;
        default:
            return -EINVAL;
    }

    if( link_id == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[link_id];
    }
    link_base = plink->base;

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        return -ENOLINK;
    }

    /* put into CIM update mode */
    reg = FPD_READ(pfpd->remap, link_base + LIF_LINK_CONTROL);
    if( chg_mode == FPD_CHANGE_MODE_IMMEDIATELY ) {
        reg |= LIFCR_CHG_MODE_NOW;
    }
    else {
        reg &= ~LIFCR_CHG_MODE_NOW;
    }
    reg |=  LIFCR_UPDATE_MODE;
    FPD_WRITE(pfpd->remap, link_base + LIF_LINK_CONTROL, reg);

    return result;
}

/******************************************************************************
 *  ROUTINE:        fpd_paragon_exit_cim_update()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Take the FPGA Link Interface out of CIM FW Update mode.
 *
 *  PARAMETERS:
 *
 *      pfpd      Pointer to the FPGA Protocol device data structure.
 *      link_id   Link Interface ID
 *
 *  RETURNS:
 *
 *     	0       Command is successful.
 *     -errno   Command failed.
 *
 *****************************************************************************/
int
fpd_paragon_exit_cim_update( fpd_device_t *pfpd, int link_id )
{
    fpd_linkif_t *plink;
    u32 link_base;
    u32 reg;

    if( link_id == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[link_id];
    }
    link_base = plink->base;

    /* requires that Link IF must be enabled first */
    if( !plink->enabled ) {
        return -ENOLINK;
    }

    /* make sure that no packet is left in the buffers */
    while( plink->cim.txbuf_bytes_left != FPD_PDATA_BUFFER_SIZE ) {
        /* lazy way for now but later on replace this with timeout */
        schedule();
    }

    /* delay for a bit to make sure the FPGA has send it - lazy way */
    mdelay(100);

    /* exit out of CIM update mode */
    reg = FPD_READ(pfpd->remap, link_base + LIF_LINK_CONTROL);
    reg &= ~LIFCR_UPDATE_MODE;
    FPD_WRITE(pfpd->remap, link_base + LIF_LINK_CONTROL, reg);

    return 0;
}
