/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  FPGA Protocol Driver CIM/Priority Data Buffer Interface
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/delay.h>

#include "fpd.h"
#include "fpd_ioctl.h"
#include "fpd_reg.h"
#include "fpd_pdata.h"
#include "fpd_thread.h"
#include "txcim_thread.h"
#include "debug.h"

#ifndef MIN
#define MIN(a,b)                               ((a < b)?a:b)
#endif


/******************************************************************************
 *  ROUTINE:        pdata_init()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Initializes the Protocol Data interface
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *      parg    Pointer to data to be stored in private member for later
 *              access 
 *
 *  RETURNS:
 *
 *     	0       Initialization is successful.
 *     -errno   Initialization failed.
 *
 *****************************************************************************/
int
pdata_init( fpd_pdata_t *pcim, u32 link_if, u32 btype, void *parg )
{
    /* allocate tx and rx buffers for each link IF */
    pcim->txbuf = (u8 *)kmalloc(FPD_PDATA_BUFFER_SIZE, GFP_KERNEL);
    if( !pcim->txbuf ) {
        FPD_ERROR("Failed to allocate LinkIF TXbuf\n");
        return -ENOMEM;
    }

    pcim->rxbuf = (u8 *)kmalloc(FPD_PDATA_BUFFER_SIZE, GFP_KERNEL);
    if( !pcim->rxbuf ) {
        FPD_ERROR("Failed to allocate LinkIF RXbuf\n");
        kfree(pcim->txbuf);
        return -ENOMEM;
    }

    memset(pcim->txbuf, 0, FPD_PDATA_BUFFER_SIZE);
    memset(pcim->rxbuf, 0, FPD_PDATA_BUFFER_SIZE);
    pcim->link_if            = link_if;
    pcim->btype              = btype;
    pcim->txbuf_rdptr        = pcim->txbuf;
    pcim->txbuf_wrptr        = pcim->txbuf;
    pcim->txbuf_end          = (u8 *)(pcim->txbuf + FPD_PDATA_BUFFER_SIZE);
    pcim->txbuf_bytes_left   = FPD_PDATA_BUFFER_SIZE;
    pcim->rxbuf_rdptr        = pcim->rxbuf;
    pcim->rxbuf_wrptr        = pcim->rxbuf;
    pcim->rxbuf_end          = (u8 *)(pcim->rxbuf + FPD_PDATA_BUFFER_SIZE);
    pcim->rxbuf_bytes_unused = FPD_PDATA_BUFFER_SIZE;
    fpd_lock_init(&pcim->tx_lock, 0);
    fpd_lock_init(&pcim->rx_lock, 0);
    atomic_set(&pcim->req_tx, 0);
    atomic_set(&pcim->rxbuf_no_space, 0);

    pcim->private = parg;

    /* 
    FPD_INFO("Link %d, Type %s, TXBuf=%p, RXBuf=%p\n",
             link_if, (btype==FPD_CIM_BUFFER)?"CIM":"PRI",
             pcim->txbuf, pcim->rxbuf);
    */

    return 0;
}

/******************************************************************************
 *  ROUTINE:        pdata_cleanup()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Cleans up the Protocol Data interface module.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Cleanup is successful.
 *     -1       Cleanup failed.
 *
 *****************************************************************************/
int
pdata_cleanup( fpd_pdata_t *pcim )
{
    kfree(pcim->txbuf);
    kfree(pcim->rxbuf);
    return 0;
}

/******************************************************************************
 *  ROUTINE:        pdata_reset()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Resets the Protocol Data interface module.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Reset is successful.
 *     -1       Reset failed.
 *
 *****************************************************************************/
int
pdata_reset( fpd_pdata_t *pcim )
{
    fpd_lock_acquire(&pcim->tx_lock);
    pcim->txbuf_rdptr        = pcim->txbuf;
    pcim->txbuf_wrptr        = pcim->txbuf;
    pcim->txbuf_bytes_left   = FPD_PDATA_BUFFER_SIZE;
    atomic_set(&pcim->req_tx, 0);
    fpd_lock_release(&pcim->tx_lock);

    fpd_lock_acquire(&pcim->rx_lock);
    pcim->rxbuf_rdptr        = pcim->rxbuf;
    pcim->rxbuf_wrptr        = pcim->rxbuf;
    pcim->rxbuf_bytes_unused = FPD_PDATA_BUFFER_SIZE;
    atomic_set(&pcim->rxbuf_no_space, 0);
    fpd_lock_release(&pcim->rx_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        pdata_check_events()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Checks if the CIM/Protocol Buffer have some more data to be process. 
 *      It may have stopped due to prior short read request.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       No event.
 *      1       An event requires processing.
 *
 *  NOTE:
 *
 *      The event semaphore should already be acquired at this point.
 *
 *****************************************************************************/
int
pdata_check_events( fpd_pdata_t *pcim )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    FPD_event_t *pevent;
    int bytes_unread;
    int has_event = 0;

    fpd_lock_acquire(&pcim->rx_lock);

    /* determine # of bytes not yet read */
    bytes_unread = FPD_PDATA_BUFFER_SIZE - pcim->rxbuf_bytes_unused;

    if( bytes_unread > 0) {
        FPD_DEBUG(3, "Link%d %s bytesUnread=%d\n", pcim->link_if, pcim->btype?"Protocol":"CIM", bytes_unread);
        pevent = &pfpd->event;
        if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
            if( pcim->btype == FPD_CIM_BUFFER ) {
                pevent->bgnd_link_if.rx_cim = 1;
                pevent->event_mask |= FPD_EVENT_MASK_BGND_LINK;
            }
        }
        else {
            if( pcim->btype == FPD_CIM_BUFFER )
                pevent->link_if[pcim->link_if].rx_cim = 1;
            else
                pevent->link_if[pcim->link_if].rx_priority = 1;

            pevent->event_mask |= (FPD_EVENT_MASK_LINK0 << pcim->link_if);
        }

        /* copy current events to event lock resource */
        fpd_lock_write_resource(&pfpd->event_lock, pevent->event_mask);

        has_event = 1;
    }

    fpd_lock_release(&pcim->rx_lock);

    return has_event;
}

/******************************************************************************
 *  ROUTINE:        pdata_disconnect_check()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      A verification before disconnecting a Link Interface that the TX
 *      Buffer in both the driver and FPGA is already empty.  If not, this
 *      will wait until it empties or timeout and do a force disconnection.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Safe to disconnect.
 *     -1       Buffers are not empty.
 *
 *****************************************************************************/
int
pdata_disconnect_check( fpd_pdata_t *pcim )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    fpd_timer_t *ptimer;
#endif
    u32 status_addr;
    u32 reg, avail_bytes;
    u32 driver_wait_time, fpga_wait_time;
    u32 timer_timeout_value = WAIT_100MS;
    u8  all_clear = 0;
    u8  tries = 1;

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
    ptimer = &plink->timer;
#endif

    switch( pcim->btype ) {
        case FPD_CIM_BUFFER:
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;
        case FPD_PRIORITY_BUFFER:
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;
        default:
            return -EINVAL;
    }

    /* determine wait time necessary before forcing disconnect */
    driver_wait_time = atomic_read(&pfpd->driver_txbuf_wait_time)/timer_timeout_value;
    fpga_wait_time = atomic_read(&pfpd->fpga_txbuf_wait_time)/timer_timeout_value;
    FPD_DEBUG(2, "driver_wait=%d, fpga_wait=%d\n", driver_wait_time, fpga_wait_time);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    /*
     * For Linux Kernel 2.4, msleep() is not supported. fpd_timer.c handles
     * the timeout.  However, the timeout is based on 10ms interval.  So,
     * the timeout value has to be divided by 10.
     */
    timer_timeout_value = WAIT_100MS/10;
#endif

    do {
        /* get lock */
        fpd_lock_acquire(&pcim->tx_lock);

        /* check for driver's tx buffer */
        if( pcim->txbuf_bytes_left == FPD_PDATA_BUFFER_SIZE ) {
            FPD_DEBUG(2, "Link%d Driver %s TXBuf All Clear\n",
                      pcim->link_if,(pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");

            /* in case it was used before */
            tries = 1;
            do {
                /* check for FPGA's tx buffer */
                reg = FPD_READ(pfpd->remap, status_addr);
                avail_bytes = (reg & LIFCBSR_TX_BYTE_AVAIL) >> 13;
                if( avail_bytes == 512 ) {
                    FPD_DEBUG(2, "Link%d FPGA %s TXBuf All Clear\n",
                              pcim->link_if, (pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");
                    all_clear = 1;
                }
                else {
                    FPD_DEBUG(2, "Link%d FPGA %s TXBuf has data\n",
                              pcim->link_if, (pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");

                    /* wait until FPGA timeout occurs */
                    if( tries <= fpga_wait_time ) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
                        msleep(timer_timeout_value);
#else
                        init_waitqueue_head(&ptimer->wait_queue);
                        ptimer->timeout_value = timer_timeout_value;
                        fpd_setup_timer( ptimer );

                        /* wait for the timeout event */
                        if( wait_event_interruptible(ptimer->wait_queue,
                                                     (!atomic_read(&ptimer->running) ||
                                                      signal_pending(current))) ) {
                            fpd_cleanup_timer( ptimer );
                            fpd_lock_release(&pcim->tx_lock);
                            return -2;
                        }
#endif
                        ++tries;
                    }
                    else {
                        fpd_lock_release(&pcim->tx_lock);
                        FPD_DEBUG(2, "Link%d still has data on FPGA %s TXBuf\n",
                                  pcim->link_if,
                                  (pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");
                        return -1;
                    }
                }
            } while( !all_clear );
        }
        else {
            /* release lock */
            fpd_lock_release(&pcim->tx_lock);

            FPD_DEBUG(2, "Link%d Driver %s TXBuf has data\n",
                      pcim->link_if,(pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");

            /* wait until FPGA timeout occurs */
            if( tries <= driver_wait_time ) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
                msleep(timer_timeout_value);
#else
                init_waitqueue_head(&ptimer->wait_queue);
                ptimer->timeout_value = timer_timeout_value;
                fpd_setup_timer( ptimer );

                /* wait for the timeout event */
                if( wait_event_interruptible(ptimer->wait_queue,
                                             (!atomic_read(&ptimer->running) ||
                                              signal_pending(current))) ) {
                    fpd_cleanup_timer( ptimer );
                    return -2;
                }
#endif
                ++tries;
            }
            else {
                FPD_DEBUG(2, "Link%d still has data on Driver %s TXBuf\n",
                          pcim->link_if,
                          (pcim->btype==FPD_CIM_BUFFER)?"CIM":"PRI");
                return -1;
            }
        }
    } while( !all_clear );

    /* release lock */
    fpd_lock_release(&pcim->tx_lock);

    return 0;
}

#ifdef PROTOCOL_DATA_XMIT_IMMEDIATE
/******************************************************************************
 *  ROUTINE:        pdata_transmit_immediately()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Transmits a protocol data packet to the FPGA TX Buffer without going
 *      through the TXCIM thread.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Transmission is successful.
 *     -EINVAL  Invalid buffer type.
 *     -ENOSPC  Data will not fit into the FPGA's transmit buffer.
 *
 *****************************************************************************/
static int
pdata_transmit_immediately( fpd_pdata_t *pcim, int len, u8 *wrbuf )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
    u32 reg, avail_bytes, cnt;
    u32 buf_addr, status_addr, eop_addr;
    int revised_len;
    u8 *pbuf = wrbuf;
    u8  databuf[512];

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }

    switch( pcim->btype ) {
        case FPD_CIM_BUFFER:
            buf_addr = plink->base + LIF_TX_CIM_BUFFER_DATA;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_CIM_BUFFER_EOP;
            break;
        case FPD_PRIORITY_BUFFER:
            buf_addr = plink->base + LIF_TX_PRIORITY_BUFFER_DATA;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_PRIORITY_BUFFER_EOP;
            break;
        default:
            return -EINVAL;
    }

    /* get available bytes for the transmitter */
    reg = FPD_READ(pfpd->remap, status_addr);
    FPD_DDEBUG(pfpd->id, 3, "XI status ADDR = 0x%08x, reg = 0x%08x\n",
               status_addr, reg);
    avail_bytes = (reg & LIFCBSR_TX_BYTE_AVAIL) >> 13;

    /* check if there is space for the pkt + its length */
    if( avail_bytes >= (len + sizeof(u32)) ) {
        /* copy user-space buffer to kernel buffer */
        if( copy_from_user((void *)databuf, (const void *)pbuf, len) != 0 )
        {
            FPD_ERROR("CIM Data Write copy_from_user failed\n");
            fpd_lock_release(&pcim->tx_lock);
            return -EFAULT;
        }

        /* deduct the last packet word from the length */
        if( len & 0x3 ) {
            revised_len = len & ~0x3;
        }
        else {
            revised_len = len - sizeof(u32);
        }

        FPD_DDEBUG(pfpd->id, 3, "XI CIM TX Len %d Rev Len %d\n",
                   len, revised_len);

        /* write pkt length first */
        FPD_WRITE(pfpd->remap, buf_addr, len);

        for(cnt = 0; cnt < revised_len; cnt+=sizeof(u32)) {
            FPD_DDEBUG(pfpd->id, 3, "XI CIM TX Buffer Data %08x\n",
                       readl(databuf+cnt));

            /* write data pkt 32-bit at a time */
            FPD_WRITE(pfpd->remap, buf_addr, readl(databuf+cnt));
        }

        FPD_DDEBUG(pfpd->id, 3, "XI CIM TX EOP Data %08x\n",
                   readl(databuf+cnt));

        /* write last 32-bit data to TX EOP Register */
        FPD_WRITE(pfpd->remap, eop_addr, readl(databuf+cnt));
    }
    else {
        /* no space in FPGA */
        return -ENOSPC;
    }

    return 0;
}
#endif /* PROTOCOL_DATA_XMIT_IMMEDIATE */

/******************************************************************************
 *  ROUTINE:        pdata_write()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Writes a protocol data packet to the Driver's circular TX Buffer
 *      and notifies the TX Data Thread to do the real transmission.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *      len     Number of bytes requested.
 *      wrbuf   Buffer containing the data.
 *      bufloc  Buffer location: user-space or kernel-space.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully written to the TX buffer.
 *     	0       No data is written.
 *     -ENOSPC  Run out of space in driver's TX Buffer.
 *     -EFAULT  Failed to write.
 *
 *  NOTE:
 *
 *      Read pointer is always 4-byte aligned.
 *      Write pointer has to make sure that it ends in a 4-byte alignment.
 *
 *****************************************************************************/
int
pdata_write( fpd_pdata_t *pcim, int len, const u8 *wrbuf, bufloc_t bufloc )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    u32 addr_mask;
    u32 remainder;
    u32 size = 0;
    const u8 *pbuf = wrbuf;
    u8 *orig_wrptr = pcim->txbuf_wrptr;
    u32 orig_bytes_left = pcim->txbuf_bytes_left;
    int bytes_wrote = 0;
    int round = 1;

    /**********************************************/
    /* packet shall be stored in 4-byte alignment */
    /**********************************************/

    /* get lock */
    fpd_lock_acquire(&pcim->tx_lock);

#ifdef PROTOCOL_DATA_XMIT_IMMEDIATE
    /* check if data can be written directly */
    if( pcim->txbuf_bytes_left == FPD_PDATA_BUFFER_SIZE ) {
        int result = pdata_transmit_immediately(pcim, len, wrbuf);

        switch( result ) {
            case 0:
                fpd_lock_release(&pcim->tx_lock);
                return len;
            case -EINVAL:
            case -EFAULT:
            default:
                fpd_lock_release(&pcim->tx_lock);
                return result;
            case -ENOSPC:
                /* no space in FPGA to write complete data */
                break;
        }
    }
#endif /* PROTOCOL_DATA_XMIT_IMMEDIATE */

    /* verify that write will occur on a 4-byte alignment */
    if( (u32)pcim->txbuf_wrptr & 0x03 ) {
        FPD_ERROR("CIM TX Buffer Write Ptr Alignment error!\n");

        /* Damn it! Adjust it then */
        addr_mask = (u32)pcim->txbuf_wrptr & 0x03;
        switch( addr_mask ) {
            case 1:
                pcim->txbuf_wrptr += 3;
                pcim->txbuf_bytes_left -= 3;
                break;
            case 2:
                pcim->txbuf_wrptr += 2;
                pcim->txbuf_bytes_left -= 2;
                break;
            case 3:
                pcim->txbuf_wrptr += 1;
                pcim->txbuf_bytes_left -= 1;
                break;
            default:
                /* yipee! it is aligned */
                break;
        }
    }

    for( ; len > 0; len -= size, pbuf += size, round++ ) {
        FPD_DDEBUG(pfpd->id, 3,
                   "CIM Write wrptr=%p, rdptr=%p, start=%p, end=%p\n",
                   pcim->txbuf_wrptr, pcim->txbuf_rdptr,
                   pcim->txbuf, pcim->txbuf_end);


        FPD_DDEBUG(pfpd->id, 3,
                   "CIM Write Len %d Avail %d Wrote %d Round %d\n",
                   len, pcim->txbuf_bytes_left, bytes_wrote, round);

        /* things to do for the first round */
        if( round == 1 ) {
            /* handle condition: no space left to store data */
            if( (len + sizeof(u32)) > pcim->txbuf_bytes_left ) {
                FPD_DDEBUG(pfpd->id, 2, "No space for CIM Data Write!\n");
                fpd_lock_release(&pcim->tx_lock);
                return -ENOSPC;
            }

            /* prepend the packet length */
            *(u32 *)pcim->txbuf_wrptr = len & 0x3FF;
            pcim->txbuf_wrptr += sizeof(u32);
            pcim->txbuf_bytes_left -= sizeof(u32);
            if( (u32)pcim->txbuf_wrptr >= (u32)pcim->txbuf_end ) {
                /* reached end, reset to start of txbuf */
                pcim->txbuf_wrptr = pcim->txbuf;
            }
        }
        else {
            /*
             * check if the previous size is not 4-byte aligned
             * and we reached the end already
             */
            if( size & 0x3 ) {
                FPD_ERROR("FIX THIS size=%d, len=%d\n", size, len);
            }
            if( len > pcim->txbuf_bytes_left ) {
                FPD_ERROR("SHIT Len %d Avail %d Wrote %d Round %d\n",
                          len, pcim->txbuf_bytes_left, bytes_wrote, round);
            }
        }

        if( (u32)pcim->txbuf_wrptr >= (u32)pcim->txbuf_rdptr ) {
            FPD_DDEBUG(pfpd->id, 3, "CIM Write wrptr >= rdptr\n");
            /* get remaining size from the wrptr to the end of the buffer */
            remainder = (u32)pcim->txbuf_end - (u32)pcim->txbuf_wrptr;
        }
        else {
            FPD_DDEBUG(pfpd->id, 3, "CIM Write wrptr < rdptr\n");
            /* available space start from wrptr and ends at rdptr */
            remainder = (u32)pcim->txbuf_rdptr - (u32)pcim->txbuf_wrptr;
        }

        /* determine the length of the copy for this round */
        size = MIN(len, remainder);
        FPD_DDEBUG(pfpd->id, 3, "CIM Write size %d (LEN %d, REM %d)\n",
                   size, len, remainder);

        if( bufloc == USER_SPACE ) {
            if( copy_from_user((void *)pcim->txbuf_wrptr, (const void *)pbuf, size) != 0 )
            {
                FPD_ERROR("CIM Data Write copy_from_user failed\n");

                /* 
                 * Do not allow partial writes!
                 * Revert back to original location of wrptr
                 */
                pcim->txbuf_wrptr = orig_wrptr;
                pcim->txbuf_bytes_left = orig_bytes_left;

                fpd_lock_release(&pcim->tx_lock);
                return -EFAULT;
            }
        }
        else {
            /* kernel space */
            if( memcpy((void *)pcim->txbuf_wrptr, (void *)pbuf, size) != pcim->txbuf_wrptr )
            {
                FPD_ERROR("CIM Data Write memcpy failed\n");

                /* 
                 * Do not allow partial writes!
                 * Revert back to original location of wrptr
                 */
                pcim->txbuf_wrptr = orig_wrptr;
                pcim->txbuf_bytes_left = orig_bytes_left;

                fpd_lock_release(&pcim->tx_lock);
                return -EFAULT;
            }
        }

        /* move wrptr */
        pcim->txbuf_wrptr += size;
        pcim->txbuf_bytes_left -= size;
        if( pcim->txbuf_bytes_left < 0 ) {
            FPD_ERROR("SERIOUS PROBLEM HERE txbuf bytes left=%d, size=%d\n", pcim->txbuf_bytes_left, size);
        }
        if( (u32)pcim->txbuf_wrptr >= (u32)pcim->txbuf_end ) {
            /* reached end, reset to start of txbuf */
            pcim->txbuf_wrptr = pcim->txbuf;
        }

        bytes_wrote += size;

    }

    FPD_DDEBUG(pfpd->id, 3,
               "CIM Write completed Avail %d Wrote %d TXbuf wrptr %p\n",
               pcim->txbuf_bytes_left, bytes_wrote, pcim->txbuf_wrptr);

    /* make sure that write ends on a 4-byte alignment */
    addr_mask = (u32)pcim->txbuf_wrptr & 0x03;
    switch( addr_mask ) {
        case 1:
            pcim->txbuf_wrptr += 3;
            pcim->txbuf_bytes_left -= 3;
            break;
        case 2:
            pcim->txbuf_wrptr += 2;
            pcim->txbuf_bytes_left -= 2;
            break;
        case 3:
            pcim->txbuf_wrptr += 1;
            pcim->txbuf_bytes_left -= 1;
            break;
        default:
            /* yipee! it is aligned */
            break;
    }

    if( (u32)pcim->txbuf_wrptr >= (u32)pcim->txbuf_end ) {
        /* reached end, reset to start of txbuf */
        pcim->txbuf_wrptr = pcim->txbuf;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "CIM Write fix wrptr Avail %d Wrote %d TXbuf wrptr %p\n",
               pcim->txbuf_bytes_left, bytes_wrote, pcim->txbuf_wrptr);

    /* release lock */
    fpd_lock_release(&pcim->tx_lock);

    /* notify TX Data Thread to write new pkts to the FPGA Protocol device */
    if( bytes_wrote > 0 ) {
        atomic_inc(&pcim->req_tx);
#if 1
        if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
            txcim_thread_wakeup(pfpd->bgnd_link);
        }
        else {
            txcim_thread_wakeup(pfpd->link[pcim->link_if]);
        }
#else
        /* testing multiple packets write to FPGA */
        if( pcim->txbuf_bytes_left == 0 ) {
            if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
                txcim_thread_wakeup(pfpd->bgnd_link);
            }
            else {
                txcim_thread_wakeup(pfpd->link[pcim->link_if]);
            }
        }
#endif
    }

    return bytes_wrote;
}

/******************************************************************************
 *  ROUTINE:        pdata_transmit()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Transmits a protocol data packet to the FPGA TX Buffer.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Transmission is successful.
 *     -EINVAL  Invalid buffer type.
 *
 *  NOTE:
 *
 *      Read pointer is always 4-byte aligned.
 *      Write pointer has to make sure that it ends in a 4-byte alignment.
 *
 *****************************************************************************/
int
pdata_transmit( fpd_pdata_t *pcim )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
    u32 reg, avail_bytes, cnt;
    u32 buf_addr, status_addr, eop_addr;
    u32 addr_mask;
    int rx_avail;
    int len, revised_len;
    static int prior_pktlen = 0;

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }

    switch( pcim->btype ) {
        case FPD_CIM_BUFFER:
            buf_addr = plink->base + LIF_TX_CIM_BUFFER_DATA;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_CIM_BUFFER_EOP;
            break;
        case FPD_PRIORITY_BUFFER:
            buf_addr = plink->base + LIF_TX_PRIORITY_BUFFER_DATA;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            eop_addr = plink->base + LIF_TX_PRIORITY_BUFFER_EOP;
            break;
        default:
            return -EINVAL;
    }

    /* get lock */
    fpd_lock_acquire(&pcim->tx_lock);

    /******************************************/
    /* packets are stored in 4-byte alignment */
    /******************************************/

    /*
     * 1. Write data until rdptr and wrptr are the same.
     * 2. If rdptr and wrptr are the same, process only if there are no
     *    bytes left to store any new data from the user-app.
     */
    while( pcim->txbuf_rdptr != pcim->txbuf_wrptr ||
           pcim->txbuf_bytes_left == 0 ) {
        FPD_DDEBUG(pfpd->id, 3, "CIM TX rdptr = %p wrptr = %p bytes_left %d\n",
                   pcim->txbuf_rdptr, pcim->txbuf_wrptr,
                   pcim->txbuf_bytes_left);

        /* verify that read will occur on a 4-byte alignment */
        if( (u32)pcim->txbuf_rdptr & 0x03 ) {
            FPD_ERROR("CIM TX Buffer Read Ptr Alignment error!\n");

            /* Damn it! Adjust it then */
            addr_mask = (u32)pcim->txbuf_rdptr & 0x03;
            switch( addr_mask ) {
                case 1:
                    pcim->txbuf_rdptr += 3;
                    break;
                case 2:
                    pcim->txbuf_rdptr += 2;
                    break;
                case 3:
                    pcim->txbuf_rdptr += 1;
                    break;
                default:
                    /* yipee! it is aligned */
                    break;
            }
        }

        /* get available bytes for the transmitter */
        reg = FPD_READ(pfpd->remap, status_addr);
        FPD_DDEBUG(pfpd->id, 3, "status ADDR = 0x%08x, reg = 0x%08x\n",
                   status_addr, reg);
        avail_bytes = (reg & LIFCBSR_TX_BYTE_AVAIL) >> 13;

        /*
         * The first 32-bit word that will be read on the circular TX Buffer
         * will be the packet length.
         */
        len = *(int *)pcim->txbuf_rdptr;

        /* get total number of bytes not yet transmitted */
        rx_avail = FPD_PDATA_BUFFER_SIZE - pcim->txbuf_bytes_left;

        FPD_DDEBUG(pfpd->id, 3, "CIM TX Len %d unTXed %d FPGA Avail %d\n",
                   len, rx_avail, avail_bytes);

        if( len > FPD_MAX_PACKET_SIZE || len < 0 ) {
            FPD_ERROR("CIM TXBuf inconsistent! RESET LINK IF!\n");
            FPD_ERROR("CIM TX Len %d unTXed %d FPGA Avail %d prevlen %d\n",
                      len, rx_avail, avail_bytes, prior_pktlen);
            goto pdata_transmit_end;
        }
        prior_pktlen = len;

        /* check if there is space for the pkt + its length */
        if( avail_bytes >= (len + sizeof(u32)) ) {
            /* deduct the last packet word from the length */
            if( len & 0x3 ) {
                revised_len = len & ~0x3;
            }
            else {
                revised_len = len - sizeof(u32);
            }

            FPD_DDEBUG(pfpd->id, 3, "CIM TX Len %d Rev Len %d\n",
                       len, revised_len);

            /* write pkt length first */
            FPD_WRITE(pfpd->remap, buf_addr, len);
            pcim->txbuf_rdptr += sizeof(u32);
            pcim->txbuf_bytes_left += sizeof(u32);
            if( pcim->txbuf_rdptr >= pcim->txbuf_end ) {
                /* reached end, reset to start of txbuf */
                pcim->txbuf_rdptr = pcim->txbuf;
            }

            for(cnt = 0; cnt < revised_len; cnt+=sizeof(u32)) {
                FPD_DDEBUG(pfpd->id, 3, "CIM TX Buffer Data %08x\n",
                           readl(pcim->txbuf_rdptr));

                /* write data pkt 32-bit at a time */
                FPD_WRITE(pfpd->remap, buf_addr, readl(pcim->txbuf_rdptr));

                pcim->txbuf_rdptr += sizeof(u32);
                pcim->txbuf_bytes_left += sizeof(u32);
                if( pcim->txbuf_rdptr >= pcim->txbuf_end ) {
                    /* reached end, reset to start of txbuf */
                    pcim->txbuf_rdptr = pcim->txbuf;
                }
            }

            FPD_DDEBUG(pfpd->id, 3, "CIM TX EOP Data %08x\n",
                       readl(pcim->txbuf_rdptr));

            /* write last 32-bit data to TX EOP Register */
            FPD_WRITE(pfpd->remap, eop_addr, readl(pcim->txbuf_rdptr));

            pcim->txbuf_rdptr += sizeof(u32);
            pcim->txbuf_bytes_left += sizeof(u32);
            if( pcim->txbuf_rdptr >= pcim->txbuf_end ) {
                /* reached end, reset to start of txbuf */
                pcim->txbuf_rdptr = pcim->txbuf;
            }
        }
        else {
            /* handle condition: pkts need to be sent but no space in FPGA */
            FPD_DDEBUG(pfpd->id, 2, "No space in FPGA for TX CIM Data\n");

            /*
             * clear interrupt in case it is already set right now
             * 
             * Is it possible that it transitioned to an empty state already
             * at this point so that this will actually cause problems?
             */
            switch( pcim->btype ) {
                case FPD_CIM_BUFFER:
                    reg = LIFISR_TX_CIM_BUF;
                    break;
                case FPD_PRIORITY_BUFFER:
                    reg = LIFISR_TX_PRI_BUF;
                    break;
                default:
                    /* this should never happen at this point */
                    break;
            }
            FPD_WRITE(pfpd->remap, plink->base+LIF_LINK_INTERRUPT_STATUS, reg);

            /*
             * Enable TX Buffer Interrupt to learn when the TX Buffer is
             * empty and trigger the TX CIM thread to begin processing again.
             */
            reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
            switch( pcim->btype ) {
                case FPD_CIM_BUFFER:
                    reg |= LIFCR_TX_CIM_BUF_INT_EN;
                    break;
                case FPD_PRIORITY_BUFFER:
                    reg |= LIFCR_TX_PRI_BUF_INT_EN;
                    break;
                default:
                    /* this should never happen at this point */
                    break;
            }
            FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

            goto pdata_transmit_end;
        }

        FPD_DDEBUG(pfpd->id, 3, "CIM TX completed rdptr = %p wrptr %p\n",
                   pcim->txbuf_rdptr, pcim->txbuf_wrptr);
    }

 pdata_transmit_end:
    /* release lock */
    fpd_lock_release(&pcim->tx_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        pdata_read()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read data from the circular RX Buffer.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *      len     Number of bytes requested.
 *      appbuf  Buffer to store the data that is allocated in user space.
 *
 *  RETURNS:
 *
 *     	> 0     Number of bytes successfully read.
 *     	0       No data is available at this time.
 *     -EFAULT  Failed to read
 *
 *  NOTE:
 *
 *      Write pointer is always 4-byte aligned.
 *      Read pointer has to make sure that it ends in a 4-byte alignment.
 *
 *****************************************************************************/
int
pdata_read( fpd_pdata_t *pcim, int len, u8 *appbuf )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
    u8 *orig_rdptr = pcim->rxbuf_rdptr;
    u32 orig_bytes_unused = pcim->rxbuf_bytes_unused;
    u32 addr_mask;
    u32 remainder;
    u32 size;
    u8 *pbuf = appbuf;
    int bytes_unread;
    int bytes_read = 0;
    int round = 1;

    /* get lock */
    fpd_lock_acquire(&pcim->rx_lock);

    /******************************************/
    /* packets are stored in 4-byte alignment */
    /******************************************/

    /* verify that the read will occur on a 4-byte alignment */
    if( (u32)pcim->rxbuf_rdptr & 0x03 ) {
        FPD_ERROR("CIM RX Buffer Read Ptr Alignment error!\n");

        /* Damn it! Adjust it then */
        addr_mask = (u32)pcim->rxbuf_rdptr & 0x03;
        switch( addr_mask ) {
            case 1:
                pcim->rxbuf_rdptr += 3;
                break;
            case 2:
                pcim->rxbuf_rdptr += 2;
                break;
            case 3:
                pcim->rxbuf_rdptr += 1;
                break;
            default:
                /* yipee! it is aligned */
                break;
        }
    }

    /* determine # of bytes not yet read */
    bytes_unread = FPD_PDATA_BUFFER_SIZE - pcim->rxbuf_bytes_unused;
    FPD_DDEBUG(pfpd->id, 3, "CIM Read unread %d\n", bytes_unread);

    /* handle condition: no data in RX buffer */
    if( bytes_unread <= 0) {
        FPD_DDEBUG(pfpd->id, 2, "CIM Read has no data available\n");
        bytes_read = 0;
        goto pdata_read_error;
    }

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }

    if( plink->protocol == FPD_PROTOCOL_VM ) {
        /* determine if user requested too much data */
        if( len > bytes_unread ) {
            len = bytes_unread;
        }
    }
    else {
        /*
         * For paragon data, user can only read one packet at a time. Also, it
         * is not allowed to do partial read of a packet.
         *
         * For paragon data, the first 32-bit word that will be read on the
         * circular RX Buffer will be the packet length.
         */
        len = *(int *)pcim->rxbuf_rdptr;
        pcim->rxbuf_rdptr += sizeof(u32);
        pcim->rxbuf_bytes_unused += sizeof(u32);
        if( (u32)pcim->rxbuf_rdptr >= (u32)pcim->rxbuf_end ) {
            /* reached end, reset to start of rxbuf */
            pcim->rxbuf_rdptr = pcim->rxbuf;
        }
    }

    for( ; len > 0; len -= size, pbuf += size, round++ ) {
        FPD_DDEBUG(pfpd->id, 3,
                   "CIM Read rdptr=%p, wrptr=%p, start=%p, end=%p\n",
                   pcim->rxbuf_rdptr, pcim->rxbuf_wrptr,
                   pcim->rxbuf, pcim->rxbuf_end);


        FPD_DDEBUG(pfpd->id, 3,
                   "CIM Read Len %d Unused %d Read %d Round %d\n",
                   len, pcim->rxbuf_bytes_unused, bytes_read, round);

        if( (u32)pcim->rxbuf_rdptr >= (u32)pcim->rxbuf_wrptr ) {
            FPD_DDEBUG(pfpd->id, 3, "CIM Read rdptr >= wrptr\n");
            /* get remaining size from the rdptr to the end of the buffer */
            remainder = (u32)pcim->rxbuf_end - (u32)pcim->rxbuf_rdptr;
        }
        else {
            FPD_DDEBUG(pfpd->id, 3, "CIM Read rdptr < wrptr\n");
            /* unread bytes start from rdptr and ends at wrptr */
            remainder = (u32)pcim->rxbuf_wrptr - (u32)pcim->rxbuf_rdptr;
        }

        /* determine the length of the copy for this round */
        size = MIN(len, remainder);
        FPD_DDEBUG(pfpd->id, 3, "CIM Read size %d (LEN %d, REM %d)\n",
                   size, len, remainder);

        if( copy_to_user((void *)pbuf, (const void *)pcim->rxbuf_rdptr, size) != 0 )
        {
            FPD_ERROR("CIM Read copy_to_user failed\n");

            if( plink->protocol == FPD_PROTOCOL_VM ) {
                /* handle condition: first read passed but this next round 
                 * fails
                 */
                if( round == 1 ) {
                    bytes_read = -EFAULT;
                    goto pdata_read_error;
                }
                else {
                    goto pdata_read_cleanup;
                }
            }
            else {
                /* 
                 * Do not allow partial read!
                 * Revert back to original location of rdptr
                 */
                pcim->rxbuf_rdptr = orig_rdptr;
                pcim->rxbuf_bytes_unused = orig_bytes_unused;

                bytes_read = -EFAULT;
                goto pdata_read_error;
            }
        }

        /* move rdptr */
        pcim->rxbuf_rdptr += size;
        pcim->rxbuf_bytes_unused += size;
        if( (u32)pcim->rxbuf_rdptr >= (u32)pcim->rxbuf_end ) {
            /* reached end, reset to start of rxbuf */
            pcim->rxbuf_rdptr = pcim->rxbuf;
        }

        bytes_read += size;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "CIM Read completed Unused %d Read %d RXbuf rdptr %p\n",
               pcim->rxbuf_bytes_unused, bytes_read, pcim->rxbuf_rdptr);

 pdata_read_cleanup:
    /* make sure that read ends on a 4-byte alignment */
    addr_mask = (u32)pcim->rxbuf_rdptr & 0x03;
    switch( addr_mask ) {
        case 1:
            pcim->rxbuf_rdptr += 3;
            pcim->rxbuf_bytes_unused += 3;
            break;
        case 2:
            pcim->rxbuf_rdptr += 2;
            pcim->rxbuf_bytes_unused += 2;
            break;
        case 3:
            pcim->rxbuf_rdptr += 1;
            pcim->rxbuf_bytes_unused += 1;
            break;
        default:
            /* yipee! it is aligned */
            break;
    }

    if( (u32)pcim->rxbuf_rdptr >= (u32)pcim->rxbuf_end ) {
        /* reached end, reset to start of rxbuf */
        pcim->rxbuf_rdptr = pcim->rxbuf;
    }

    FPD_DDEBUG(pfpd->id, 3,
               "CIM Read fix rdptr Unused %d Read %d RXbuf rdptr %p\n",
               pcim->rxbuf_bytes_unused, bytes_read, pcim->rxbuf_rdptr);

    if( atomic_read(&pcim->rxbuf_no_space) ) {
        fpd_linkif_t *plink;
        u32 reg;
        u32 status_addr;
        u32 intr_mask;
        u32 avail_bytes;
        int hdr_size;

        if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
            plink = pfpd->bgnd_link;
        }
        else {
            plink = pfpd->link[pcim->link_if];
        }

        switch( pcim->btype ) {
            case FPD_CIM_BUFFER:
                status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
                intr_mask = LIFCR_RX_CIM_BUF_INT_EN;
                break;
            case FPD_PRIORITY_BUFFER:
                status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
                intr_mask = LIFCR_RX_PRI_BUF_INT_EN;
                break;
            default:
                return -EINVAL;
        }

        /*
         * More data is available from FPGA but there was no space before.
         * Some space opened up but have to make sure it is enough to 
         * store the new packet.
         */
        reg = FPD_READ(pfpd->remap, status_addr);
        if(( reg & LIFCBSR_RX_PKT_AVAIL ) && ( reg & LIFCBSR_RX_SOP )) {
            avail_bytes = ((reg & LIFCBSR_RX_PKT_CNT) >> 3);

            if( plink->protocol == FPD_PROTOCOL_VM ) {
                hdr_size = 0;
            }
            else {
                /* For paragon, buffer contains the packet size as well */
                hdr_size = sizeof(u32);
            }

            if( (avail_bytes + hdr_size) <= pcim->rxbuf_bytes_unused ) {
                atomic_set(&pcim->rxbuf_no_space, 0);

                /* enable RX buffer interrupt */
                reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
                reg |= intr_mask;
                FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);
            }
        }
        else {
            FPD_ERROR("INCONSISTENT RX BUFFER SPACE!\n");
#if 0
            atomic_set(&pcim->rxbuf_no_space, 0);

            /* enable RX buffer interrupt */
            reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
            reg |= intr_mask;
            FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);
#endif
        }
    }

 pdata_read_error:
    /* release lock */
    fpd_lock_release(&pcim->rx_lock);

    return bytes_read;
}

/******************************************************************************
 *  ROUTINE:        pdata_read_buffer_data_register()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read data directly from the CIM/Priority Data Buffer of the 
 *      FPGA Protocol device.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Succeeded.
 *     -EINVAL  Invalid buffer type defined inside data structure.
 *     -ENOSPC  Run out of space in driver's RX Buffer.
 *
 *  NOTE:
 *
 *      Write pointer is always 4-byte aligned.
 *      Read pointer has to make sure that it ends in a 4-byte alignment.
 *      Write pointer shall only be updated after a complete packet.
 *
 *****************************************************************************/
static int
pdata_read_buffer_data_register( fpd_pdata_t *pcim )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
    int avail_bytes;
    u32 rx_status_word;
    u32 addr_mask;
    u32 buf_addr, status_addr;
    int hdr_size;

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }

    switch( pcim->btype ) {
        case FPD_CIM_BUFFER:
            buf_addr = plink->base + LIF_RX_CIM_BUFFER_DATA;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;
        case FPD_PRIORITY_BUFFER:
            buf_addr = plink->base + LIF_RX_PRIORITY_BUFFER_DATA;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;
        default:
            return -EINVAL;
    }

    /* get lock */
    fpd_lock_acquire(&pcim->rx_lock);

    /* verify that the write will occur on a 4-byte alignment */
    if( (u32)pcim->rxbuf_wrptr & 0x03 ) {
        FPD_ERROR("CIM RX Buffer Write Ptr Alignment error!\n");

        /* Damn it! Adjust it then */
        addr_mask = (u32)pcim->rxbuf_wrptr & 0x03;
        switch( addr_mask ) {
            case 1:
                pcim->rxbuf_wrptr += 3;
                break;
            case 2:
                pcim->rxbuf_wrptr += 2;
                break;
            case 3:
                pcim->rxbuf_wrptr += 1;
                break;
            default:
                /* yipee! it is aligned */
                break;
        }
    }

    /*
     * Read the Receive Status Word from the RX CIM Buffer Data
     * register to obtain the byte count for the packet that follows.
     */
    avail_bytes = ((FPD_READ(pfpd->remap, status_addr) & LIFCBSR_RX_PKT_CNT) >> 3);

#if 1
    FPD_DDEBUG(pfpd->id, 3, "CIM RX Len %d bytes_unused %d\n",
               avail_bytes, pcim->rxbuf_bytes_unused);
#else
    FPD_WARNING("CIM RX Len %d bytes_unused %d\n",
                avail_bytes, pcim->rxbuf_bytes_unused);
#endif

    if( plink->protocol == FPD_PROTOCOL_VM ) {
        hdr_size = 0;
    }
    else {
        /* For paragon, circular buffer contains the packet size as well */
        hdr_size = sizeof(u32);
    }

    /* handle condition: no space in buffer to store data */
    if( (avail_bytes + hdr_size) > pcim->rxbuf_bytes_unused ) {
        u32 reg;

        FPD_DDEBUG(pfpd->id, 2, "CIM RX Buffer run out of space!\n");

        /*
         * notify RX buffer's reader that if space becomes available 
         * somebody needs to do processing of the received packets from
         * the FPGA
         */
        atomic_set(&pcim->rxbuf_no_space, 1);

        /* disable RX buffer interrupt for now */
        reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
        switch( pcim->btype ) {
            case FPD_CIM_BUFFER:
                reg &= ~(LIFCR_RX_CIM_BUF_INT_EN);
                break;
            case FPD_PRIORITY_BUFFER:
                reg &= ~(LIFCR_RX_PRI_BUF_INT_EN);
                break;
            default:
                break;
        }
        FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

        fpd_lock_release(&pcim->rx_lock);
        return -ENOSPC;
    }

    /* still need to take the rx status word from the RX buffer data */
    rx_status_word = FPD_READ(pfpd->remap, buf_addr);
    if( avail_bytes != rx_status_word ) {
        /* this should not happen */
        FPD_ERROR("Buffer Status RX Packet Size %d != Buffer Data RX Status Word %d\n",
                  avail_bytes, rx_status_word);
    }

    if( plink->protocol == FPD_PROTOCOL_PARAGON ) {
        /* prepend the packet length */
        *(u32 *)pcim->rxbuf_wrptr = rx_status_word;
        pcim->rxbuf_wrptr += sizeof(u32);
        pcim->rxbuf_bytes_unused -= sizeof(u32);
        if( (u32)pcim->rxbuf_wrptr >= (u32)pcim->rxbuf_end ) {
            /* reached end, reset wrptr to start of buffer */
            pcim->rxbuf_wrptr = pcim->rxbuf;
        }
    }

    while( avail_bytes > 0 ) {
        FPD_DDEBUG(pfpd->id, 3,
                   "CIM RX wrptr = %p rdptr = %p unread %d unused %d\n",
                   pcim->rxbuf_wrptr, pcim->rxbuf_rdptr,
                   avail_bytes, pcim->rxbuf_bytes_unused);

        /*
         * Read the RX CIM Buffer Data register to read the packet
         * from the buffer.
         */
        *(volatile u32 *)pcim->rxbuf_wrptr = FPD_READ_SWAP(pfpd->remap, buf_addr);

        avail_bytes -= sizeof(u32);

        /* manage wrptr location */
        pcim->rxbuf_wrptr += sizeof(u32);
        pcim->rxbuf_bytes_unused -= sizeof(u32);
        if( pcim->rxbuf_wrptr >= pcim->rxbuf_end ) {
            /* reset wrptr to start of buffer */
            pcim->rxbuf_wrptr = pcim->rxbuf;
        }
    }

    FPD_DDEBUG(pfpd->id, 3, "CIM RX completed wrptr = %p rdptr = %p unused %d\n",
               pcim->rxbuf_wrptr, pcim->rxbuf_rdptr,
               pcim->rxbuf_bytes_unused);

    /* release lock */
    fpd_lock_release(&pcim->rx_lock);

    return 0;
}

/******************************************************************************
 *  ROUTINE:        pdata_receive()
 ******************************************************************************
 *
 *  DESCRIPTION:
 *
 *      Read data directly from the CIM Data Buffer or Priority Buffer 
 *      of the FPGA Protocol device.
 *
 *  PARAMETERS:
 *
 *      pcim    Pointer to the protocol data structure.
 *
 *  RETURNS:
 *
 *     	0       Succeeded.
 *     -EINVAL  Invalid buffer type defined inside data structure.
 *     -ENOSPC  Run out of space in driver's RX Buffer.
 *
 *****************************************************************************/
int
pdata_receive( fpd_pdata_t *pcim )
{
    fpd_device_t *pfpd = (fpd_device_t *)pcim->private;
    fpd_linkif_t *plink;
    char in_progress = 1;
    int result = 0;
    u32 buf_addr, status_addr;
    u32 reg;

    /**********************************************/
    /* packet shall be stored in 4-byte alignment */
    /**********************************************/

    if( pcim->link_if == FPD_LINK_IF_ID_BGND ) {
        plink = pfpd->bgnd_link;
    }
    else {
        plink = pfpd->link[pcim->link_if];
    }

    switch( pcim->btype ) {
        case FPD_CIM_BUFFER:
            buf_addr = plink->base + LIF_RX_CIM_BUFFER_DATA;
            status_addr = plink->base + LIF_CIM_BUFFER_STATUS;
            break;
        case FPD_PRIORITY_BUFFER:
            buf_addr = plink->base + LIF_RX_PRIORITY_BUFFER_DATA;
            status_addr = plink->base + LIF_PRIORITY_BUFFER_STATUS;
            break;
        default:
            return -EINVAL;
    }

    /*
     * Read RX CIM Buffer Status register and check that both RX_PKT_AVAIL
     * and RX_SOP are set.
     */
    do {
        reg = FPD_READ(pfpd->remap, status_addr);

        if( reg & LIFCBSR_RX_PKT_AVAIL ) {
            if( reg & LIFCBSR_RX_SOP ) {
                if(( result = pdata_read_buffer_data_register(pcim) ) < 0 ) {
                    in_progress = 0;
                }
            }
            else {
                char resync_completed = 0;

                FPD_ERROR("RX CIM Data Packet SOP missing\n");
#if 1
                /*
                 * If the RX_SOP bit is not set, resynchronize by reading from
                 * the RX CIM Buffer Data register until the RX_SOP bit is set.
                 */
                do {
                    if( !(reg & LIFCBSR_RX_EMPTY) ) {
                        reg = FPD_READ(pfpd->remap, buf_addr);
                        FPD_WARNING("Data 0x%08x\n", reg);
                    }
                    reg = FPD_READ(pfpd->remap, status_addr);
                    if( reg & LIFCBSR_RX_EMPTY ) {
                        if( reg & LIFCBSR_RX_PKT_AVAIL ) {
                            FPD_ERROR("RX_PKT_AVAIL and RX_EMPTY bits are both set\n");
                        }
                        FPD_WARNING("RX SOP failed to resynchronize 1\n");
                        resync_completed = -1;
                        in_progress = 0;
                    }
                    else if( reg & LIFCBSR_RX_PKT_AVAIL ) {
                        if( reg & LIFCBSR_RX_SOP ) {
                            /* resynchronization succeeded */
                            resync_completed = 1;
                        }
                    }
                    else {
                        /* forget it */
                        FPD_ERROR("RX SOP failed to resynchronize 2\n");
                        resync_completed = -1;
                        in_progress = 0;
                    }
                } while( !resync_completed );

                if( resync_completed > 0 ) {
                    FPD_WARNING("RX SOP resynchronized\n");
                    if((result = pdata_read_buffer_data_register(pcim)) < 0 ) {
                        in_progress = 0;
                    }
                }
#else
                /* disable RX CIM Buffer interrupt */
                reg = FPD_READ(pfpd->remap, plink->base + LIF_LINK_CONTROL);
                reg &= ~(LIFCR_RX_CIM_BUF_INT_EN | LIFCR_TX_TIMEOUT_INT_EN);
                FPD_WRITE(pfpd->remap, plink->base + LIF_LINK_CONTROL, reg);

                in_progress = 0;
#endif
            }
        }
        else {
            in_progress = 0;
        }
    } while( in_progress );

    return result;
}
