/**
 * gpio.c
 *
 * Top-Level GPIO kernel module
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *                        Michael Baumann <miba@peppercon.de>
 *
 * TODO
 * - support level trigger irq if needed
 * - add request/release/watch for kernel access if needed/possible
 */

#include <linux/version.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/wait.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# include <linux/workqueue.h>
#endif
#include <asm/io.h>
#include <asm/semaphore.h>
#include <asm/atomic.h>

#include <pp_gpio.h>
#include "gpio_common.h"
#include "gpio_core.h"

#ifdef PP_BOARD_KIRA
# include "gpio_core_faraday.h"
# include "gpio_core_vsc.h"
#else
# include "gpio_core_ibm.h"
# if defined(LARA_KIMAMD) || defined(LARA_KIMMSI)
#   include "gpio_core_opencores.h"
# elif defined(PRODUCT_ICPMMD)
#   include "gpio_core_xr17d158.h"
# endif
#endif /* !PP_BOARD_KIRA */

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define GPIO_MAJOR		((u8)247)
#define GPIO_MAX_DEVICES	16
#define	GPIO_KERNEL_SUPPORT	1

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

/**
 * init data to build a list of supported cores
 */
typedef struct {
    u8			minor;		/* device minor number for this core */
    u8			core_nr;	/* core number, unique for a core/driver type */

    /* init/cleanup functions, determine core to be used */
    core_init_fn_t	init;

    u8			lazy_init;	/* don't init on module load, init on open */
} gpio_core_init_t;

/* device private data */
struct gpio_dev_s {
    int			initialized;
    atomic_t		refcnt;
    gpio_core_t *	core;		/* core specific data and operations */
    spinlock_t		mtx;		/* device lock */

    gpio_core_init_t*   init_data;	/* to be saved for lazy init */
    
    u_long		requested;	/* mask of reserved bits for exclusive write */
    struct list_head	watch_list;	/* file descs which are waiting for IRQs */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    struct tq_struct	dsr;		/* deferred service routing for intr handling in non-irq mode */
#else
    struct work_struct	dsr;		/* deferred service routing for intr handling in non-irq mode */
#endif
    u_long		intr_state;	/* transfers intr state from ISR to DSR */
    spinlock_t		intr_state_lock;/* locks intr_state access */
};

/* file descriptor private data */
typedef struct {
    gpio_dev_t *	dev;
    u_long		requested;	/* mask of reserved bits for fd's exclusive write */
    struct list_head	watch_anchor;	/* anchor for inserting into dev's watch_list */
    u_long		watch_mask;	/* mask of bits to be watched */
    u_long		watch_modified;	/* watched bits that have been modified since last read */
    wait_queue_head_t   watch_wait;	/* wait queue for poll/select */
#ifdef GPIO_KERNEL_SUPPORT
    void 		(*watchk_cb)(void * arg); /* call back function */
    void 		*watchk_arg;		  /* argument used for watch function */
#endif /* GPIO_KERNEL_SUPPORT */   
} gpio_file_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

/**
 * list of cores to use, gpio device list gets build from that
 */
gpio_core_init_t core_init_list[] = {
    /* minor, core number, init function */
#ifdef PP_BOARD_KIRA
    { PP_GPIO_DEV_FARADAY,	0, gpio_core_faraday_init,	0 },
# ifdef PP_FEAT_VSC_SINGLE_PIXEL
    { PP_GPIO_DEV_VSC,		0, gpio_core_vsc_init,		0 },
# endif
#else
    { PP_GPIO_DEV_IBM,		0, gpio_core_ibm_init,		0 },    
# if defined(LARA_KIMAMD) || defined(LARA_KIMMSI)
    { PP_GPIO_DEV_OPENCORES_0,	0, gpio_core_opencores_init,	1 },
    { PP_GPIO_DEV_OPENCORES_1,	1, gpio_core_opencores_init,	1 },
# elif defined(PRODUCT_ICPMMD)
    { PP_GPIO_DEV_XR17_0,	0, gpio_core_xr17d158_init,	1 },
    { PP_GPIO_DEV_XR17_1,	1, gpio_core_xr17d158_init,	1 },
# endif    
#endif /* ! PP_BOARD_KIRA */
    
    { 0, 0, NULL }
};

static gpio_dev_t gpio_devs[GPIO_MAX_DEVICES];

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

int init_module(void);
void cleanup_module(void);
static int gpio_init(void);
static void gpio_cleanup(void);
static int gpio_open(struct inode *inode, struct file *file);
static int gpio_release(struct inode *inode, struct file *file);
static int gpio_ioctl(struct inode *inode, struct file *file, u_int cmd, u_long arg);
static u_int gpio_poll(struct file *file, poll_table *polltab);

/* exported functions for kernel gpio access */

int pp_gpio_init(u_char dev_no);

int pp_gpio_get(u_char dev_no, u_long *value);
int pp_gpio_set(u_char dev_no, u_long set, u_long clr, u_long tri);

int pp_gpio_is_set(u_char dev_no, u_long *value);
int pp_gpio_is_clr(u_char dev_no, u_long *value);
int pp_gpio_is_tri(u_char dev_no, u_long *value);

int pp_gpio_bit_set(u_char dev_no, u_char bit, u_char value);
int pp_gpio_bit_set_enable(u_char dev_no, u_char bit, u_char value);
int pp_gpio_bit_get(u_char dev_no, u_char bit, u_char *value);

int pp_gpio_alternate_set(u_char dev_no, u_long set, u_long clr);
int pp_gpio_alternate_get(u_char dev_no, u_long *value);

EXPORT_SYMBOL(pp_gpio_init);
EXPORT_SYMBOL(pp_gpio_get);
EXPORT_SYMBOL(pp_gpio_set);
EXPORT_SYMBOL(pp_gpio_is_set);
EXPORT_SYMBOL(pp_gpio_is_clr);
EXPORT_SYMBOL(pp_gpio_is_tri);
EXPORT_SYMBOL(pp_gpio_bit_set);
EXPORT_SYMBOL(pp_gpio_bit_set_enable);
EXPORT_SYMBOL(pp_gpio_bit_get);
EXPORT_SYMBOL(pp_gpio_alternate_set);
EXPORT_SYMBOL(pp_gpio_alternate_get);

/* internal functions which do the actual work (and call the cores) */

static inline int gpio_get(gpio_dev_t *dev, u_long *value);
static inline int gpio_set(gpio_dev_t *dev, u_long set, u_long clr, u_long tri);

static inline int gpio_is_set(gpio_dev_t *dev, u_long *value);
static inline int gpio_is_clr(gpio_dev_t *dev, u_long *value);
static inline int gpio_is_tri(gpio_dev_t *dev, u_long *value);

static inline int gpio_bit_set(gpio_dev_t *dev, u_char bit, u_char value);
static inline int gpio_bit_set_enable(gpio_dev_t *dev, u_char bit, u_char value);
static inline int gpio_bit_get(gpio_dev_t *dev, u_char bit, u_char *value);

static inline int gpio_alternate_set(gpio_dev_t *dev, u_long set, u_long clr);
static inline int gpio_alternate_get(gpio_dev_t *dev, u_long *value);

static inline int gpio_irq_ack(gpio_dev_t *dev, u_long mask);

/* internal functions */

static int  gpio_init_device(gpio_dev_t *dev);
static inline gpio_dev_t* gpio_lookup_device(u_char dev_no);

static void gpio_isr(struct gpio_core_s *core, u_long state);
static void gpio_dsr(void *ctx);

static void watch(gpio_dev_t* dev, gpio_file_t* priv, u_long mask);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Michael Baumann <miba@peppercon.de>, Ralf Guenther <rgue@peppercon.de>");
MODULE_DESCRIPTION("Linux kernel module for GPIO interfaces");
MODULE_LICENSE("GPL");

/*
 * Initialize the module
 */
int init_module(void)
{
    return gpio_init();
}

/*
 * Cleanup module
 */
void cleanup_module(void)
{
    gpio_cleanup();
}

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations gpio_ops = {
    owner:	THIS_MODULE,
    open:	gpio_open,
    release:	gpio_release,
    ioctl:	gpio_ioctl,
    poll:	gpio_poll
};

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

static int
gpio_init(void)
{
    gpio_core_init_t* init_data = core_init_list;    
    int ret = SUCCESS, nr_cores = 0, r;
    gpio_dev_t *dev;

    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(GPIO_MAJOR, NAME, &gpio_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto bail;
    }

    /* ---- initialize all cores --------------------------------------- */
    memset(gpio_devs, 0, sizeof(gpio_devs));
    
    while (init_data && init_data->init != NULL) {
	gpio_core_t *core;
	
	dev = &gpio_devs[init_data->minor];

	if (dev->initialized) {
	    WARN("Minor nr %d already initialized, strange, check init_data!",
		 init_data->minor);
	}
	
	atomic_set(&dev->refcnt, 0);
        INIT_LIST_HEAD(&dev->watch_list);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	INIT_TQUEUE(&dev->dsr, gpio_dsr, dev);
#else
	INIT_WORK(&dev->dsr, gpio_dsr, dev);
#endif
        spin_lock_init(&dev->mtx);
        spin_lock_init(&dev->intr_state_lock);
	
	if ((core = kmalloc(sizeof(gpio_core_t), GFP_KERNEL)) == NULL) {
	    ERR("Could not alloc memory for core");
	    ret = -ENOMEM;
	    goto bail;
	}
	memset(core, 0, sizeof(gpio_core_t));
	
	/* connect core and device */
	dev->core = core;
	core->dev = dev;
	dev->init_data = init_data;

	/* if device is not lazy-initialized, do it now */
	if (init_data->lazy_init == 0) {
	    if (gpio_init_device(dev) != 0) {
		ERR("Could not initialize device");
		ret = -ENODEV;
		goto bail;
	    }
	}
	
	init_data++;
	nr_cores++;
    }
    
    if (nr_cores == 0) {
	ERR("No cores available");
	ret = -ENODEV;
	goto bail;
    } else {
	INFO("Overall number cores: %d", nr_cores);
    }
    
 bail:
    if (ret != SUCCESS) {
	gpio_cleanup();
    }
    return ret;
}

static void
gpio_cleanup(void)
{
    int i = 0;
    int r;
    
    while (i < GPIO_MAX_DEVICES) {
	gpio_dev_t *dev = &gpio_devs[i];

	if (dev->core && dev->core->cleanup) {
	    dev->core->cleanup(dev->core);	    
	}

	if (dev->core != NULL) {
	    kfree(dev->core);
	}

	i++;
    }

    if ((r = unregister_chrdev(GPIO_MAJOR, NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }
}

/* ------------------------------------------------------------------------- *
 * the driver operations 
 * ------------------------------------------------------------------------- */

static int
gpio_open(struct inode * inode, struct file * file)
{
    int ret = SUCCESS;
    gpio_file_t *priv;
    gpio_dev_t *dev;
    int minor = MINOR(inode->i_rdev);
    int new_refcnt;
    u_long flags;
   
    if (minor < 0 || minor >= GPIO_MAX_DEVICES) {
	ERR("Unsupported GPIO device (minor number out of range)");
	return -ENODEV;
    }

    dev = &gpio_devs[minor];

    if (gpio_init_device(dev) != 0) {
	ERR("Could not initialize GPIO device");
	return -ENODEV;
    }
    
    spin_lock_irqsave(&dev->mtx, flags);
    if ((priv = kmalloc(sizeof(gpio_file_t), GFP_KERNEL)) == NULL) {
	ERR("Could not allocate memory for file data");
	ret = -ENOMEM;
	goto bail;
    }

    memset(priv, 0, sizeof(gpio_file_t));
    priv->dev = dev;
    INIT_LIST_HEAD(&priv->watch_anchor);
    init_waitqueue_head(&priv->watch_wait);
    file->private_data = priv;   

    new_refcnt = atomic_inc_return(&dev->refcnt);
    if (new_refcnt == 1) {
	/* initialization on first open */
	
    }

    if (dev->core->open) dev->core->open(dev->core, new_refcnt);
    
    DBG("GPIO device (minor %d) opened", minor);
    
 bail:
    spin_unlock_irqrestore(&dev->mtx, flags);
    return ret;
}

static int
gpio_release(struct inode *inode, struct file *file)
{
    gpio_file_t *priv = (gpio_file_t*) file->private_data;
    gpio_dev_t *dev = priv->dev;
    gpio_core_t *core = dev->core;
    int ret = SUCCESS;
    int new_refcnt;
    u_long flags;

    spin_lock_irqsave(&dev->mtx, flags);

    new_refcnt = atomic_dec_return(&dev->refcnt);

    watch(dev, priv, 0); /* remove from dev's watch list */
    dev->requested &= ~priv->requested; /* free requested bits */
    kfree(priv); file->private_data = NULL;
    
    if (core->release) core->release(core, new_refcnt);

    if (new_refcnt == 0) {
	/* last reference cleanup */
    }
    
    DBG("GPIO device (minor %d) closed", MINOR(inode->i_rdev));
    spin_unlock_irqrestore(&dev->mtx, flags);
    return ret;
}

static int
gpio_ioctl(struct inode *inode, struct file *file, u_int cmd, u_long arg)
{
    gpio_file_t *priv = (gpio_file_t*) file->private_data;
    gpio_dev_t *dev = priv->dev;
    gpio_core_t *core = dev->core;
    u_long flags;
    int ret = SUCCESS;

    spin_lock_irqsave(&dev->mtx, flags);

    switch (cmd) {
      case GPIO_IOCTL_REQUEST:
	  {
	      u_long v;
	      if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      if (v & dev->requested) { ret = -EBUSY; break; }
	      priv->requested |= v;
	      dev->requested |= v;
	      break;
	  }

      case GPIO_IOCTL_RELEASE:
	  {
	      u_long v;
	      if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      if ((v & priv->requested) != v) { ret = -EINVAL; break; }
	      priv->requested &= ~v;
	      dev->requested &= ~v;
	      break;
	  }

      case GPIO_IOCTL_IS_REQUESTED:
	  {
	      u_long v = dev->requested;
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }

      case GPIO_IOCTL_WATCH:
	  {
	      u_long v;
	      if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      if ((core->caps & GPIO_CAPS_IRQ) == 0 || core->irq_enable == NULL) {
		  ret = -EINVAL; break;
	      }
      
	      watch(dev, priv, v);
	      break;
	  }

      case GPIO_IOCTL_IRQCONF:
	  {
	      gpio_ioctl_irqconf_t c;

	      if (copy_from_user(&c, (void*)arg, sizeof(c))) { ret = -EFAULT; break; }
	      if ((c.level & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((c.edge & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((c.pos & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((c.neg & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((core->caps & GPIO_CAPS_IRQ) == 0 || core->irq_edge == NULL) {		     
		      ret = -EINVAL; break;
	      }

	      /* some sanity checks */

	      if (c.level & c.edge) {
		  ERR("Either choose level trigger (0x%08lx) _or_ edge trigger (0x%08lx) for each pin.",
		      c.level, c.edge);
		  ret = -EINVAL;
		  break;	  
	      }
	      
	      if (c.level & c.pos & c.neg) {
		  ERR("Level triggered IRQ (0x%08lx) does not make sense with both pos (0x%08lx) and neg (0x%08lx).",
		      c.level, c.pos, c.neg);
		  ret = -EINVAL;
		  break;
	      }
	      
	      if (core->irq_edge(core, c.level, c.edge, c.pos, c.neg) != 0) {
		  ret = -EIO;
	      }
	      break;	      
	  }
	  
      case GPIO_IOCTL_IN:
	  {
	      u_long v;

	      if (gpio_get(dev, &v) != 0) {
		  ret = -EIO;
		  break;		  
	      }

	      gpio_irq_ack(dev, priv->watch_modified);
	      priv->watch_modified = 0; /* reset all "modified" flags */

	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_OUT:
	  {
	      gpio_ioctl_out_t o;
	      if (copy_from_user(&o, (void*)arg, sizeof(o))) { ret = -EFAULT; break; }
	      if ((o.set & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((o.clr & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((o.tri & ~priv->requested) & dev->requested) { ret = -EACCES; break; }

	      if (gpio_set(dev, o.set, o.clr, o.tri) != 0) {
		  ret = -EIO;
	      }
	      break;
	  }

      case GPIO_IOCTL_SET:
	{
	    u_long v;
	    if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	    if ((v & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	    
	    if (gpio_set(dev, v, 0, 0) != 0) {
		ret = -EIO;
	    }
	    break;
	}
	
      case GPIO_IOCTL_CLR:
	  {
	      u_long v;
	      if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      if ((v & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      
	      if (gpio_set(dev, 0, v, 0) != 0) {
		  ret = -EIO;
	      }
	      break;
	  }
	  
      case GPIO_IOCTL_TRI:
	  {
	      u_long v;
	      if (get_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      if ((v & ~priv->requested) & dev->requested) { ret = -EACCES; break; }

	      if (gpio_set(dev, 0, 0, v) != 0) {
		  ret = -EIO;
	      }
	      break;
	  }
	  
      case GPIO_IOCTL_IS_SET:
	  {
	      u_long v;

	      if (gpio_is_set(dev, &v) != 0) {
		  ret = -EIO;
		  break;
	      }
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_IS_CLR:
	  {
	      u_long v;

	      if (gpio_is_clr(dev, &v) != 0) {
		  ret = -EIO;
		  break;
	      }
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_IS_TRI:
	  {
	      u_long v;

	      if (gpio_is_tri(dev, &v) != 0) {
		  ret = -EIO;
		  break;
	      }
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_GET_BIT:
	  {
	      gpio_ioctl_bit_t b;
	      
	      if (copy_from_user(&b, (void*)arg, sizeof(b))) { ret = -EFAULT; break; }

	      if (gpio_bit_get(dev, b.idx, &b.val) != 0) {
		  ret = -EIO;
		  break;
	      }
	      
	      gpio_irq_ack(dev, 1 << b.idx);
	      priv->watch_modified &= ~(1 << b.idx); /* reset this "modified" flags */
	      
	      if (copy_to_user((void*)arg, &b, sizeof(b))) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_SET_BIT:
	  {
	      gpio_ioctl_bit_t b;

	      if (copy_from_user(&b, (void*)arg, sizeof(b))) { ret = -EFAULT; break; }

	      if (((1 << b.idx) & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if (gpio_bit_set(dev, b.idx, b.val) != 0) {
		  ret = -EIO;
	      }	            
	      break;
	  }

      case GPIO_IOCTL_SET_BIT_ENBL:
	  {
	      gpio_ioctl_bit_t b;

	      if (copy_from_user(&b, (void*)arg, sizeof(b))) { ret = -EFAULT; break; }
	      if (((1 << b.idx) & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if (gpio_bit_set_enable(dev, b.idx, b.val) != 0) {
		  ret = -EIO;
	      }	                 
	      break;
	  }
	  
      case GPIO_IOCTL_SET_ALT:
	  {
	      gpio_ioctl_alternate_t a;
	      if (copy_from_user(&a, (void*)arg, sizeof(a))) { ret = -EFAULT; break; }
	      
	      if ((a.set & ~priv->requested) & dev->requested) { ret = -EACCES; break; }
	      if ((a.clr & ~priv->requested) & dev->requested) { ret = -EACCES; break; }

	      if (gpio_alternate_set(dev, a.set, a.clr) != 0) {
		  ret = -EIO;
	      }
	      break;
	  }
	  
      case GPIO_IOCTL_GET_ALT:
	  {
	      u_long v;
	      if (core->get_alternate == NULL || (core->caps & GPIO_CAPS_ALTERNATE_FUNCTIONS) == 0) {
		  ret = -EINVAL;
		  break;
	      }

	      if (gpio_alternate_get(dev, &v) != 0) {
		  ret = -EIO;
		  break;
	      }
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
      case GPIO_IOCTL_GET_MODIFIED:
	  {
	      u_long v = priv->watch_modified;
	      if (put_user(v, (u_long*)arg)) { ret = -EFAULT; break; }
	      break;
	  }
	  
	  /* TODO(miba), add GPIO_IOCTL_GET_REG and GPIO_IOCTL_SET_REG if needed */
	  
      default:
	  /* invalid cmd */
	  ret = -EINVAL;
    }
    
    spin_unlock_irqrestore(&dev->mtx, flags);
    return ret;
}

static u_int
gpio_poll(struct file *file, poll_table *polltab)
{
    gpio_file_t *priv = (gpio_file_t*) file->private_data;
    u_int mask = 0;

    poll_wait(file, &priv->watch_wait, polltab);
    if (priv->watch_modified) mask |= POLLIN;

    return mask;
}

/* ------------------------------------------------------------------------- *
 * exported functions for kernel gpio access
 * ------------------------------------------------------------------------- */

int
pp_gpio_init(u_char dev_no)
{
    gpio_dev_t *dev;
    int ret = -1;

    if ((dev = gpio_lookup_device(dev_no)) == NULL) {
	ERR("Unsupported GPIO device");
	goto bail;
    }

    if (gpio_init_device(dev) != 0) {
	ERR("Could not initialize GPIO device");
	goto bail;
    }
        
    ret = 0;
 bail:
    return ret;
}

int
pp_gpio_get(u_char dev_no, u_long *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);
    ret = gpio_get(dev, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;   
}

int
pp_gpio_set(u_char dev_no, u_long set, u_long clr, u_long tri)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);
    ret = gpio_set(dev, set, clr, tri);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_is_set(u_char dev_no, u_long *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_is_set(dev, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_is_clr(u_char dev_no, u_long *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_is_clr(dev, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_is_tri(u_char dev_no, u_long *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_is_tri(dev, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_bit_set(u_char dev_no, u_char bit, u_char value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_bit_set(dev, bit, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_bit_set_enable(u_char dev_no, u_char bit, u_char value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_bit_set_enable(dev, bit, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_bit_get(u_char dev_no, u_char bit, u_char *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);
    ret = gpio_bit_get(dev, bit, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_alternate_set(u_char dev_no, u_long set, u_long clr)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_alternate_set(dev, set, clr);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

int
pp_gpio_alternate_get(u_char dev_no, u_long *value)
{
    gpio_dev_t *dev = gpio_lookup_device(dev_no);
    u_long flags;
    int ret = -1;

#ifdef ENABLE_CHECKS
    if (dev == NULL) return ret;
#endif

    spin_lock_irqsave(&dev->mtx, flags);    
    ret = gpio_alternate_get(dev, value);
    spin_unlock_irqrestore(&dev->mtx, flags);

    return ret;
}

/* ------------------------------------------------------------------------- *
 * functions which do the actual work (and call the cores)
 * ------------------------------------------------------------------------- */

static inline int
gpio_get(gpio_dev_t *dev, u_long *value)
{
    gpio_core_t *core = dev->core;
    int ret = -1;

    if (core->get_input(core, value) != 0) {	
	goto bail;
    }

    ret = 0;
 bail:
    return ret;
}

static inline int
gpio_set(gpio_dev_t *dev, u_long set, u_long clr, u_long tri)
{
    gpio_core_t *core = dev->core;
    int ret = -1;

    if (core->set_enable(core, 0, tri | set | clr) != 0 ||
	core->set_output(core, set, clr) != 0 ||
	core->set_enable(core, set | clr, 0) != 0) {
	
	goto bail;
    }

    ret = 0;    
 bail:
    return ret;
}

static inline int
gpio_is_set(gpio_dev_t *dev, u_long *value)
{
    gpio_core_t *core = dev->core;
    u_long o, e;
    int ret = -1;
    
    if (core->get_enable(core, &e) != 0  ||
	core->get_output(core, &o) != 0) {
	
	goto bail;
    }
    *value = o & e;
    
    ret = 0;    
 bail:
    return ret;
}

static inline int
gpio_is_clr(gpio_dev_t *dev, u_long *value)
{
    gpio_core_t *core = dev->core;
    u_long o, e;
    int ret = -1;

    if (core->get_enable(core, &e) != 0  ||
	core->get_output(core, &o) != 0) {
	goto bail;
    }     
    *value = ~o & e;
    
    ret = 0;    
 bail:
    return ret;
}

static inline int
gpio_is_tri(gpio_dev_t *dev, u_long *value)
{
    gpio_core_t *core = dev->core;
    u_long e;
    int ret = -1;

    if (core->get_enable(core, &e) != 0) {
	goto bail;
    }     
    *value = ~e;
	      
    ret = 0;
 bail:
    return ret;
}

static inline int
gpio_bit_set(gpio_dev_t *dev, u_char bit, u_char value)
{
    gpio_core_t *core = dev->core;
    u_long mask = 1 << bit;

    return core->set_output(core, value ? mask : 0, value ? 0 : mask);
}

static inline int
gpio_bit_set_enable(gpio_dev_t *dev, u_char bit, u_char value)
{
    gpio_core_t *core = dev->core;
    u_long mask = 1 << bit;

    return core->set_enable(core, value ? mask : 0, value ? 0 : mask);
}

static inline int
gpio_bit_get(gpio_dev_t *dev, u_char bit, u_char *value)
{
    gpio_core_t *core = dev->core;
    u_long v;
    int ret = -1;

    if (core->get_input(core, &v) != 0) {
	goto bail;
    }    
    *value = (v & (1 << bit)) ? 1 : 0;
    
    ret = 0;
 bail:
    return ret;
}

static inline int
gpio_alternate_set(gpio_dev_t *dev, u_long set, u_long clr)
{
    gpio_core_t *core = dev->core;
    int ret = -1;

    if (core->set_alternate && (core->caps & GPIO_CAPS_ALTERNATE_FUNCTIONS)) {
	if (core->set_alternate(core, set, clr) != 0) {
	    goto bail;
	}
    } else {
	goto bail;
    }      
        
    ret = 0;    
 bail:
    return ret;   
}

static inline int
gpio_alternate_get(gpio_dev_t *dev, u_long *value)
{
    gpio_core_t *core = dev->core;
    int ret = -1;

    if (value == NULL ||
	core->get_alternate == NULL ||
	((core->caps & GPIO_CAPS_ALTERNATE_FUNCTIONS) == 0) ||
	core->get_alternate(core, value) != 0) {

	goto bail;
    }
    
    ret = 0;    
 bail:
    return ret;
}

static inline int
gpio_irq_ack(gpio_dev_t *dev, u_long mask)
{
    gpio_core_t *core = dev->core;
    int ret = -1;
    
    if (core->irq_ack) {
	ret = core->irq_ack(core, mask);
    }

    return ret;
}

/* ------------------------------------------------------------------------- *
 * private/helper functions
 * ------------------------------------------------------------------------- */

static int
gpio_init_device(gpio_dev_t *dev)
{
    gpio_core_t *core = dev->core;
    u_long flags;
    int ret = -1;    
    
    if (dev->init_data == NULL) {
	ERR("GPIO core not available");
	return ret;
    }

    spin_lock_irqsave(&dev->mtx, flags);

    if (dev->initialized) {
	ret = 0;
	goto bail;
    }
    
    if (dev->init_data->init(core, dev->init_data->core_nr) != 0) {
	ERR("Could not intialize core");
	goto bail;
    }
    
    core->nr = dev->init_data->core_nr;
    
    /* check if core did set all mandatory ops */
    if (!(core->get_input  &&
	  core->get_output &&
	  core->get_enable &&
	  core->set_output &&
	  core->set_enable)) {
	
	ERR("Core did not define all mandatory ops");
	goto bail;
    }	
    
    core->isr_cb = gpio_isr;
    dev->initialized = 1;

    ret = 0;
    
 bail:
    spin_unlock_irqrestore(&dev->mtx, flags);
    return ret;
}

static inline gpio_dev_t*
gpio_lookup_device(u_char dev_no)
{
#ifdef ENABLE_CHECKS
    gpio_dev_t *dev;
    u_long flags;

    if (dev_no >= GPIO_MAX_DEVICES) { return NULL; }
    
    dev = &gpio_devs[dev_no];

    spin_lock_irqsave(&dev->mtx, flags);
    if (dev->initialized == 0) {
	spin_unlock_irqrestore(&dev->mtx, flags);	
	return NULL;
    }
    spin_unlock_irqrestore(&dev->mtx, flags);

    return dev;
#else
    return &gpio_devs[dev_no];
#endif
}

static void
gpio_isr(gpio_core_t *core, u_long state)
{
    u_long flags;
    gpio_dev_t *dev = core->dev;
    
     //if (!gpio_get_reg_bits(dev->gpio, GPIO_CTRL, 2)) return; /* not our intr */
    DBG("enter generic gpio isr");
    
    spin_lock_irqsave(&dev->intr_state_lock, flags);
    DBG("isr: intr_flags old=0x%08lx this=0x%08lx new=0x%08lx", dev->intr_state,
	state, dev->intr_state | state);

    dev->intr_state |= state;
  
    spin_unlock_irqrestore(&dev->intr_state_lock, flags);
    
    /* issue deferred intr routine in non-irq mode */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    schedule_task(&dev->dsr);
#else
    schedule_work(&dev->dsr);
#endif
    DBG("leave gpio isr");
}

static void
gpio_dsr(void *ctx)
{
    gpio_dev_t *dev = ctx;
    u_int intr_state;
    u_long flags;

    spin_lock_irqsave(&dev->intr_state_lock, flags);
    intr_state = dev->intr_state;
    dev->intr_state = 0;
    spin_unlock_irqrestore(&dev->intr_state_lock, flags);

    DBG("dsr: intr_flags %08x", intr_state);

    spin_lock_irqsave(&dev->mtx, flags);

    {
        struct list_head *e;
        for (e = dev->watch_list.next; e != &dev->watch_list; e = e->next) {
            gpio_file_t *priv = list_entry(e, gpio_file_t, watch_anchor);
            if (intr_state & priv->watch_mask) {
                priv->watch_modified |= intr_state & priv->watch_mask;
#ifdef GPIO_KERNEL_SUPPORT
                if (priv->watchk_cb) {
                    priv->watchk_cb(priv->watchk_arg);
                } else {
                    wake_up (&priv->watch_wait);
                }
#else
                wake_up(&priv->watch_wait);
#endif
            }
        }
    }

    spin_unlock_irqrestore(&dev->mtx, flags);
    DBG("leave gpio dsr");
}

/** Adds or removes watchers to/from dev's list */
static void watch(gpio_dev_t* dev, gpio_file_t* priv, u_long mask)
{
    priv->watch_mask = mask;
    /* prevent double adding */
    if (!list_empty(&priv->watch_anchor)) list_del_init(&priv->watch_anchor);
    if (mask) list_add(&priv->watch_anchor, &dev->watch_list);

    {
        /* calc new resulting mask */
        u_long m = 0;
        struct list_head *e;
        for (e = dev->watch_list.next; e != &dev->watch_list; e = e->next) {
            gpio_file_t *priv = list_entry(e, gpio_file_t, watch_anchor);
            m |= priv->watch_mask;
        }

        /* enable interrupts */
	if (dev->core->irq_enable) dev->core->irq_enable(dev->core, m, ~m);
    }
}

#ifdef GPIO_KERNEL_SUPPORT
int
gpio_watchk (uint8_t core_nr, void (*watchk_cb) (void *arg),
	     void * watchk_arg, unsigned long mask)
{
    gpio_dev_t *dev;
    struct list_head *e;
    gpio_file_t *watchk_priv = NULL;
    int new_refcnt = 0;

    if (core_nr >= 2) {
        BUG();
    }
    dev = &gpio_devs[core_nr];
    
    /* check if registered */
    for (e = dev->watch_list.next; e != &dev->watch_list; e = e->next) {
        gpio_file_t *priv = list_entry (e, gpio_file_t, watch_anchor);
        if ((priv->watchk_cb == watchk_cb) && 
            (priv->watchk_arg == watchk_arg)) {
            watchk_priv = priv;
			break;
        }
    }

    if ((watchk_priv == NULL) && (mask)) {

        if (!dev->initialized) {
            ERR ("GPIO device was not initialized\n");
            return -ENODEV;
        }

        watchk_priv = kmalloc (sizeof(gpio_file_t), GFP_KERNEL);
        if (watchk_priv == NULL) {
            ERR ("Can not allocate private for watchk\n");
            return -ENOMEM;
        }

        memset (watchk_priv, 0, sizeof (gpio_file_t));
        watchk_priv->dev = dev;
        INIT_LIST_HEAD (&watchk_priv->watch_anchor);
        init_waitqueue_head (&watchk_priv->watch_wait);

        new_refcnt = atomic_inc_return (&dev->refcnt);
        if (dev->core->open) dev->core->open(dev->core, new_refcnt);
    }

    if (watchk_priv == NULL) {
        ERR("Could not find handle for watchk\n");
        return -ENOENT;
    } 

    watchk_priv->watchk_cb = watchk_cb;
    watchk_priv->watchk_arg = watchk_arg;

    watch (dev, watchk_priv, mask);
    
    /* check if it is delete watch */
    if (mask == 0) {
        
        new_refcnt = atomic_dec_return (&dev->refcnt);
        dev->requested &= ~watchk_priv->requested;
        kfree (watchk_priv);

        if (dev->core->release) dev->core->release (dev->core, new_refcnt);
    }

    return 0;
}

EXPORT_SYMBOL(gpio_watchk);
#endif /* GPIO_KERNEL_SUPPORT */
