/**
 * gpio_core.h
 *
 * Common header file for all low-level GPIO core drivers
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 */

#ifndef _GPIO_CORE_H
#define _GPIO_CORE_H

#include <linux/types.h>

/* ------------------------------------------------------------------------- *
 * data types
 * ------------------------------------------------------------------------- */

#define GPIO_CORE_NAME_SIZE	16

/* possible capabilities of a core */
#define GPIO_CAPS_IRQ			0x01
#define GPIO_CAPS_ALTERNATE_FUNCTIONS	0x02

typedef struct gpio_dev_s gpio_dev_t;
typedef struct gpio_core_s gpio_core_t;

struct gpio_core_s {
    gpio_dev_t	*dev;		/* link to device structure */
    void*	private_data;	/* may be filled by the core */
    u32		caps;		/* this core's capabilities */
    u8		nr_bits;	/* number of gpios this core can handle */
    char	name[GPIO_CORE_NAME_SIZE];
    char        nr;
    
    /* core operations (mandatory) */
    void	(*cleanup)	(gpio_core_t *core);
    
    int		(*open)		(gpio_core_t *core, u_int new_use_count);
    void	(*release)	(gpio_core_t *core, u_int new_use_count);
    
    int		(*get_input)	(gpio_core_t *core, u_long *value);
    int		(*get_output)	(gpio_core_t *core, u_long *value);
    int		(*get_enable)	(gpio_core_t *core, u_long *value);
    int		(*set_output)	(gpio_core_t *core, u_long set, u_long clr);
    int		(*set_enable)	(gpio_core_t *core, u_long set, u_long clr);

    /* core operations (optional) */
    int		(*irq_enable)   (gpio_core_t *core, u_long set, u_long clr);
    int		(*irq_edge)	(gpio_core_t *core, u_long level, u_long edge,
				 u_long pos, u_long neg);
    int         (*irq_ack)      (gpio_core_t *core, u_long mask);

    /* alternate functions shared with gpio pins (optional) */
    int		(*get_alternate)(gpio_core_t *core, u_long *value);
    int		(*set_alternate)(gpio_core_t *core, u_long set, u_long clr);
    
    /* isr callback, filled by hilevel device and called from the core */
    void        (*isr_cb)       (gpio_core_t *core, u_long state);
};

typedef int  (*core_init_fn_t)   (gpio_core_t *core, u_char core_nr);

#endif /* _GPIO_CORE_H */
