/**
 * gpio_core_ibm.c
 *
 * Low-level driver for the IBM PPC405 GPIO core
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 *
 */

#include <linux/ioport.h>
#include <asm/io.h>
#include <asm/ibm405.h>
#include <linux/kernel.h>

#include "gpio_common.h"
#include "gpio_core_ibm.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define GPIO_IBM_NR_BITS	32
#define GPIO_IBM_NAME		"ibm"

#define PPC405_GPIO_REGS_ADDR		((u32)0xEF600700)
#define PPC405_GPIO_REGS_SIZE		((size_t)0x2C)

#define PPC405_GPIO_REG_OR		0
#define PPC405_GPIO_REG_TCR		1
#define PPC405_GPIO_REG_ODR		6 
#define PPC405_GPIO_REG_IR		7
/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u32			io_base;
    volatile u32 *	regs;
    int			mem_region_requested;
} data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

data_t data[] = {
    { PPC405_GPIO_REGS_ADDR, NULL, 0 },
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void ibm_cleanup(gpio_core_t *core);

static int ibm_get_input(gpio_core_t *core, u_long *value);
static int ibm_get_output(gpio_core_t *core, u_long *value);
static int ibm_get_enable(gpio_core_t *core, u_long *value);
static int ibm_set_output(gpio_core_t *core, u_long set, u_long clr);
static int ibm_set_enable(gpio_core_t *core, u_long set, u_long clr);
static int ibm_get_alternate(gpio_core_t *core, u_long *value);
static int ibm_set_alternate(gpio_core_t *core, u_long set, u_long clr);

#ifdef ENABLE_CHECKS
static void check_alternate(gpio_core_t *core, u_long mask);
#endif

/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline u32
gpio_get_reg(data_t *gpio, int reg)
{
    return gpio->regs[reg];
}

static __inline void
gpio_set_reg(data_t *gpio, int reg, u32 val)
{
    gpio->regs[reg] = val;
}

static __inline u32
gpio_get_reg_bits(data_t *gpio, int reg, u32 mask)
{
    return gpio_get_reg(gpio, reg) & mask;
}

static __inline void
gpio_set_reg_bits(data_t *gpio, int reg, u32 set_mask, u32 clr_mask)
{
    gpio_set_reg(gpio, reg, (gpio_get_reg(gpio, reg) & ~clr_mask) | set_mask);
}

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */
int
gpio_core_ibm_init(gpio_core_t *core, u8 core_nr)
{
    int ret = -1;
    int nr_cores = sizeof(data) / sizeof(data_t);
    data_t *pdata;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	return ret;
    }

    core->private_data = pdata = &data[core_nr];
    core->cleanup      = ibm_cleanup;
    
    if (request_mem_region(pdata->io_base,
			   PPC405_GPIO_REGS_SIZE, NAME) == NULL) {
	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    pdata->io_base, PPC405_GPIO_REGS_SIZE);
	
	goto bail;
    }
    pdata->mem_region_requested = 1;
    
    if ((pdata->regs = ioremap_nocache(pdata->io_base,
				       PPC405_GPIO_REGS_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    pdata->io_base, PPC405_GPIO_REGS_SIZE);
	goto bail;	
    }

    core->caps = GPIO_CAPS_ALTERNATE_FUNCTIONS;
    core->nr_bits = GPIO_IBM_NR_BITS;
    strncpy(core->name, GPIO_IBM_NAME, sizeof(core->name) - 1);
    core->name[sizeof(core->name) - 1] = '\0';

    core->get_input  = ibm_get_input;
    core->get_output = ibm_get_output;
    core->get_enable = ibm_get_enable;
    core->set_output = ibm_set_output;
    core->set_enable = ibm_set_enable;
    core->get_alternate = ibm_get_alternate;
    core->set_alternate = ibm_set_alternate;
    core->irq_ack    = NULL;
    core->irq_enable = NULL;
    core->irq_edge   = NULL;
    
    ret = SUCCESS;
    INFO("GPIO core '%s' (%d) initialized", core->name, core_nr);
    
    
 bail:
    if (ret != SUCCESS) {
	ibm_cleanup(core);
    }
    return ret;    
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
ibm_cleanup(gpio_core_t *core)
{
    data_t *pdata = (data_t*) core->private_data;

    if (pdata == NULL) {	
	return; // we were never really initialized
    }

    if (pdata->regs != NULL) {
	iounmap((void*)pdata->regs);
	pdata->regs = NULL;
    }
  
    if (pdata->mem_region_requested) {
	release_mem_region(pdata->io_base, PPC405_GPIO_REGS_SIZE);
    }

    INFO("GPIO core '%s' cleanup done", core->name);    
}

static int
ibm_get_input(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;
    
    *value = gpio_get_reg(pdata, PPC405_GPIO_REG_IR);

    return 0;    
}

static int
ibm_get_output(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;
    
    *value = gpio_get_reg(pdata, PPC405_GPIO_REG_OR);

    return 0;       
}

static int
ibm_get_enable(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;
    
    *value = gpio_get_reg(pdata, PPC405_GPIO_REG_TCR);
    
    return 0;           
}

static int
ibm_set_output(gpio_core_t *core, u_long set, u_long clr)
{
    data_t *pdata = (data_t*) core->private_data;

#if defined(ENABLE_CHECKS)
    check_alternate(core, set | clr);
#endif
    gpio_set_reg_bits(pdata, PPC405_GPIO_REG_OR, set, clr);
    
    return 0;
}

static int
ibm_set_enable(gpio_core_t *core, u_long set, u_long clr)
{
    data_t *pdata = (data_t*) core->private_data;
  
#if defined(ENABLE_CHECKS)
    check_alternate(core, set | clr);
#endif
    gpio_set_reg_bits(pdata, PPC405_GPIO_REG_TCR, set, clr);
    
    return 0;
}

static int
ibm_get_alternate(gpio_core_t *core, u_long *value)
{
    u32 cpc0_cr0 = mfdcr(DCRN_CPC0_CR0);
    
    int i;
    u32 val = 0;

    if (value == 0) return 0;
  
    // GPIO 0 is reserved

    // GPIO 1-9 are in bit 4
    for (i = 1; i < 10; i++) {
	val |= ((cpc0_cr0 >> (31 - 4)) & 1) << i;
    }
    // GPIO 10-24 are in bit 5-19
    for (i = 10; i < 25; ++i) {
	val |= ((cpc0_cr0 >> (31 - 5 - (i - 10))) & 1) << i;	
    }

    *value = val;
    
    return 0;
}

static int
ibm_set_alternate(gpio_core_t *core, u_long set, u_long clr)
{
    u_int32_t r_set = 0, r_clr = 0;
    int i;

    u32 cpc0_cr0 = mfdcr(DCRN_CPC0_CR0);
    // GPIO 0 is reserved

    // GPIO 1-9 are in bit 4: no usable alternate function
    
    // GPIO 10-24 are in bit 5-19
    for (i = 10; i < 25; ++i) {
	r_set |= ((set >> i) & 1) << (31 - 5 - (i - 10));
	r_clr |= ((clr >> i) & 1) << (31 - 5 - (i - 10));
    }
    //printk("clr: %08x, set: %08x\n", r_clr, r_set);
    //printk("cr0 changed from :%08x to %08x\n", cpc0_cr0, (cpc0_cr0 & ~r_set) | r_clr);
    
    /* please note, that the meaning is different in the CRC0 register:
     * 0: alternate function
     * 1: GPIO
     */
    mtdcr(DCRN_CPC0_CR0, (cpc0_cr0 & ~r_set) | r_clr);
    return 0;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */

#if defined(ENABLE_CHECKS)
static void
check_alternate(gpio_core_t *core, u_long mask)
{
    u_long alt;

    ibm_get_alternate(core, &alt);

    if ((alt & mask) == mask) {
	WARN("Trying to access GPIOs set to alternate functions (alt=0x%08lx, mask=0x%08lx)",
	     alt, mask);
    }
}
#endif
