/**
 * gpio_core_opencores.c
 *
 * Low-level driver for the OpenCores GPIO core
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 *
 * (based on the GPIO Module from rgue@peppercon.de)
 */

#include <linux/version.h>
#include <linux/pci.h>
#include <linux/interrupt.h>

#ifdef __powerpc__
# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#   include <asm/ppc405_pcimcs.h>
# else
#   include <syslib/ppc405_pcimcs.h>
# endif
#endif

#include "gpio_common.h"
#include "gpio_core_opencores.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define OC_PCI_VENDOR_ID	0x1743
#define OC_PCI_DEVICE_ID	0x1003
#define OC_PCI_BAR_NO		1
#define OC_IO_SIZE		(10 * 4) /* regs are dword aligned */
#define OC_NR_BITS		32
#define OC_IRQ			27
#define OC_NAME			"OpenCores"

/* GPIO register offsets */

#define GPIO_IN     0x00
#define GPIO_OUT    0x01
#define GPIO_OE     0x02
#define GPIO_INTE   0x03
#define GPIO_PTRIG  0x04
#define GPIO_AUX    0x05
#define GPIO_CTRL   0x06
#define GPIO_INTS   0x07
#define GPIO_ECLK   0x08
#define GPIO_NEC    0x09

/* bits of GPIO_CTRL */
#define GPIO_CTRL_INTE  ((u8)(1 << 0))
#define GPIO_CTRL_INTS  ((u8)(1 << 1))

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u32			io_base_offset;
    volatile u32 *	regs;
    int			isr_registered;
    int			mem_region_requested;
    struct pci_dev *	pci;
} oc_data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

oc_data_t oc_data[] = {
    { 0x980000, NULL, 0, 0, NULL },
    { 0x990000, NULL, 0, 0, NULL }
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void oc_cleanup(gpio_core_t *core);

static int oc_get_input(struct gpio_core_s *core, u_long *value);
static int oc_get_output(struct gpio_core_s *core, u_long *value);
static int oc_get_enable(struct gpio_core_s *core, u_long *value);
static int oc_set_output(struct gpio_core_s *core, u_long set, u_long clr);
static int oc_set_enable(struct gpio_core_s *core, u_long set, u_long clr);
static int oc_irq_enable(struct gpio_core_s *core, u_long set, u_long clr   );

static irqreturn_t oc_isr(int irq, void *dev_id, struct pt_regs *regs);

/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline u32
gpio_get_reg(oc_data_t *gpio, int reg)
{
    return readl(&gpio->regs[reg]);
}

static __inline void
gpio_set_reg(oc_data_t *gpio, int reg, u32 val)
{
    writel(val, &gpio->regs[reg]);
}

static __inline u32
gpio_get_reg_bits(oc_data_t *gpio, int reg, u32 mask)
{
    return gpio_get_reg(gpio, reg) & mask;
}

static __inline void
gpio_set_reg_bits(oc_data_t *gpio, int reg, u32 set_mask, u32 clr_mask)
{
    gpio_set_reg(gpio, reg, (gpio_get_reg(gpio, reg) & ~clr_mask) | set_mask);
}

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
gpio_core_opencores_init(gpio_core_t *core, u8 core_nr)
{
    int ret = -1, r;
    int nr_cores = sizeof(oc_data) / sizeof(oc_data_t);
    u32 io_base, io_end;
    oc_data_t *pdata;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	goto bail;
    }

    core->private_data = pdata = &oc_data[core_nr];
    core->cleanup      = oc_cleanup;
   
    /* find hw */
    pdata->pci = pci_find_device(OC_PCI_VENDOR_ID, OC_PCI_DEVICE_ID, NULL);
    if (pdata->pci == NULL) {
        ERR("failed to find PCI device (0x%04x:0x%04x) containing GPIO core",
            OC_PCI_VENDOR_ID, OC_PCI_DEVICE_ID);
        goto bail;
    }

    io_base = pci_resource_start(pdata->pci, OC_PCI_BAR_NO);
    io_end  = pci_resource_end(pdata->pci, OC_PCI_BAR_NO);

    if (io_base + pdata->io_base_offset + OC_IO_SIZE > (io_end + 1)) {
	ERR("not enough io space for GPIO core (have %d @ %04x:0x%04x, need %d)",
	    io_end - io_base, io_base, pdata->io_base_offset, OC_IO_SIZE);
	goto bail;
    }

    if (request_mem_region(io_base + pdata->io_base_offset,
			   OC_IO_SIZE, NAME) == NULL) {
	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    io_base + pdata->io_base_offset, OC_IO_SIZE);
	goto bail;
    }
    pdata->mem_region_requested = 1;

    if ((pdata->regs = ioremap_nocache(io_base + pdata->io_base_offset,
				       OC_IO_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    io_base + pdata->io_base_offset, OC_IO_SIZE);
	goto bail;	
    }

#ifdef __powerpc__
    /* check if io range is (machine check save) accessible */
    {
        u32 val;
        int acc_err;
        val = mcs_in_le32((u32*)&pdata->regs[GPIO_INTE], &acc_err);
        if (acc_err != 0) {
            ERR("failed to read GPIO registers");
            goto bail;
        }
        mcs_out_le32(val, (u32*)&pdata->regs[GPIO_INTE], &acc_err);
        if (acc_err != 0) {
            ERR("failed to write GPIO registers");
            goto bail;
        }
    }
#endif

    if ((r = request_irq(OC_IRQ, oc_isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
			 NAME, core)) < 0) {
	
	ERR("failed to request irq %d (err %d)", OC_IRQ, r);
	goto bail;
    }
    pdata->isr_registered = 1;

    gpio_set_reg(pdata, GPIO_INTE, 0); /* disable all bit IRQs */
    gpio_set_reg(pdata, GPIO_CTRL, 1); /* enable IRQ globally */
    
    core->caps = GPIO_CAPS_IRQ;
    core->nr_bits = OC_NR_BITS;
    strncpy(core->name, OC_NAME, sizeof(core->name) - 1);
    core->name[sizeof(core->name) - 1] = '\0';

    core->get_input  = oc_get_input;
    core->get_output = oc_get_output;
    core->get_enable = oc_get_enable;
    core->set_output = oc_set_output;
    core->set_enable = oc_set_enable;
    core->irq_enable = oc_irq_enable;
    core->get_alternate = NULL;
    core->set_alternate = NULL;
    
    ret = SUCCESS;
    INFO("GPIO core %s (%d) initialized", core->name, core_nr);
    
 bail:
    if (ret != 0) {
	oc_cleanup(core);
    }
    return ret;
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
oc_cleanup(gpio_core_t *core)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    if (pdata == NULL) {
	return; // we were never really initialized
    }
    
    if (pdata->regs != NULL) {
	gpio_set_reg(pdata, GPIO_CTRL, 0); /* disable IRQ globally */
    }

    if (pdata->isr_registered) {
	free_irq(OC_IRQ, core); pdata->isr_registered = 0;
    }

    if (pdata->regs != NULL) {
	iounmap((void*)pdata->regs);
	pdata->regs = NULL;
    }

    if (pdata->pci && pdata->mem_region_requested) {
	release_mem_region(pci_resource_start(pdata->pci, OC_PCI_BAR_NO) + pdata->io_base_offset,
			   OC_IO_SIZE);
    }

    INFO("GPIO core %s cleanup done", core->name);
}

static int
oc_get_input(gpio_core_t *core, u_long *value)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;
  
    if (value) *value = gpio_get_reg(pdata, GPIO_IN);

    return 0;
}

static int
oc_get_output(gpio_core_t *core, u_long *value)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_OUT);

    return 0;
}

static int
oc_get_enable(gpio_core_t *core, u_long *value)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_OE);

    return 0;
}

static int
oc_set_output(gpio_core_t *core, u_long set, u_long clr)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_OUT, set, clr);

    return 0;
}

static int
oc_set_enable(gpio_core_t *core, u_long set, u_long clr)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_OE, set, clr);

    return 0;
}

static int
oc_irq_enable(gpio_core_t *core, u_long set, u_long clr)
{
    oc_data_t *pdata = (oc_data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_INTE, set, clr);

    return 0;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */


static irqreturn_t
oc_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    gpio_core_t *core = (gpio_core_t*) dev_id;
    oc_data_t *pdata = (oc_data_t*) core->private_data;
    u_long intr_state = gpio_get_reg(pdata, GPIO_INTS);

    /* this core does a clear-on-read */
    // GPIO core has been changed: IRQs are acked by read
    // so, next lines are obsolete (and even dangerous)
    //gpio_set_reg(dev->gpio, GPIO_INTS, 0); /* ack IRQs */
    //gpio_set_reg_bits(dev->gpio, GPIO_CTRL, 0, 2); /* ack global IRQ */
   
    if (!intr_state) return IRQ_NONE; /* not our intr */
    
    if (core->isr_cb) core->isr_cb(core, intr_state);

    return IRQ_HANDLED;
}
