/**
 * gpio_core_vsc.c
 *
 * Low-level driver for the OpenCores based GPIO in the VSC
 * mutliplexed with video input path B (in case of single pixel)
 *
 * (c) 2005 Peppercon AG, Michael Baumann <miba@peppercon.de>
 *
 */

#include <linux/ioport.h>
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <asm/io.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/arch-cpe/cpe/kira100.h>
#else
# include <asm/arch-faraday/platform-a320/kira100.h>
#endif
#ifdef __powerpc__
# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#   include <asm/ppc405_pcimcs.h>
# else
#   include <syslib/ppc405_pcimcs.h>
# endif
#endif

#include "gpio_common.h"
#include "gpio_core_vsc.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define GPIO_BASE		(CPE_VSCREG_BASE + (0x30 * 4))
#define GPIO_IO_SIZE		(6 * 4)
#define GPIO_NR_BITS		19
#define GPIO_IRQ		28
#define CORE_NAME		"vsc gpio"

/* GPIO register offsets (32bit words) */

#define GPIO_IN     0x00
#define GPIO_OUT    0x01
#define GPIO_OE     0x02
#define GPIO_TRIG   0x03
#define GPIO_INTS   0x04
#define GPIO_INTE   0x05

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u32			io_base;
    volatile u32 *	regs;
    int			isr_registered;
    int			mem_region_requested;
} data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

data_t data[] = {
    { GPIO_BASE, NULL, 0, 0 },
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void cleanup(gpio_core_t *core);

static int get_input(struct gpio_core_s *core, u_long *value);
static int get_output(struct gpio_core_s *core, u_long *value);
static int get_enable(struct gpio_core_s *core, u_long *value);
static int set_output(struct gpio_core_s *core, u_long set, u_long clr);
static int set_enable(struct gpio_core_s *core, u_long set, u_long clr);
static int irq_enable(struct gpio_core_s *core, u_long set, u_long clr);
static int irq_edge(gpio_core_t *core, u_long level, u_long edge,
		    u_long pos, u_long neg);

static irqreturn_t isr(int irq, void *dev_id, struct pt_regs *regs);

/* ------------------------------------------------------------------------- *
 * register access inline functions
 * ------------------------------------------------------------------------- */

static __inline u32
gpio_get_reg(data_t *gpio, int reg)
{
    return readl(&gpio->regs[reg]);
}

static __inline void
gpio_set_reg(data_t *gpio, int reg, u32 val)
{
    writel(val, &gpio->regs[reg]);
}

static __inline u32
gpio_get_reg_bits(data_t *gpio, int reg, u32 mask)
{
    return gpio_get_reg(gpio, reg) & mask;
}

static __inline void
gpio_set_reg_bits(data_t *gpio, int reg, u32 set_mask, u32 clr_mask)
{
    gpio_set_reg(gpio, reg, (gpio_get_reg(gpio, reg) & ~clr_mask) | set_mask);
}

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
gpio_core_vsc_init(gpio_core_t *core, u8 core_nr)
{
    int ret = -1, r;
    int nr_cores = sizeof(data) / sizeof(data_t);
    data_t *pdata;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	goto bail;
    }

    core->private_data = pdata = &data[core_nr];
    core->cleanup      = cleanup;
   
    if (request_mem_region(pdata->io_base, GPIO_IO_SIZE, NAME) == NULL) {	
	ERR("memory region @ 0x%08x (size %d) already in use",
	    pdata->io_base, GPIO_IO_SIZE);
	goto bail;
    }
    pdata->mem_region_requested = 1;

    if ((pdata->regs = ioremap_nocache(pdata->io_base,
				       GPIO_IO_SIZE)) == NULL) {

	ERR("could not remap GPIO regs @ 0x%08x (size %d)",
	    pdata->io_base, GPIO_IO_SIZE);
	goto bail;	
    }

    if ((r = request_irq(GPIO_IRQ, isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
			 NAME, core)) < 0) {
	
	ERR("failed to request irq %d (err %d)", GPIO_IRQ, r);
	goto bail;
    }
    pdata->isr_registered = 1;

    gpio_set_reg(pdata, GPIO_INTE, 0); /* disable all bit IRQs */
    
    core->caps = GPIO_CAPS_IRQ;
    core->nr_bits = GPIO_NR_BITS;
    strncpy(core->name, CORE_NAME, sizeof(core->name) - 1);
    core->name[sizeof(core->name) - 1] = '\0';

    core->get_input  = get_input;
    core->get_output = get_output;
    core->get_enable = get_enable;
    core->set_output = set_output;
    core->set_enable = set_enable;
    core->irq_enable = irq_enable;
    core->irq_edge   = irq_edge;
    core->irq_ack    = NULL;
    core->get_alternate = NULL;
    core->set_alternate = NULL;
    
    ret = SUCCESS;
    INFO("GPIO core '%s' (%d) initialized", core->name, core_nr);
    
 bail:
    if (ret != 0) {
	cleanup(core);
    }
    return ret;
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
cleanup(gpio_core_t *core)
{
    data_t *pdata = (data_t*) core->private_data;

    if (pdata == NULL) {
	return; // we were never really initialized
    }
    
    if (pdata->regs != NULL) {
	gpio_set_reg(pdata, GPIO_INTE, 0); /* disable all bit IRQs */
    }

    if (pdata->isr_registered) {
	free_irq(GPIO_IRQ, core); pdata->isr_registered = 0;
    }

    if (pdata->regs != NULL) {
	iounmap((void*)pdata->regs);
	pdata->regs = NULL;
    }

    if (pdata->mem_region_requested) {
	release_mem_region(pdata->io_base, GPIO_IO_SIZE);
    }

    INFO("GPIO core %s cleanup done", core->name);
}

static int
get_input(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;
  
    if (value) *value = gpio_get_reg(pdata, GPIO_IN);

    return 0;
}

static int
get_output(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_OUT);

    return 0;
}

static int
get_enable(gpio_core_t *core, u_long *value)
{
    data_t *pdata = (data_t*) core->private_data;

    if (value) *value = gpio_get_reg(pdata, GPIO_OE);

    return 0;
}

static int
set_output(gpio_core_t *core, u_long set, u_long clr)
{
    data_t *pdata = (data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_OUT, set, clr);

    return 0;
}

static int
set_enable(gpio_core_t *core, u_long set, u_long clr)
{
    data_t *pdata = (data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_OE, set, clr);

    return 0;
}

static int
irq_enable(gpio_core_t *core, u_long set, u_long clr)
{
    data_t *pdata = (data_t*) core->private_data;

    gpio_set_reg_bits(pdata, GPIO_INTE, set, clr);

    return 0;
}

static int
irq_edge(gpio_core_t *core, u_long level, u_long edge,
	 u_long pos, u_long neg)
{
    data_t *pdata = (data_t*) core->private_data;
    u_long both = edge & (pos & neg);
    int ret = -1;  
    
    if (both) {
	ERR("Core '%s' does not support both IRQ edges", core->name);
	goto bail;
    }

    if (level) {
	ERR("Core '%s' does not support level trigger", core->name);
	goto bail;
    }

    gpio_set_reg_bits(pdata, GPIO_TRIG, neg, pos);

    ret = 0;
    
 bail:    
    return ret;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */


static irqreturn_t
isr(int irq, void *dev_id, struct pt_regs *regs)
{
    gpio_core_t *core = (gpio_core_t*) dev_id;
    data_t *pdata = (data_t*) core->private_data;
    u_long intr_state = gpio_get_reg(pdata, GPIO_INTS);

    /* this core does a clear-on-read */
   
    if (!intr_state) return IRQ_NONE; /* not our intr */
    
    if (core->isr_cb) core->isr_cb(core, intr_state);

    return IRQ_HANDLED;
}
