/**
 * pp_gpio.h
 *
 * Public header file for GPIO core interface
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *                        Michael Baumann <miba@peppercon.de>
 *
 */

#ifndef __GPIO_H__
#define __GPIO_H__

#include <linux/types.h>

typedef struct {
    unsigned char idx;
    unsigned char val;
} gpio_ioctl_bit_t;

typedef struct {
    unsigned long set;
    unsigned long clr;
    unsigned long tri;
} gpio_ioctl_out_t;

typedef struct {
    unsigned long level; /* switch pin to level trigger */
    unsigned long edge;	 /* switch pin to edge trigger */
    unsigned long pos;	 /* rising edge or active high level */
    unsigned long neg;	 /* falling edge or active low level */
} gpio_ioctl_irqconf_t;

typedef struct {
    unsigned long set;
    unsigned long clr;
} gpio_ioctl_alternate_t;

#define GPIO_IOCTL_MAGIC	'G'
#define GPIO_IOCTL_REQUEST      _IOW (GPIO_IOCTL_MAGIC,  2, unsigned long)
#define GPIO_IOCTL_RELEASE      _IOW (GPIO_IOCTL_MAGIC,  3, unsigned long)
#define GPIO_IOCTL_IS_REQUESTED _IOR (GPIO_IOCTL_MAGIC,  4, unsigned long)
#define GPIO_IOCTL_WATCH        _IOW (GPIO_IOCTL_MAGIC,  5, unsigned long)
#define GPIO_IOCTL_IRQCONF	_IOW (GPIO_IOCTL_MAGIC,  6, gpio_ioctl_irqconf_t)

#define GPIO_IOCTL_IN           _IOR (GPIO_IOCTL_MAGIC, 10, unsigned long)
#define GPIO_IOCTL_OUT          _IOW (GPIO_IOCTL_MAGIC, 12, gpio_ioctl_out_t)
#define GPIO_IOCTL_SET          _IOW (GPIO_IOCTL_MAGIC, 13, unsigned long)
#define GPIO_IOCTL_CLR          _IOW (GPIO_IOCTL_MAGIC, 14, unsigned long)
#define GPIO_IOCTL_TRI          _IOW (GPIO_IOCTL_MAGIC, 15, unsigned long)
#define GPIO_IOCTL_IS_SET       _IOR (GPIO_IOCTL_MAGIC, 16, unsigned long)
#define GPIO_IOCTL_IS_CLR       _IOR (GPIO_IOCTL_MAGIC, 17, unsigned long)
#define GPIO_IOCTL_IS_TRI       _IOR (GPIO_IOCTL_MAGIC, 18, unsigned long)

#define GPIO_IOCTL_GET_BIT      _IOWR(GPIO_IOCTL_MAGIC, 20, gpio_ioctl_bit_t)
#define GPIO_IOCTL_SET_BIT      _IOW (GPIO_IOCTL_MAGIC, 21, gpio_ioctl_bit_t)
#define GPIO_IOCTL_SET_BIT_ENBL _IOW (GPIO_IOCTL_MAGIC, 22, gpio_ioctl_bit_t)

#define GPIO_IOCTL_SET_ALT	_IOW (GPIO_IOCTL_MAGIC, 23, gpio_ioctl_alternate_t)
#define GPIO_IOCTL_GET_ALT	_IOR (GPIO_IOCTL_MAGIC, 24, unsigned long)

#define GPIO_IOCTL_GET_MODIFIED	_IOR (GPIO_IOCTL_MAGIC, 25, unsigned long)

/* possible gpio cores/devices (conform to their minor numbers) */
#define PP_GPIO_DEV_IBM		0	/* be careful with bit ordering, use _IBM macros */
#define PP_GPIO_DEV_FARADAY	1
#define PP_GPIO_DEV_OPENCORES_0	2
#define PP_GPIO_DEV_OPENCORES_1	3
#define PP_GPIO_DEV_XR17_0	4
#define PP_GPIO_DEV_XR17_1	5
#define PP_GPIO_DEV_VSC		6

#define PP_GPIO_MAX_NR_DEVS	7

#define PP_GPIO_BIT(num)		(num)
#define PP_IBM_GPIO_BIT(num)		(31-(num))

#define PP_GPIO_MASK(num)		(1 << PP_GPIO_BIT(num))
#define PP_IBM_GPIO_MASK(num)		(1 << PP_IBM_GPIO_BIT(num))

/*
 * Functions to cover GPIO access from kernel
 */
#ifdef __KERNEL__

/* exported functions for kernel gpio access */
int pp_gpio_init(u_char dev_no);

int pp_gpio_get(u_char dev_no, u_long *value);
int pp_gpio_set(u_char dev_no, u_long set, u_long clr, u_long tri);

int pp_gpio_is_set(u_char dev_no, u_long *value);
int pp_gpio_is_clr(u_char dev_no, u_long *value);
int pp_gpio_is_tri(u_char dev_no, u_long *value);

int pp_gpio_bit_set(u_char dev_no, u_char bit, u_char value);
int pp_gpio_bit_get(u_char dev_no, u_char bit, u_char *value);

int pp_gpio_alternate_set(u_char dev_no, u_long set, u_long clr);
int pp_gpio_alternate_get(u_char dev_no, u_long *value);

#else /* !__KERNEL__ */

/*
 * Helper function to cover IOCTL (only in user mode)
 */
#include <sys/ioctl.h>

static __inline__ int gpio_ioctl_wr_ulong(int fd, int cmd, unsigned long val)
{
    return ioctl(fd, cmd, &val);
}

static __inline__ int gpio_ioctl_rd_ulong(int fd, int cmd, unsigned long *val)
{
    return ioctl(fd, cmd, val);
}

/** Requests multiple IO bits for exclusive write */
#define gpio_ioctl_request(fd, v)   gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_REQUEST, v)
/** Release reserved IO bits */
#define gpio_ioctl_release(fd, v)   gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_RELEASE, v)
/** Returns mask of reserved IO bits (reserved by any client) */
#define gpio_ioctl_is_requested(fd, v) gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_IS_REQUESTED, v)
/** Enable IRQ for multiple IO bits (used for poll) */
#define gpio_ioctl_watch(fd, v)     gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_WATCH, v)

/** Reads all IN bits at once */
#define gpio_ioctl_in(fd, v)       gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_IN, v)

/** Sets multiple OUT bits to 1 */
#define gpio_ioctl_set(fd, v)   gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_SET, v)
/** Sets multiple OUT bits to 0 */
#define gpio_ioctl_clr(fd, v)   gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_CLR, v)
/** Sets multiple OUT bits to Z */
#define gpio_ioctl_tri(fd, v)   gpio_ioctl_wr_ulong(fd, GPIO_IOCTL_TRI, v)

/** Returns all OUT bits set to 1 */
#define gpio_ioctl_is_set(fd, v)   gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_IS_SET, v)
/** Returns all OUT bits set to 0 */
#define gpio_ioctl_is_clr(fd, v)   gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_IS_CLR, v)
/** Returns all OUT bits set to Z */
#define gpio_ioctl_is_tri(fd, v)   gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_IS_TRI, v)

/** Returns all bits set to alternate functions */
#define gpio_ioctl_get_alt(fd, v)  gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_GET_ALT, v)

/** Returns all bits which have been modified at least once since last read */
#define gpio_ioctl_get_modified(fd, v) gpio_ioctl_rd_ulong(fd, GPIO_IOCTL_GET_MODIFIED, v)

/** Sets multiple OUT bits at once */
static __inline__ int gpio_ioctl_out(int fd, unsigned long _set, unsigned long _clr, unsigned long _tri)
{
    gpio_ioctl_out_t ioc = { set: _set, clr: _clr, tri: _tri };
    return ioctl(fd, GPIO_IOCTL_OUT, &ioc);
}

/** Configure GPIO alternate functions  */
static __inline__ int gpio_ioctl_alt_func(int fd, unsigned long _set, unsigned long _clr)
{
    gpio_ioctl_alternate_t ioc = { set: _set, clr: _clr };
    return ioctl(fd, GPIO_IOCTL_SET_ALT, &ioc);
}

/** Gets single IN bit via idx */
static __inline__ int gpio_ioctl_get_bit(int fd, unsigned char _idx)
{
    gpio_ioctl_bit_t ioc = { idx: _idx, val: 0 };
    int r = ioctl(fd, GPIO_IOCTL_GET_BIT, &ioc);
    return r < 0 ? r : ioc.val;
}

/** Configure IRQ triggers */
static __inline__ int gpio_ioctl_irqconf(int fd, unsigned long _level, unsigned long _edge,
					 unsigned long _pos, unsigned long _neg)
{
    gpio_ioctl_irqconf_t conf = { level: _level, edge: _edge, pos: _pos, neg: _neg };

    return ioctl(fd, GPIO_IOCTL_IRQCONF, &conf);
}

/** Sets single OUT bit via idx */
static __inline__ int gpio_ioctl_set_bit(int fd, unsigned char _idx, unsigned char _val)
{
    gpio_ioctl_bit_t ioc = { idx: _idx, val: _val };
    return ioctl(fd, GPIO_IOCTL_SET_BIT, &ioc);
}

static __inline__ int gpio_ioctl_set_bit_enable(int fd, unsigned char _idx, unsigned char _val)
{
    gpio_ioctl_bit_t ioc = { idx: _idx, val: _val };
    return ioctl(fd, GPIO_IOCTL_SET_BIT_ENBL, &ioc);
}
    
#endif /* !__KERNEL__ */

#endif /* __GPIO_H__ */
