#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/i2c-dev.h>
#include <linux/errno.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/poll.h>
#include <asm/uaccess.h>

#include "debug.h"
#include "pp_i2c.h"
#include "i2c_common.h"

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */

/* Until we really know how to halt and restart the i2c bus,
   we should ignore overflows. Just discard the oldest value
   in the case of an overflow. */
#define IGNORE_OVERFLOW	1

/* I2C master operations can possibly cause an error because
   the bus is not free or arbitration is lost. In this case,
   i2c_transfer will fail. We will retry the operation. */
#define MASTER_RETRIES	0

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG(fmt, x...)	I2C_DBG("i2c-common", fmt, ##x)
#define DBG2(fmt, x...)	I2C_DBG2("i2c-common", fmt, ##x)
#define WARN(fmt, x...)	I2C_WARN("i2c-common", fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Macros
 * ------------------------------------------------------------------------- */

#define DOWN(sem) if (down_interruptible(sem)) return -ERESTARTSYS;

#define UP(sem)	up(sem);


/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int i2c_common_slave_write(struct i2c_logical_device* dev,
				  struct file* file, const char* buffer,
				  size_t size, loff_t* offset);
static int i2c_common_master_write(struct i2c_logical_device* dev,
				   struct file* file, const char* buffer,
				   size_t size, loff_t* offset);
static int i2c_common_master_write_address(struct i2c_logical_device* dev,
					   char* buffer, size_t size);
static int i2c_common_master_write_no_address(struct i2c_logical_device* dev,
					      char* buffer, size_t size);
static int i2c_common_slave_read(struct i2c_logical_device* dev,
				 struct file* file, const char* buffer,
				 size_t size, loff_t* offset);
static int i2c_common_master_read(struct i2c_logical_device* dev,
				  struct file* file, const char* buffer,
				  size_t size, loff_t* offset);
static int i2c_common_master_read_address(struct i2c_logical_device* dev,
					  char* buffer, size_t size);
static int i2c_common_master_read_no_address(struct i2c_logical_device* dev,
					     char* buffer, size_t size);
static int i2c_common_flush_buffer(struct i2c_logical_device* dev,
				   struct file* file);
static int i2c_common_smbus_transfer(struct i2c_logical_device* dev,
				     struct i2c_smbus_ioctl_data* op);
static int i2c_initialize_slave_buffers(struct i2c_logical_device* dev);

/* ------------------------------------------------------------------------- *
 * I2C structures
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static unsigned short normal_i2c[] = { I2C_OWN_ADDRESS, I2C_CLIENT_END };
static unsigned short normal_i2c_range[] = { I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short probe_i2c_range[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c_range[] = { I2C_CLIENT_END };
static unsigned short force_i2c[] = { 0, I2C_OWN_ADDRESS, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range,
	probe_i2c, probe_i2c_range,
	ignore_i2c, ignore_i2c_range,
	force_i2c
};
#else
static unsigned short normal_i2c[] = { I2C_OWN_ADDRESS, I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
static unsigned short force_i2c[] = { 0, I2C_OWN_ADDRESS, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, probe_i2c, ignore_i2c, force_i2c
};
#endif

/* ------------------------------------------------------------------------- *
 * I2C functions
 * ------------------------------------------------------------------------- */

int
i2c_common_init(struct i2c_logical_device* dev,
                struct i2c_driver* driver,
                struct i2c_slave_device* slave_dev)
{
	int rc = 0;
	int r;
	
	/* fill in addr_data */
	addr_data.normal_i2c[0] = dev->slave_address;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	addr_data.normal_i2c_range[0] = addr_data.normal_i2c_range[1] = dev->slave_address;
#endif

	/* TODO for all devices other than the generic ones we could probe again */
	addr_data.force[0] = ANY_I2C_BUS;
	addr_data.force[1] = dev->slave_address;
      
        // workaround to check if slave is registered
        slave_dev->adapter = NULL;
	
	/* register i2c driver */
	dev->i2c_initialized = 1;
	if ((r = i2c_add_driver(driver))) {
		WARN("%s: i2c driver registration failed (%d).\n", dev->name, r);
		dev->i2c_initialized = 0;
		rc = -ENODEV;
		goto fail;
	}

	/* check if the driver is installed */
	if (dev->data == NULL) {
                /* The probed driver is not included in the fpga ip core. This
                 * happens quite often, so dont issue a warning but a debug message. */
                WARN("%s: device not present, skipping\n", dev->name);
		//WARN("%s: no client detected.\n", dev->name);

                // use ENXIO instead of ENODEV to distinguish between 'bad' and regular initialization errors
		rc = -ENXIO;
		goto fail;
	}
	
	/* register the device in the i2c slave layer */
	slave_dev->adapter = dev->adapter;
	if ((r = i2c_slave_add_device(slave_dev)) != 0) {
		WARN("%s: i2c slave device registration failed (%d).\n", dev->name, r);
		rc = -ENODEV;
		goto fail;
	}

	spin_lock_init(&dev->slave_write_lock);
	spin_lock_init(&dev->slave_read_lock);
	init_MUTEX(&dev->sem);
	
	if ((rc = i2c_initialize_slave_buffers(dev)) != 0) {
		WARN("%s: could not initialize slave mode ringbuffers.\n", dev->name);
		goto fail;
	}
        
        init_waitqueue_head(&dev->wq_read);
        init_waitqueue_head(&dev->wq_write);

	DBG("%s: device loaded successfully!\n", dev->name);

	return 0;

fail:
	i2c_common_cleanup(dev, driver, slave_dev);
	if (rc != -ENXIO) WARN("%s: device didn't load successfully %d!\n", dev->name,rc);
	return rc;
}

static int i2c_initialize_slave_buffers(struct i2c_logical_device* dev) {
	unsigned long flags;

	if (dev->receiver_mode & I2C_MODE_SLAVE) {
		spin_lock_irqsave(&dev->slave_write_lock, flags);
		if (dev->slave_write_buffer == NULL) {
			dev->slave_write_buffer = kmalloc(sizeof(struct ringbuffer), GFP_KERNEL);
		}
		if (dev->slave_write_buffer == NULL) {
			spin_unlock_irqrestore(&dev->slave_write_lock, flags);
			return -ENOMEM;
		}
		ringbuffer_init(dev->slave_write_buffer, IGNORE_OVERFLOW);
		spin_unlock_irqrestore(&dev->slave_write_lock, flags);
	}
	if (dev->transmitter_mode & I2C_MODE_SLAVE) {
		spin_lock_irqsave(&dev->slave_read_lock, flags);
		if (dev->slave_read_buffer == NULL) {
			dev->slave_read_buffer = kmalloc(sizeof(struct ringbuffer), GFP_KERNEL);
		}
		if (dev->slave_read_buffer == NULL) {
			spin_unlock_irqrestore(&dev->slave_read_lock, flags);
			return -ENOMEM;
		}
		ringbuffer_init(dev->slave_read_buffer, IGNORE_OVERFLOW);
		spin_unlock_irqrestore(&dev->slave_read_lock, flags);
	}
	
	return 0;
}

int
i2c_common_cleanup(struct i2c_logical_device* dev,
		   struct i2c_driver* driver,
		   struct i2c_slave_device* slave_dev)
{
	int ret = 0;
	unsigned long flags;
	
	/* delete the driver */
	if (dev->i2c_initialized == 1) {
		if ((ret = i2c_del_driver(driver))) {
			WARN("%s: i2c driver deregistration failed.\n", dev->name);
			return ret;
		}
		dev->i2c_initialized--;
	}
	
	/* delete the device from the slave layer */
        if (slave_dev->adapter != NULL) {
        	if ((ret = i2c_slave_remove_device(slave_dev)) != 0) {
        		WARN("%s: i2c slave device deregistration failed (%d).\n", dev->name, ret);
        	}
        }
	
	/* delete the ringbuffers */
	spin_lock_irqsave(&dev->slave_write_lock, flags);
	if (dev->slave_write_buffer != NULL) {
		kfree(dev->slave_write_buffer);
	}
	spin_unlock_irqrestore(&dev->slave_write_lock, flags);
	
	spin_lock_irqsave(&dev->slave_read_lock, flags);
	if (dev->slave_read_buffer != NULL) {
		kfree(dev->slave_read_buffer);
	}
	spin_unlock_irqrestore(&dev->slave_read_lock, flags);

	return ret;
}

int
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
i2c_common_attach_adapter(struct i2c_logical_device* dev,
			  struct i2c_adapter *adapter,
			  int (*found_proc) (struct i2c_adapter *, int, int))
#else
i2c_common_attach_adapter(struct i2c_logical_device* dev,
			  struct i2c_adapter *adapter,
			  i2c_client_found_addr_proc *found_proc)
#endif
{
	DBG("i2c adapter attached, calling probe: %s\n", adapter->name);

	if (!dev->i2c_initialized) {
		DBG("Not attaching to adapter %s, device not (yet) initialized\n",
			adapter->name);
		return 0;
	}

	/* don't attach the client if it is already attached */
	if (dev->adapter) {
		DBG("Device already attached.\n");
		return 0;
	}
	
	/* don't attach the client if adapter name doesn't match */
	if (dev->desired_adapter_name[0] && strcmp(dev->desired_adapter_name, adapter->name)) {
		DBG("Expected adapter: %s\n", dev->desired_adapter_name);
		return 0;
	}

	return i2c_probe_force(adapter, &addr_data, found_proc);
}

int
i2c_common_i2c_detect_client(struct i2c_logical_device* dev,
			     struct i2c_driver* driver,
			     struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			     unsigned short flags,
#endif
			     int kind)
{
	struct i2c_client* new_client = NULL;
	struct i2c_device_data* data;
	int err = 0;
	
	DBG("i2c detect client addr 0x%04x, kind %d\n", address, kind);
	
	if (!(new_client = kmalloc(sizeof(struct i2c_client) + sizeof(struct i2c_device_data),
			       GFP_KERNEL))) {
		err = -ENOMEM;
		goto fail;
	}
	memset(new_client, 0, sizeof(struct i2c_client) + sizeof(struct i2c_device_data));
	
	dev->data = data = (struct i2c_device_data*) (new_client + 1);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	(void) flags; /* keep compiler silent */
	strcpy(new_client->name, driver->name);
	new_client->data = data;
	new_client->id  = 0; /* ??? */
#else
	strlcpy(new_client->name, driver->name, I2C_NAME_SIZE);
	i2c_set_clientdata(new_client, data);
#endif
	new_client->addr = address;
	new_client->adapter = adapter;
	new_client->driver = driver;
	new_client->flags = 0;
	if ((err = i2c_attach_client_force(new_client))) {
		goto fail_free;
	}    
	
	data->client = new_client;
	
	dev->adapter = adapter;
	
	return 0;
	
fail_free:    
	if (new_client) {
		kfree(new_client);
		new_client = NULL;
	}
	
fail:
	return err;    
}

int
i2c_common_i2c_detach_client(struct i2c_logical_device* dev,
			     struct i2c_client *client)
{
	int err;
	
	if ((err = i2c_detach_client(client))) {
		WARN("i2c client detach failed\n");
		goto fail;
	}
	
	kfree(client);
	dev->adapter = NULL;
	
	return 0;

fail:
	kfree(client);
	
	return err;
}


/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

int
i2c_common_open(struct i2c_logical_device* dev,
		struct inode * inode, struct file * file)
{
	DOWN(&dev->sem);
	
	DBG2("opening device %s\n", dev->name);
		
	file->private_data = dev;
	
	UP(&dev->sem);
	
	return 0;
}

int
i2c_common_release(struct i2c_logical_device* dev,
		   struct inode * inode, struct file * file)
{
	DOWN(&dev->sem);
	
	UP(&dev->sem);
	
	return 0;
}

loff_t
i2c_common_llseek(struct i2c_logical_device* dev,
		  struct file * file, loff_t offset, int whence)
{
	/* don't allow seeking here */
	return -ESPIPE;
}

int
i2c_common_ioctl(struct i2c_logical_device* dev,
		 struct inode * inode, struct file * file,
		 uint cmd, ulong arg)
{
	int retval = 0;

	DOWN(&dev->sem);
	
	switch (cmd) {
		case I2C_DEVICE_SET_TRANSMITTER_MODE:
			dev->transmitter_mode = (u16)arg;
			retval = i2c_initialize_slave_buffers(dev);
			break;
		case I2C_DEVICE_SET_RECEIVER_MODE:
			dev->receiver_mode = (u16)arg;
			retval = i2c_initialize_slave_buffers(dev);
			break;
		case I2C_SLAVE_FLUSH_BUFFER:
			i2c_common_flush_buffer(dev, file);
			retval = 0;
			break;
		case I2C_SLAVE:
		case I2C_SLAVE_FORCE:
			if (arg > 0x7f) {
				retval = -EINVAL;
				break;
			}
			dev->slave_address = arg;
			dev->data->client->addr = arg;
			retval = 0;
			break;
		case I2C_SMBUS:
			retval = i2c_common_smbus_transfer(dev,
				(struct i2c_smbus_ioctl_data *) arg);
			break;
		default:
		    if (dev->adapter->algo->algo_control != NULL) {
			retval = dev->adapter->algo->algo_control(dev->adapter,
								  cmd, arg);
		    } else {
			retval = -EINVAL;
		    }
	}
	
	UP(&dev->sem);
	
	return retval;
}

int
i2c_common_read(struct i2c_logical_device* dev,
		struct file* file, char* buffer,
		size_t size, loff_t* offset)
{
	int ret = 0;
	
	DOWN(&dev->sem);

	if (dev->receiver_mode & I2C_MODE_SLAVE) {
		ret = i2c_common_slave_read(dev,
			file, buffer, size, offset);
	}
	else if (dev->receiver_mode & I2C_MODE_MASTER) {
		ret = i2c_common_master_read(dev,
			file, buffer, size, offset);
	}
	else {
		ret = -EIO;
	}

	UP(&dev->sem);
	
	return ret;
}

int
i2c_common_write(struct i2c_logical_device* dev,
		 struct file* file, const char* buffer,
		 size_t size, loff_t* offset)
{
	int ret = 0;
	
	DOWN(&dev->sem);

	if (dev->transmitter_mode & I2C_MODE_SLAVE) {
		ret = i2c_common_slave_write(dev,
			file, buffer, size, offset);
	}
	else if (dev->transmitter_mode & I2C_MODE_MASTER) {
		ret = i2c_common_master_write(dev,
			file, buffer, size, offset);
	}
	else {
		ret = -EIO;
	}

	UP(&dev->sem);
	
	return ret;
}

int
i2c_common_poll(struct i2c_logical_device* dev,
		struct file * file, poll_table * wait)
{
	int mask = 0;
	int empty, full;
	unsigned long flags;
	
	if (dev->receiver_mode & I2C_MODE_SLAVE) {
		poll_wait(file, &dev->wq_read, wait);
		
		/* if no data in the slave write buffer, don't allow reading */
		spin_lock_irqsave(&dev->slave_write_lock, flags);
		empty = ringbuffer_is_empty(dev->slave_write_buffer);
		spin_unlock_irqrestore(&dev->slave_write_lock, flags);
		
		if (!empty) {
			mask |= POLLIN | POLLRDNORM;
		}
	}
	else {
		/* master mode, always readable and writable */
		mask |= POLLIN | POLLRDNORM;
	}
	
	if (dev->transmitter_mode & I2C_MODE_SLAVE) {
		poll_wait(file, &dev->wq_write, wait);
		
		/* if slave read buffer is full, don't allow writing */
		spin_lock_irqsave(&dev->slave_read_lock, flags);
		full = ringbuffer_is_full(dev->slave_read_buffer);
		spin_unlock_irqrestore(&dev->slave_read_lock, flags);
		
		if (!full) {
			mask |= POLLOUT | POLLWRNORM;
		}
	}
	else {
		/* master mode, always readable and writable */
		mask |= POLLOUT | POLLWRNORM;
	}
	
	return mask;
}

/* ------------------------------------------------------------------------- *
 * SMBus emulation
 * ------------------------------------------------------------------------- */

static int
i2c_common_smbus_transfer(struct i2c_logical_device* dev,
			  struct i2c_smbus_ioctl_data* op)
{
	struct i2c_smbus_ioctl_data data_arg;
	int datasize, res;
	struct i2c_client * client = dev->data->client;
	union i2c_smbus_data temp;

	if (copy_from_user(&data_arg, op,
	                   sizeof(struct i2c_smbus_ioctl_data))) {
		return -EFAULT;
	}
	
	if ((data_arg.size != I2C_SMBUS_BYTE) && 
	    (data_arg.size != I2C_SMBUS_QUICK) &&
	    (data_arg.size != I2C_SMBUS_BYTE_DATA) && 
	    (data_arg.size != I2C_SMBUS_WORD_DATA) &&
	    (data_arg.size != I2C_SMBUS_PROC_CALL) &&
	    (data_arg.size != I2C_SMBUS_BLOCK_DATA) &&
	    (data_arg.size != I2C_SMBUS_I2C_BLOCK_DATA)) {
		DBG("i2c-dev.o: size out of range (%x) in ioctl I2C_SMBUS.\n",
		       data_arg.size);
		return -EINVAL;
	}
	/* Note that I2C_SMBUS_READ and I2C_SMBUS_WRITE are 0 and 1, 
	   so the check is valid if size==I2C_SMBUS_QUICK too. */
	if ((data_arg.read_write != I2C_SMBUS_READ) && 
	    (data_arg.read_write != I2C_SMBUS_WRITE)) {
		DBG("i2c-dev.o: read_write out of range (%x) in ioctl I2C_SMBUS.\n",
		       data_arg.read_write);
		return -EINVAL;
	}

	/* Note that command values are always valid! */

	if ((data_arg.size == I2C_SMBUS_QUICK) ||
	    ((data_arg.size == I2C_SMBUS_BYTE) && 
	    (data_arg.read_write == I2C_SMBUS_WRITE)))
		/* These are special: we do not use data */
		return i2c_smbus_xfer(client->adapter, client->addr,
		                      client->flags,
		                      data_arg.read_write,
		                      data_arg.command,
		                      data_arg.size, NULL);

	if (data_arg.data == NULL) {
		DBG("i2c-dev.o: data is NULL pointer in ioctl I2C_SMBUS.\n");
		return -EINVAL;
	}

	if ((data_arg.size == I2C_SMBUS_BYTE_DATA) ||
	    (data_arg.size == I2C_SMBUS_BYTE))
		datasize = sizeof(data_arg.data->byte);
	else if ((data_arg.size == I2C_SMBUS_WORD_DATA) || 
	         (data_arg.size == I2C_SMBUS_PROC_CALL))
		datasize = sizeof(data_arg.data->word);
	else /* size == I2C_SMBUS_BLOCK_DATA */
		datasize = sizeof(data_arg.data->block);

	if ((data_arg.size == I2C_SMBUS_PROC_CALL) || 
	    (data_arg.read_write == I2C_SMBUS_WRITE)) {
		if (copy_from_user(&temp, data_arg.data, datasize))
			return -EFAULT;
	}
	res = i2c_smbus_xfer(client->adapter,client->addr,client->flags,
	      data_arg.read_write,
	      data_arg.command,data_arg.size,&temp);
	if (! res && ((data_arg.size == I2C_SMBUS_PROC_CALL) || 
		      (data_arg.read_write == I2C_SMBUS_READ))) {
		if (copy_to_user(data_arg.data, &temp, datasize))
			return -EFAULT;
	}
	return res;

}

/* ------------------------------------------------------------------------- *
 * I2C Slave mode callbacks
 * ------------------------------------------------------------------------- */

int
i2c_common_slave_write_cb(struct i2c_logical_device* dev,
			  char* buf, int size)
{
	int ret = 0;
	int mysize = size;
	unsigned long flags;
	
	spin_lock_irqsave(&dev->slave_write_lock, flags);
	
	if (!(dev->receiver_mode & I2C_MODE_SLAVE)) {
		WARN("%s: device is not configured for slave mode.\n", dev->name);
	}
	else if ((ret = ringbuffer_put_entry(dev->slave_write_buffer, buf, &mysize)) < 0) {
		WARN("%s: could not put slave write data, buffer full!\n", dev->name);
	}
	else {
		DBG2("%s: put %d bytes in slave write queue.\n", dev->name, mysize);
		DUMP_BUF(dev->name, buf, mysize);
		ret = mysize;
	}
	
	spin_unlock_irqrestore(&dev->slave_write_lock, flags);
	
	/* new data for reading, wakeup a sleeping select or poll */
	wake_up_interruptible(&dev->wq_read);

	return ret;
}

int
i2c_common_slave_read_cb(struct i2c_logical_device* dev,
			 char* buf, int size)
{
	int ret = 0;
	int mysize = size;
	unsigned long flags;
	
	spin_lock_irqsave(&dev->slave_read_lock, flags);
	
	if (!(dev->transmitter_mode & I2C_MODE_SLAVE)) {
		WARN("%s: device is not configured for slave mode.\n", dev->name);
	}
	else if ((ret = ringbuffer_get_entry(dev->slave_read_buffer, buf, &mysize)) < 0) {
		WARN("%s: could not get slave read data, buffer empty!\n", dev->name);
	}
	else {
		DBG2("%s: got %d bytes from slave read queue.\n", dev->name, mysize);
		DUMP_BUF(dev->name, buf, mysize);
		ret = mysize;
	}
	
	spin_unlock_irqrestore(&dev->slave_read_lock, flags);
	
	/* new space for writing, wakeup a sleeping select or poll */
	wake_up_interruptible(&dev->wq_write);
	
	return ret;
}

/* ------------------------------------------------------------------------- *
 * actual write functions
   Note that these are the write function from user view, means the user
   wants to write data to the I2C. A write function results in an I2C
   master write or a slave read (because a master wants to read data from
   us which we previously wrote to a buffer).
 * ------------------------------------------------------------------------- */

static int
i2c_common_slave_write(struct i2c_logical_device* dev,
		       struct file* file, const char* buffer,
		       size_t size, loff_t* offset)
{
	int ret = 0;
	unsigned char buf[BUFFER_ENTRYSIZE];
	unsigned long flags;
	size_t mysize = min((int)size, BUFFER_ENTRYSIZE);
	
	if (copy_from_user(buf, buffer, mysize)) {
		ret = -EFAULT;
		goto bail;
	}
	
	spin_lock_irqsave(&dev->slave_read_lock, flags);
	
	if (ringbuffer_put_entry(dev->slave_read_buffer, buf, &mysize) < 0) {
		WARN("%s: could not write slave read data, buffer full!\n", dev->name);
		ret = 0;	/* signal EOF */
	}
	else {
		DBG2("%s: put %d bytes in slave read queue.\n", dev->name, mysize);
		ret = mysize;
	}
	
	spin_unlock_irqrestore(&dev->slave_read_lock, flags);
	
	*offset += ret;
	
	if (dev->adapter) {
		unblock_adapter(dev->adapter);
	}

bail:
	return ret;
}

static int
i2c_common_master_write(struct i2c_logical_device* dev,
			struct file* file, const char* buffer,
			size_t size, loff_t* offset)
{
	int (*write_func)(struct i2c_logical_device*, char*, size_t) =
		(dev->transmitter_mode & I2C_MODE_MASTER_ADDRESS) ?
		i2c_common_master_write_address : 
		i2c_common_master_write_no_address;
	int ret = 0;
	
	unsigned char buf[BUFFER_ENTRYSIZE];
	size_t mysize = min((int)size, BUFFER_ENTRYSIZE);
	
	if (copy_from_user(buf, buffer, mysize)) {
		ret = -EFAULT;
		goto bail;
	}
	
	ret = write_func(dev, buf, mysize);
	if (ret <= 0) {
		goto bail;
	}
	
	DBG2("%s: wrote %d bytes as master write operation.\n", dev->name, ret);
	*offset += ret;
	
bail:
	return ret;
}

/* ------------------------------------------------------------------------- *
 * actual read functions
   Note that these are the read function from user view, means the user
   wants to read data from the I2C. A read function results in an I2C
   master read or a slave write (because a master wants to write data to
   us which we afterwards read into a buffer).
 * ------------------------------------------------------------------------- */

static int
i2c_common_slave_read(struct i2c_logical_device* dev,
		      struct file* file, const char* buffer,
		      size_t size, loff_t* offset)
{
	int ret = 0;
	unsigned char buf[BUFFER_ENTRYSIZE];
	unsigned long flags;
	size_t mysize = min((int)size, BUFFER_ENTRYSIZE);
	
	DBG2("%s: reading %d bytes as slave read operation.\n", dev->name, mysize);

	spin_lock_irqsave(&dev->slave_write_lock, flags);
	
	if (ringbuffer_get_entry(dev->slave_write_buffer, buf, &mysize) < 0) {
		DBG("%s: could not get slave write data, buffer empty!\n", dev->name);
		ret = 0;	/* signal EOF */
	}
	else {
		DBG2("%s: got %d bytes from slave write queue.\n", dev->name, mysize);
		ret = mysize;
	}
	
	spin_unlock_irqrestore(&dev->slave_write_lock, flags);
	
	if (copy_to_user((char *)buffer, buf, mysize)) {
		ret = -EFAULT;
		goto bail;
	}
	
	*offset += ret;
	
	if (dev->adapter) {
		unblock_adapter(dev->adapter);
	}

	DBG2("%s: received %d bytes as slave read operation.\n", dev->name, mysize);

bail:
	return ret;
}

static int
i2c_common_master_read(struct i2c_logical_device* dev,
		       struct file* file, const char* buffer,
		       size_t size, loff_t* offset)
{
	int (*read_func)(struct i2c_logical_device*, char*, size_t) =
		(dev->receiver_mode & I2C_MODE_MASTER_ADDRESS) ?
		i2c_common_master_read_address : 
		i2c_common_master_read_no_address;
	int ret = 0;
	
	unsigned char buf[BUFFER_ENTRYSIZE];
	size_t mysize = min((int)size, BUFFER_ENTRYSIZE);
	
	ret = read_func(dev, buf, mysize);
	if (ret <= 0) {
		goto bail;
	}
	
	if (copy_to_user((char *)buffer, buf, ret)) {
		ret = -EFAULT;
		goto bail;
	}
	
	DBG2("%s: received %d bytes as master read operation.\n", dev->name, ret);
	*offset += ret;
	
bail:
	return ret;
}

/* ------------------------------------------------------------------------- *
 * low level I2C master mode functions
 * ------------------------------------------------------------------------- */

static inline int
i2c_common_transfer(struct i2c_adapter * adapter, struct i2c_msg * msg, int num)
{
	int retries = 0;
	int ret;
	
	if (!adapter) {
		return -ENODEV;
	}
	
	do {
		ret = i2c_transfer(adapter, msg, num);
		if (ret > 0) return ret;
		DBG("i2c_transfer failed, retry # %d\n", retries + 1);
	} while (retries++ < MASTER_RETRIES);
	
	return ret;
}

static int
i2c_common_master_write_address(struct i2c_logical_device* dev,
				char* buffer, size_t size)
{
	struct i2c_msg msg[2];
	unsigned char addr_msg[] = { (dev->master_address & 0xFF00) >> 8, dev->master_address & 0x00FF };
	int ret;
	
	memset(msg, 0, 2*sizeof(struct i2c_msg));
	
	msg[0].addr  = dev->slave_address;
	msg[0].buf   = addr_msg;
	msg[0].len   = 2;
	
	msg[1].addr  = dev->slave_address;
	msg[1].buf   = buffer;
	msg[1].len   = size;
	
	DBG2("writing %d bytes as master write transfer with address to 0x%02x\n", size, dev->slave_address);
	ret = i2c_common_transfer(dev->adapter, msg, 2);
	DBG2("ret %d\n", ret);

	if (ret < 2) return ret;
	return size;
}

static int
i2c_common_master_write_no_address(struct i2c_logical_device* dev,
				   char* buffer, size_t size)
{
	struct i2c_msg msg;
	int ret;
	
	memset(&msg, 0, sizeof(struct i2c_msg));
	
	msg.addr  = dev->slave_address;
	msg.buf   = buffer;
	msg.len   = size;
	
	DBG2("writing %d bytes as master write transfer without address to 0x%02x\n", size, dev->slave_address);
	ret = i2c_common_transfer(dev->adapter, &msg, 1);
	DBG2("ret %d\n", ret);

	if (ret < 1) return ret;
	return size;
}

static int
i2c_common_master_read_address(struct i2c_logical_device* dev,
			       char* buffer, size_t size)
{
	struct i2c_msg msg[2];
	unsigned char addr_msg[] = { (dev->master_address & 0xFF00) >> 8, dev->master_address & 0x00FF };
	int ret;
	
	memset(msg, 0, 2*sizeof(struct i2c_msg));
	
	msg[0].addr  = dev->slave_address;
	msg[0].buf   = addr_msg;
	msg[0].len   = 2;
	
	msg[1].addr  = dev->slave_address;
	msg[1].buf   = buffer;
	msg[1].flags = I2C_M_RD;
	msg[1].len   = size;
	
	DBG2("reading %d bytes as master read transfer with address\n", size);
	ret = i2c_common_transfer(dev->adapter, msg, 2);
	DBG2("ret %d\n", ret);

	if (ret < 2) return ret;
	return size;
}

static int
i2c_common_master_read_no_address(struct i2c_logical_device* dev,
				  char* buffer, size_t size)
{
	struct i2c_msg msg;
	int ret;
	
	memset(&msg, 0, sizeof(struct i2c_msg));
	
	msg.addr  = dev->slave_address;
	msg.buf   = buffer;
	msg.flags = I2C_M_RD;
	msg.len   = size;
	
	DBG2("reading %d bytes as master read transfer without address\n", size);
	ret = i2c_common_transfer(dev->adapter, &msg, 1);
	DBG2("ret %d\n", ret);

	if (ret < 1) return ret;
	return size;
}

/* ------------------------------------------------------------------------- *
 * misc functions
 * ------------------------------------------------------------------------- */

static int
i2c_common_flush_buffer(struct i2c_logical_device* dev, struct file* file)
{
	unsigned long flags;
	
	spin_lock_irqsave(&dev->slave_write_lock, flags);
	if (dev->slave_write_buffer) {
		ringbuffer_init(dev->slave_write_buffer, IGNORE_OVERFLOW);
	}
	spin_unlock_irqrestore(&dev->slave_write_lock, flags);
	
	spin_lock_irqsave(&dev->slave_read_lock, flags);
	if (dev->slave_read_buffer) {
		ringbuffer_init(dev->slave_read_buffer, 0);
	}
	spin_unlock_irqrestore(&dev->slave_read_lock, flags);
	
	if (dev->adapter) {
		unblock_adapter(dev->adapter);
	}
	
	return 0;
}


