/*****************************************************************************
 * lara_common.c
 *
 * Copyright (c) 2001-2003 Peppercon AG, miba@peppercon.de
 *
 * Contains common routines for all LARA-like devices regardless of which
 * video engine (TFT/FIFO or VSC) they are using
 *****************************************************************************/

/* ------------------------------------------------------------------------- *
 * kernel includes
 * ------------------------------------------------------------------------- */

#include <linux/version.h>
#include <linux/bigphysarea.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/rtc.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/irq.h>
#include <pp_kernel_common.h>

#ifdef __powerpc__
# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#  include <platforms/peppercon.h>
#  include <asm/ppc405_pcimcs.h>
# else
#  include <platforms/4xx/peppercon.h>
#  include <syslib/ppc405_pcimcs.h>
# endif
# include <asm/time.h>
# include <asm/ppc4xx_dma.h>
#endif

#ifdef __arm__
# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#  include <asm/arch-cpe/cpe_bd.h>
# else
#  include <asm/arch/pp_bd.h>
#  include <asm/arch/timer.h>
# endif
# include <asm/arch/cpe_clk.h>
# include <asm/arch/cpe_gettime.h>
#endif

/* ------------------------------------------------------------------------- *
 * own includes
 * ------------------------------------------------------------------------- */

#include "lara.h"
#include "lara_int.h"

#if defined (PP_FEAT_RESET_TO_DEFAULTS_PINS)
# include "gpio/gpio_hilevel_functions.h"
# if defined PRODUCT_ASMIDC
#  define RESET_TO_DEFAULTS_HIGHACTIVE 0
# endif
# if defined PRODUCT_ERICG4
#  define RESET_TO_DEFAULTS_HIGHACTIVE 1
# endif
#endif

#include "lara_common.h"
#include "vsc_regs.h"

/* ------------------------------------------------------------------------- *
 * global constants, macros and datatypes
 * ------------------------------------------------------------------------- */

#define DRV_NAME			"lara_common"
#define LARA_COMMON_MAJOR		((u8)241)

#define PPC_PCI_SLOT			0
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
/* seems to be a bug in pci kernel code */
# define PPC_PCI_FUNC			1
#else
# define PPC_PCI_FUNC			0
#endif

#ifdef __powerpc__
#define PPC_TBR(u, l) {					\
	u32 u2;						\
	asm volatile("1: mftbu %0\n"			\
		     "   mftb  %1\n"			\
		     "   mftbu %2\n"			\
		     "   cmpw  %0, %2\n"		\
		     "   bne   1b\n"			\
		     : "=r" (u), "=r" (l), "=r" (u2)	\
		     );					\
    }

#endif

#define HWID_PROC_ENTRY			"hardware_id"
#define noPPC_TIME_MEASUREMENT

#include "lara_common.h"

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */
typedef struct {
    char			name[15];
    int				defaults_set;
    int                         serial_debug_enabled;
    int                         config_mode_enabled;
    int                         power_cim_detected;
    spinlock_t			propchange_lock;
    struct list_head		propchange_list;
    struct timer_list           beep_timer;
    struct pci_dev *		ppc_pci_dev;
    struct pci_dev *		hfc_pci_dev;
    u32				ppc_pcibar2;
    u32				hfc_pcibar1;
    u8			    	hwid;
    clock_info_t		clock_info;
    spinlock_t		        clock_lock;
} lara_dev_common_t;

static lara_dev_common_t dev = {
    name:			DRV_NAME,
    defaults_set:		DEFAULTS_NONE,
    serial_debug_enabled:       0,
    config_mode_enabled:        0,
    power_cim_detected:         0,
    ppc_pci_dev:                NULL,
    hfc_pci_dev:                NULL,
    ppc_pcibar2:                0,
    hfc_pcibar1:                0,
    propchange_lock:            SPIN_LOCK_UNLOCKED,
    propchange_list:            {},
    beep_timer:                 {},
    hwid:                       0,
    clock_info:                 {},
    clock_lock:			SPIN_LOCK_UNLOCKED,
};

/*
 * used to keep pid registered for getting 
 * the property change signal
 */
typedef struct {
    struct list_head listnode;
    int pid;
} sigpid_t;

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */

int         init_module(void);
void        cleanup_module(void);
int         lara_common_init(void);
static void lara_common_cleanup(void);
static int  lara_common_ioctl(struct inode *, struct file *, uint, ulong);
static int  lara_common_open(struct inode *, struct file *);
static int  lara_common_release(struct inode *, struct file *);

static int  propchange_add(int pid);
static int  propchange_remove(int pid);
static void propchange_cleanup(void);
void propchange_fire(unsigned short prop, unsigned short param);

#ifdef __powerpc__
# ifdef PPC_TIME_MEASUREMENT
static inline void ppc_tbr(u32* upper, u32* lower);
static unsigned long long ppc_timebase(void);
static unsigned long long ppc_timediff_ns(unsigned long long t1,					  
					  unsigned long long t2);
# endif
static void fpga_write_byte(u_int8_t value);
#endif

static int hwid_proc_read(char*, char**, off_t, int, int*, void*);

#if defined (PRODUCT_FLASHX4) || ( defined (PRODUCT_KIMTESTERMST) && defined (PP_BOARD_LARA))
static void gpio_isr(int irq, void * dev_id, struct pt_regs * regs);
static int last_tm = 0;
#endif

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

#ifdef MODULE
MODULE_AUTHOR("miba@peppercon.de");
MODULE_DESCRIPTION("LARA device driver for basic functionality");
MODULE_LICENSE("GPL");

EXPORT_SYMBOL(propchange_fire);

/*
 * Initialize the module
 */
int init_module(void)
{
    return lara_common_init();
}

/*
 * Cleanup - unregister the appropriate file from /proc
 */
void cleanup_module(void)
{
    int r;

    lara_common_cleanup();

    if ((r = unregister_chrdev(LARA_COMMON_MAJOR, dev.name)) < 0) {
	D(D_ALWAYS, "failed to unregister driver (%d)\n", r);
    } else {
	D(D_ALWAYS, "driver unregistered\n");
    }
}
#endif	/* MODULE */

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations lara_common_ops = {
    owner:   THIS_MODULE,
    ioctl:   lara_common_ioctl,
    open:    lara_common_open,
    release: lara_common_release
};

/* ------------------------------------------------------------------------- *
 * common initialization
 * ------------------------------------------------------------------------- */

int
lara_common_init(void)
{
    int rc = SUCCESS;
    int r;
#if defined (PP_FEAT_RESET_TO_DEFAULTS_PINS)
    u_char defaults_set;
#endif

    D(D_ALWAYS, "driver init, debuglevel=%d\n", DEBUGLEVEL);
	
    /* ---- list and mutex initializations ----------------------------- */
    INIT_LIST_HEAD(&dev.propchange_list);

    /* ---- get some data from bootloader ---------------------------------- */
#ifdef __powerpc__
    {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	bd_t * bd = (bd_t *)__res;
#else
	bd_t * bd = (bd_t *)&__res;
#endif

	dev.defaults_set = bd->bi_def_jumper_set;
	dev.serial_debug_enabled = bd->bi_debug_mode;
	dev.config_mode_enabled = bd->bi_config_mode;
	dev.power_cim_detected = 0;
	dev.hwid = bd->bi_hwid;

	dev.clock_info.cpu = bd->bi_procfreq;
	dev.clock_info.arch.ppc.plb = bd->bi_plb_busfreq;
	dev.clock_info.arch.ppc.pci = bd->bi_pci_busfreq;
    }
#else
    {
        bd_arm_t * bd = (bd_arm_t *)__res;

    	dev.defaults_set = bd->pp_info.bi_def_jumper_set;
#if defined (PP_FEAT_RESET_TO_DEFAULTS_PINS)
	r = pp_gpio_bit_get(PP_GPIO_DEFAULTS_DEV, PP_GPIO_DEFAULTS, &defaults_set);
	if (r == 0) dev.defaults_set |= ((defaults_set == RESET_TO_DEFAULTS_HIGHACTIVE) ? DEFAULTS_JUMPER : 0);
#endif
	dev.serial_debug_enabled = bd->pp_info.bi_debug_mode;
	dev.config_mode_enabled = bd->pp_info.bi_config_mode;
	dev.power_cim_detected = bd->pp_info.bi_power_cim;
	dev.hwid = bd->pp_info.bi_hwid;

	dev.clock_info.cpu = BD_CPU_CLK(bd);
	dev.clock_info.arch.arm.ahb = BD_AHB_CLK(bd);
	dev.clock_info.arch.arm.apb = BD_APB_CLK(bd);
    }
#endif
    D(D_VERBOSE, "defaults state = %d\n", dev.defaults_set);
    D(D_VERBOSE, "serial debug state = %d\n", dev.serial_debug_enabled);
    D(D_VERBOSE, "config mode state = %d\n", dev.config_mode_enabled);
    D(D_VERBOSE, "power cim = %d\n", dev.power_cim_detected);
    D(D_VERBOSE, "hardware id = %d\n", dev.hwid);
#ifdef __powerpc__
    D(D_VERBOSE, "clocks: CPU=%d, PLB=%d, PCI=%d\n",
      dev.clock_info.cpu, dev.clock_info.arch.ppc.plb, dev.clock_info.arch.ppc.pci);
#else
    D(D_VERBOSE, "clocks: CPU=%d, AHB=%d, APB=%d \n",
      dev.clock_info.cpu, dev.clock_info.arch.arm.ahb, dev.clock_info.arch.arm.apb);
#endif
    
#ifdef __powerpc__
    /* ---- find the various devices on our PCI bus ------------------------ */
    dev.ppc_pci_dev = pci_find_slot(0, PCI_DEVFN(PPC_PCI_SLOT, PPC_PCI_FUNC));
    if (dev.ppc_pci_dev == NULL) {
	D(D_ERROR, "PPC PCI device not detected (slot:%d, func:%d)\n", PPC_PCI_SLOT, PPC_PCI_FUNC );
	rc = -ENODEV;
	goto error_out;
    }
    /* ---- enable extended PCI error reporting ---------------------------- */
    {
	u8 val8;
	mcs_pci_read_config_byte(dev.ppc_pci_dev, 0x48, &val8);
	mcs_pci_write_config_byte(dev.ppc_pci_dev, 0x48, val8 | 0x5d);
    }
#endif
    
    /* ---- create proc fs entry for hardware id --------------------------- */
    if (create_proc_read_entry(HWID_PROC_ENTRY, 0, NULL,
			       hwid_proc_read, NULL) == NULL) {
	D(D_ERROR, "Could not create /proc entry for hardware id\n");
	rc = -ENODEV;
	goto error_out;
    }

#if __powerpc__
    /* TODO(miba) move this to a 'start' ioctl */
#if defined(PRODUCT_FLASHX4)
    if ( request_irq(FLASH_PROG_IRQ2, gpio_isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
		    "FLASH PROG", NULL ) ) {
	    D(D_ERROR, "couldn't register irq handler for flash prog\n");
	    rc = -ENODEV;
	    goto error_out;
	}    
#endif
    D(D_NOTICE, "Found Hardware Version: %02x\n", dev.hwid);
#if defined (PRODUCT_KIMTESTERMST) && defined (PP_BOARD_LARA)
    if ( request_irq(KIM_TESTER_GPIO_IRQ, gpio_isr, SA_SHIRQ | SA_SAMPLE_RANDOM,
		    "LARA GPIO", NULL ) ) {
	    D(D_ERROR, "couldn't register gpio irq handler for kimtester master\n");
	    rc = -ENODEV;
	    goto error_out;
	}    
#endif    
#endif
       
    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(LARA_COMMON_MAJOR, dev.name, &lara_common_ops)) < 0) {
	D(D_ERROR, "failed to register driver (%d)\n", r);
	rc = -ENODEV;
	goto error_out;
    }

    D(D_ALWAYS, "kernel module loaded successfully!\n");
    return rc;
    
 error_out:
    lara_common_cleanup();
    D(D_ALWAYS, "kernel module didn't load successfully %d!\n", rc);
    return rc;
}

void
lara_common_cleanup(void)
{
    propchange_cleanup();

    remove_proc_entry(HWID_PROC_ENTRY, NULL);
}

int
lara_common_ioctl(struct inode * inode, struct file * file,
		  uint cmd, ulong arg)
{
    unsigned char state;
    
    switch (cmd) {
      case ERICIOCPROPCHANGESIGNAL:
	  if(arg == 1) return propchange_add(current->pid);
	  else return propchange_remove(current->pid);
#ifdef __powerpc__
      case ERICIOCFPGAWRITEBYTE:
	  {
	      u_int8_t fpga_val;
	      
	      if (get_user(fpga_val, (u_int8_t *)arg)) {
		  return -EFAULT;
	      }
	      
	      fpga_write_byte(fpga_val);
	      return SUCCESS;
	  }
#endif
      case ERICIOCGETHARDWAREREVISION:
	  if (put_user(dev.hwid, (u_int8_t *)arg)) {
	      return -EFAULT;
	  }
	  return SUCCESS;

      case ERICIOCCHECKDEFAULTS:
	  if (put_user(dev.defaults_set, (int *)arg)) {
	      return -EFAULT;
	  }
	  return SUCCESS;

      case ERICIOCCHECKSERIALDEBUG:
	  if (put_user(dev.serial_debug_enabled, (int *)arg)) {
	      return -EFAULT;
          }
	  return SUCCESS;

      case ERICIOCCHECKPOWERCIM:
	  if (put_user(dev.power_cim_detected, (int *)arg)) {
	      return -EFAULT;
          }
	  return SUCCESS; 
	  
      case ERICIOCCHECKCONFIGMODE:
	  if (put_user(dev.config_mode_enabled, (int *)arg)) {
	      return -EFAULT;
          }
	  return SUCCESS;

      case ERICIOCKMEISPRESENT:
	  if (put_user(1, (int *)arg)) {
	      return -EFAULT;
	  }
	  return SUCCESS;

      case PPIOCGETCLOCKINFO:
	  if (copy_to_user((clock_info_t *)arg, &dev.clock_info, sizeof(clock_info_t))) {
	      return -EFAULT;
	  }
	  return SUCCESS;
#ifdef __arm__
      case PPIOCGETCLOCKVALUE:
	  {
              bd_arm_t * bd = (bd_arm_t *)__res;
	      u_long flags;
	      uint64_t v;
	      uint32_t t;
	      
	      // lock to prevent the timer irq between reading the timer (1) register and the jiffies value
	      spin_lock_irqsave(&dev.clock_lock, flags);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	      t = cpe_timer_get_counter(1); // these are the clock ticks since the last jiffies update
#else
	      t = timer_fttmr010_get_counter(0); // these are the clock ticks since the last jiffies update
#endif

	      // note, that the counter counts from APB_CLK downto 0
	      v = (uint64_t)((BD_APB_CLK(bd)/HZ) - t) + ((uint64_t)jiffies)*(uint64_t)(BD_APB_CLK(bd)/HZ);

	      spin_unlock_irqrestore(&dev.clock_lock, flags);
	      if (copy_to_user((uint64_t *)arg, &v, sizeof(v))) {
		  return -EFAULT;
	      }
	  }
	  return SUCCESS;
#endif

      case PPIOCSTATUSLEDBLINKING:
	  if (get_user(state, (unsigned char *)arg)) return -EFAULT;
#ifdef PP_FEAT_SYSOK_BLINKING
	  pp_status_led_blinking(state);
	  return SUCCESS; 
#else
	  D(D_ERROR, "PPIOCSTATUSLEDBLINKING not implemented (!PP_FEAT_SYSOK_BLINKING)\n");
	  return -EIO;
#endif

      case PPIOCGETKIRAMINORREVISION:
#ifdef __arm__
	  {
	      u32* pmu_regs = (u32*) CPE_PMU_VA_BASE;	      
	      u_char minor = (pmu_regs[KIRA_PMU_REG_MINORREV] & KIRA_PMU_REG_MINORREV_MINORREV) ? 1 : 0;

	      D(D_VERBOSE, "KIRA_PMU_REG_MINORREV=%08x\n",
		      pmu_regs[KIRA_PMU_REG_MINORREV]);
	      
	      if (put_user(minor, (u_char *)arg)) {
		  return -EFAULT;
	      }
	      
	      return SUCCESS;
	  }
#else
	  return -ENODEV;
#endif

      default:
	  D(D_ERROR, "Invalid ioctl request %08x\n", cmd);
	  return -EINVAL;
    }
}

static int
lara_common_open(struct inode * inode, struct file * file)
{
    return SUCCESS;
}

static int
lara_common_release(struct inode * inode, struct file * file)
{
    return SUCCESS;
}

/* ------------------------------------------------------------------------- *
 * Property Change Support method
 * it's just like in java: add, remove, fire... ;-)
 * ------------------------------------------------------------------------- */

/*
 * fire a Property Change Signal
 */
void
propchange_fire(unsigned short prop, unsigned short param)
{
    struct siginfo info;
    struct list_head *entry;
    u_long flags;
    int ret;
    info.si_signo = SIGPROPCHG;
    info.si_errno = 0;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    info.si_code  = SI_QUEUE;
#else
/* FIXME: using this new si_code is not tested - dont remove this warning until you are sure it works properly */
    info.si_code  = __SI_CODE(__SI_PPPROP, SI_KERNEL);
#endif
    spin_lock_irqsave(&dev.propchange_lock, flags);
    list_for_each(entry, &dev.propchange_list) {
	sigpid_t * sigpid = list_entry(entry, sigpid_t, listnode);
	struct task_struct * proc = find_task_by_pid(sigpid->pid);
	if(NULL == proc) {
	    entry = entry->prev;
	    list_del(&sigpid->listnode);
	    kfree(sigpid);
	    continue;
	}
	info.si_int = (prop << 16) | param;
	ret = send_sig_info(SIGPROPCHG, &info, proc);
	D(D_VERBOSE, "sent signal %d (%d,%d) to pid: %d, ret=%d\n", 
	  SIGPROPCHG, prop, param, proc->pid, ret);
    }
    spin_unlock_irqrestore(&dev.propchange_lock, flags);
}

/**
 * add a pid to the property change list
 */
int
propchange_add(int newpid)
{
    sigpid_t* sigpid;
    u_long flags;
    if(NULL == (sigpid = kmalloc(sizeof(sigpid_t), GFP_KERNEL))) 
	return -ENOMEM;
    sigpid->pid = newpid;
    spin_lock_irqsave(&dev.propchange_lock, flags);
    D(D_VERBOSE, "adding pid: %d to property change listeners\n", newpid);
    list_add(&sigpid->listnode, &dev.propchange_list);
    spin_unlock_irqrestore(&dev.propchange_lock, flags);
    return 0;
}

/**
 * remove a pid from the property change list
 */
int
propchange_remove(int delpid)
{
    struct list_head *entry;
    u_long flags;
    spin_lock_irqsave(&dev.propchange_lock, flags);
    list_for_each(entry, &dev.propchange_list) {
	sigpid_t * sigpid = list_entry(entry, sigpid_t, listnode);
	if(sigpid->pid == delpid) {
            D(D_VERBOSE, "removing pid: %d to property change listeners\n", delpid);
	    entry = entry->prev;
	    list_del(&sigpid->listnode);
	    kfree(sigpid);
	    spin_unlock_irqrestore(&dev.propchange_lock, flags);
	    return 0;
	}
    }
    spin_unlock_irqrestore(&dev.propchange_lock, flags);
    return -EINVAL;
}

/**
 * return all the allocated memory
 */
void
propchange_cleanup()
{
    struct list_head *entry;
    list_for_each(entry, &dev.propchange_list) {
	sigpid_t * sigpid = list_entry(entry, sigpid_t, listnode);
	entry = entry->prev;
	list_del(&sigpid->listnode);
	kfree(sigpid);
    }
}

/*
 * return hardware revision id using a /proc entry
 */
int
hwid_proc_read(char *page, char **start, off_t offset, int count,
	       int *eof, void *data)
{
    int len = 0;

    len += sprintf(page, "%02x\n", dev.hwid);
    
    *eof = 1;
    return len;
}

#ifdef __powerpc__
/* ------------------------------------------------------------------------- *
 * high resolution timer support
 * ------------------------------------------------------------------------- */

# ifdef PPC_TIME_MEASUREMENT
/*
 * read time base register
 */
inline void
ppc_tbr(u32* upper, u32* lower)
{
    u32 u, l;
    PPC_TBR(u, l);
    *upper = u;
    *lower = l;
}

/*
 * returns the ppc timebase register as
 * long long value
 */
unsigned
long long ppc_timebase(void)
{
    unsigned long long t;
    unsigned long u, u0, l;
    do { // loop required as per PPC manual
	u0 = get_tbu();
	l = get_tbl();
	u = get_tbu();
    } while(u0 != u); // reread if rollover occurred
    t = ((unsigned long long)u << 32) | l;
    return t;
}

/*
 * takes two timebase values and calculates the
 * difference in nano seconds.
 * We don't take care of wrap arounds and overflows
 */
unsigned long long
ppc_timediff_ns(unsigned long long t1,
		unsigned long long t2)
{
    /* nano seconds per clock tick */
    unsigned long ppc_ns_per_tick = 1000000000 / dev.clock_info.cpu;
    unsigned long long tmp;
    if(t1 > t2) { tmp = t1; t1 = t2; t2 = tmp; }
    return (t2 - t1) * ppc_ns_per_tick;
}
# endif /* PPC_TIME_MEASUREMENT */

static void
fpga_write_byte(u_int8_t value)
{
    int mask;

    /* Program a byte,from LSb to MSb */
    for (mask = 1; mask <= 128; mask <<= 1) {
	pp_gpio_bit_set(PP_GPIO_DEV_IBM, PP_GPIO_FPGA_TDI, value & mask);
	    
	/* generate an pos clk edge */
	pp_gpio_bit_set(PP_GPIO_DEV_IBM, PP_GPIO_FPGA_TCK, 1);

	/* generate an neg clk edge */
	pp_gpio_bit_set(PP_GPIO_DEV_IBM, PP_GPIO_FPGA_TCK, 0);
    }
}
#endif /* __powerpc__ */

#if defined (PRODUCT_FLASHX4) || ( defined (PRODUCT_KIMTESTERMST) && defined (PP_BOARD_LARA))
static void
gpio_isr(int irq, void * dev_id, struct pt_regs * regs)
{
    if ( (jiffies - last_tm) < 3 ) {
	last_tm = jiffies;
	return;
    }

    last_tm = jiffies;
    
    /* wait some time to avoid getting a wrong gpio */
    udelay(10000);

#if defined (PRODUCT_FLASHX4)
    int sw1 = pp_gpio_bit_get(PP_GPIO_DEV_IBM, 5);
    int sw2 = pp_gpio_bit_get(PP_GPIO_DEV_IBM, 6);
    int sw3 = pp_gpio_bit_get(PP_GPIO_DEV_IBM, 7);
    int sw4 = pp_gpio_bit_get(PP_GPIO_DEV_IBM, 9);

    //D(D_ERROR, "%x %x %x %x\n", sw1, sw2, sw3, sw4);

    if ( !sw1 && sw2 && sw3 && sw4 ) {
	propchange_fire(PP_PROP_SWITCH_PRESSED, 1);
    } else if ( !sw2 ) {
	propchange_fire(PP_PROP_SWITCH_PRESSED, 2);
    } else if ( !sw3 ) {
	propchange_fire(PP_PROP_SWITCH_PRESSED, 4);
    } else if ( !sw4 ) {
	propchange_fire(PP_PROP_SWITCH_PRESSED, 8);
    }
#endif
#if defined (PRODUCT_KIMTESTERMST)
    {
	unsigned short prop_flags = 0;
	u_char sw1, sw2, sw3;

	/* get gpio states of buttons */
	pp_gpio_bit_get(PP_GPIO_KTST_DEV, PP_GPIO_SW_1, &sw1) << 0;
	pp_gpio_bit_get(PP_GPIO_KTST_DEV, PP_GPIO_SW_2, &sw2) << 1;
	pp_gpio_bit_get(PP_GPIO_KTST_DEV, PP_GPIO_SW_3, &sw3) << 2;

	prop_flags = !sw1 << 0 | !sw2 << 1 | !sw3 << 2;
	if (prop_flags) {
	    propchange_fire(PP_PROP_SWITCH_PRESSED, prop_flags);
	    D(D_NOTICE, "PP_PROP_SWITCH_PRESSED: %d\n", prop_flags);
	}
    }
#endif
}
#endif

