/**
 * log.h
 *
 * Logging macros for kernel modules
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 *
 * Needs following pre-defines:
 * - NAME: name of logging module, will precede the log messages
 * - LOG_COLORED: colored logging (if defined)
 * - LOG_DBG: do DBG logging (if defined), supress DBG if not defined
 */

#ifndef __LOG_H__
#define __LOG_H__

#ifdef NAME
#define PREFIX NAME ": "
#else
#define PREFIX
#endif

#if defined (LOG_COLORED) && !defined (PP_BUILD_TYPE_FINAL)
# define COLOR(b, c) "\x1b[" b ";3" c "m"
#else
# define COLOR(b, c)
#endif

#define BLACK   "0"
#define RED     "1"
#define GREEN   "2"
#define YELLOW  "3"
#define BLUE    "4"
#define MAGENTA "5"
#define CYAN    "6"
#define WHITE   "7"
#define DK      "0"
#define LT      "1"

#define CERR    COLOR(DK, RED)
#define CWARN   COLOR(DK, YELLOW)
#define CINFO   COLOR(DK, GREEN)
#define CDBG    COLOR(LT, BLACK) // dark gray
#if defined (LOG_COLORED) && !defined (PP_BUILD_TYPE_FINAL)
# define CNORM  "\x1b[0m"
#else
# define CNORM
#endif

#define ERR(format, args...)    printk(PREFIX CERR  "ERR  " format CNORM "\n", ##args)
#define WARN(format, args...)   printk(PREFIX CWARN "WARN " format CNORM "\n", ##args)
#define INFO(format, args...)   printk(PREFIX CINFO "INFO " format CNORM "\n", ##args)

#ifdef COND_DBG
# define DBG(format, args...)   if (debug) printk(PREFIX CDBG  "DBG  " format CNORM "\n", ##args)
#else
# if NDEBUG || !defined(LOG_DBG)
#  define DBG(format, args...)
# else
#  define DBG(format, args...)  printk(PREFIX CDBG  "DBG  " format CNORM "\n", ##args)
# endif
#endif

#include <linux/time.h>

#define TERR(format, args...)    { struct timeval tv; do_gettimeofday(&tv); printk("%06d.%03d " PREFIX CERR  "ERR  " format CNORM "\n", (int)tv.tv_sec, (int)tv.tv_usec / 1000, ##args); }
#define TWARN(format, args...)   { struct timeval tv; do_gettimeofday(&tv); printk("%06d.%03d " PREFIX CWARN "WARN " format CNORM "\n", (int)tv.tv_sec, (int)tv.tv_usec / 1000, ##args); }
#define TINFO(format, args...)   { struct timeval tv; do_gettimeofday(&tv); printk("%06d.%03d " PREFIX CINFO "INFO " format CNORM "\n", (int)tv.tv_sec, (int)tv.tv_usec / 1000, ##args); }
#if NDEBUG || !defined(LOG_DBG)
#define TDBG(format, args...)
#else
#define TDBG(format, args...)    { struct timeval tv; do_gettimeofday(&tv); printk("%06d.%03d " PREFIX CDBG  "DBG  " format CNORM "\n", (int)tv.tv_sec, (int)tv.tv_usec / 1000, ##args); }
#endif

#endif // __LOG_H__
