/**
 * lpc_bt.c
 *
 * KIRA100 LPC (slave) core driver for Block Transfer protocol
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/sched.h>

#include "lpc_core.h"
#include "lpc_bt.h"

#define SUCCESS 0

#define NAME                    "lpc_bt"
#define noLOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

#define bt_h2b_atn()            lpc_get_reg_bits(bt->lpc, LPC_BTCR, LPC_H2B_ATN)
#define bt_clr_h2b_atn()        lpc_set_reg_bits(bt->lpc, LPC_BTCR, 0, LPC_H2B_ATN)
#define bt_b2h_atn()            lpc_get_reg_bits(bt->lpc, LPC_BTCR, LPC_B2H_ATN)
#define bt_set_b2h_atn()        lpc_set_reg_bits(bt->lpc, LPC_BTCR, LPC_B2H_ATN, 0)
#define bt_h_busy()             lpc_get_reg_bits(bt->lpc, LPC_BTCR, LPC_H_BUSY)
#define bt_set_b_busy(v)        lpc_set_reg_bits(bt->lpc, LPC_BTCR, v ? LPC_B_BUSY : 0, LPC_B_BUSY)
#define bt_set_bevt_atn()       lpc_set_reg_bits(bt->lpc, LPC_BTCR, LPC_BEVT_ATN, 0)
#define bt_in_cnt()             lpc_get_reg(bt->lpc, LPC_BTFVSR0)
#define bt_out_cnt()            lpc_get_reg(bt->lpc, LPC_BTFVSR1)
#define bt_in()                 lpc_get_reg(bt->lpc, LPC_BTDTR)
#define bt_out(v)               lpc_set_reg(bt->lpc, LPC_BTDTR, v)

/*
 * Global data
 */

typedef struct {
    lpc_t *lpc;

    wait_queue_head_t *wait;
} bt_t;

static int used = 0;

/*
 * Public functions
 */

int bt_init(lpc_t *lpc, int chan, uint16_t host_io_base, wait_queue_head_t *wait, void **pbt_data)
{
    bt_t *bt;

    if (chan != 3) return -EINVAL; /* only supported by LPC3 */
    if (used) return -EBUSY;

    bt = kmalloc(sizeof(bt_t), GFP_KERNEL);
    memset(bt, 0, sizeof(bt_t));

    bt->lpc = lpc;
    bt->wait = wait;

    /* set LPC3 host base address */
    lpc_set_reg(bt->lpc, LPC_LADR3H, (host_io_base >> 8) & 0xff);
    lpc_set_reg(bt->lpc, LPC_LADR3L, (host_io_base & 0xfe) | 0x02);

    /* enable LPC3 and switch to BT */
    lpc_set_reg_bits(bt->lpc, LPC_HICR4, LPC_BT3ENBL, 0);
    lpc_set_reg_bits(bt->lpc, LPC_HICR0, LPC_LPC3E, 0);

    /* clean interrupt flags and data register */
    lpc_set_reg(bt->lpc, LPC_STR3, 0);
    lpc_set_reg(bt->lpc, LPC_BTCR, 0);
    lpc_set_reg_bits(bt->lpc, LPC_BTSR0, 0, LPC_BTI_MASK0);
    lpc_set_reg_bits(bt->lpc, LPC_BTSR1, 0, LPC_BTI_MASK1);

    *pbt_data = bt;
    used = 1;
    return SUCCESS;
}

int bt_cleanup(void *bt_data)
{
    bt_t *bt = bt_data;

    /* disable LPC3 and BT */
    lpc_set_reg_bits(bt->lpc, LPC_HICR0, 0, LPC_LPC3E);
    lpc_set_reg_bits(bt->lpc, LPC_HICR4, 0, LPC_BT3ENBL);

    kfree(bt);
    used = 0;
    return SUCCESS;
}

int bt_start(void *bt_data)
{
    bt_t *bt = bt_data;

    /* enable interrupts */
    lpc_set_reg(bt->lpc, LPC_BTCSR0, LPC_FRDI | LPC_HRDI | LPC_HWRI | LPC_HBTWI | LPC_HBTRI); /* TODO: we actually dont need them all */
    lpc_set_reg(bt->lpc, LPC_BTCSR1, LPC_B2HIE | LPC_H2BIE | LPC_CRWPIE | LPC_CRRPIE);
    lpc_set_reg_bits(bt->lpc, LPC_HICR2, LPC_IBFIE3, 0);
    return SUCCESS;
}

int bt_stop(void *bt_data)
{
    bt_t *bt = bt_data;

    /* disable irqs */
    lpc_set_reg_bits(bt->lpc, LPC_HICR2, 0, LPC_IBFIE3);
    lpc_set_reg_bits(bt->lpc, LPC_BTCSR0, 0, LPC_BTI_MASK0);
    lpc_set_reg_bits(bt->lpc, LPC_BTCSR1, 0, LPC_BTI_MASK1);
    return SUCCESS;
}

int bt_read(void *bt_data, uint8_t *buf, int size)
{
    bt_t *bt = bt_data;
    int i;

    DBG("read: wait for H2B_ATN...");
    wait_event_interruptible(*bt->wait, bt_h2b_atn());
    DBG("read: set B_BUSY");
    bt_set_b_busy(1);

    for (i = 0; bt_in_cnt() > 0; i++) {
        uint8_t d = bt_in();
        DBG("read: data 0x%02x received", d);
        if (i < size) buf[i] = d;
    }

    DBG("read: clear H2B_ATN");
    bt_clr_h2b_atn();

    DBG("read: clear B_BUSY");
    bt_set_b_busy(0);

    DBG("read: finished (%d bytes read)", i);
    return i < size ? i : size;
}

int bt_write(void *bt_data, const uint8_t *buf, int size)
{
    bt_t *bt = bt_data;
    int i;

    DBG("write: wait for !H_BUSY...");
    /* cannot do a wait_event_interruptible(bt->wait, !bt_h_busy())
       because the H_BUSY flag does not generate a slave intr when cleared by host */
    while (bt_h_busy()) schedule_timeout(msecs_to_jiffies(1)); /* 1ms */

    for (i = 0; i < size; i++) {
        DBG("write: transfer data 0x%02x", buf[i]);
        bt_out(buf[i]);
    }

    DBG("write: set B2H_ATN");
    bt_set_b2h_atn();

    DBG("write: wait for !B2H_ATN...");
    wait_event_interruptible(*bt->wait, !bt_b2h_atn());

    DBG("write finished (%d bytes written)", i);
    return i;
}

int bt_can_read(void *bt_data)
{
    bt_t *bt = bt_data;
    return bt_h2b_atn();
}

int bt_can_write(void *bt_data)
{
    bt_t *bt = bt_data;
    return !bt_h_busy();
}

int bt_host_atn(void *bt_data, int set)
{
    bt_t *bt = bt_data;

    if (set) bt_set_bevt_atn();
    // else: BT cannot clear SMS_ATN bit - ignorred
    return SUCCESS;
}

/*
 * ISR
 */
 
int bt_event(void *bt_data)
{
    bt_t *bt = bt_data;
    uint8_t btsr0 = lpc_get_reg(bt->lpc, LPC_BTSR0);
    uint8_t btsr1 = lpc_get_reg(bt->lpc, LPC_BTSR1);
//    uint8_t btcr = lpc_get_reg(bt->lpc, LPC_BTCR);

    /* ack all (known) irqs */
    lpc_set_reg(bt->lpc, LPC_BTSR0, ~btsr0);
    lpc_set_reg(bt->lpc, LPC_BTSR1, ~btsr1);

    /* host has written to full FIFO: warn and ignore */
    if (btsr0 & LPC_FRDI) WARN("FIFO overflow while host is writing - ignored");

    /* host has set CLR_xx_PTR flags: we must reset them manually (H8 LPC core differs from IPMI spec here!) */
    if (btsr1 & LPC_CRWPIE) lpc_set_reg_bits(bt->lpc, LPC_BTCR, 0, LPC_CLR_WR_PTR);
    if (btsr1 & LPC_CRRPIE) lpc_set_reg_bits(bt->lpc, LPC_BTCR, 0, LPC_CLR_RD_PTR);

    /* B2H_ATN or H2B_ATN changed: wake up any pending state machines */
    if (btsr1 & (LPC_B2HI | LPC_H2BI)) wake_up(bt->wait);

/* for debugging only
    uint8_t btfvsr0 = lpc_get_reg(bt->lpc, LPC_BTFVSR0);
    uint8_t btfvsr1 = lpc_get_reg(bt->lpc, LPC_BTFVSR1);
    DBG("irq BTSR0=%02x BTSR1=%02x BTCR=%02x H2Bfifo=%d B2Hfifo=%d", btsr0, btsr1, btcr, btfvsr0, btfvsr1);
*/

    return SUCCESS;
}
