/**
 * lpc_kcs.c
 *
 * KIRA100 LPC (slave) driver for Keyboard Controller Style protocol
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <linux/time.h>
#else
# include <linux/errno.h>
# include <linux/wait.h>
# include <linux/sched.h>
#endif

#include "lpc_core.h"
#include "lpc_kcs.h"

#define SUCCESS 0

#define USE_IRQ

#define NAME                    "lpc_kcs"
#define noLOG_COLORED
#define noLOG_DBG
#include "log.h"

/* comment regarding the timeout value:
   The original value of this was 200ms. When doing a firmware
   upgrade via KCS the BMC sometimes needs up to 6 seconds
   to answer the request. Perhaps this is related to the KIRA R01
   DMA hack. In all cases, the reason of the long BMC hangs
   needs to be investigated and the value has to be decreased
   after that!
                      daba, 06/07/14                            */ 
#define KCS_BMC_RESP_TIMEOUT 10000 // ms (max. allowed response time of BMC)

#define KCS_MAX_TRANSFER_LEN 1024

#define SPAN_MS(t0, t1) \
    (((t1).tv_sec - (t0).tv_sec) * 1000 + ((t1).tv_usec - (t0).tv_usec) / 1000);

#define KCS_CMD_GET_STATUS  ((uint8_t)0x60)
#define KCS_CMD_ABORT       ((uint8_t)0x60)
#define KCS_CMD_WRITE_START ((uint8_t)0x61)
#define KCS_CMD_WRITE_END   ((uint8_t)0x62)
#define KCS_CMD_READ        ((uint8_t)0x68)

#define KCS_STATE_IDLE  ((uint8_t)(0x0 << 6))
#define KCS_STATE_READ  ((uint8_t)(0x1 << 6))
#define KCS_STATE_WRITE ((uint8_t)(0x2 << 6))
#define KCS_STATE_ERROR ((uint8_t)(0x3 << 6))
#define KCS_STATE_MASK  ((uint8_t)(0x3 << 6))

#define KCS_SMS_ATN     ((uint8_t)(1 << 2))

#define KCS_ERR_NONE    ((uint8_t)0x00)
#define KCS_ERR_ABORT   ((uint8_t)0x01)
#define KCS_ERR_ILLEGAL ((uint8_t)0x02)
#define KCS_ERR_OVERRUN ((uint8_t)0x06)
#define KCS_ERR_FAIL    ((uint8_t)0xff)

// regs
static uint8_t LPC_STR[3]       = { LPC_STR1,       LPC_STR2,       LPC_STR3        };
static uint8_t LPC_IDR[3]       = { LPC_IDR1,       LPC_IDR2,       LPC_IDR3        };
static uint8_t LPC_ODR[3]       = { LPC_ODR1,       LPC_ODR2,       LPC_ODR3        };
static uint8_t LPC_LADRH[3]     = { LPC_LADR12H,    LPC_LADR12H,    LPC_LADR3H      };
static uint8_t LPC_LADRL[3]     = { LPC_LADR12L,    LPC_LADR12L,    LPC_LADR3L      };

// bits
static uint8_t LPC_LPCE[3]      = { LPC_LPC1E,      LPC_LPC2E,      LPC_LPC3E       };
static uint8_t LPC_KCSENBL[3]   = { LPC_KCS1ENBL,   LPC_KCS2ENBL,   LPC_KCS3ENBL    };
static uint8_t LPC_IBFIE[3]     = { LPC_IBFIE1,     LPC_IBFIE2,     LPC_IBFIE3      };
static uint8_t LPC_IBF[3]       = { LPC_IBF1,       LPC_IBF2,       LPC_IBF3A       };
static uint8_t LPC_OBF[3]       = { LPC_OBF1,       LPC_OBF2,       LPC_OBF3A       };
static uint8_t LPC_CD[3]        = { LPC_CD1,        LPC_CD2,        LPC_CD3         };

#define kcs_set_state(s)    lpc_set_reg_bits(kcs->lpc, LPC_STR[kcs->chan], s, KCS_STATE_MASK)
#define kcs_get_state()     lpc_get_reg_bits(kcs->lpc, LPC_STR[kcs->chan], KCS_STATE_MASK)
#define kcs_ibf()           lpc_get_reg_bits(kcs->lpc, LPC_STR[kcs->chan], LPC_IBF[kcs->chan])
#define kcs_obf()           lpc_get_reg_bits(kcs->lpc, LPC_STR[kcs->chan], LPC_OBF[kcs->chan])
#define kcs_clear_obf()     lpc_set_reg_bits(kcs->lpc, LPC_STR[kcs->chan], 0, LPC_OBF[kcs->chan])
#define kcs_got_cmd()       lpc_get_reg_bits(kcs->lpc, LPC_STR[kcs->chan], LPC_CD[kcs->chan])
#define kcs_got_data()      (!lpc_get_reg_bits(kcs->lpc, LPC_STR[kcs->chan], LPC_CD[kcs->chan]))
#define kcs_in()            lpc_get_reg(kcs->lpc, LPC_IDR[kcs->chan])
#define kcs_out(v)          lpc_set_reg(kcs->lpc, LPC_ODR[kcs->chan], v)
#define kcs_set_sms_atn(v)  lpc_set_reg_bits(kcs->lpc, LPC_STR[kcs->chan], v ? KCS_SMS_ATN : 0, KCS_SMS_ATN)

#if 0
#undef kcs_in
#undef kcs_out

#define kcs_in()            _kcs_in(kcs)
#define kcs_out(v)          _kcs_out(kcs, v)

static uint8_t _kcs_in(kcs_t *kcs)
{
    uint8_t d = lpc_get_reg(kcs->lpc, LPC_IDR[kcs->chan]);
    DBG("------------------------- get %s %02x", kcs_got_cmd() ? "cmd" : "data", d);
    return d;
}

static void _kcs_out(kcs_t *kcs, uint8_t d)
{
    DBG("------------------------- put data %02x", d);
    lpc_set_reg(kcs->lpc, LPC_ODR[kcs->chan], d);
}
#endif

/*
 * Private
 */

// current step - BMC's view
typedef enum {
    idle = 0,
    read, read_end,
    read_wait, write_wait,
    write, write_end,
    abort, abort_end
} kcs_step_t;

static const char *kcs_step_str[] = {
    "idle",
    "read", "read_end",
    "read_wait", "write_wait",
    "write", "write_end",
    "abort", "abort_end"
};

// kcs state - host's view
static const char *kcs_state_str[] = {
    "IDLE_STATE",
    "READ_STATE",
    "WRITE_STATE",
    "ERROR_STATE"
};

/*
 * Global data
 */

typedef struct {
    lpc_t       *lpc;
    int         chan;

    kcs_step_t  step;
    struct timeval wait_start_time; // used to track BMC response times
    spinlock_t  wait_time_lock;
    uint8_t     error;
    uint8_t     in_buf[KCS_MAX_TRANSFER_LEN];
    int         in_pos;
    int         in_size;
    uint8_t     out_buf[KCS_MAX_TRANSFER_LEN];
    int         out_pos;
    int         out_size;

    wait_queue_head_t *wait;
} kcs_t;

static int chan_used[3] = { 0, 0, 0 };

/*
 * Local helper functions 
 */
 
static inline struct timeval kcs_get_wait_time(kcs_t *kcs)
{
    unsigned long flags;
    struct timeval tv;
    spin_lock_irqsave(&kcs->wait_time_lock, flags);
    tv = kcs->wait_start_time;
    spin_unlock_irqrestore(&kcs->wait_time_lock, flags);
    return tv;
}

static inline void kcs_update_wait_time(kcs_t *kcs)
{
    unsigned long flags;
    spin_lock_irqsave(&kcs->wait_time_lock, flags);
    do_gettimeofday(&kcs->wait_start_time);
    spin_unlock_irqrestore(&kcs->wait_time_lock, flags);
    return;
}
 
/*
 * Public functions
 */

int kcs_init(lpc_t *lpc, int chan, uint16_t host_io_base, wait_queue_head_t *wait, void **pkcs_data)
{
    kcs_t *kcs;

    if (chan < 1 || chan > 3) return -EINVAL;
    if (chan_used[chan - 1]) return -EBUSY;

    kcs = kmalloc(sizeof(kcs_t), GFP_KERNEL);
    memset(kcs, 0, sizeof(kcs_t));

    kcs->lpc = lpc;
    kcs->chan = chan - 1;
    kcs->wait = wait;

    kcs->step = idle;
    kcs->out_size = kcs->out_pos = 0;
    kcs->in_size = kcs->in_pos = 0;

    /* select addr register for chan1/2 */
    if (kcs->chan == 0) lpc_set_reg_bits(kcs->lpc, LPC_HICR4, 0, LPC_LADR12SEL);
    else if (kcs->chan == 1) lpc_set_reg_bits(kcs->lpc, LPC_HICR4, LPC_LADR12SEL, 0);
    /* set host base address */
    lpc_set_reg(kcs->lpc, LPC_LADRH[kcs->chan], (host_io_base >> 8) & 0xff);
    lpc_set_reg(kcs->lpc, LPC_LADRL[kcs->chan], (host_io_base & 0xfe) | 0x02);

    /* switch chan3 str to random access */
    if (kcs->chan == 2) lpc_set_reg_bits(kcs->lpc, LPC_HISEL, LPC_SELSTR3, 0);
    /* enable LPC channel and switch to KCS */
    lpc_set_reg_bits(kcs->lpc, LPC_HICR4, LPC_KCSENBL[kcs->chan], 0);
    lpc_set_reg_bits(kcs->lpc, LPC_HICR0, LPC_LPCE[kcs->chan], 0);

    /* clean interrupt flags and data register */
    lpc_set_reg(kcs->lpc, LPC_STR[kcs->chan], 0);

    kcs->wait_time_lock = SPIN_LOCK_UNLOCKED;
    
    chan_used[kcs->chan] = 1;
    *pkcs_data = kcs;
    return SUCCESS;
}

int kcs_cleanup(void *kcs_data)
{
    kcs_t *kcs = kcs_data;

    /* disable LPC and KCS */
    lpc_set_reg_bits(kcs->lpc, LPC_HICR0, 0, LPC_LPCE[kcs->chan]);
    lpc_set_reg_bits(kcs->lpc, LPC_HICR4, 0, LPC_KCSENBL[kcs->chan]);

    kfree(kcs);
    chan_used[kcs->chan] = 0;
    return SUCCESS;
}

int kcs_start(void *kcs_data)
{
    kcs_t *kcs = kcs_data;

#ifdef USE_IRQ
    /* enable interrupts */
    lpc_set_reg_bits(kcs->lpc, LPC_HICR2, LPC_IBFIE[kcs->chan], 0);
#endif
    return SUCCESS;
}

int kcs_stop(void *kcs_data)
{
    kcs_t *kcs = kcs_data;

#ifdef USE_IRQ
    /* disable irqs */
    lpc_set_reg_bits(kcs->lpc, LPC_HICR2, 0, LPC_IBFIE[kcs->chan]);
#endif
    return SUCCESS;
}

int kcs_read(void *kcs_data, uint8_t *buf, int size)
{
    kcs_t *kcs = kcs_data;

    /* wait for incoming data */
#ifdef USE_IRQ
    if (wait_event_interruptible(*kcs->wait, kcs->step == read_wait) < 0)
        ERR("timeout while receiving message @ byte %d", kcs->in_pos);
#else
    while (kcs->step != read_wait) {
        kcs_event(kcs);
        if (current->need_resched) schedule();
    }
#endif

    {
        struct timeval tv;
        do_gettimeofday(&tv);
        int dt = SPAN_MS(kcs_get_wait_time(kcs), tv);
        if (dt > KCS_BMC_RESP_TIMEOUT) {
            WARN("BMC did not respond within %dms in step %s (read() occurs after %dms)",
                 KCS_BMC_RESP_TIMEOUT, kcs_step_str[kcs->step], dt);
        }
    }

    memcpy(buf, kcs->in_buf, min(kcs->in_size, size));
    kcs->step = write_wait;
    kcs_update_wait_time(kcs);

    /* clear sms attention flag */
    //kcs_set_sms_atn(0); - should be done by higher level software

    return min(kcs->in_size, size);
}

int kcs_write(void *kcs_data, const uint8_t *buf, int size)
{
    kcs_t *kcs = kcs_data;

    if (size <= 0 || size > sizeof(kcs->out_buf)) return 0;

    /* ready to send? */
    if (kcs->step != write_wait) return 0;

    {
        struct timeval tv;
        do_gettimeofday(&tv);
        int dt = SPAN_MS(kcs_get_wait_time(kcs), tv);
        if (dt > KCS_BMC_RESP_TIMEOUT) {
            WARN("BMC did not respond within %dms in step %s (write() occurs after %dms)",
                 KCS_BMC_RESP_TIMEOUT, kcs_step_str[kcs->step], dt);
        }
    }

    memcpy(kcs->out_buf, buf, size);
    kcs->out_pos = 0;
    kcs->out_size = size;
    kcs->step = write;
    DBG("transfering data 0x%02x (#%d)", kcs->out_buf[kcs->out_pos], kcs->out_pos);
    kcs_out(kcs->out_buf[kcs->out_pos++]);

    /* wait for data transmit completion */
#ifdef USE_IRQ
    if (wait_event_interruptible_timeout(*kcs->wait, kcs->step != write && kcs->step != write_end, 1*HZ) < 0)
        ERR("timeout while sending message @ byte %d", kcs->out_pos);
#else
    {
        unsigned long timeout = jiffies + 1*HZ;
        while (kcs->step != idle) {
            kcs_event(kcs);
            if (current->need_resched) schedule();
            if (time_after(jiffies, timeout)) {
                ERR("timeout while sending message @ byte %d", kcs->out_pos);
                break;
            }
        }
    }
#endif

    return kcs->out_pos;
}

int kcs_can_read(void *kcs_data)
{
    kcs_t *kcs = kcs_data;
#ifdef USE_IRQ
    return kcs->step == read_wait;
#else
    return 1;
#endif
}

int kcs_can_write(void *kcs_data)
{
    kcs_t *kcs = kcs_data;
    return kcs->step == write_wait;
}

int kcs_host_atn(void *kcs_data, int set)
{
    kcs_t *kcs = kcs_data;

    /* wait for data transmit completion */
    if (wait_event_interruptible_timeout(*kcs->wait, kcs->step == idle, 1 * HZ) < 0)
        ERR("timeout while waiting for idle (step %d)", kcs->step);
    if (kcs->step != idle) return -EFAULT;

    /* set sms attention flag */
    kcs_set_sms_atn(set ? 1 : 0);

    /* generate non-communication intr at host */
    kcs_out('I');

    return SUCCESS;
}

/*
 * ISR
 */

int kcs_event(void *kcs_data)
{
    kcs_t *kcs = kcs_data;

    /*
     * KCS step machine according to IPMI 2.0 spec chapter 9.15.
     * See that for details. Just devide the flow diagrams at that points where
     * host writes any data (and thereby generate the KCS event (IBF IRQ).
     *
     * - set new KCS state
     * - put acknowledge to trigger host irq
     * - write data (this let host engine continue immediately!)
     * - other stuff
     */

    if (kcs_ibf()) {
        uint8_t old_state = kcs_get_state() >> 6;

        if (kcs->step == read_wait || kcs->step == write_wait) {
            struct timeval tv;
            do_gettimeofday(&tv);
            int dt = SPAN_MS(kcs_get_wait_time(kcs), tv);
            if (dt > KCS_BMC_RESP_TIMEOUT) {
                WARN("BMC did not respond within %dms in step %s (client brakes the silence after %dms)",
                     KCS_BMC_RESP_TIMEOUT, kcs_step_str[kcs->step], dt);
            }
        }

        /* clear Output Buffer Full to prevent host from reading old stuff */
        kcs_clear_obf();

        if (kcs_got_cmd()) {
            uint8_t cmd;
            kcs_set_state(KCS_STATE_WRITE);
            kcs_out('C'); /* trigger host irq */

            cmd = kcs_in();
            switch (cmd) {
            case KCS_CMD_ABORT:
                DBG("got cmd ABORT in step %s (%s)", kcs_step_str[kcs->step], kcs_state_str[old_state]);
                kcs->error = KCS_ERR_NONE;
                if (kcs->step != idle) {
                    WARN("step %s canceled", kcs_step_str[kcs->step]);
                    kcs->error = KCS_ERR_ABORT;
                }
                kcs->step = abort;
                break;

            case KCS_CMD_WRITE_START:
                DBG("got cmd WRITE_START in step %s (%s)", kcs_step_str[kcs->step], kcs_state_str[old_state]);
                kcs->error = KCS_ERR_NONE;
                if (kcs->step != idle) {
                    WARN("step %s canceled", kcs_step_str[kcs->step]);
                }
                kcs->in_pos = kcs->in_size = 0;
                kcs->step = read; // host write == BMC read
                break;

            case KCS_CMD_WRITE_END:
                DBG("got cmd WRITE_END in step %s (%s)", kcs_step_str[kcs->step], kcs_state_str[old_state]);
                if (kcs->step != read) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("got unexpected cmd WRITE_END in step %s", kcs_step_str[kcs->step]);
                    kcs->error = KCS_ERR_ILLEGAL;
                    kcs->step = idle;
                    break;
                }
                kcs->step = read_end; // host write == BMC read
                break;

            default:
                kcs_set_state(KCS_STATE_ERROR);
                ERR("got unknown cmd 0x%02x in step %s (%s)", cmd, kcs_step_str[kcs->step], kcs_state_str[old_state]);
                kcs->error = KCS_ERR_ILLEGAL;
                kcs->step = idle;
            }
        } else {
            // is data:
            uint8_t data;

            switch (kcs->step) {
            case read:
                /* next data byte by host */

                /* state KCS_STATE_WRITE remaining */
                kcs_out('W'); /* trigger host irq */

                data = kcs_in();
                DBG("got data 0x%02x (#%d) in step %s (%s)", data, kcs->in_pos,
                    kcs_step_str[kcs->step], kcs_state_str[old_state]);

                /* in buf overflow? (let space left for last data byte) */
                if (kcs->in_pos >= sizeof(kcs->in_buf) - 2) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("input buffer overflow (%d bytes)", kcs->in_pos);
                    kcs->error = KCS_ERR_OVERRUN;
                    kcs->step = idle;
                    break;
                }

                kcs->in_buf[kcs->in_pos++] = data;
                break;

            case read_end:
                /* last data byte by host*/

                kcs_set_state(KCS_STATE_READ);
                /* dont trigger host irq here: will be done by first write byte */

                data = kcs_in();
                DBG("got data 0x%02x (#%d - last) in step %s (%s)", data, kcs->in_pos,
                    kcs_step_str[kcs->step], kcs_state_str[old_state]);
                kcs->in_buf[kcs->in_pos++] = data;
                kcs->in_size = kcs->in_pos;
                kcs->step = read_wait;
                kcs_update_wait_time(kcs);
                break;

            case write:
                /* client sends data to host */

                /* state KCS_STATE_READ remaining */
                kcs_out(kcs->out_buf[kcs->out_pos++]);

                data = kcs_in();
                DBG("got data 0x%02x (READ) in step %s (%s) - transfering data 0x%02x (#%d)",
                    data, kcs_step_str[kcs->step], kcs_state_str[old_state],
                    kcs->out_buf[kcs->out_pos - 1], kcs->out_pos - 1);
                if (data != KCS_CMD_READ) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("expected 0x68 (READ) got 0x%02x", data);
                    kcs->error = KCS_ERR_ILLEGAL;
                    kcs->step = idle;
                    break;
                }

                /* was the last byte? */
                if (kcs->out_pos >= kcs->out_size) kcs->step = write_end;
                break;

            case write_end:
                /* client has finished sending to host */

                kcs_set_state(KCS_STATE_IDLE);
                kcs_out('R');

                data = kcs_in();
                DBG("got data 0x%02x in step %s (%s) - data transfer completed",
                    data, kcs_step_str[kcs->step], kcs_state_str[old_state]);
                if (/*0 &&*/ data != KCS_CMD_READ) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("expected READ got 0x%02x", data);
                    kcs->error = KCS_ERR_ILLEGAL;
                    kcs->step = idle;
                    break;
                }
                kcs->step = idle;
                break;

            case abort:
                /* error handling initiated by host */

                kcs_set_state(KCS_STATE_READ);
                kcs_out(kcs->error);
                kcs->error = KCS_ERR_NONE;

                data = kcs_in();
                DBG("got data 0x%02x in step %s (%s) - transfering status 0x%02x",
                    data, kcs_step_str[kcs->step], kcs_state_str[old_state], 0);
                if (data != 0) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("expected 0 got 0x%02x", data);
                    kcs->error = KCS_ERR_ILLEGAL;
                    kcs->step = idle;
                    break;
                }
                kcs->step = abort_end;
                break;

            case abort_end:
                /* error handling last stage */

                kcs_set_state(KCS_STATE_IDLE);
                kcs_out('A');

                data = kcs_in();
                DBG("got data 0x%02x in step %s (%s) - status transfer completed",
                    data, kcs_step_str[kcs->step], kcs_state_str[old_state]);
                if (data != KCS_CMD_READ) {
                    kcs_set_state(KCS_STATE_ERROR);
                    ERR("expected READ got 0x%02x", data);
                    kcs->error = KCS_ERR_ILLEGAL;
                    kcs->step = idle;
                    break;
                }
                kcs->step = idle;
                break;

            default:
                kcs_set_state(KCS_STATE_ERROR);
                kcs_out('E');

                data = kcs_in();
                ERR("got unexpected data 0x%02x in step %s (%s)",
                    data, kcs_step_str[kcs->step], kcs_state_str[old_state]);
                kcs->step = idle;
            }
        }

        /* wake pending write/read() calls */
        wake_up(kcs->wait);

        return SUCCESS;
    }
    return -1;
}
