/**
 * lpc_smic.c
 *
 * KIRA100 LPC (slave) core driver for System Management Interface Chip protocol
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/sched.h>

#include "lpc_core.h"
#include "lpc_smic.h"

#define SUCCESS 0

#define NAME                    "lpc_smic"
#define noLOG_COLORED
#define noLOG_BLACK_BG
#define noLOG_DBG
#include "log.h"

#define SMIC_CC_SMS_GET_STATUS  ((uint8_t)0x40)
#define SMIC_CC_SMS_WR_START    ((uint8_t)0x41)
#define SMIC_CC_SMS_WR_NEXT     ((uint8_t)0x42)
#define SMIC_CC_SMS_WR_END      ((uint8_t)0x43)
#define SMIC_CC_SMS_RD_START    ((uint8_t)0x44)
#define SMIC_CC_SMS_RD_NEXT     ((uint8_t)0x45)
#define SMIC_CC_SMS_RD_END      ((uint8_t)0x46)

#define SMIC_SC_SMS_RDY         ((uint8_t)0xc0)
#define SMIC_SC_SMS_WR_START    ((uint8_t)0xc1)
#define SMIC_SC_SMS_WR_NEXT     ((uint8_t)0xc2)
#define SMIC_SC_SMS_WR_END      ((uint8_t)0xc3)
#define SMIC_SC_SMS_RD_START    ((uint8_t)0xc4)
#define SMIC_SC_SMS_RD_NEXT     ((uint8_t)0xc5)
#define SMIC_SC_SMS_RD_END      ((uint8_t)0xc6)

#define SMIC_ERR_NONE           ((uint8_t)0x00)
#define SMIC_ERR_ABORTED        ((uint8_t)0x01)
#define SMIC_ERR_UNEXPECTED     ((uint8_t)0x02)
#define SMIC_ERR_NORESP         ((uint8_t)0x03)
#define SMIC_ERR_ILLEGAL        ((uint8_t)0x04)
#define SMIC_ERR_OVERFLOW       ((uint8_t)0x05)


#define smic_set_smi(v)         lpc_set_reg_bits(smic->lpc, LPC_SMICFLG, v ? LPC_SMI : 0, LPC_SMI)
#define smic_busy()             lpc_set_reg_bits(smic->lpc, LPC_SMICFLG, LPC_BUSY)
#define smic_clr_busy()         lpc_set_reg_bits(smic->lpc, LPC_SMICFLG, 0, LPC_BUSY)
#define smic_set_tx_rdy(v)      lpc_set_reg_bits(smic->lpc, LPC_SMICFLG, v ? LPC_TX_DATA_RDY : 0, LPC_TX_DATA_RDY)
#define smic_set_rx_rdy(v)      lpc_set_reg_bits(smic->lpc, LPC_SMICFLG, v ? LPC_RX_DATA_RDY : 0, LPC_RX_DATA_RDY)
#define smic_get_ctrl()         lpc_get_reg(smic->lpc, LPC_SMICCSR)
#define smic_get_data()         lpc_get_reg(smic->lpc, LPC_SMICDTR)
#define smic_set_state(v)       lpc_set_reg(smic->lpc, LPC_SMICCSR, v)
#define smic_set_data(v)        lpc_set_reg(smic->lpc, LPC_SMICDTR, v)
#define smic_ack_busyi()        lpc_set_reg_bits(smic->lpc, LPC_SMICIR0, 0, LPC_BUSYI)

#define smic_resp(s)            do { DBG("send state %02x", s); smic_set_state(s); smic_clr_busy(); } while (0)
#define smic_resp_code(s, c)    do { smic_set_data(c); smic_resp(s); } while (0)
#define smic_resp_err(e)        smic_resp_code(SMIC_SC_SMS_RDY, e)

/*
 * Global data
 */

typedef enum { idle = 0, start, next, end, wait } smic_state_t;

/* debug helper strings */
#if defined (LOG_DBG)
static const char *smic_state_str[] = { "idle", "start", "next", "end", "wait" };

static const char* smic_cs_str(uint8_t cs)
{
    switch (cs) {
    case SMIC_CC_SMS_GET_STATUS:    return "CC_SMS_GET_STATUS";
    case SMIC_CC_SMS_WR_START:      return "CC_SMS_WR_START";
    case SMIC_CC_SMS_WR_NEXT:       return "CC_SMS_WR_NEXT";
    case SMIC_CC_SMS_WR_END:        return "CC_SMS_WR_END";
    case SMIC_CC_SMS_RD_START:      return "CC_SMS_RD_START";
    case SMIC_CC_SMS_RD_NEXT:       return "CC_SMS_RD_NEXT";
    case SMIC_CC_SMS_RD_END:        return "CC_SMS_RD_END";

    case SMIC_SC_SMS_RDY:           return "SC_SMS_RDY";
    case SMIC_SC_SMS_WR_START:      return "SC_SMS_WR_START";
    case SMIC_SC_SMS_WR_NEXT:       return "SC_SMS_WR_NEXT";
    case SMIC_SC_SMS_WR_END:        return "SC_SMS_WR_END";
    case SMIC_SC_SMS_RD_START:      return "SC_SMS_RD_START";
    case SMIC_SC_SMS_RD_NEXT:       return "SC_SMS_RD_NEXT";
    case SMIC_SC_SMS_RD_END:        return "SC_SMS_RD_END";
    default:                        return "<unknown>";
    }
};
#endif

typedef struct {
    lpc_t *lpc;
    wait_queue_head_t *wait;

    smic_state_t b2h_state;
    smic_state_t h2b_state;
} smic_t;

static int used = 0;

/*
 * Public functions
 */

int smic_init(lpc_t *lpc, int chan, uint16_t host_io_base, wait_queue_head_t *wait, void **psmic_data)
{
    smic_t *smic;

    if (chan != 3) return -EINVAL; /* only supported by LPC3 */
    if (used) return -EBUSY;

    smic = kmalloc(sizeof(smic_t), GFP_KERNEL);
    memset(smic, 0, sizeof(smic_t));

    smic->lpc = lpc;
    smic->wait = wait;

    /* set LPC3 host base address */
    lpc_set_reg(smic->lpc, LPC_LADR3H, (host_io_base >> 8) & 0xff);
    lpc_set_reg(smic->lpc, LPC_LADR3L, (host_io_base & 0xfe) | 0x02);

    /* enable LPC3 and switch to SMIC */
    lpc_set_reg_bits(smic->lpc, LPC_HICR4, LPC_SMIC3ENBL, 0);
    lpc_set_reg_bits(smic->lpc, LPC_HICR0, LPC_LPC3E, 0);

    /* clean interrupt flags and data register */
    lpc_set_reg(smic->lpc, LPC_STR3, 0);
    lpc_set_reg(smic->lpc, LPC_SMICFLG, 0);
    lpc_set_reg(smic->lpc, LPC_SMICCSR, 0);
    lpc_set_reg(smic->lpc, LPC_SMICDTR, 0);
    lpc_set_reg_bits(smic->lpc, LPC_SMICIR0, 0, LPC_SMICI_MASK);

    *psmic_data = smic;
    used = 1;
    return SUCCESS;
}

int smic_cleanup(void *smic_data)
{
    smic_t *smic = smic_data;

    /* disable LPC3 and SMIC */
    lpc_set_reg_bits(smic->lpc, LPC_HICR0, 0, LPC_LPC3E);
    lpc_set_reg_bits(smic->lpc, LPC_HICR4, 0, LPC_SMIC3ENBL);

    kfree(smic);
    used = 0;
    return SUCCESS;
}

int smic_start(void *smic_data)
{
    smic_t *smic = smic_data;

    /* enable interrupts */
    lpc_set_reg_bits(smic->lpc, LPC_SMICIR1, LPC_BUSYIE, LPC_SMICI_MASK);
    lpc_set_reg_bits(smic->lpc, LPC_HICR2, LPC_IBFIE3, 0);
    return SUCCESS;
}

int smic_stop(void *smic_data)
{
    smic_t *smic = smic_data;

    /* disable irqs */
    lpc_set_reg_bits(smic->lpc, LPC_HICR2, 0, LPC_IBFIE3);
    lpc_set_reg_bits(smic->lpc, LPC_SMICIR1, 0, LPC_SMICI_MASK);
    return SUCCESS;
}

int smic_read(void *smic_data, uint8_t *buf, int size)
{
    smic_t *smic = smic_data;
    int p = 0;
    if (size <= 0) return 0;

    wait_event_interruptible(*smic->wait, smic->h2b_state == start);
    buf[p] = smic_get_data();
    DBG("recv: %02x (first)", buf[p]);
    smic->h2b_state = wait;
    smic_set_tx_rdy(1);
    smic_resp(SMIC_SC_SMS_WR_START);

    while (1) {
        p++;
        wait_event_interruptible(*smic->wait, smic->h2b_state != wait);

        switch (smic->h2b_state) {
        case next:
            if (p == size) {
                ERR("read buffer overflow");
                smic->h2b_state = idle;
                smic_resp_err(SMIC_ERR_OVERFLOW);
                return p;
            }
            buf[p] = smic_get_data();
            DBG("recv: %02x", buf[p]);
            smic->h2b_state = wait;
            smic_set_tx_rdy(1);
            smic_resp(SMIC_SC_SMS_WR_NEXT);
            break;

        case end:
            if (p == size) {
                ERR("read buffer overflow");
                smic->h2b_state = idle;
                smic_resp_code(SMIC_SC_SMS_WR_NEXT, SMIC_ERR_OVERFLOW);
                return p;
            }
            buf[p] = smic_get_data();
            DBG("recv: %02x (last)", buf[p]);
            smic->h2b_state = idle;
            smic_resp_code(SMIC_SC_SMS_WR_END, SMIC_ERR_NONE);
            return p + 1;

        default:
            ERR("read canceled");
            smic->h2b_state = idle;
            smic_resp_err(SMIC_ERR_UNEXPECTED);
            return p;
        }
    }
}

int smic_write(void *smic_data, const uint8_t *buf, int size)
{
    smic_t *smic = smic_data;
    int p = 0;
    if (size <= 0) return 0;

    smic_set_rx_rdy(1);

    wait_event_interruptible(*smic->wait, smic->b2h_state == start);
    smic->b2h_state = wait;
    smic_set_rx_rdy(1);
    smic_resp_code(p + 1 >= size ? SMIC_SC_SMS_RD_END: SMIC_SC_SMS_RD_START, buf[p]);
    DBG("xmit: %02x%s", buf[p], p + 1 >= size ? " (first and last)" : " (first)");

    while (1) {
        p++;
        wait_event_interruptible(*smic->wait, smic->b2h_state != wait);

        switch (smic->b2h_state) {
        case next:
            smic->b2h_state = wait;
            smic_set_rx_rdy(1);
            smic_resp_code(p + 1 >= size ? SMIC_SC_SMS_RD_END : SMIC_SC_SMS_RD_NEXT, buf[p]);
            DBG("xmit: %02x%s", buf[p], p + 1 >= size ? " (last)" : "");
            break;

        case end:
            smic->b2h_state = idle;
            smic_resp_err(SMIC_ERR_NONE);
            DBG("xmit: completed");
            return p;

        default:
            ERR("write canceled");
            smic->b2h_state = idle;
            smic_resp_err(SMIC_ERR_UNEXPECTED);
            return p;
        }
    }
}

int smic_can_read(void *smic_data)
{
    smic_t *smic = smic_data;
    return smic->h2b_state == start;
}

int smic_can_write(void *smic_data)
{
    smic_t *smic = smic_data;
    return smic->b2h_state == start;
}

int smic_host_atn(void *smic_data, int set)
{
    smic_t *smic = smic_data;
    smic_set_smi(set ? 1 : 0);
    return SUCCESS;
}

/*
 * ISR
 */

int smic_event(void *smic_data)
{
    smic_t *smic = smic_data;
    uint8_t smicir0 = lpc_get_reg(smic->lpc, LPC_SMICIR0);

    if (smicir0 & LPC_CTLWI) DBG("host has written ctrl");
    if (smicir0 & LPC_HDTWI) DBG("host has written data");
    if (smicir0 & LPC_STARI) DBG("host has read status");
    if (smicir0 & LPC_HDTRI) DBG("host has read data");
    if (smicir0 & LPC_BUSYI) DBG("host has set busy");

    /* host has written control code */
    if (smicir0 & LPC_BUSYI) {
        uint8_t ctrl;

        smic_ack_busyi(); /* ack irq */
        smic_set_tx_rdy(0);
        ctrl = smic_get_ctrl();
        DBG("got ctrl %s (%d) (h2b: %s, b2h: %s)",
            smic_cs_str(ctrl), ctrl, smic_state_str[smic->h2b_state], smic_state_str[smic->b2h_state]);

        switch (ctrl) {
        case SMIC_CC_SMS_GET_STATUS:
            if (smic->h2b_state == idle && smic->b2h_state == idle) {
                smic_resp_err(SMIC_ERR_NONE);
            } else {
                ERR("unexpected CC_SMS_GET_STATUS while transfer in progess");
                smic->h2b_state = idle;
                smic->b2h_state = idle;
                smic_resp_err(SMIC_ERR_UNEXPECTED);
            }
            break;

        case SMIC_CC_SMS_WR_START:
            smic->h2b_state = start;
            break;

        case SMIC_CC_SMS_WR_NEXT:
            if (smic->h2b_state == wait) {
                smic->h2b_state = next;
            } else {
                ERR("unexpected CC_SMS_WR_NEXT while no transfer in progess");
                smic->h2b_state = idle;
                smic_resp_err(SMIC_ERR_UNEXPECTED);
            }
            break;

        case SMIC_CC_SMS_WR_END:
            if (smic->h2b_state == wait) {
                smic->h2b_state = end;
            } else {
                ERR("unexpected CC_SMS_WR_END while no transfer in progess");
                smic->h2b_state = idle;
                smic_resp_err(SMIC_ERR_UNEXPECTED);
            }
            break;

        case SMIC_CC_SMS_RD_START:
            smic->b2h_state = start;
            break;

        case SMIC_CC_SMS_RD_NEXT:
            if (smic->b2h_state == wait) {
                smic->b2h_state = next;
            } else {
                ERR("unexpected CC_SMS_RD_NEXT while no transfer in progess");
                smic->b2h_state = idle;
                smic_resp_err(SMIC_ERR_UNEXPECTED);
            }
            break;

        case SMIC_CC_SMS_RD_END:
            if (smic->b2h_state == wait) {
                smic->b2h_state = end;
            } else {
                ERR("unexpected CC_SMS_WR_END while no transfer in progess");
                smic->b2h_state = idle;
                smic_resp_err(SMIC_ERR_UNEXPECTED);
            }
            break;

        default:
            ERR("illegal control code %02x", ctrl);
            smic->h2b_state = idle;
            smic->b2h_state = idle;
            smic_resp_err(SMIC_ERR_ILLEGAL);
        }

        wake_up(smic->wait);
        return SUCCESS;
    }
    return -1;
}
