#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/poll.h>

/* linux ether net driver module includes */
#include <linux/delay.h>
#include <linux/pci.h>

#include <asm/io.h>
#include <asm/irq.h>

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/arch/cpe/peppercon.h>
# include <asm/arch/cpe_int.h>
#define cond_resched() if (current->need_resched) schedule()
#else
# include <asm/arch/platform/peppercon.h>
# define MOD_INC_USE_COUNT
# define MOD_DEC_USE_COUNT
#endif
#endif

#include <linux/ethtool.h>
#include <linux/socket.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>

#ifdef NVIDIA_IO55
# define NAME "io55"
# define I2C_NAME "IO55SB_I2C"
#else
# define NAME "mcp55"
# define I2C_NAME "MCP55SB_I2C"
#endif
#define LOG_COLORED
#define noLOG_DBG
#define noUNUSED_FUNCTIONS

#include "log.h"
#include "mcp55_sb_lan.h"

#if 0
#include <pp_gpio.h>
#define trig_init() { if (pp_gpio_init(PP_GPIO_DEV_FARADAY) < 0) ERR("GPIO dev not open"); }
#define trig_fire() { pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 1 << 22, 0); udelay(1); pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 0, 1 << 22); }
#define trig2_fire() {pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 1 << 22, 0); udelay(1); pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 0, 1 << 22); udelay(1); pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 1 << 22, 0); udelay(1); pp_gpio_set(PP_GPIO_DEV_FARADAY, 0, 0, 1 << 22); }
#else
#define trig_init()
#define trig_fire()
#define trig2_fire()
#endif

/* ------------------------------------------------------------------------- *
 * Constants
 * ------------------------------------------------------------------------- */

/* Which core is mapped to the sideband i2c interface */
#ifdef PP_BOARD_KIRA
# define I2C_ADAPTER_NAME		"FIA320 I2C 2"
#else
# define I2C_SB_LAN_I2C_NAME		"Asics I2C 2"
#endif

#define DEF_IRQ				23
#ifdef NVIDIA_IO55
# define I2C_SLAVE_ADDR			0x71
#else
# define I2C_SLAVE_ADDR			0x70
#endif

#define MAX_FRAG_SIZE			224
#define I2C_SB_LAN_MAC			0
#define I2C_MAX_DATA_SIZE		(MAX_FRAG_SIZE + 2)
#define MAX_RETRY_COUNT			32

#define noUSE_PEC			/* unused for now */
#define noDUMP_DATA

#define SUCCESS	 0
#define FAIL    -1

// detailed debug logging
#define DBG2 DBG
//#define DBG2(...)

#define DELAY() udelay(100)

/* IO55 driver registers at MCP55 master driver (just one slave can register yet)
 * - only master registers ISR
 * - master does ARA and notifies slave
 * - FIXME: MCP55 and IO55 should be handled in one driver!
 */
#ifdef NVIDIA_IO55
extern int mcp55_register_slave(int saddr, int (*cb)(void*), void* ctx);
extern int mcp55_unregister_slave(int saddr);
#else
static int slave_saddr = 0;
static int (*slave_cb)(void*);
static void *slave_ctx;
int mcp55_register_slave(int saddr, int (*cb)(void*), void* ctx)
{
    if (slave_saddr != 0) return -1;
    slave_saddr = saddr;
    slave_cb = cb;
    slave_ctx = ctx;
    return 0;
}
int mcp55_unregister_slave(int saddr)
{
    if (slave_saddr != saddr) return -1;
    slave_saddr = 0;
    return 0;
}
EXPORT_SYMBOL(mcp55_register_slave);
EXPORT_SYMBOL(mcp55_unregister_slave);
#endif

/* ------------------------------------------------------------------------- *
 * realtime job types
 * ------------------------------------------------------------------------- */

typedef int (*job_func_t)(void*);

typedef struct job_s {
    struct list_head    anchor;
    job_func_t          func;
    void                *ctx;
    int                 ret;
    struct semaphore    *done;
} job_t;

/* ------------------------------------------------------------------------- *
 * ethernet driver structure
 * ------------------------------------------------------------------------- */

/*
 * receive packet layout
 */

typedef struct mcp55_recv_pkt_s {
    uint8_t status;
    uint8_t count;
    uint8_t data[MAX_FRAG_SIZE]; 
} __attribute__ ((packed)) mcp55_recv_pkt_t;

typedef struct mcp55_priv_s {
    /* general */
    atomic_t refcnt;
    spinlock_t lock;
    int is_init;
    int job_thd_running;
    
    /* network device data */
    struct net_device *dev;
    struct net_device_stats stats;
    int mac_set;
    u_char mac_count;
    u8 ip_address[4];
           
    /* job queue */
    struct list_head        job_queue;
    spinlock_t              job_queue_lock;
    wait_queue_head_t       job_queue_wait;

    /* async jobs */
    job_t                   recv_job;
    job_t                   send_job;
    job_t                   reinit_job;

    /* skb storage */
    struct sk_buff          *recv_skb;
    struct sk_buff          *send_skb;
    int                     send_offs;   
    
    /* i2c/IRQ specific stuff */
    struct i2c_client*  i2c_client;
    int			i2c_slave_addr;
    int			max_data_size;
} mcp55_priv_t;

/* ------------------------------------------------------------------------- *
 * Function prototypes
 * ------------------------------------------------------------------------- */

static int  mcp55_mod_init(void);
static void mcp55_mod_cleanup(void);
static int  mcp55_init_dev(struct net_device *dev);
static void mcp55_cleanup_dev(struct net_device *dev);

static int  mcp55_open(struct net_device *dev);
static int  mcp55_release(struct net_device *dev);
static int  mcp55_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);
static int  mcp55_set_mac_addr(struct net_device *dev, void *addr);
static struct net_device_stats* mcp55_stats(struct net_device *dev);
static int  mcp55_tx(struct sk_buff *skb, struct net_device *dev);

static int  mcp55_check_status(struct net_device *dev, u_char *queue_rx);
#ifndef NVIDIA_IO55
static irqreturn_t mcp55_isr(int irq, void *dev_id, struct pt_regs *regs);
#endif

/* ------------------------------------------------------------------------- *
 * realtime job functions
 * ------------------------------------------------------------------------- */
static void add_job(mcp55_priv_t *priv, job_t *job);
#ifdef UNUSED_FUNCTIONS
static void add_job_head(mcp55_priv_t *priv, job_t *job);
#endif
static int exec_job(mcp55_priv_t *priv, job_func_t func, void *ctx);
static int jobs_avail(mcp55_priv_t *priv);
static int get_next_job(mcp55_priv_t *priv, job_t **job);
static int do_job(job_t *job);

static int send(void* ctx);
static int recv(void* ctx);
static int init(void* ctx);
static int done(void* ctx);
static int set_mac(void *ctx);
#define term ((job_func_t)NULL) // special func for job thread termination

static int job_thread(void* arg);

// init args
#define FIRST_INIT  ((void*)0)
#define RE_INIT     ((void*)1)

/* ------------------------------------------------------------------------- *
 * high level PMU access functions
 * ------------------------------------------------------------------------- */

static int pmu_cfg_read(u_char reg);
static int pmu_cfg_write(u_char reg, u_char data);
static int pmu_mac_write(uint8_t mac[6]);
static int pmu_ipv4_write(uint8_t ip[4]);

#ifdef UNUSED_FUNCTIONS
static int pmu_mac_read(uint8_t *mac);
static int pmu_ipv4_read(uint8_t *ip);
#endif

/* ------------------------------------------------------------------------- *
 * I2C wrapper
 * ------------------------------------------------------------------------- */

static int i2c_access_write(char* buffer, size_t size);
static int i2c_access_proc(char* buffer, size_t out_size, size_t in_size);

static int i2c_write_block(mcp55_priv_t *priv, u8 command,
			   u8 byte_count, const u8 * data);
static int i2c_read_fragment(mcp55_priv_t *priv, u8 command,
			     u8 *count, u8 *data);

#ifdef UNUSED_FUNCTIONS
static int i2c_access_read(char* buffer, size_t size);
#endif

/* ------------------------------------------------------------------------- *
 * I2C low level routines
 * ------------------------------------------------------------------------- */

static int mcp55_i2c_attach_adapter(struct i2c_adapter *adapter);
static int mcp55_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
				     unsigned short flags,
#endif
				     int kind);
static int mcp55_i2c_detach_client(struct i2c_client *client);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void mcp55_i2c_inc_use (struct i2c_client *client);
static void mcp55_i2c_dec_use (struct i2c_client *client);
#endif
static int mcp55_i2c_command(struct i2c_client *client,
			     unsigned int cmd, void *arg);

/* ------------------------------------------------------------------------- *
 * PEC (packet error checking) stuff
 * ------------------------------------------------------------------------- */

#if USE_PEC
const static u8 crc8table[];

#define crc8(x,crc) crc8table[(u8)(crc ^ x)]
#define calc_crc8(data,len) update_crc8(data,len,0)

static u8 update_crc8 (u8 *data, size_t len, u8 crc);

#define  PEC_BUF(sp) \
{ \
    if (SB_LAN_VERIFY_RW(sp, WRITE)) {                  \
        int size = sp->xfer_size;                       \
        u8 *buf = sp->xfer_buffer;                      \
        buf[size] = update_crc8 (buf,                   \
                           size,                        \
                           crc8table[I2C_SLAVE_ADDR << 1]);  \
    }                                                   \
    sp->xfer_size++;                                    \
}

#else /* USE_PEC */

#define PEC_BUF(sp)

#endif /* USE_PEC */

/* i2c low level functions */
#define MASTER_RETRIES  5

#define DOWN(sem) if (down_interruptible(sem)) return -ERESTARTSYS;
#define UP(sem) up(sem);

/* ------------------------------------------------------------------------- *
 * I2C driver structure
 * ------------------------------------------------------------------------- */

static struct i2c_driver mcp55_i2c_driver = {
    name:		I2C_NAME,
    id:			0xFFD1,	/* f000-ffff for local use */
    flags:		I2C_DF_NOTIFY,
    attach_adapter:	mcp55_i2c_attach_adapter,
    detach_client:	mcp55_i2c_detach_client,
    command:		mcp55_i2c_command,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    inc_use:		mcp55_i2c_inc_use,
    dec_use:		mcp55_i2c_dec_use,
#endif
};

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Qiming Zhu <<QZhu@nvidia.com>, Michael Baumann <miba@peppercon.de>");
MODULE_DESCRIPTION("Linux network driver for NVidia MCP55 "
                   "platform management controller sideband interface");

module_init(mcp55_mod_init);
module_exit(mcp55_mod_cleanup);

static struct net_device net_dev;
static mcp55_priv_t mcp55_priv;

/* ------------------------------------------------------------------------- *
 * Initialization, Network driver functions
 * ------------------------------------------------------------------------- */

static int
mcp55_mod_init(void)
{
    int r;
    
    /* init net_device struct */
    memset(&net_dev, 0, sizeof(net_dev));
    strcpy(net_dev.name, NAME);
    net_dev.init = mcp55_init_dev;

    if ((r = register_netdev(&net_dev)) < 0) {
        ERR("register_netdev() failed (err %d)", r);
        mcp55_cleanup_dev(&net_dev);
        return -EIO;
    }
    return SUCCESS;
}

static void 
mcp55_mod_cleanup(void)
{
    mcp55_cleanup_dev(&net_dev);
    
    unregister_netdev(&net_dev);
}

static int
mcp55_init_dev(struct net_device *dev)
{
    mcp55_priv_t *priv;
    int ret = SUCCESS, r;

    /* setup ethernet defaults */
    ether_setup(dev);

    /* fill other standard network_dev fields */
    dev->open               = mcp55_open;
    dev->stop               = mcp55_release;
    dev->hard_start_xmit    = mcp55_tx;
    dev->get_stats          = mcp55_stats;
    dev->do_ioctl           = mcp55_ioctl;
    dev->set_mac_address    = mcp55_set_mac_addr;
    dev->base_addr          = 0; /* will be set below */
    dev->irq                = 0; /* will be set below */
    SET_MODULE_OWNER(dev);

    /* init privat data */
    dev->priv = &mcp55_priv;
    priv = (mcp55_priv_t*)dev->priv;
    memset(priv, 0, sizeof(mcp55_priv_t));

    priv->dev = dev;
    spin_lock_init (&priv->lock);
    memcpy (priv->ip_address, "\xc0\xa8\x1\x17", sizeof (priv->ip_address));

    /* find hardware, register i2c driver */
    if ((r = i2c_add_driver(&mcp55_i2c_driver))) {
    	WARN("%s: i2c driver registration failed (%d).", dev->name, r);
    	ret = -ENODEV;
    	goto bail;
    }

    /* check if the driver is installed */
    if (priv->i2c_client == NULL) {
        WARN("%s: device not present, skipping", dev->name);
    
        // use ENXIO instead of ENODEV to distinguish between 'bad' and regular initialization errors
    	ret = -ENXIO;
    	goto bail;
    }   

    /* init job related stuff */
    INIT_LIST_HEAD(&priv->job_queue);
    spin_lock_init(&priv->job_queue_lock);
    init_waitqueue_head(&priv->job_queue_wait);
   
    /* init async jobs (all other field remain 0) */
    INIT_LIST_HEAD(&priv->recv_job.anchor);
    priv->recv_job.func = recv;
    priv->recv_job.ctx = NULL;
    INIT_LIST_HEAD(&priv->send_job.anchor);
    priv->send_job.func = send;
    priv->send_job.ctx = NULL;
    INIT_LIST_HEAD(&priv->reinit_job.anchor);
    priv->reinit_job.func = init;
    priv->reinit_job.ctx = RE_INIT;

    priv->max_data_size = I2C_MAX_DATA_SIZE;

    /* start realtime job thread */
    r = kernel_thread(job_thread, dev, 0);
    if (r < 0) {
        ERR("unable to start real-time job thread (err %d)", r);
        goto bail;
    }
        
 bail:
    return ret;
}

static void
mcp55_cleanup_dev(struct net_device *dev)
{
    mcp55_priv_t *priv = dev->priv;
    if (priv && priv->job_thd_running) exec_job(priv, term, NULL);

    if (i2c_del_driver(&mcp55_i2c_driver)) {
	WARN("%s: i2c driver deregistration failed.", dev->name);
    }
}

/* ------------------------------------------------------------------------- *
 * Device functions
 * ------------------------------------------------------------------------- */

static struct net_device_stats *
mcp55_stats(struct net_device *dev)
{
    mcp55_priv_t *sp = dev->priv;
    return &sp->stats;
}

static int
mcp55_open(struct net_device *dev)
{
    mcp55_priv_t *priv = dev->priv;
    int ret = SUCCESS;
    unsigned long flags;
    
    MOD_INC_USE_COUNT;
    DBG ("mcp55_open()");
    
    spin_lock_irqsave(&priv->lock, flags);
   
    if (atomic_inc_return(&priv->refcnt) == 1) {
	/* first time init */
        priv->recv_skb = dev_alloc_skb(dev->mtu + ETH_HLEN);
	priv->send_skb = NULL;
	
        /* sync exec init job */
        ret = exec_job(priv, init, FIRST_INIT);
        if (ret == -EAGAIN) {
            WARN("init not yet succeeded, retrying in background");
            ret = SUCCESS;
        } else if (ret != SUCCESS) {
            ERR("init failed");
            goto bail;
        }	

#ifdef NVIDIA_IO55
        if (mcp55_register_slave(I2C_SLAVE_ADDR, recv, NULL) < 0) {
            ERR("can't register at master driver mcp55_sb_lan");
        }
#else
	/* request interrupt */
	int r;
	DBG("Requesting IRQ %d", DEF_IRQ);

# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	cpe_int_set_irq(DEF_IRQ, LEVEL, L_ACTIVE);
# else
	set_irq_type(DEF_IRQ, IRQT_LOW);
# endif
	
	if ((r = request_irq(DEF_IRQ, mcp55_isr, SA_SHIRQ | SA_SAMPLE_RANDOM, NAME, dev)) < 0) {
	    ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
	    ret = -EBUSY;
	    goto bail;
	}
#endif

	/* start upper layer */
	netif_start_queue(dev);

	INFO("Loaded");
    }


    trig_init();
    spin_unlock_irqrestore(&priv->lock, flags);

    return ret;

 bail:
    spin_unlock_irqrestore(&priv->lock, flags);
    mcp55_release(dev);
    return ret;
}

static int
mcp55_release(struct net_device *dev)
{
    mcp55_priv_t *priv = dev->priv;
    unsigned long flags;
    
    DBG ("mcp55_release()");

    spin_lock_irqsave(&priv->lock, flags);
       
    if (atomic_dec_return(&priv->refcnt) == 0) {
	
	netif_stop_queue(dev);

        // job to sync with job thread and mark driver as unloaded
        exec_job(priv, done, NULL);
	
#ifdef NVIDIA_IO55
        if (mcp55_unregister_slave(I2C_SLAVE_ADDR) < 0) {
            ERR("can't unregister at master driver mcp55_sb_lan");
        }
#else
	free_irq(DEF_IRQ, dev);
#endif

        if (priv->recv_skb) {
            dev_kfree_skb(priv->recv_skb);
            priv->recv_skb = NULL;
        }

        if (priv->send_skb) {
            dev_kfree_skb(priv->send_skb);
            priv->send_skb = NULL;
        }

	INFO("Unloaded");
    }

    spin_unlock_irqrestore(&priv->lock, flags);
    
    MOD_DEC_USE_COUNT;
    return 0;
}

static int
mcp55_tx(struct sk_buff *skb, struct net_device *dev)
{
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    
    netif_stop_queue(dev); 

    /* start sending in non-irq mode */
    priv->send_skb = skb;
    priv->send_offs = 0;
    add_job(priv, &priv->send_job);

    return SUCCESS;
}

static int
mcp55_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    switch (cmd) {
      case SIOCETHTOOL: {
	  struct ethtool_cmd ecmd;
	  struct ethtool_value edata;
	  
	  if (copy_from_user(&ecmd, rq->ifr_data, sizeof(ecmd))) {
	      return -EFAULT;
	  }

	  memset(&edata, 0, sizeof(edata));
	  edata.cmd = ETHTOOL_GLINK;
	  edata.data = netif_carrier_ok(dev);
	  if (copy_to_user(rq->ifr_data, &edata, sizeof(edata))) {
	      return -EFAULT;
	  }
	  
	  break;
      }
      default:
	  return -EOPNOTSUPP;
    }

    return 0;
}

static int
mcp55_set_mac_addr (struct net_device *dev, void *addr)
{
    struct sockaddr *hwaddr = (struct sockaddr *)addr;

    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    return exec_job(priv, set_mac, &hwaddr->sa_data);
    
    return 0;
}

static int
mcp55_check_status(struct net_device *dev, u_char *queue_rx)
{
    int asr, isr;

    *queue_rx = 0;
    
    if ((asr = pmu_cfg_read(NV_CFG_PARA_ASR)) < 0) {
	WARN("ASR status read failed");
	return FAIL;
    }

    DBG("ASR=0x%02x", asr);

    if (asr & NV_ALERT_MAC_RX_READY) {
	DBG2("NV_ALERT_MAC_RX_READY");	
	*queue_rx = 1;
    }
    
    if (asr & NV_ALERT_INT_ALERT) {
	DBG2("NV_ALERT_INT_ALERT");
	if ((isr = pmu_cfg_read(NV_CFG_PARA_ASR)) < 0) {
	    WARN("ISR status read failed");
	    return FAIL;
	}

	DBG("ISR=0x%02x\n", isr);
    }

    if (asr & NV_ALERT_LINK_STATUS) {
	u_char msr = pmu_cfg_read(NV_CFG_PARA_MSR);
	u_char link_states = (msr & 0xF0) >> 4;

	INFO("NV_ALERT_LINK_STATUS (0x%02x)", link_states);
	
	if (link_states == 0x00) {
	    netif_carrier_off(dev);
	} else {
	    netif_carrier_on(dev);
	}	
    }

    if (asr & NV_ALERT_WATCHDOG_RESET) {
	INFO("Watchdog reset from MCP55");
	/* FIXME: do something */
    }
    
    if (asr & NV_ALERT_NVRAM_ERROR) {
	ERR("NVRAM error from MCP55");
	/* FIXME: do something, or not */
    }

    if (asr & NV_ALERT_MAC_TX_ERROR) {
	ERR("TX ERROR 0x%02x",
	    (pmu_cfg_read(NV_CFG_PARA_TSR) & NV_TSR_MAC_TX_STATUS_MASK) >> NV_TSR_MAC_TX_STATUS_SHIFT);	
    }
    
    return SUCCESS;
}

/* ------------------------------------------------------------------------- *
 * I2C sideband lan xmit handlers
 * ------------------------------------------------------------------------- */
#ifdef DUMP_DATA
static void dump (u_int8_t *buf, int size)
{
    int i, j;
    for (i=0; i<size; i+=16) {
        for (j=i; j<i+16; j++) {
            if (j<size) {
                printk ("%02x ", buf[j]);
            } else {
                printk ("   ");
            }
        }
        for (j=i; j<i+16; j++) {
            if (j<size) {
                printk ("%c", (buf[j]>' ') ? (buf[j]) : ('.')); 
            }
        }
        printk ("\n"); 
    }
}
#endif

#ifndef NVIDIA_IO55
static irqreturn_t
mcp55_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    struct net_device *dev = (struct net_device *) dev_id;
    mcp55_priv_t *priv = (mcp55_priv_t*) dev->priv;
    
    DBG("IRQ");

    disable_irq_nosync(DEF_IRQ);

    /* start receiving in non-irq mode */
    add_job(priv, &priv->recv_job);

    return IRQ_HANDLED;
}
#endif

/* ------------------------------------------------------------------------- *
 * the actual jobs which do the work
 * ------------------------------------------------------------------------- */

static int
send(void* ctx)
{
    struct net_device *dev = &net_dev;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    struct sk_buff *skb = (struct sk_buff*)priv->send_skb;

    int first = priv->send_offs == 0;
    int last = skb->len - priv->send_offs <= priv->max_data_size;
    int len = last ? skb->len - priv->send_offs : priv->max_data_size;
    uint8_t cmd = NV_ETH_FRM_WR | (first ? NV_ETH_FRM_WR_FF : 0) | (last ? NV_ETH_FRM_WR_LF : 0);
    int r;

    if (!priv->is_init) {
        priv->stats.tx_errors++;
        goto exit;
    }

    DBG2("sending fragment of frame %ld (offs=%d frag-len=%d pkt-len=%d)",
        priv->stats.tx_packets, priv->send_offs, len, skb->len);

    trig2_fire();
    if ((r = i2c_write_block(priv, cmd, len, skb->data + priv->send_offs))) {
        ERR("send: failed to send frame - offset %d (err %d)", priv->send_offs, r);
        priv->stats.tx_errors++;
        goto exit;
    }
    
    priv->send_offs += len;

    if (first) {
        dev->trans_start = jiffies;
    }

    // more fragments to send?
    if (!last) {
        add_job(priv, &priv->send_job);
        return SUCCESS;
    } else {
        /* statistics */
        DBG2("ethernet frame %ld successfully sent (%d bytes)",
	     priv->stats.tx_packets, skb->len);
        //DBG("  %02x %02x %02x %02x", 
        //    skb->data[0], skb->data[1], skb->data[2], skb->data[3]);
        priv->stats.tx_packets++;
        priv->stats.tx_bytes += skb->len;
    }

exit:
    /* free the buffer */
    dev_kfree_skb(priv->send_skb);
    priv->send_skb = NULL;

    /* switch on transmission again */
    netif_wake_queue(dev);

    return SUCCESS;
}

static int
recv(void* ctx)
{
    struct net_device *dev = &net_dev;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    struct sk_buff *skb = (struct sk_buff*)priv->recv_skb;

    mcp55_recv_pkt_t recv_pkt;
    uint8_t count;
    int r, more = 0;
    
    if (!priv->is_init) {
	return -1;
    }

#ifndef NVIDIA_IO55
    if (slave_saddr != 0) {
        // make ARA to check, if ours or slave's device has caused the alert
        u8 ara = 0;
        struct i2c_msg msg = { 0x0c, I2C_M_RD, 1, &ara };

        if (i2c_transfer(priv->i2c_client->adapter, &msg, 1) < 0) {
            ara = 0;
        }
        ara >>= 1; // make it a 7bit address

        if (ara == slave_saddr) {
            slave_cb(slave_ctx);
            goto bail;
        }
    }
#endif

    trig_fire();
    if ((r = i2c_read_fragment(priv, NV_ETH_FRM_RD, &count, (u8*)&recv_pkt)) == SUCCESS) {
	DBG2("recv status 0x%02x, count=%d", recv_pkt.status, count);

	udelay(200);
	
	if (count == 0) {
	    /* seems we asked for a frame but weren't alerted */
	    if (recv_pkt.status & (NV_ETH_FRM_RD_STS_LAST_FRAGMENT | NV_ETH_FRM_RD_STS_STATUS_CHANGE)) {
		u_char queue_rx = 0;
		
		mcp55_check_status(dev, &queue_rx);
		if (queue_rx) {
		    add_job(priv, &priv->recv_job);
		}
		goto bail;
	    }
	}
	
	if (recv_pkt.status & NV_ETH_FRM_RD_STS_FIRST_FRAGMENT) {
	    /* first packet fragment: prepare skb */
	    if (skb->len != 0) {
		ERR("recv: new frame while old not completed, discard old frame (%d bytes)",
		    skb->len);
		skb_trim(skb, 0);
	    }
	}

	/* process the single fragment */
	DBG2("recv fragment of frame %ld (offs=%d len=%d)",
	     priv->stats.rx_packets, priv->recv_skb->len, count);
	
	if (priv->recv_skb->tail + count > priv->recv_skb->end) {
	    ERR("recv: frame exceeds max length (offs=%d frag-len=%d max=%d)",
		priv->recv_skb->len, count, dev->mtu + ETH_HLEN);
	} else {
	    memcpy(__skb_put(priv->recv_skb, count), recv_pkt.data, count);
	}
	
	if (recv_pkt.status & NV_ETH_FRM_RD_STS_LAST_FRAGMENT) {
	    DBG2("frame %ld successfully received (%d bytes)",
		 priv->stats.rx_packets, skb->len);

	    dev->last_rx = jiffies;
	    
	    /* statistics */
	    priv->stats.rx_packets++;
	    priv->stats.rx_bytes += skb->len;
	    
	    /* send packet upstairs */
	    skb->dev = dev;
	    skb->protocol = eth_type_trans(skb, dev);
	    netif_rx(skb);
	    
	    /* get new buffer (size: payload + header, without CRC!) */
	    priv->recv_skb = dev_alloc_skb(dev->mtu + ETH_HLEN);

	    if (recv_pkt.status & NV_ETH_FRM_RD_STS_MAC_RX_READY) {
		DBG2("more packets to come, requeuing");
		more = 1;
	    }
   	} else {
	    more = 1;
	}
	
	if (recv_pkt.status & NV_ETH_FRM_RD_STS_STATUS_CHANGE) {
	    u_char queue_rx = 0;

	    /* if we are already queued don't do it again */
	    mcp55_check_status(dev, &queue_rx);
	    more |= queue_rx;
	}
    } else {
        ERR("recv: failed to read fragment");
    }

    /* queue ourself again to get the next fragment or packet */
    if (more) {
	add_job(priv, &priv->recv_job);
	return SUCCESS;
    }

 bail:
#ifndef NVIDIA_IO55
    enable_irq(DEF_IRQ);
#endif
    return SUCCESS;
}

static int
init(void* ctx)
{
    struct net_device *dev = &net_dev;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    int r = SUCCESS, i;
    u_char auto_mask = 0;

    if (ctx == RE_INIT) {
        // don't do re-init when has been already called
        if (!priv->is_init) return -1;
    }

    /* disable alerting */
    disable_irq(DEF_IRQ);

    /* get MAC count and enable autoselect among them */
    if ((r = pmu_cfg_read(NV_CFG_PARA_MSR)) == FAIL) {
	ERR("Could not get MAC count");
	goto bail;
    }
    
    priv->mac_count = ((r & NV_MSR_MAC_COUNT_MASK) >> NV_MSR_MAC_COUNT_SHIFT) + 1;
    INFO("MAC count = %d", priv->mac_count);

    for (i=0; i<priv->mac_count; i++) {
	auto_mask |= 0x01 << i;
    }
    if ((r = pmu_cfg_write(NV_CFG_PARA_MCR, auto_mask)) != SUCCESS) {
	ERR("Could not enable auto MAC switching");
	goto bail;
    }

    {
	u_char msr = pmu_cfg_read(NV_CFG_PARA_MSR);
	u_char link_states = (msr & 0xF0) >> 4;

	INFO("NV_ALERT_LINK_STATUS (0x%02x)", link_states);
	
	if (link_states == 0x00) {
	    netif_carrier_off(dev);
	} else {
	    netif_carrier_on(dev);
	}	
    }
    
    /* config filter settings */
    if ((r = pmu_cfg_write(NV_CFG_PARA_FCR, NV_FCR_MAC_RX_EN)) != SUCCESS) {
	ERR("Could not config filter settings");
	goto bail;
    }

    /* MAC features register, no arp and ping response for now */
    if ((r = pmu_cfg_write(NV_CFG_PARA_MFR, 0)) != SUCCESS) {
	ERR("Could not config filter settings");
	goto bail;
    }
    
    /* setup recv frame fragment size */
    if ((r = pmu_cfg_write(NV_CFG_PARA_FSR, MAX_FRAG_SIZE)) != SUCCESS) {
	ERR("Could not set recv fragment size");
	goto bail;
    }
    
    /* setup alert mask register */
    if ((r = pmu_cfg_write(NV_CFG_PARA_AMR,
			   NV_ALERT_MAC_RX_READY   |
			   NV_ALERT_LINK_STATUS    |
			   NV_ALERT_WATCHDOG_RESET |
			   NV_ALERT_NVRAM_ERROR    |
			   NV_ALERT_MAC_TX_ERROR)) != SUCCESS) {
	
	    ERR("Could not setup alert mask");
	    goto bail;
    }
   
    /* setup initial mac */
    if ((r = pmu_mac_write(dev->dev_addr)) != SUCCESS) {
	ERR("Could not setup initial mac");
	goto bail;
    }

    /* setup IP */
    if ((r = pmu_ipv4_write(priv->ip_address)) != SUCCESS) {
	ERR("Could not setup initial IP");
	goto bail;
    }
  
 bail:
    if (r != SUCCESS) {
        // queues another init job (will be executed after this init has returned)

	set_current_state(TASK_INTERRUPTIBLE);
        schedule_timeout(10 * HZ);

        add_job(priv, &priv->reinit_job);

        r = -EAGAIN;
    } else {
	u_char queue_rx;
	
        /* enable alerting (only if init succeeded) */
        enable_irq(DEF_IRQ);

	// first time init passed: mark driver as loaded
	priv->is_init = 1;

	mcp55_check_status(dev, &queue_rx);
	if (queue_rx) {
	    add_job(priv, &priv->recv_job);
	}
	
	INFO("MCP55 initialized");
    }
   
    return r;
}

static int
done(void* ctx)
{
    struct net_device *dev = &net_dev;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;

    // mark driver as unloaded
    priv->is_init = 0;
    return SUCCESS;       
}

static int
set_mac(void *ctx)
{
    struct net_device *dev = &net_dev;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;
    uint8_t *new_mac_addr = ctx;

    // store mac addr
    memcpy(dev->dev_addr, new_mac_addr, dev->addr_len);
    priv->mac_set = 1;

    if (pmu_mac_write(new_mac_addr) != SUCCESS) {
	ERR("Error setting MAC address tp PMU");
	return FAIL;
    }
    
    INFO("MAC addr set to %02x:%02x:%02x:%02x:%02x:%02x",
         new_mac_addr[0], new_mac_addr[1], new_mac_addr[2],
         new_mac_addr[3], new_mac_addr[4], new_mac_addr[5]);

    return SUCCESS;    
}

/* ------------------------------------------------------------------------- *
 * job thread (high prio!)
 * ------------------------------------------------------------------------- */

static int job_thread(void * arg)
{
    struct net_device *dev = (struct net_device *) arg;
    mcp55_priv_t *priv = (mcp55_priv_t*)dev->priv;


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
        /* Release all our userspace resources */
        daemonize();
        reparent_to_init();
	strcpy(current->comm, NAME);
#else
        daemonize(NAME);
#endif

    //  I2C: realtime priority to reduce delays between message bytes!
    current->policy = SCHED_RR;
    current->rt_priority = 50;

    priv->job_thd_running = 1;
    
    while (1) {
        wait_event(priv->job_queue_wait, jobs_avail(priv));

        job_t *job;
        while (get_next_job(priv, &job) == SUCCESS) {
            // do the job (termination job returns FAIL here!)
            if (do_job(job) != SUCCESS) goto exit;
        }
    }
    
exit:
    DBG("Job thread exiting");
    return 0;
}

/* ------------------------------------------------------------------------- *
 * realtime job functions
 * ------------------------------------------------------------------------- */

#ifdef UNUSED_FUNCTIONS
// add a highest-priority job to job queue for asynchron execution
void add_job_head(mcp55_priv_t *priv, job_t *job)
{
    unsigned long flags;

    // add job into queue
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&job->anchor)) {
        ERR("add_job_head: job %s already queued", 
            job->func == send ? "SEND" :
            job->func == recv ? "RECV" :
            job->func == init ? "INIT" :
            job->func == set_mac ? "SET_MAC" : "OTHER");
    }
    else list_add(&job->anchor, &priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    // wake up job thread
    wake_up(&priv->job_queue_wait);
}
#endif

// add a job to job queue for asynchron execution
static void add_job(mcp55_priv_t *priv, job_t *job)
{
    unsigned long flags;

    // add job into queue
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&job->anchor)) {
        ERR("add_job: job %s already queued", 
            job->func == send ? "SEND" :
            job->func == recv ? "RECV" :
            job->func == init ? "INIT" :
            job->func == set_mac ? "SET_MAC" : "OTHER");
    }
    else list_add_tail(&job->anchor, &priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    // wake up job thread
    wake_up(&priv->job_queue_wait);
}

// syncrhronly executes a job and returns result
static int exec_job(mcp55_priv_t *priv, job_func_t func, void *ctx)
{
    DECLARE_MUTEX_LOCKED(done);
    job_t job = { .anchor = LIST_HEAD_INIT(job.anchor), .func = func, .ctx = ctx, .done = &done };

    // issue job
    add_job(priv, &job);

    // wait for job completion
    down(&done);

    // return job result
    return job.ret;
}

// checks, if there are job in the queue (called from job thread)
static int jobs_avail(mcp55_priv_t *priv)
{
    int ret;
    unsigned long flags;
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    ret = !list_empty(&priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);
    return ret;
}

// removes oldest job from the queue and returns it (called from job thread)
static int get_next_job(mcp55_priv_t *priv, job_t **job)
{
    int ret = FAIL;
    unsigned long flags;

    // return next job (queue head) if present
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&priv->job_queue)) {
        *job = list_entry(priv->job_queue.next, job_t, anchor);
        list_del(priv->job_queue.next);
        INIT_LIST_HEAD(&(*job)->anchor); // make job as non-queued
        ret = SUCCESS;
    }
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    return ret;
}

// perform a job (called from job thread)
static int do_job(job_t *job)
{
    // do the job
    if (job->func != term) job->ret = job->func(job->ctx);

    // signal completion (for synchron jobs only)
    if (job->done) up(job->done);

    // return FAIL to signal job thread termination
    return job->func != term ? SUCCESS : FAIL;
}

/* ------------------------------------------------------------------------- *
 * high level PMU access functios
 * ------------------------------------------------------------------------- */

static int
pmu_cfg_read(u_char reg)
{
    u_char buf[3];
    int ret = -1;

    buf[0] = NV_CFG_PARA_RD;
    buf[1] = reg;
    
    if ((ret = i2c_access_proc(buf, 2, 1)) <= 0) {
	ERR("PMU CFG RD @ 0x%02x failed (%d)", reg, ret);
	return FAIL;
    }

    DBG2("PMU CFG RD @ 0x%02x = 0x%02x (%d)", reg, buf[0], ret);    
    
    return buf[0];
}

static int
pmu_cfg_write(u_char reg, u_char data)
{
    u_char buf[3];
    int ret = -1;

    buf[0] = NV_CFG_PARA_WR;
    buf[1] = reg;
    buf[2] = data;
    
    if ((ret = i2c_access_write(buf, 3)) <= 0) {
	ERR("PMU CFG WR @ 0x%02x failed (%d)", reg, ret);
	return FAIL;
    }

    DBG2("PMU CFG WR @ 0x%02x : 0x%02x (%d)", reg, data, ret);

    return SUCCESS;    
}

static int
pmu_mac_write(uint8_t mac[6])
{
    u_char buf[16];
    int ret = -1;

    buf[0] = NV_MAC_ADDR_WR;
    memcpy(&buf[1], mac, 6);

    if ((ret = i2c_access_write(buf, 7)) <= 0) {
	ERR("PMU MAC WRITE failed (%d)", ret);
	return FAIL;
    }
    
    return SUCCESS;
}

#ifdef UNUSED_FUNCTIONS
static int
pmu_mac_read(uint8_t *mac)
{
    u_char buf[16];
    int ret = -1;

    buf[0] = NV_MAC_ADDR_RD;

    if ((ret = i2c_access_proc(buf, 1, 6)) <= 0) {
	ERR("PMU MAC READ failed (%d)", ret);
	return FAIL;
    }
    
    memcpy(mac, buf, 6);    
    return SUCCESS;
}

static int
pmu_ipv4_read(uint8_t *ip)
{
    u_char buf[16];
    int ret = -1;

    buf[0] = NV_IP_ADDR_RD;

    if ((ret = i2c_access_proc(buf, 1, 4)) <= 0) {
	ERR("PMU IPV4 READ failed (%d)", ret);
	return FAIL;
    }

    memcpy(ip, buf, 4);   
    return ret;
}
#endif

static int
pmu_ipv4_write(uint8_t ip[4])
{
    u_char buf[16];
    int ret = -1;

    buf[0] = NV_IP_ADDR_WR;
    memcpy(&buf[1], ip, 4);

    if ((ret = i2c_access_write(buf, 5)) <= 0) {
	ERR("PMU IPV4 WRITE failed (%d)", ret);
	return FAIL;
    }
    
    return SUCCESS;
}
 

/* ------------------------------------------------------------------------- *
 * I2C access foundations
 * ------------------------------------------------------------------------- */

static inline int
i2c_access_transfer(struct i2c_adapter * adapter, struct i2c_msg * msg, int num)
{
    int retries = 0;
    int ret;
    
    if (!adapter) {
        return -ENODEV;
    }
    
    do {
        ret = i2c_transfer(adapter, msg, num);
        if (ret > 0) return ret;
        DBG ("i2c_transfer failed, retry # %d", retries + 1);
        DELAY();
    } while (retries++ < MASTER_RETRIES);
    
    return ret;
}

static int
i2c_access_write(char* buffer,
                 size_t size)
{
    struct i2c_msg msg;
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));
    
    msg.addr  = mcp55_priv.i2c_slave_addr;
    msg.buf   = buffer;
    msg.len   = size;
    
    ret = i2c_access_transfer(mcp55_priv.i2c_client->adapter, &msg, 1);

    if (ret < 1) return ret;
    return size;
}

#ifdef UNUSED_FUNCTIONS
static int
i2c_access_read(char* buffer,
                size_t size)
{
    struct i2c_msg msg;
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));
    
    msg.addr  = mcp55_priv.i2c_slave_addr;
    msg.buf   = buffer;
    msg.flags = I2C_M_RD;
    msg.len   = size;
    
    ret = i2c_access_transfer(mcp55_priv.i2c_client->adapter, &msg, 1);

    if (ret < 1) return ret;
    return size;
}
#endif

static int
i2c_access_proc(char* buffer,
                size_t out_size,
                size_t in_size)
{
    struct i2c_msg msg[2];
    int ret;
    
    memset(&msg, 0, sizeof(struct i2c_msg));

    msg[0].addr  = mcp55_priv.i2c_slave_addr;
    msg[0].buf   = buffer;
    msg[0].len   = out_size;
        
    msg[1].addr  = mcp55_priv.i2c_slave_addr;
    msg[1].buf   = buffer;
    msg[1].flags = I2C_M_RD;
    msg[1].len   = in_size;
    
    ret = i2c_access_transfer(mcp55_priv.i2c_client->adapter, msg, 2);

    if (ret < 1) return ret;
    return in_size;
}

/* block read/write functions, to replace */

/* NOTE: the i2c_smbus_... functions of the linux kernel only supports
 * messages up to 32 bytes length, block reads are not supported at all.
 * So we have to implement a SMBus emulation via I2C function calls here.
 */
static int
i2c_write_block(mcp55_priv_t *priv, u8 command, u8 byte_count, const u8 * data)
{
    int ret = 0;

    /* sanity check */
    if (byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_write_block: byte count of %d exceeds limit of %d.",
             byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    DBG2("i2c_write_block: len: %d, data (%02x, %02x, %02x)",
	 byte_count, data[0], data[1], data[2]);

    /* construct an I2C message
       byte 1: command
       byte 2: byte count
       byte 3 .. byte n: data */
    u8 msg_buf[2 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg = {
        addr:   priv->i2c_slave_addr,
        flags:  0,
        len:    byte_count + 2, /* 2 bytes header + data */
        buf:    msg_buf,
    };

    /* fill the I2C message */
    msg_buf[0] = command;
    msg_buf[1] = byte_count;
    memcpy(msg_buf + 2, data, byte_count);

    /* transfer the I2C message */
    int retry = MASTER_RETRIES;
    do {
        ret = i2c_transfer(priv->i2c_client->adapter, &msg, 1);
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C write transfer: %d", ret);
        else WARN("Error during I2C write transfer: %d", ret);
	DELAY();
    } while (retry-- > 0);
    if (ret < 0) goto bail;

bail:
    return ret;
}

static int
i2c_read_fragment(mcp55_priv_t *priv, u8 command, u8 *byte_count, u8 *data)
{
    int ret = 0;

    /* construct 2 I2C messages:
       message 1: write command byte
       message 2: read data bytes */    
    u8 msg_buf[1 + I2C_MAX_DATA_SIZE];
    struct i2c_msg msg[2] = {
        { priv->i2c_slave_addr, 0, 1, &command },
        { priv->i2c_slave_addr, I2C_M_RD, 1 + I2C_MAX_DATA_SIZE, msg_buf },
    };

    /* transfer the I2C messages */
    int retry = MASTER_RETRIES;
    do {
        ret = i2c_transfer(priv->i2c_client->adapter, msg, 2);
        if (ret >= 0) { ret = 0; break; }
        if (!retry) ERR("Error during I2C read transfer: %d", ret);
        else WARN("Error during I2C read transfer: %d", ret);
	DELAY();
    } while (retry-- > 0);
    if (ret < 0) goto bail;

    /* fill out return values;
       byte 0: status
       byte 1: byte count for this fragment
       byte 2 .. byte n: data */
    *byte_count = msg_buf[1];

    /* sanity check */
    if (*byte_count > I2C_MAX_DATA_SIZE) {
        WARN("i2c_read_fragment: byte count of %d exceeds limit of %d.",
             *byte_count, I2C_MAX_DATA_SIZE);
        ret = -EINVAL;
        goto bail;
    }

    memcpy(data, msg_buf, *byte_count + 2);

    DBG2("i2c_read_fragment: cmd: %02x, len: %d, data (%02x, %02x, %02x)",
	 command, *byte_count, data[0], data[1], data[2]);

bail:
    return ret;
}

/* ------------------------------------------------------------------------- *
 * I2C kernel stuff
 * ------------------------------------------------------------------------- */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static unsigned short normal_i2c[] = { I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short normal_i2c_range[] = { I2C_SLAVE_ADDR, I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short probe_i2c_range[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c_range[] = { I2C_CLIENT_END };
// TODO: we could probe here, not force...
static unsigned short force_i2c[] = { ANY_I2C_BUS, I2C_SLAVE_ADDR, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range,
	probe_i2c, probe_i2c_range,
	ignore_i2c, ignore_i2c_range,
	force_i2c
};
#else
static unsigned short normal_i2c[] = { I2C_SLAVE_ADDR, I2C_CLIENT_END };
static unsigned short probe_i2c[] = { I2C_CLIENT_END };
static unsigned short ignore_i2c[] = { I2C_CLIENT_END };
// TODO: we could probe here, not force...
static unsigned short force_i2c[] = { ANY_I2C_BUS, I2C_SLAVE_ADDR, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, probe_i2c, ignore_i2c, force_i2c
};
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void mcp55_i2c_inc_use (struct i2c_client *client) {
}

static void mcp55_i2c_dec_use (struct i2c_client *client) {
}
#endif

static int mcp55_i2c_command(struct i2c_client *client,
		               unsigned int cmd, void *arg) {
    /* no commands defined */
    return 0;
}

static int mcp55_i2c_attach_adapter(struct i2c_adapter *adapter) {
    /* don't attach the client if it is already attached */
    if (mcp55_priv.i2c_client) {
        DBG("Device already has adapter attached.");
        return 0;
    }
    
    DBG("i2c adapter attached, calling probe: %s", adapter->name);
    
    /* don't attach the client if adapter name doesn't match */
    if (strcmp(I2C_ADAPTER_NAME, adapter->name) != 0) {
        DBG("Expected adapter: %s, skipping", I2C_ADAPTER_NAME);
        return 0;
    }

    return i2c_probe_force(adapter, &addr_data, &mcp55_i2c_detect_client);
}

static int mcp55_i2c_detect_client(struct i2c_adapter *adapter, int address, 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			             unsigned short flags,
#endif
			             int kind) {
    struct i2c_client* new_client = NULL;
    int err = 0;
    
    DBG("i2c detect client addr 0x%02x, kind %d", address, kind);
    
    if (!(new_client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL))) {
    	err = -ENOMEM;
    	goto fail;
    }
    memset(new_client, 0, sizeof(struct i2c_client));
    
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    strcpy(new_client->name, I2C_NAME);
    new_client->data = &mcp55_priv;
    new_client->id  = 0; /* ??? */
#else
    strlcpy(new_client->name, I2C_NAME, I2C_NAME_SIZE);
    i2c_set_clientdata(new_client, &mcp55_priv);
#endif
    new_client->addr = address;
    new_client->adapter = adapter;
    new_client->driver = &mcp55_i2c_driver;
    new_client->flags = 0;

    if ((err = i2c_attach_client_force(new_client))) {
    	goto fail_free;
    }    
    
    mcp55_priv.i2c_client = new_client;
    mcp55_priv.i2c_slave_addr = address;
    
    return 0;
    
 fail_free:    
    if (new_client) {
    	kfree(new_client);
    	new_client = NULL;
    }
    
 fail:
    return err;
}

static int mcp55_i2c_detach_client(struct i2c_client *client) {
    int err;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    mcp55_priv_t *priv = client->data;
#else
    mcp55_priv_t *priv = (mcp55_priv_t*)  i2c_get_clientdata (client);
#endif
    
    DBG("i2c detach client %s", client->name);

    if ((err = i2c_detach_client(client))) {
    	WARN("i2c client detach failed");
    	goto fail;
    }
    
    kfree(client);

 fail:
    priv->i2c_client = NULL;
    
    return err;
}

#if USE_PEC

/* ------------------------------------------------------------------------- *
 * Calculate CRC-8 x^8+x^2+x^+1
 * ------------------------------------------------------------------------- */

const static u8 crc8table[] =
{
    0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15,
    0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
    0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65,
    0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
    0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5,
    0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
    0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85,
    0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
    0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2,
    0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
    0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2,
    0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
    0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32,
    0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
    0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42,
    0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
    0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C,
    0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
    0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC,
    0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
    0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C,
    0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
    0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C,
    0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
    0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B,
    0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
    0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B,
    0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
    0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB,
    0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
    0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB,
    0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
};

static u8 
update_crc8 (u8 *data, size_t len, u8 crc)
{
    int   i;

    for (i = 0; i < len; i++)
    {
        crc = crc8 (data[i], crc);
    }

    return crc;
}

#endif /* USE_PEC */
