/*****************************************************************************
 * msp_adapter.c
 *
 * Copyright (c) 2005-2006 Peppercon AG, miba@peppercon.de, inva@peppercon.de
 *
 * Converts file io operations to mass storage protocol read/write requests
 *
 * Devices: /dev/msp_fe_x (b, 250, x) 'image' files containing the redirected devices
 *          /dev/msp_be   (c, 250, 0) backend connector for MSP thread
 *
 * TODO:
 * - use the msp adapter also for PPC
 *****************************************************************************/
#define MAJOR_NR		250
#define DRV_NAME		"msp_adapter"

#define log2(n)			ffz(~(n))

/* some defines which must exist before including blk.h */
#define DEVICE_NAME		DRV_NAME

#define BASE_SECTOR_SIZE	512
#define MAX_NR_BLOCKS		128
#define MAX_DEV_COUNT		PP_FEAT_USB_MASS_STORAGE_NO

#define BLK_REQ_ERROR		0
#define BLK_REQ_SUCCESS		1

#include <linux/blkdev.h>

#include <linux/fs.h>
#include <linux/ioctl.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/slab.h>
#include <linux/types.h>

#include <asm/bitops.h>
#include <asm/semaphore.h>

#include "msp_adapter.h"
#include "common_debug.h"

/* ------------------------------------------------------------------------- *
 * global constants, macros and datatypes
 * ------------------------------------------------------------------------- */

#ifdef SUCCESS
#  undef  SUCCESS
#endif
#define SUCCESS 0

typedef enum {
    REQ_STATE_IDLE,
    REQ_STATE_READ_PENDING,
    REQ_STATE_WRITE_PENDING,
    REQ_STATE_WRITEDONE_PENDING,
    REQ_STATE_DONE,
} req_state_t;

typedef struct {
    struct list_head listnode;
    struct request *req;
} req_entry_t;

/*
 * data structure per usage of the module (open/release)
 */
typedef struct {
} priv_data_t;

/*
 * global data used by all clients
 */
typedef struct {
    char			name[15];
    int				cd_registered;		/* character device registered */
    int				bd_registered;		/* block device registered */
    u8				fe_connected;		/* connected clients */
    u8				be_connected;		/* connected backends (usually 1) */
    wait_queue_head_t		poll_wq;

    request_queue_t		*rqueue;		/* common request queue */
    struct request		*current_req;		/* request being currently served */
    req_state_t			req_state;		/* the current request in progress */
    pp_msp_file_request_t	file_req;		/* request data */
    pp_msp_file_response_t	file_resp;		/* response data */

    struct semaphore		sem_fe;			/* frontend lock */
    struct semaphore		sem_be;			/* backend lock */

    volatile u8			devs_used;
    spinlock_t			dev_lock;
} msp_adap_t;

/*
 * data used per specific block device
 */
typedef struct {
    u8			index;
    volatile u8		active;
    volatile u8		cleanup;

    u32			block_size;
    u32			block_count;
    u8			sect_divider;
} msp_dev_t;

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */

static msp_adap_t adap = {
    name:		DRV_NAME,
    dev_lock:		SPIN_LOCK_UNLOCKED,
    req_state:		REQ_STATE_IDLE,
    file_req:		{},
    file_resp:		{},
};

static const msp_dev_t msp_dev_init_tmpl = {
    sect_divider:	1
};

static msp_dev_t devs[MAX_DEV_COUNT];
static struct gendisk *disks[MAX_DEV_COUNT];

int bd_opt_read_ahead  = 32;
int bd_opt_max_blocks  = 128;
MODULE_PARM(bd_opt_read_ahead, "i");
MODULE_PARM_DESC(bd_opt_read_ahead, "Number of blocks to read ahead (default 32, max 128)");
MODULE_PARM(bd_opt_max_blocks, "i");
MODULE_PARM_DESC(bd_opt_max_blocks, "Number of blocks per mass storage request (default 128, max 128)");

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */
static int  msp_init(void);
static void msp_cleanup(void);

/* character device operations */
static int  msp_be_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg);
static int  msp_be_open(struct inode *inode, struct file *file);
static int  msp_be_release(struct inode *inode, struct file *file);
static unsigned int msp_be_poll(struct file *file, struct poll_table_struct *wait);

/* block device operations */
static int msp_bd_open(struct inode *inode, struct file *file);
static int msp_bd_release(struct inode *inode, struct file *file);
static int msp_bd_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg);
#if 0
static int msp_bd_check_change(kdev_t dev);
static int msp_bd_revalidate(kdev_t dev);
#endif

static void msp_bd_request(request_queue_t *queue);

/* internal functions */
static void wakeup_userspace(void);

static int msp_bd_prepare_transfer(void);
static int msp_bd_transfer_done(void);
static void msp_bd_cleanup_request(struct request *req, int status);

static int msp_bd_create_blockdevice(msp_dev_t *dev, u_long size_kb, u_long blksize, u_long sectsize);
static int msp_bd_remove_blockdevice(msp_dev_t *dev);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("miba@peppercon.de");
MODULE_DESCRIPTION("MSP adapter driver");

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations msp_be_ops = {
    owner:   THIS_MODULE,
    ioctl:   msp_be_ioctl,
    poll:    msp_be_poll,
    open:    msp_be_open,
    release: msp_be_release,
    read:    NULL,
    write:   NULL,
    llseek:  NULL
};

static struct block_device_operations msp_bdops = {
    owner:		THIS_MODULE,
    open:		msp_bd_open,
    release:		msp_bd_release,
    ioctl:		msp_bd_ioctl,
#if 0
    check_media_change: msp_bd_check_change,
    revalidate:		msp_bd_revalidate,
#endif
};

/* ------------------------------------------------------------------------- *
 * driver operations
 * ------------------------------------------------------------------------- */

static int
msp_init(void)
{
    int ret = SUCCESS, r, i;

    init_waitqueue_head(&adap.poll_wq);
    init_MUTEX(&adap.sem_fe);
    init_MUTEX(&adap.sem_be);

    // register the character device
    if ((r = register_chrdev(MAJOR_NR, adap.name, &msp_be_ops))) {
	D(D_ERROR, "failed to register char dev driver (%d).\n", r);
	ret = -EBUSY;
	goto error_out;
    }
    adap.cd_registered = 1;

    // register the block device
    if ((r = register_blkdev(MAJOR_NR, adap.name))) {
	D(D_ERROR, "failed to register block dev driver (%d).\n", r);
	ret = -EBUSY;
	goto error_out;
    }
    adap.bd_registered = 1;

    adap.rqueue = blk_init_queue(msp_bd_request, &adap.dev_lock);
    blk_queue_max_sectors(adap.rqueue, bd_opt_max_blocks);

    for (i = 0; i < MAX_DEV_COUNT; i++) {
	msp_dev_t *dev = &devs[i];
	struct gendisk *disk;

	*dev = msp_dev_init_tmpl;
	dev->index = i;

	disk = disks[i] = alloc_disk(1);
	disk->major = MAJOR_NR;
	disk->first_minor = i;
	disk->fops = &msp_bdops;
	sprintf(disk->disk_name, "msp_fe_%d", i);
	disk->private_data = dev;
	disk->queue = adap.rqueue;
	add_disk(disk);
    }
    return 0;

error_out:
    msp_cleanup();
    return ret;
#if 0
    int ret = SUCCESS, r, i;


    init_MUTEX(&adap.sem_fe);
    init_MUTEX(&adap.sem_be);


    /* ---- pre-init dev specific data  ------------------------------ */
    for (i = 0; i < MAX_DEV_COUNT; i++) {
	devs[i] = msp_dev_init_tmpl;
	devs[i].index = i;
    }

    if (msp_bd_alloc_blkdev_arrays() != 0) {
	D(D_ERROR, "failed to init kernel blockdev data arrays\n");
	ret = -ENOMEM;
	goto error_out;
    }

    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(MAJOR_NR, adap.name, &msp_be_ops)) < 0) {
	D(D_ERROR, "failed to register char dev driver (%d)\n", r);
	ret = -ENODEV;
	goto error_out;
    }

    D(D_NOTICE, "msp options: read_ahead=%d, max_blocks=%d\n",
      bd_opt_read_ahead, bd_opt_max_blocks);

    return ret;

error_out:
    msp_cleanup();
    return ret;
#endif
}
module_init(msp_init);

static void
msp_cleanup(void)
{
    int r, i;

    for (i = 0; i < MAX_DEV_COUNT; i++) {
	del_gendisk(disks[i]);
	put_disk(disks[i]);
    }

    blk_put_queue(adap.rqueue);

    if (adap.cd_registered) {
	if ((r = unregister_chrdev(MAJOR_NR, adap.name))) {
	    D(D_ERROR, "failed to unregister char dev driver (%d)\n", r);
	}
    }

    if (adap.bd_registered) {
	if ((r = unregister_blkdev(MAJOR_NR, adap.name))) {
	    D(D_ERROR, "failed to unregister block dev driver (%d)\n", r);
	}
    }

    D(D_ALWAYS, "driver cleanup done\n");
}
module_exit(msp_cleanup);

static int
msp_be_open(struct inode *inode, struct file *file)
{
    priv_data_t *pdata;
    unsigned int minor = MINOR(inode->i_rdev);
    int ret = SUCCESS;

    if (minor != 0) {
	  D(D_ERROR, "Unsupported minor number %d\n", minor);
	  ret = -EFAULT;
	  goto bail;
    }

    if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
    adap.be_connected = 1;
    D(D_VERBOSE, "New MSP BE client\n");
    up(&adap.sem_be);

    if (file->private_data == NULL) {
	pdata = kmalloc(sizeof(priv_data_t), GFP_KERNEL);
	file->private_data = (void*) pdata;
    }

 bail:
    return ret;
}

static int
msp_be_release(struct inode *inode, struct file *file)
{
    priv_data_t *pdata = (priv_data_t*) file->private_data;

    if (pdata) {
	if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
	adap.be_connected = 0;
	D(D_VERBOSE, "Removed MSP BE client\n");
	up(&adap.sem_be);
	kfree(pdata);
    }

#if 0
    int i;
    for (i = 0; i < MAX_DEV_COUNT; i++) {
	msp_dev_t *dev = &devs[i];
	msp_bd_remove_blockdevice(dev);
    }
#endif

    return SUCCESS;
}

static int
msp_be_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg)
{
    pp_msp_file_info_t file_info;
    int ret = SUCCESS;

    if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
    switch (cmd) {
      case PP_MSP_IOC_GETREQUEST:
	  if (msp_bd_prepare_transfer() != 0) {
	      ret = -ENODATA;
	      goto bail;
	  }

	  if (copy_to_user((char*)arg, (u8*)&adap.file_req, sizeof(pp_msp_file_request_t))) {
	      D(D_ERROR, "cannot copy to user\n");
	      ret = -EFAULT;
	      goto bail;
	  }
	  break;

      case PP_MSP_IOC_SETRESPONSE:
	  if (copy_from_user(&adap.file_resp, (pp_msp_file_response_t*)arg,
			     sizeof(pp_msp_file_response_t))) {
	      D(D_ERROR, "cannot copy from user file_resp\n");
	      ret = -EFAULT;
	      goto bail;
	  }
	  D(D_BLABLA,"PP_MSP_IOC_SETRESPONSE: dev=%d, response=%d, length=%d\n",
	    adap.file_resp.dev_no, adap.file_resp.response, adap.file_resp.length);

	  if (msp_bd_transfer_done() != 0) {
	      D(D_ERROR, "Error during finish of transfer\n");
	      ret = -EFAULT;
	      goto bail;
	  }
	  break;

      case PP_MSP_IOC_SETFILEINFO:
	  if (copy_from_user(&file_info, (pp_msp_file_info_t*)arg, sizeof(pp_msp_file_info_t))) {
	      D(D_ERROR, "cannot copy file info\n");
	      ret = -EFAULT;
	      goto bail;
	  }

	  if (file_info.block_length == 0 && file_info.block_count == 0) {
	      D(D_VERBOSE, "Cleaning up backend\n");

	      msp_bd_remove_blockdevice(&devs[file_info.dev_no]);
	  } else {
	      if ((0x01 << log2(file_info.block_length)) != file_info.block_length) {
		  D(D_ERROR, "Unsupported block size of %d, must be power of 2\n",
		    file_info.block_length);
	      }

	      devs[file_info.dev_no].block_size = file_info.block_length;
	      devs[file_info.dev_no].block_count = file_info.block_count;

	      D(D_VERBOSE, "PP_MSP_IOC_SETFILEINFO: new block_size=%u, block_count=%u\n",
		file_info.block_length, file_info.block_count);

	      if (msp_bd_create_blockdevice(&devs[file_info.dev_no],
					    (uint32_t)(((uint64_t)file_info.block_length * (uint64_t)file_info.block_count) / (uint64_t)1024),
					    file_info.block_length, file_info.block_length) != 0)
		  {
		      ret = -ENODEV;
		      D(D_ERROR, "Could not create new block device\n");
		      goto bail;
		  }
	  }

	  break;

      default:
	  D(D_ERROR, "Invalid ioctl request\n");
	  break;
    }

 bail:
    up(&adap.sem_be);
    return ret;
}

static unsigned int
msp_be_poll(struct file *file, struct poll_table_struct *wait)
{
    u_long flags;
    int empty;

    if (adap.req_state == REQ_STATE_WRITEDONE_PENDING) {
	return POLLIN | POLLRDNORM;
    }

    spin_lock_irqsave(&adap.dev_lock, flags);
    empty = elv_queue_empty(adap.rqueue);
    spin_unlock_irqrestore(&adap.dev_lock, flags);

    if (empty) poll_wait(file, &adap.poll_wq, wait);

    spin_lock_irqsave(&adap.dev_lock, flags);
    empty = elv_queue_empty(adap.rqueue);
    spin_unlock_irqrestore(&adap.dev_lock, flags);

    return empty? 0 :  POLLIN | POLLRDNORM;
}

/* ------------------------------------------------------------------------- *
 * driver operations (block device)
 * ------------------------------------------------------------------------- */

static int
msp_bd_open(struct inode *inode, struct file *file)
{
    unsigned int minor = MINOR(inode->i_rdev);
    int ret = SUCCESS;

    if (minor >= MAX_DEV_COUNT) {
	D(D_ERROR, "Minor number out of range (max device count=%d)\n",
	  MAX_DEV_COUNT);
    }

    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;

    if (adap.be_connected == 0) {
	D(D_ERROR, "No backend connected, open refused\n");
	ret = -ENODEV;
	up(&adap.sem_fe);
	goto bail;
    }

    adap.fe_connected++;
    D(D_VERBOSE, "New MSP FE blockdev client (count=%d)\n", adap.fe_connected);
    up(&adap.sem_fe);

bail:
    return ret;
}

static int
msp_bd_release(struct inode *inode, struct file *file)
{
    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;

    adap.fe_connected--;
    D(D_VERBOSE, "Removed MSP FE blockdev client (count=%d)\n", adap.fe_connected);
    up(&adap.sem_fe);

    return SUCCESS;
}

static int
msp_bd_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg)
{
    int ret = -EINVAL;

    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;

    D(D_ERROR, "Invalid ioctl request\n");

    up(&adap.sem_fe);  
    return ret;
}

#if 0
static int
msp_bd_check_change(kdev_t dev)
{
    // TODO(miba) implement me!
    return 0;
}

static int
msp_bd_revalidate(kdev_t dev)
{
    // TODO(miba) implement me!
    return 0;
}
#endif

static void
msp_bd_request(request_queue_t *queue)
{
    /* just trigger userspace to fetch a new request from our queue */
    wakeup_userspace();
}

/* ------------------------------------------------------------------------- *
 * internal functions
 * ------------------------------------------------------------------------- */

static void
wakeup_userspace(void) {
    wake_up_interruptible(&adap.poll_wq);
}

static int
msp_bd_prepare_transfer()
{
    msp_dev_t *dev;
    u_long flags;
    int ret = -1;

    if (adap.req_state == REQ_STATE_WRITEDONE_PENDING) {
	adap.file_req.cmd = PP_MSP_WRITE_DONE;
	D(D_BLABLA, "Preparing WRITE_DONE\n");
	ret = 0;
        goto bail;
    }

    if (adap.current_req == NULL) {
	/* we need a new buffer, get next request */
	spin_lock_irqsave(&adap.dev_lock, flags);
	adap.current_req = elv_next_request(adap.rqueue);
	spin_unlock_irqrestore(&adap.dev_lock, flags);
	if (!adap.current_req) {
	    D(D_BLABLA, "No new request available\n");
	    goto bail;
	}
    }

    dev = adap.current_req->rq_disk->private_data;

    /* get the whole request (a number of buffers) from the client */
    switch (rq_data_dir(adap.current_req)) {
      case READ:
	  adap.file_req.dev_no		= dev->index;
	  adap.file_req.cmd		= PP_MSP_READ;
	  adap.file_req.length		= adap.current_req->hard_nr_sectors / dev->sect_divider;
	  adap.file_req.block_length	= dev->block_size;
	  adap.file_req.block_address	= adap.current_req->hard_sector / dev->sect_divider; 
	  break;

      case WRITE:
	  adap.file_req.dev_no		= dev->index;
	  adap.file_req.cmd		= PP_MSP_WRITE;
	  adap.file_req.length		= adap.current_req->hard_nr_sectors / dev->sect_divider;
	  adap.file_req.block_length	= dev->block_size;
	  adap.file_req.block_address	= adap.current_req->hard_sector / dev->sect_divider;  
	  break;

      default:
	  goto bail_cleanup;
    }

    adap.req_state = rq_data_dir(adap.current_req) == READ ? REQ_STATE_READ_PENDING : REQ_STATE_WRITE_PENDING;

    D(D_BLABLA,"Preparing req %p, cmd %d, sector %d, len %d count %d -> %d\n",
      adap.current_req, (int)rq_data_dir(adap.current_req), adap.file_req.block_address,
      adap.file_req.block_length, adap.file_req.length, 
      adap.file_req.block_length * adap.file_req.length);

    ret = 0;

bail:
    return ret;

bail_cleanup:
    msp_bd_cleanup_request(adap.current_req, BLK_REQ_ERROR);
    adap.current_req = NULL;
    return ret;
}

static int
msp_bd_transfer_done(void)
{
    struct request *req;
    u_long offset = 0, flags;
    int finished = 0;

    if (adap.req_state == REQ_STATE_WRITEDONE_PENDING) {
	/* just get the ack in this case */
	adap.req_state = REQ_STATE_IDLE;
	return 0;
    }

    req = adap.current_req;

    /* return error for all buffers */
    if (adap.file_resp.response == PP_MSP_FILE_DATA_BAD) {
	finished = !end_that_request_first(req, 0, req->hard_nr_sectors);
    } else {
	do {
	    switch (adap.file_req.cmd) {
		case PP_MSP_READ:
		    copy_from_user(req->buffer, adap.file_resp.data + offset, req->hard_cur_sectors * 512);
		    break;
		case PP_MSP_WRITE:
		    copy_to_user(adap.file_resp.data + offset, req->buffer, req->hard_cur_sectors * 512);
		    break;
		default:
		    ;
	    }
	    offset += req->hard_cur_sectors * 512;
	    finished = !end_that_request_first(req, 1, req->hard_cur_sectors);
	} while (!finished);
    }

    spin_lock_irqsave(&adap.dev_lock, flags);
    if (finished) {
	blkdev_dequeue_request(req);
	end_that_request_last(req);
    }
    spin_unlock_irqrestore(&adap.dev_lock, flags);

    adap.current_req = NULL;

    if (adap.file_req.cmd == PP_MSP_WRITE) {
	adap.req_state = REQ_STATE_WRITEDONE_PENDING;
    } else {
	adap.req_state = REQ_STATE_IDLE;
    }

    return 0;
}

static int
msp_bd_create_blockdevice(msp_dev_t *dev, u_long size_kb, u_long blksize, u_long sectsize)
{
    int ret = -1;
    u_long flags;

    spin_lock_irqsave(&adap.dev_lock, flags);
    if (dev->active) {
	D(D_ERROR, "Cannot create new block device while old one registered!\n");
	goto error_out;
    }

    if (bd_opt_read_ahead > MAX_NR_BLOCKS) {
	bd_opt_read_ahead = MAX_NR_BLOCKS;
    }

    if (bd_opt_max_blocks > MAX_NR_BLOCKS) {
	bd_opt_max_blocks = MAX_NR_BLOCKS;
    }

    if (sectsize % BASE_SECTOR_SIZE) {
	D(D_ERROR,"Sector size (%ld) must be multiple of %d\n", sectsize, BASE_SECTOR_SIZE);
	goto error_out;
    }

    dev->sect_divider = sectsize / BASE_SECTOR_SIZE;

    D(D_BLABLA, "Create blockdev %d, size=%ld kb, blksize=%ld, sectsize=%ld, read_ahead=%d, max_blocks=%d, sect_div=%d\n",
      dev->index, size_kb, blksize, sectsize, bd_opt_read_ahead, bd_opt_max_blocks, dev->sect_divider);

    /* set block device parameters */
#if 0
    read_ahead[MAJOR_NR] = bd_opt_read_ahead;   
    max_sectors[MAJOR_NR][dev->index] = bd_opt_max_blocks;
    blk_size[MAJOR_NR][dev->index] = size_kb;
    blksize_size[MAJOR_NR][dev->index] = blksize;
    hardsect_size[MAJOR_NR][dev->index] = sectsize;
#endif
    set_capacity(disks[dev->index], 2*size_kb);
//    struct block_device *bdev = bdget(MKDEV(MAJOR_NR, dev->index));
//    set_blocksize(bdev, blksize);
//    bdput(bdev);

    ret = 0;
    dev->active = 1;
    adap.devs_used++;

    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;

error_out:
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;
}

static int
msp_bd_remove_blockdevice(msp_dev_t *dev)
{
//    struct list_head *pos, *n;
    u_long flags;
    int ret = -1;

    if (!dev->active) {
	D(D_VERBOSE, "Blockdevice not registered, ignoring.\n");
	return 0;
    }

    D(D_VERBOSE, "Removing blockdevice %d\n", dev->index);

    spin_lock_irqsave(&adap.dev_lock, flags);
    dev->cleanup = 1;

    /* cleanup requests pending for this device in our queue */
#if 0
    list_for_each_safe(pos, n, &adap.req_list) {
	req_entry_t *req = list_entry(&pos, req_entry_t, listnode);
	if (DEVICE_NR(req->req->rq_dev) == dev->index) {
	    msp_bd_cleanup_request(req->req, BLK_REQ_ERROR);
	    list_del(pos);
	    kfree(req);
	}
    }

    fsync_dev(MKDEV(MAJOR_NR, dev->index));

    if (--adap.devs_used == 0) {
	blk_cleanup_queue(BLK_DEFAULT_QUEUE(MAJOR_NR));
	
	if ((ret = unregister_blkdev(MAJOR_NR, adap.name)) < 0) {
	    D(D_ERROR, "failed to unregister block dev driver (%d)\n", ret);
	    goto bail;
	}
    }
#endif

    ret = 0;
    dev->active = 0;
    dev->cleanup = 0;

//bail:
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;
}

static void
msp_bd_cleanup_request(struct request *req, int status)
{
    spin_lock(&adap.dev_lock);
    end_request(req, (status == 0));
    spin_unlock(&adap.dev_lock);
}
