/*****************************************************************************
 * msp_adapter.c
 *
 * Copyright (c) 2005 Peppercon AG, miba@peppercon.de
 *
 * Converts file io operations to mass storage protocol read/write requests
 *
 * Devices: /dev/msp_fe_x (b, 250, x) 'image' files containing the redirected devices
 *          /dev/msp_be   (c, 250, 0) backend connector for MSP thread
 *
 * TODO:
 * - use the msp adapter also for PPC
 * - convert to kernel 2.6
 *****************************************************************************/
#define MAJOR_NR		250
#define DRV_NAME		"msp_adapter"

#define log2(n)			ffz(~(n))

/* some defines which must exist before including blk.h */
#define DEVICE_NAME		DRV_NAME
#define DEVICE_NR(device)	MINOR(device)
#define DEVICE_REQUEST		msp_bd_request

#define BASE_SECTOR_SIZE	512
#define MAX_NR_BLOCKS		128
#define MAX_DEV_COUNT		PP_FEAT_USB_MASS_STORAGE_NO

#define BLK_REQ_ERROR		0
#define BLK_REQ_SUCCESS		1

#include <linux/blk.h>

#include <linux/fs.h>
#include <linux/ioctl.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/slab.h>
#include <linux/types.h>

#include <asm/bitops.h>
#include <asm/semaphore.h>

#include "msp_adapter.h"
#include "common_debug.h"

/* ------------------------------------------------------------------------- *
 * global constants, macros and datatypes
 * ------------------------------------------------------------------------- */

#ifdef SUCCESS
#  undef  SUCCESS
#endif
#define SUCCESS 0

typedef enum {
    REQ_STATE_IDLE,
    REQ_STATE_READ_PENDING,
    REQ_STATE_WRITE_PENDING,
    REQ_STATE_WRITEDONE_PENDING,
    REQ_STATE_DONE,
} req_state_t;

typedef struct {
    struct list_head listnode;
    struct request *req;
} req_entry_t;

/*
 * data structure per usage of the module (open/release)
 */
typedef struct {
} priv_data_t;

/*
 * global data used by all clients
 */
typedef struct {
    char			name[15];    
    u8				fe_connected;		/* connected clients */
    u8				be_connected;		/* connected backends (usually 1) */
    u8				wakeup_userspace;	
    wait_queue_head_t		poll_wq;

    req_state_t			req_state;		/* the current request in progress */
    pp_msp_file_request_t	file_req;		/* request data */
    pp_msp_file_response_t	file_resp;		/* response data */

    struct semaphore		sem_fe;			/* frontend lock */
    struct semaphore		sem_be;			/* backend lock */

    volatile u8			devs_used;
    spinlock_t			dev_lock;
    
    spinlock_t			req_list_lock;
    struct list_head		req_list;
    req_entry_t		       *current_req;
    u_long			nr_requests;
} msp_adap_t;

/*
 * data used per specific block device
 */
typedef struct {
    u8			index;   
    volatile u8		active;
    volatile u8		cleanup;
   
    u32			block_size;
    u32			block_count;   
    u8			sect_divider;
} msp_dev_t;

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */

static msp_adap_t adap = {
    name:		DRV_NAME,
    fe_connected:	0,
    be_connected:	0,
    wakeup_userspace:	0,
    poll_wq:            {},
    req_state:		REQ_STATE_IDLE,
    file_req:		{},
    file_resp:		{},
    sem_fe:		{},
    sem_be:		{},
    devs_used:		0,
    dev_lock:		SPIN_LOCK_UNLOCKED,
    req_list_lock:	SPIN_LOCK_UNLOCKED,
    req_list:		{},
    current_req:	NULL,
    nr_requests:	0,   
};

static const msp_dev_t msp_dev_init_tmpl = {
    index:		0,    
    active:		0,
    cleanup:		0,

    block_size:		0,
    block_count:	0,
    sect_divider:	1,
};
			
static msp_dev_t devs[MAX_DEV_COUNT];

int bd_opt_read_ahead  = 32;
int bd_opt_max_blocks  = 128;
MODULE_PARM(bd_opt_read_ahead, "i");
MODULE_PARM_DESC(bd_opt_read_ahead, "Number of blocks to read ahead (default 32, max 128)");
MODULE_PARM(bd_opt_max_blocks, "i");
MODULE_PARM_DESC(bd_opt_max_blocks, "Number of blocks per mass storage request (default 128, max 128)");

/* ------------------------------------------------------------------------- *
 * function prototypes
 * ------------------------------------------------------------------------- */

int         init_module(void);
void        cleanup_module(void);

/* character device operations */
int         msp_init(void);
static void msp_cleanup(void);
static int  msp_be_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg);
static int  msp_be_open(struct inode *inode, struct file *file);
static int  msp_be_release(struct inode *inode, struct file *file);
static unsigned int msp_be_poll(struct file *file, struct poll_table_struct *wait);

/* block device operations */
static int msp_bd_open(struct inode *inode, struct file *file);
static int msp_bd_release(struct inode *inode, struct file *file);
static int msp_bd_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg);
static int msp_bd_check_change(kdev_t dev);
static int msp_bd_revalidate(kdev_t dev);

/* internal functions */
static void wakeup_userspace(void);

static int msp_bd_prepare_transfer(void);
static int msp_bd_transfer_done(void);
static void msp_bd_cleanup_request(struct request *req, int status);

static int msp_bd_create_blockdevice(msp_dev_t *dev, u_long size_kb, u_long blksize, u_long sectsize);
static int msp_bd_remove_blockdevice(msp_dev_t *dev);

static int msp_bd_alloc_blkdev_arrays(void);
static void msp_bd_cleanup_blkdev_arrays(void);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

#ifdef MODULE
MODULE_AUTHOR("miba@peppercon.de");
MODULE_DESCRIPTION("MSP adapter driver");

/*
 * Initialize the module
 */
int init_module(void)
{
    return msp_init();
}

/*
 * Cleanup
 */
void cleanup_module(void)
{
    msp_cleanup();
}
#endif	/* MODULE */

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations msp_be_ops = {
    owner:   THIS_MODULE,
    ioctl:   msp_be_ioctl,
    poll:    msp_be_poll,
    open:    msp_be_open,
    release: msp_be_release,
    read:    NULL,
    write:   NULL,
    llseek:  NULL
};

static struct block_device_operations msp_bdops = {
    open:		msp_bd_open,
    release:		msp_bd_release,
    ioctl:		msp_bd_ioctl,
    check_media_change: msp_bd_check_change,
    revalidate:		msp_bd_revalidate,
};

/* ------------------------------------------------------------------------- *
 * driver operations
 * ------------------------------------------------------------------------- */

int
msp_init(void)
{
    int ret = SUCCESS, r, i;

    init_waitqueue_head(&adap.poll_wq);

    init_MUTEX(&adap.sem_fe);
    init_MUTEX(&adap.sem_be);

    INIT_LIST_HEAD(&adap.req_list);

    /* ---- pre-init dev specific data  ------------------------------ */
    for (i = 0; i < MAX_DEV_COUNT; i++) {
	devs[i] = msp_dev_init_tmpl;
	devs[i].index = i;
    }
    
    if (msp_bd_alloc_blkdev_arrays() != 0) {
	D(D_ERROR, "failed to init kernel blockdev data arrays\n");
	ret = -ENOMEM;
	goto error_out;
    }
        
    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(MAJOR_NR, adap.name, &msp_be_ops)) < 0) {
	D(D_ERROR, "failed to register char dev driver (%d)\n", r);
	ret = -ENODEV;
	goto error_out;
    }

    D(D_NOTICE, "msp options: read_ahead=%d, max_blocks=%d\n",
      bd_opt_read_ahead, bd_opt_max_blocks);
        
    return ret;
    
 error_out:
    msp_cleanup();
    return ret;
}

static void
msp_cleanup(void)
{
    int r;

    msp_bd_cleanup_blkdev_arrays();
    
    if ((r = unregister_chrdev(MAJOR_NR, adap.name)) < 0) {
	D(D_ERROR, "failed to unregister char dev driver (%d)\n", r);
    }

    D(D_ALWAYS, "driver cleanup done\n");
}

static int
msp_be_open(struct inode *inode, struct file *file)
{
    priv_data_t *pdata;
    unsigned int minor = MINOR(inode->i_rdev);
    int ret = SUCCESS;

    if (minor != 0) {
	  D(D_ERROR, "Unsupported minor number %d\n", minor);
	  ret = -EFAULT;
	  goto bail;
    }
    
    if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
    adap.be_connected = 1;
    D(D_VERBOSE, "New MSP BE client\n");
    up(&adap.sem_be);
    
    if (file->private_data == NULL) {
	pdata = kmalloc(sizeof(priv_data_t), GFP_KERNEL);
	
	file->private_data = (void*) pdata;
    }

 bail:
    return ret;
}

static int
msp_be_release(struct inode *inode, struct file *file)
{
    priv_data_t *pdata = (priv_data_t*) file->private_data;
    int i;

    if (pdata) {
	if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
	adap.be_connected = 0;
	D(D_VERBOSE, "Removed MSP BE client\n");
	up(&adap.sem_be);

	kfree(pdata);
    }

    for (i = 0; i < MAX_DEV_COUNT; i++) {
	msp_dev_t *dev = &devs[i];
	msp_bd_remove_blockdevice(dev);
    }
    
    return SUCCESS;   
}

static int
msp_be_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg)
{
    pp_msp_file_info_t file_info;
    int ret = SUCCESS;

    if (down_interruptible(&adap.sem_be) < 0) return -EINTR;
    switch (cmd) {
      case PP_MSP_IOC_GETREQUEST:
	  if (msp_bd_prepare_transfer() != 0) {
	      ret = -ENODATA;
	      goto bail;
	  }

	  if (copy_to_user((char*)arg, (u8*)&adap.file_req, sizeof(pp_msp_file_request_t))) {
	      D(D_ERROR, "cannot copy to user\n");
	      ret = -EFAULT;
	      goto bail;
	  }
	  break;
	  
      case PP_MSP_IOC_SETRESPONSE:
	  if (copy_from_user(&adap.file_resp, (pp_msp_file_response_t*)arg,
			     sizeof(pp_msp_file_response_t))) {
	      D(D_ERROR, "cannot copy from user file_resp\n");
	      ret = -EFAULT;
	      goto bail;
	  }
	  D(D_BLABLA, "PP_MSP_IOC_SETRESPONSE: dev=%d, response=%d, length=%d\n",
	    adap.file_resp.dev_no, adap.file_resp.response, adap.file_resp.length);

	  if (msp_bd_transfer_done() != 0) {
	      D(D_ERROR, "Error during finish of transfer\n");
	      ret = -EFAULT;
	      goto bail;
	  }	  
	  break;

      case PP_MSP_IOC_SETFILEINFO:
	  if (copy_from_user(&file_info, (pp_msp_file_info_t*)arg, sizeof(pp_msp_file_info_t))) {
	      D(D_ERROR, "cannot copy file info\n");
	      ret = -EFAULT;
	      goto bail;
	  }

	  if (file_info.block_length == 0 && file_info.block_count == 0) {
	      D(D_VERBOSE, "Cleaning up backend\n");

	      msp_bd_remove_blockdevice(&devs[file_info.dev_no]);
	  } else {
	      if ((0x01 << log2(file_info.block_length)) != file_info.block_length) {
		  D(D_ERROR, "Unsupported block size of %d, must be power of 2\n",
		    file_info.block_length);
	      }

	      devs[file_info.dev_no].block_size = file_info.block_length;
	      devs[file_info.dev_no].block_count = file_info.block_count;
	      
	      D(D_VERBOSE, "PP_MSP_IOC_SETFILEINFO: new block_size=%u, block_count=%u\n",
		file_info.block_length, file_info.block_count);
	      
	      if (msp_bd_create_blockdevice(&devs[file_info.dev_no],
					    (uint32_t)(((uint64_t)file_info.block_length * (uint64_t)file_info.block_count) / (uint64_t)1024),
					    file_info.block_length, file_info.block_length) != 0)
		  {
		      ret = -ENODEV;
		      D(D_ERROR, "Could not create new block device\n");
		      goto bail;
		  }
	  }
	  
	  break;
	  
      default:
	  D(D_ERROR, "Invalid ioctl request\n");
	  break;
    }

 bail:
    up(&adap.sem_be);
    return ret;
}

static unsigned int
msp_be_poll(struct file *file, struct poll_table_struct *wait)
{
    int ret = SUCCESS;
    u_long flags;
   
    spin_lock_irqsave(&adap.req_list_lock, flags);
    if (adap.nr_requests == 0 && adap.wakeup_userspace == 0 &&
	adap.req_state == REQ_STATE_IDLE) {
	/* this will not wait here, waiting takes place after returning from this function */	
	poll_wait(file, &adap.poll_wq, wait);
    } else {
	if (adap.wakeup_userspace) {
	    adap.wakeup_userspace = 0;
	}
	ret = POLLIN | POLLRDNORM;
    }
    spin_unlock_irqrestore(&adap.req_list_lock, flags);
    
    return ret;
}

/* ------------------------------------------------------------------------- *
 * driver operations (block device)
 * ------------------------------------------------------------------------- */

static int
msp_bd_open(struct inode *inode, struct file *file)
{
    unsigned int minor = MINOR(inode->i_rdev);
    int ret = SUCCESS;

    if (minor >= MAX_DEV_COUNT) {
	D(D_ERROR, "Minor number out of range (max device count=%d)\n",
	  MAX_DEV_COUNT);
    }
    
    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;

    if (adap.be_connected == 0) {
	D(D_ERROR, "No backend connected, open refused\n");
	ret = -ENODEV;
	up(&adap.sem_fe);
	goto bail;
    }

    adap.fe_connected++;
    D(D_VERBOSE, "New MSP FE blockdev client (count=%d)\n", adap.fe_connected);
    up(&adap.sem_fe);

    MOD_INC_USE_COUNT;
    
 bail:    
    return ret;
}

static int
msp_bd_release(struct inode *inode, struct file *file)
{
    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;

    adap.fe_connected--;
    D(D_VERBOSE, "Removed MSP FE blockdev client (count=%d)\n", adap.fe_connected);
    up(&adap.sem_fe);

    if (MOD_IN_USE) {
	MOD_DEC_USE_COUNT;
    }

    return SUCCESS;       
}

static int
msp_bd_ioctl(struct inode *inode, struct file *file, uint cmd, u_long arg)
{
    int ret = -EINVAL;

    if (down_interruptible(&adap.sem_fe) < 0) return -EINTR;
    
    D(D_ERROR, "Invalid ioctl request\n");

    up(&adap.sem_fe);  
    return ret;
}

static int
msp_bd_check_change(kdev_t dev)
{
    // TODO(miba) implement me!
    return 0;
}

static int
msp_bd_revalidate(kdev_t dev)
{
    // TODO(miba) implement me!
    return 0;
}

static void
msp_bd_request(request_queue_t *queue)
{
    req_entry_t* req_entry;
    u_long flags;
    int devno;
    
    spin_lock_irqsave(&adap.dev_lock, flags);
   
    while (!list_empty(&queue->queue_head)) {
	struct request *req;
	msp_dev_t *dev;
	
	req = blkdev_entry_next_request(&queue->queue_head);

	/* check device number */
	devno = DEVICE_NR(req->rq_dev);
	if (devno >= MAX_DEV_COUNT || devs[devno].active == 0 || devs[devno].cleanup == 1) {
	    D(D_VERBOSE, "Ignoring request for device %d\n", devno);
	    goto bail;
	}
	dev = &devs[devno];
    
	/* remove req from kernel queue, means, that we'll take care of it */	
	blkdev_dequeue_request(req);

	// TODO(miba) replace with slab allocator
	req_entry = kmalloc(sizeof(req_entry_t), GFP_ATOMIC);
	req_entry->req = req;

	/* NOTE(miba): requests always arrive indexed with 512 bytes, no matter what
	   hardsect_size, but kernel will make sure correct buffers are available
	   for the specified one */
	
	D(D_BLABLA, "request on %d: cmd %d, %d sized sec %ld, count %ld -> sec %ld, count %ld\n",
	  dev->index, req->cmd, BASE_SECTOR_SIZE, req->hard_sector, req->hard_nr_sectors,
	  req->hard_sector / dev->sect_divider, req->hard_nr_sectors / dev->sect_divider);

	spin_lock(&adap.req_list_lock);
	list_add_tail(&req_entry->listnode, &adap.req_list);
	adap.nr_requests++;
	spin_unlock(&adap.req_list_lock);

	wakeup_userspace();
    }

 bail:
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return;
}

/* ------------------------------------------------------------------------- *
 * internal functions
 * ------------------------------------------------------------------------- */

static void
wakeup_userspace(void) {
    adap.wakeup_userspace = 1;
    wake_up_interruptible(&adap.poll_wq);
}

static int
msp_bd_prepare_transfer()
{
    msp_dev_t *dev;
    u_long flags;
    int ret = -1, devno;

    if (adap.req_state == REQ_STATE_WRITEDONE_PENDING) {
	adap.file_req.cmd = PP_MSP_WRITE_DONE;
	D(D_BLABLA, "Preparing WRITE_DONE\n");
	ret = 0;
        goto bail;
    }
    
    if (adap.current_req == NULL) {
	/* we need a new buffer, get next request */
	spin_lock_irqsave(&adap.req_list_lock, flags);
	if (!list_empty(&adap.req_list)) {
	    adap.current_req = list_entry(adap.req_list.next, req_entry_t, listnode);
	    list_del(adap.req_list.next);
	    adap.nr_requests--;
	} else {
	    D(D_BLABLA, "No new request available\n");
	    spin_unlock_irqrestore(&adap.req_list_lock, flags);
	    goto bail;
	}
	spin_unlock_irqrestore(&adap.req_list_lock, flags);
    }

    /* check device number */
    devno = DEVICE_NR(adap.current_req->req->rq_dev);
    if (devno >= MAX_DEV_COUNT || devs[devno].active == 0) {
	D(D_VERBOSE, "Unsupported request for %d, cleaning\n", devno);
	goto bail_cleanup;
    }
    dev = &devs[devno];
    
    /* get the whole request (a number of buffers) from the client */
    switch (adap.current_req->req->cmd) {
      case READ:
	  adap.file_req.dev_no		= dev->index;
	  adap.file_req.cmd		= PP_MSP_READ;
	  adap.file_req.length		= adap.current_req->req->hard_nr_sectors / dev->sect_divider;
	  adap.file_req.block_length	= dev->block_size;
	  adap.file_req.block_address	= adap.current_req->req->hard_sector / dev->sect_divider; 
	  break;
	  
      case WRITE:
	  adap.file_req.dev_no		= dev->index;
	  adap.file_req.cmd		= PP_MSP_WRITE;
	  adap.file_req.length		= adap.current_req->req->hard_nr_sectors / dev->sect_divider;
	  adap.file_req.block_length	= dev->block_size;
	  adap.file_req.block_address	= adap.current_req->req->hard_sector / dev->sect_divider;  
	  break;
	  
      default:
	  D(D_ERROR, "Invalid request cmd %d\n", adap.current_req->req->cmd);
	  goto bail_cleanup;
    }

    adap.req_state = (adap.current_req->req->cmd == READ) ? REQ_STATE_READ_PENDING : REQ_STATE_WRITE_PENDING;
    
    D(D_BLABLA, "Preparing req %p, cmd %d, sector %d, len %d count %d -> %d\n",
      adap.current_req->req, adap.current_req->req->cmd, adap.file_req.block_address,
      adap.file_req.block_length, adap.file_req.length, 
      adap.file_req.block_length * adap.file_req.length);
  
    ret = 0;
    
 bail:
    return ret;

 bail_cleanup:
    msp_bd_cleanup_request(adap.current_req->req, BLK_REQ_ERROR);
    kfree(adap.current_req);
    adap.current_req = NULL;
    return ret;
}

static int
msp_bd_transfer_done(void)
{
    msp_dev_t *dev;
    struct request *req;
    struct buffer_head *buf, *tmp;
    u_long offset = 0, flags;
    int ret = -1, devno;
    
    if (adap.req_state == REQ_STATE_WRITEDONE_PENDING) {
	/* just get the ack in this case */
	adap.req_state = REQ_STATE_IDLE;
	ret = 0;
	goto bail;
    }

    req = adap.current_req->req;
    buf = req->bh;

    devno = DEVICE_NR(req->rq_dev);
    if (devno >= MAX_DEV_COUNT || devs[devno].active == 0 || devno != adap.file_resp.dev_no) {
	D(D_ERROR, "Got response for wrong or unsupported device %d, cleaning\n", devno);
	msp_bd_cleanup_request(req, BLK_REQ_ERROR);
	goto bail;
    }
    dev = &devs[devno];
        
    /* return error for all buffers */
    if (adap.file_resp.response == PP_MSP_FILE_DATA_BAD) {
	while (buf != NULL) {
	    tmp = buf->b_reqnext;
	    buf->b_reqnext = NULL;
	    buf->b_end_io(buf, BLK_REQ_ERROR);
	    
	    buf = tmp;
	}
    } else {
        while (buf != NULL) {
	    u_long cur_nr = (buf->b_size / dev->block_size) * dev->sect_divider;
	    
	    if (offset + buf->b_size > adap.file_resp.length) {
		D(D_ERROR, "Buffer offset (%ld) + size (%d) exceeds response length %d!\n",
		  offset, buf->b_size, adap.file_resp.length);
		
		goto bail;
	    }

	    /*
	    D(D_BLABLA, "Finish buf %p @ offset %ld, sector %ld, count %ld, left %ld (bufsize %d)\n",
	      buf, offset, req->sector, cur_nr, req->hard_nr_sectors, buf->b_size);
	    */

	    switch (adap.file_req.cmd) {
	      case PP_MSP_READ:
		  if (copy_from_user(buf->b_data, adap.file_resp.data + offset, buf->b_size)) {
		      D(D_ERROR, "cannot copy from user data buffer\n");
		      goto bail;
		  }
		  break;
	      case PP_MSP_WRITE:
		  if (copy_to_user(adap.file_resp.data + offset, buf->b_data, buf->b_size)) {
		      D(D_ERROR, "cannot copy to user data buffer\n");
		      goto bail;
		  }
	      default:
		  break;
	    }
	
	    offset += buf->b_size;

	    /* give current buffer back to kernel, get next one, update request */
	    tmp = buf->b_reqnext;
	    buf->b_reqnext = NULL;
	    buf->b_end_io(buf, BLK_REQ_SUCCESS);
	
	    buf = tmp;
		
	    req->hard_nr_sectors -= cur_nr;
	    req->nr_sectors -= cur_nr;
	    req->hard_sector += cur_nr;
	    req->sector += cur_nr;
	}
    }
	
    spin_lock_irqsave(&io_request_lock, flags);
    end_that_request_last(req);
    spin_unlock_irqrestore(&io_request_lock, flags);
    
    kfree(adap.current_req);
    adap.current_req = NULL;	

    ret = 0;
    
    if (adap.file_req.cmd == PP_MSP_WRITE) {
	adap.req_state = REQ_STATE_WRITEDONE_PENDING;
    } else {
	adap.req_state = REQ_STATE_IDLE;
    }
    
 bail:
    //wakeup_userspace();
    return ret;
}

static int
msp_bd_create_blockdevice(msp_dev_t *dev, u_long size_kb, u_long blksize, u_long sectsize)
{
    int ret = -1;
    u_long flags;

    spin_lock_irqsave(&adap.dev_lock, flags);
    if (dev->active) {
	D(D_ERROR, "Cannot create new block device while old one registered!\n");
	goto error_out;
    }
   
    if (bd_opt_read_ahead > MAX_NR_BLOCKS) {
	bd_opt_read_ahead = MAX_NR_BLOCKS;
    }

    if (bd_opt_max_blocks > MAX_NR_BLOCKS) {
	bd_opt_max_blocks = MAX_NR_BLOCKS;
    }

    if (sectsize % BASE_SECTOR_SIZE) {
	D(D_ERROR,"Sector size (%ld) must be multiple of %d\n", sectsize, BASE_SECTOR_SIZE);
	goto error_out;
    }
    
    dev->sect_divider = sectsize / BASE_SECTOR_SIZE;
    
    D(D_NOTICE, "Create blockdev %d, size=%ld kb, blksize=%ld, sectsize=%ld, read_ahead=%d, max_blocks=%d, sect_div=%d\n",
      dev->index, size_kb, blksize, sectsize, bd_opt_read_ahead, bd_opt_max_blocks, dev->sect_divider);
   
    /* set block device parameters */
    read_ahead[MAJOR_NR] = bd_opt_read_ahead;   
    max_sectors[MAJOR_NR][dev->index] = bd_opt_max_blocks;
    blk_size[MAJOR_NR][dev->index] = size_kb;
    blksize_size[MAJOR_NR][dev->index] = blksize;
    hardsect_size[MAJOR_NR][dev->index] = sectsize;

    if (adap.devs_used == 0) {
	if ((ret = register_blkdev(MAJOR_NR, adap.name, &msp_bdops)) < 0) {
	    D(D_ERROR, "failed to register block dev driver (%d)\n", ret);
	    goto error_out;
	}

	blk_init_queue(BLK_DEFAULT_QUEUE(MAJOR_NR), msp_bd_request);
	blk_queue_headactive(BLK_DEFAULT_QUEUE(MAJOR_NR), 0);
    }

    ret = 0;
    dev->active = 1;
    adap.devs_used++;
    
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;
    
 error_out:    
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;
}

static int
msp_bd_remove_blockdevice(msp_dev_t *dev)
{      
    struct list_head *pos, *n;
    u_long flags;
    int ret = -1;
    
    if (!dev->active) {
	D(D_VERBOSE, "Blockdevice not registered, ignoring.\n");
	return 0;
    }
    
    D(D_VERBOSE, "Removing blockdevice %d\n", dev->index);
    
    spin_lock_irqsave(&adap.dev_lock, flags);
    dev->cleanup = 1;

    /* cleanup requests pending for this device in our queue */
    list_for_each_safe(pos, n, &adap.req_list) {
	req_entry_t *req = list_entry(&pos, req_entry_t, listnode);
	if (DEVICE_NR(req->req->rq_dev) == dev->index) {
	    msp_bd_cleanup_request(req->req, BLK_REQ_ERROR);
	    list_del(pos);
	    kfree(req);
	}
    }
    
    fsync_dev(MKDEV(MAJOR_NR, dev->index));

    if (--adap.devs_used == 0) {
	blk_cleanup_queue(BLK_DEFAULT_QUEUE(MAJOR_NR));
	
	if ((ret = unregister_blkdev(MAJOR_NR, adap.name)) < 0) {
	    D(D_ERROR, "failed to unregister block dev driver (%d)\n", ret);
	    goto bail;
	}
    }

    ret = 0;
    dev->active = 0;
    dev->cleanup = 0;
    
 bail:
    spin_unlock_irqrestore(&adap.dev_lock, flags);
    return ret;
}

static int
msp_bd_alloc_blkdev_arrays(void)
{
    if ((max_sectors[MAJOR_NR] = kmalloc(sizeof(int), GFP_KERNEL)) == NULL) {
	goto error_out;
    }
    
    if ((blk_size[MAJOR_NR] = kmalloc(sizeof(int), GFP_KERNEL)) == NULL) {
	goto error_out;
    }
    
    if ((blksize_size[MAJOR_NR] = kmalloc(sizeof(int), GFP_KERNEL)) == NULL) {
	goto error_out;
    }
    
    if ((hardsect_size[MAJOR_NR] = kmalloc(sizeof(int), GFP_KERNEL)) == NULL) {
	goto error_out;
    }
   
    return 0;
    
 error_out:
    msp_bd_cleanup_blkdev_arrays();
    
    return -1;
}

static void
msp_bd_cleanup_blkdev_arrays(void)
{
    if (max_sectors[MAJOR_NR] != NULL) {
	kfree(max_sectors[MAJOR_NR]);
	max_sectors[MAJOR_NR] = NULL;
    }
    
    if (blk_size[MAJOR_NR] != NULL) {
	kfree(blk_size[MAJOR_NR]);
	blk_size[MAJOR_NR] = NULL;
    }
    
    if (blksize_size[MAJOR_NR] != NULL) {
	kfree(blksize_size[MAJOR_NR]);
	blksize_size[MAJOR_NR] = NULL;
    }
    
    if (hardsect_size[MAJOR_NR] != NULL) {
	kfree(hardsect_size[MAJOR_NR]);
	hardsect_size[MAJOR_NR] = NULL;
    }
}   

static void
msp_bd_cleanup_request(struct request *req, int status)
{
    while (end_that_request_first(req, status, adap.name));
    spin_lock(&io_request_lock);
    end_that_request_last(req);
    spin_unlock(&io_request_lock);
}
