/**
 * pp_pwmtach.h
 *
 * Top-Level PWM/Tacho kernel module
 *
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>
 *			  Christian Krause <chkr@peppercon.de>
 *                        Michael Baumann <miba@peppercon.de>
 *
 * TODO
 * -core may only have 1 prescale for all tachos/pwms, reflect that in /proc
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <asm/io.h>
#include <asm/semaphore.h>
#include <asm/atomic.h>

#include <pp_pwmtach.h>
#include "pwmtach_common.h"
#include "pwmtach_core.h"

#ifdef PP_BOARD_KIRA
# include "pwmtach_core_faraday.h"
#else
# include "pwmtach_core_peppercon.h"
#endif /* !PP_BOARD_KIRA */

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */

#define PWMTACH_MAJOR		((uint8_t)248)
#define PWMTACH_MAX_DEVICES	2
#define MAX_PROC_NAME_SIZE	16

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

/**
 * init data to build a list of supported cores
 */
typedef struct {
    uint8_t minor;	/* device minor number for this core */
    uint8_t core_nr;	/* core number, unique for a core/driver type */

    /* init/cleanup functions, determine core to be used */
    core_init_fn_t	init;
} pwmtach_core_init_t;

struct pwmtach_dev_s {
    int			   initialized;
    atomic_t               refcnt;
    pwmtach_core_t        *core;	/* core specific data and operations */
    struct semaphore	   mtx;		/* device lock */
    struct proc_dir_entry *pde;		/* root proc entry */
    struct proc_data_s	   *proc_data_pwm;
    struct proc_data_s	   *proc_data_tacho;
};

typedef struct pwmtach_dev_s pwmtach_dev_t;

struct proc_data_s {
    int nr;
    pwmtach_dev_t *dev;
    struct proc_dir_entry *pde;

    uint8_t ppr;			/* save pulses_per_rev for proc access */
};

typedef struct proc_data_s proc_data_t;

/* file descriptor private data */
typedef struct {
    pwmtach_dev_t *dev;
} pwmtach_file_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

/**
 * list of cores to use, pwm/tacho device list gets build from that
 */
pwmtach_core_init_t core_init_list[] = {
#ifdef PP_BOARD_KIRA
    { 0, 0, pwmtach_core_faraday_init },
#else
    { 0, 0, pwmtach_core_peppercon_init },
#endif
    { 0, 0, NULL }
};

static pwmtach_dev_t	pwmtach_devs[PWMTACH_MAX_DEVICES];

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

int    init_module(void);
void   cleanup_module(void);

static int  pwmtach_init(void);
static void pwmtach_cleanup(void);

static int  pwmtach_open(struct inode *inode, struct file *file);
static int  pwmtach_release(struct inode *inode, struct file *file);
static int  pwmtach_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg);

static int pwm_proc_read_freq(char *page, char **start, off_t offset, int count, int *eof, void *data);
static int pwm_proc_read_duty(char *page, char **start, off_t offset, int count, int *eof, void *data);
static int pwm_proc_write_freq(struct file*file, const char *buffer, unsigned long count, void* data);
static int pwm_proc_write_duty(struct file*file, const char *buffer, unsigned long count, void* data);

static int tacho_proc_read_rpm(char *page, char **start, off_t offset, int count, int *eof, void *data);
static int tacho_proc_read_ppr(char *page, char **start, off_t offset, int count, int *eof, void *data);
static int tacho_proc_write_ppr(struct file*file, const char *buffer, unsigned long count, void* data);

static int init_proc_interface(pwmtach_dev_t *dev, int count);
static void cleanup_proc_interface(pwmtach_dev_t *dev, int count);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("rwa@peppercon.de, miba@peppercon.de");
MODULE_DESCRIPTION("Linux kernel module for PWM/Tacho interfaces");

/*
 * Initialize the module
 */
int init_module(void)
{
    return pwmtach_init();
}

/*
 * Cleanup module
 */
void cleanup_module(void)
{
    pwmtach_cleanup();
}

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations pwmtach_ops = {
    owner:	THIS_MODULE,
    open:	pwmtach_open,
    release:	pwmtach_release,
    ioctl:	pwmtach_ioctl
};

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

static int
pwmtach_init(void)
{
    pwmtach_core_init_t* init_data = core_init_list;
    int ret = SUCCESS, nr_cores = 0, r, i;
    pwmtach_dev_t *dev;

    /* ---- register the character device ------------------------------ */
    if ((r = register_chrdev(PWMTACH_MAJOR, NAME, &pwmtach_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto bail;
    }

    /* ---- initialize all cores --------------------------------------- */
    memset(pwmtach_devs, 0, sizeof(pwmtach_devs));
    
    while (init_data && init_data->init != NULL) {
	pwmtach_core_t *core;

	dev = &pwmtach_devs[init_data->minor];

	if (dev->initialized) {
	    WARN("Minor nr %d already initialized, strange, check init_data!",
		 init_data->minor);
	}

	atomic_set(&dev->refcnt, 0);
	init_MUTEX(&dev->mtx);

	if ((core = kmalloc(sizeof(pwmtach_core_t), GFP_KERNEL)) == NULL) {
	    ERR("Could not alloc memory for core");
	    ret = -ENOMEM;
	    goto bail;
	}
	memset(core, 0, sizeof(pwmtach_core_t));

	/* connect core and device */
	dev->core = core;
	core->dev = dev;

	if ((r = init_data->init(core, init_data->core_nr)) != 0) {
	    ERR("Could not intialize core");
	    ret = -ENODEV;
	    goto bail;
	}

	/* check if core did set all mandatory ops */
	if (((core->caps & PWMTACH_CAPS_HAVE_PWM)   && !(core->pwm_set_freq &&
							 core->pwm_get_freq &&
							 core->pwm_set_duty &&
							 core->pwm_get_duty))     ||
	    ((core->caps & PWMTACH_CAPS_HAVE_TACHO) && !core->tach_get_rpm))
	    {
		ERR("Core did not define all mandatory ops");
		ret = -ENODEV;
		goto bail;
	    }

	if (init_proc_interface(dev, nr_cores) != 0) {
	    ERR("Could not initialize /proc interface");
	    ret = -ENODEV;
	    goto bail;
	}

	/* set PWM for this core to 100% */
	for (i = 0; i < core->pwm_count; i++) {
	    core->pwm_set_duty(core, i, 100);
	}
	
	dev->initialized = 1;
	
	init_data++;
	nr_cores++;
    }

    if (nr_cores == 0) {
	ERR("Could not initialize any core");
	ret = -ENODEV;
	goto bail;
    } else {
	INFO("Initialized number cores: %d", nr_cores);
    }   

 bail:
    if (ret != SUCCESS) {
	pwmtach_cleanup();
    }
    return ret;

}
  
static void
pwmtach_cleanup(void)
{
    int i = 0;
    int r;
    
    while (i < PWMTACH_MAX_DEVICES) {
	pwmtach_dev_t *dev = &pwmtach_devs[i];

	cleanup_proc_interface(dev, i);

	if (dev->core && dev->core->cleanup) {
	    dev->core->cleanup(dev->core);	    
	}

	if (dev->core != NULL) {
	    kfree(dev->core);
	}

	i++;
    }

    if ((r = unregister_chrdev(PWMTACH_MAJOR, NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }    
}

/* ------------------------------------------------------------------------- *
 * the driver operations 
 * ------------------------------------------------------------------------- */

static int
pwmtach_open(struct inode *inode, struct file *file)
{
    int ret = SUCCESS;
    pwmtach_file_t *priv;
    pwmtach_dev_t *dev;
    int minor = MINOR(inode->i_rdev);
    int new_refcnt;
   
    if (minor < 0 || minor >= PWMTACH_MAX_DEVICES) {
	ERR("Unsupported PWM/Tacho device (minor number out of range)");
	return -ENODEV;
    }

    dev = &pwmtach_devs[minor];
    if (!dev->initialized) {
	ERR("PWM/Tacho device was not initialized");
	return -ENODEV;
    }

    if ((priv = kmalloc(sizeof(pwmtach_file_t), GFP_KERNEL)) == NULL) {
	ERR("Could not allocate memory for file data");
	return -ENOMEM;
    }

    memset(priv, 0, sizeof(pwmtach_file_t));
    priv->dev = dev;
    file->private_data = priv;   
    
    if (down_interruptible(&dev->mtx) < 0) return -EINTR;

    new_refcnt = atomic_inc_return(&dev->refcnt);
    if (new_refcnt == 1) {
	/* initialization on first open */
    }

    if (dev->core->open && dev->core->open(dev->core, new_refcnt) != 0) {
	ERR("Core open failed");
	goto bail;
    }

    DBG("PWM/Tacho device (minor %d) opened", minor);
    
 bail:
    up(&dev->mtx);
    return ret;
    
}

static int
pwmtach_release(struct inode *inode, struct file *file)
{
    pwmtach_file_t *priv = (pwmtach_file_t*) file->private_data;
    pwmtach_dev_t *dev = priv->dev;
    pwmtach_core_t *core = dev->core;
    int ret = SUCCESS;
    int new_refcnt;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;

    new_refcnt = atomic_dec_return(&dev->refcnt);
    kfree(priv); file->private_data = NULL;
    
    if (core->release) core->release(core, new_refcnt);

    if (new_refcnt == 0) {	
	/* last reference cleanup */
	int i;

	/* set PWM for this core to 100%, to be safe */
	for (i = 0; i < core->pwm_count; i++) {
	    core->pwm_set_duty(core, i, 100);
	}
    }

    DBG("PWM/Tacho device (minor %d) closed", MINOR(inode->i_rdev));
    up(&dev->mtx);
    return ret;    
}

static int
pwmtach_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    pwmtach_file_t *priv = (pwmtach_file_t*) file->private_data;
    pwmtach_dev_t *dev = priv->dev;
    pwmtach_core_t *core = dev->core;
    int ret = SUCCESS;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;

    switch (cmd) {
      case PP_PWMTACH_IOCTL_TACH_GET_RPM:
	  {
	      pp_tach_data_t tach_data;

	      /* get data from userspace */
	      if (copy_from_user(&tach_data, (char *)arg, sizeof(tach_data))) return -EFAULT;

	      if (tach_data.index >= core->tacho_count) {
		  return -EINVAL;
	      }
	      
	      if (core->tach_get_rpm(core, tach_data.index, tach_data.pulses_per_rev,
				     &tach_data.revs_per_minute) != 0) {
		  ret = -EIO;
		  break;
	      }
	  
	      /* put data into userspace */
	      if (copy_to_user((char *)arg, &tach_data, sizeof(tach_data))) return -EFAULT;
	      
	      break;
	  }
	  
      case PP_PWMTACH_IOCTL_PWM_SET_PARAMS:
	  {
	      pp_pwm_data_t pwm_data;

	      /* get data from userspace */
	      if (copy_from_user(&pwm_data, (char *)arg, sizeof(pwm_data))) return -EFAULT;

	      if (pwm_data.index >= core->pwm_count) {
		  return -EINVAL;
	      }
	      
	      if (core->pwm_set_freq(core, pwm_data.index, pwm_data.frequency)  != 0 ||
		  core->pwm_set_duty(core, pwm_data.index, pwm_data.duty_cycle) != 0) {
		  ret = -EIO;
	      }
	      
	      break;
	  }
	    
      case PP_PWMTACH_IOCTL_PWM_GET_PARAMS:
	  {
	      pp_pwm_data_t pwm_data;

	      /* get data from userspace */
	      if (copy_from_user(&pwm_data, (char *)arg, sizeof(pwm_data))) return -EFAULT;

	      if (pwm_data.index >= core->pwm_count) {
		  return -EINVAL;
	      }
      
	      if (core->pwm_get_freq(core, pwm_data.index, &pwm_data.frequency)  != 0 ||
		  core->pwm_get_duty(core, pwm_data.index, &pwm_data.duty_cycle) != 0) {
		  ret = -EIO;
		  break;
	      }
	      
	      /* put data into userspace */
	      if (copy_to_user((char *)arg, &pwm_data, sizeof(pwm_data))) return -EFAULT;

	      break;
	  }

      case PP_PWMTACH_IOCTL_PWM_INIT:
	  {
	      pp_pwm_init_t pwm_init;
	      
	      /* get data from userspace */
	      if (copy_from_user(&pwm_init, (char *)arg, sizeof(pwm_init))) return -EFAULT;

	      if (pwm_init.index >= core->pwm_count) {
		  return -EINVAL;
	      }

	      /* if the core supports alternate functions, set it to pwm */
	      if (core->pwm_set_alternate) {
		  core->pwm_set_alternate(core, pwm_init.index, pwm_init.activate ? 0 : 1);
	      }
	      
	      break;
	  }

      case PP_PWMTACH_IOCTL_TACH_INIT:
	  {
	      pp_tach_init_t tach_init;
	      
	      /* get data from userspace */
	      if (copy_from_user(&tach_init, (char *)arg, sizeof(tach_init))) return -EFAULT;

	      if (tach_init.index >= core->tacho_count) {
		  return -EINVAL;
	      }

	      /* if the core supports alternate functions, set it to tach */
	      if (core->tach_set_alternate) {		  
		  core->tach_set_alternate(core, tach_init.index, tach_init.activate ? 0 : 1);
	      }

	      /* activate tacho channel */
	      if (core->tach_activate_channel) {
		  core->tach_activate_channel(core, tach_init.index);
	      }

	      break;
	  }
	  
      default:
	  /* invalid cmd */
	  ret = -EINVAL;
    }

    up(&dev->mtx);
    return ret;    
}

/* ------------------------------------------------------------------------- *
 * /proc interface
 * ------------------------------------------------------------------------- */

int
pwm_proc_read_freq(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    pwmtach_core_t *core = dev->core;  
    int len = 0;
    unsigned int freq;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;
    
    core->pwm_get_freq(core, pd->nr, &freq);

    len += sprintf(page, "%d\n", freq);

    up(&dev->mtx);
    *start = NULL;
    *eof = 1;
    return len;
}

int
pwm_proc_read_duty(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    pwmtach_core_t *core = dev->core;  
    int len = 0;
    uint8_t duty;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;
    
    core->pwm_get_duty(core, pd->nr, &duty);
    
    len += sprintf(page, "%d\n", duty);

    up(&dev->mtx);
    *start = NULL;
    *eof = 1;
    return len;
}

static int
pwm_proc_write_freq(struct file *file, const char *buffer, unsigned long count, void* data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    pwmtach_core_t *core = dev->core;    
    char buf[256];
    unsigned int freq;
           
    if (count > sizeof(buf)+1) return -EINVAL;
    if (copy_from_user(buf, buffer, count)) return -EFAULT;
    buf[count] = '\0';

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;

    freq = simple_strtoul(buf, NULL, 10);
    core->pwm_set_freq(core, pd->nr, freq);

    up(&dev->mtx);
    return count;
}


static int
pwm_proc_write_duty(struct file*file, const char *buffer, unsigned long count, void* data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    pwmtach_core_t *core = dev->core;    
    char buf[256];
    uint8_t duty;
           
    if (count > sizeof(buf)+1) return -EINVAL;
    if (copy_from_user(buf, buffer, count)) return -EFAULT;
    buf[count] = '\0';

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;
    
    duty = simple_strtoul(buf, NULL, 10);

    if (duty > 100) {
	ERR("duty cycle must be between 0 and 100 percent\n");
    } else {
	core->pwm_set_duty(core, pd->nr, duty);
    }
    
    up(&dev->mtx);
    return count;
}

static int
tacho_proc_read_rpm(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    pwmtach_core_t *core = dev->core;  
    int len = 0;
    uint16_t rpm;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;
    
    core->tach_get_rpm(core, pd->nr, pd->ppr, &rpm);
    
    len += sprintf(page, "%d\n", rpm);

    up(&dev->mtx);
    *start = NULL;
    *eof = 1;
    return len;
}

static int
tacho_proc_read_ppr(char *page, char **start, off_t offset, int count, int *eof, void *data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    int len = 0;

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;

    len += sprintf(page, "%d\n", pd->ppr);

    up(&dev->mtx);
    *start = NULL;
    *eof = 1;
    return len;    
}

static int
tacho_proc_write_ppr(struct file*file, const char *buffer, unsigned long count, void* data)
{
    proc_data_t *pd = (proc_data_t*) data;
    pwmtach_dev_t *dev = pd->dev;
    char buf[256];
    
    if (count > sizeof(buf)+1) return -EINVAL;
    if (copy_from_user(buf, buffer, count)) return -EFAULT;
    buf[count] = '\0';

    if (down_interruptible(&dev->mtx) < 0) return -EINTR;
    pd->ppr = simple_strtoul(buf, NULL, 10);

    up(&dev->mtx);
    return count; 
}

/* ------------------------------------------------------------------------- *
 * private/helper functions
 * ------------------------------------------------------------------------- */

static int
init_proc_interface(pwmtach_dev_t *dev, int nr)
{
    char name[MAX_PROC_NAME_SIZE];
    int ret = -1;
    int i;
    struct proc_dir_entry *freq, *duty, *rpm, *ppr;
    
    snprintf(name, MAX_PROC_NAME_SIZE, "pwmtach%d", nr);

    if ((dev->pde = proc_mkdir(name, NULL)) == NULL) {
	DBG("proc_mkdir(%s) failed", name);
	goto bail;
    }

    /* PWM proc entries */
    dev->proc_data_pwm = kmalloc(dev->core->pwm_count * sizeof(proc_data_t), GFP_KERNEL);
    if (dev->proc_data_pwm == NULL) {
	DBG("kmalloc (pwm) failed");
	goto bail;
    }

    for (i = 0; i < dev->core->pwm_count; i++) {
	snprintf(name, MAX_PROC_NAME_SIZE, "pwm%d", i);

	dev->proc_data_pwm[i].nr  = i;
	dev->proc_data_pwm[i].dev = dev;
	dev->proc_data_pwm[i].pde = proc_mkdir(name, dev->pde);
	if (dev->proc_data_pwm[i].pde == NULL) {
	    DBG("proc_mkdir(%s) failed", name);
	    goto bail;
	}

	if ((freq = create_proc_read_entry("frequency", 0, dev->proc_data_pwm[i].pde,
					   pwm_proc_read_freq, (void*)&dev->proc_data_pwm[i])) == NULL) {
	    DBG("proc pwm 'frequency' entry failed");
	    goto bail;
	}
	freq->write_proc = pwm_proc_write_freq;
	
	if ((duty = create_proc_read_entry("duty_cycle", 0, dev->proc_data_pwm[i].pde,
					   pwm_proc_read_duty, (void*)&dev->proc_data_pwm[i])) == NULL) {
	    DBG("proc pwm 'duty' entry failed");
	    goto bail;
	}
	duty->write_proc = pwm_proc_write_duty;

    }
    
    /* Tacho proc entries */
    dev->proc_data_tacho = kmalloc(dev->core->tacho_count * sizeof(proc_data_t), GFP_KERNEL);
    if (dev->proc_data_tacho == NULL) {
	DBG("kmalloc (tacho) failed");
	goto bail;
    }
    for (i = 0; i < dev->core->tacho_count; i++) {
	snprintf(name, MAX_PROC_NAME_SIZE, "tacho%d", i);
	
	dev->proc_data_tacho[i].nr  = i;
	dev->proc_data_tacho[i].dev = dev;
	dev->proc_data_tacho[i].pde = proc_mkdir(name, dev->pde);
	if (dev->proc_data_tacho[i].pde == NULL) {
	    DBG("proc_mkdir(%s) failed", name);
	    goto bail;
	}

	if ((ppr = create_proc_read_entry("pulses", 0, dev->proc_data_tacho[i].pde,
					  tacho_proc_read_ppr, (void*)&dev->proc_data_tacho[i])) == NULL) {
	    DBG("proc tacho 'pulses' entry failed");
	    goto bail;
	}	
	ppr->write_proc = tacho_proc_write_ppr;
	
	if ((rpm = create_proc_read_entry("rpm", 0, dev->proc_data_tacho[i].pde,
					  tacho_proc_read_rpm, (void*)&dev->proc_data_tacho[i])) == NULL) {
	    DBG("proc tacho 'rpm' entry failed");
	    goto bail;
	}
    }
    
    ret = SUCCESS;
    
 bail:
    return ret;
}

static void
cleanup_proc_interface(pwmtach_dev_t *dev, int nr)
{
    int i;

    if (dev->proc_data_pwm) {
	for (i = 0; i < dev->core->pwm_count; i++) {
	    remove_proc_entry("frequency", dev->proc_data_pwm[i].pde);
	    remove_proc_entry("duty_cycle", dev->proc_data_pwm[i].pde);
	}

	kfree(dev->proc_data_pwm);
    }

    if (dev->proc_data_tacho) {
	for (i = 0; i < dev->core->tacho_count; i++) {
	    remove_proc_entry("pulses", dev->proc_data_tacho[i].pde);
	    remove_proc_entry("rpm", dev->proc_data_tacho[i].pde);
	}

	kfree(dev->proc_data_tacho);
    }    
}
