/**
 * pwmtach_core_peppercon.c
 *
 * Low-level driver for the Peppercon PWM/Tacho core
 *
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppercon.de>,
 *			  Michael Baumann <miba@peppercon.de>
 *
 */

#include <linux/version.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/vmalloc.h>
#include <asm/io.h>

#ifdef __powerpc__
# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#   include <asm/ppc405_pcimcs.h>
# else
#   include <syslib/ppc405_pcimcs.h>
# endif
#endif

#include "pwmtach_common.h"
#include "pwmtach_core_peppercon.h"

/* ------------------------------------------------------------------------- *
 * global constants and macros
 * ------------------------------------------------------------------------- */


#define MY_PCI_VENDOR_ID	0x1743
#define MY_PCI_DEVICE_ID	0x1003
#define PCI_BAR_NO		1
#define IO_BASE_OFFS		0x00960000
#define IO_SIZE			0x44
#define CLOCK_SCALER		2
#define TACH_DIV_DFL		(305 * CLOCK_SCALER)

#define PWM_COUNT		8
#define TACHO_COUNT		8

#define PWMTACH_PEPPERCON_NAME	"peppercon"

/*
 * register defines
 *
 * NOTE: To get real relative address you have to multiply these values with 4.
 */

#define PWMTACH_REG_PRESCL	0x00
#define PWMTACH_REG_PWM_BASE	0x01
#define PWMTACH_REG_PWM_LAST	0x08
#define PWMTACH_REG_TACH_BASE	0x09
#define PWMTACH_REG_TACH_LAST	0x10

/* ------------------------------------------------------------------------- *
 * type definitions
 * ------------------------------------------------------------------------- */

typedef struct {
    u_long	   io_base_phys;
    volatile u32 * io_base_virt;
    u_int	   real_max_duty;
} ft_data_t;

/* ------------------------------------------------------------------------- *
 * global data
 * ------------------------------------------------------------------------- */

ft_data_t ft_data[] = {
    { .io_base_phys	= 0,
      .io_base_virt	= NULL,
      .real_max_duty	= 100
    },
};

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void cleanup(pwmtach_core_t *core);


static int tach_get_rpm(struct pwmtach_core_s *core, uint8_t index,
			uint8_t pulses_per_rev, uint16_t *revs_per_minute);   
static int pwm_set_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int freq);
static int pwm_get_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int *freq);
static int pwm_set_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t duty_cycle);
static int pwm_get_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t *duty_cycle);

/* ------------------------------------------------------------------------- *
 * exported functions (init)
 * ------------------------------------------------------------------------- */

int
pwmtach_core_peppercon_init(pwmtach_core_t *core, u_char core_nr)
{
    int ret = -1, r;
    int nr_cores = sizeof(ft_data) / sizeof(ft_data_t);
    ft_data_t *pdata;
    struct pci_dev *pci;
    u32 prescl_reg;
    
    if (core_nr >= nr_cores) {
	ERR("Unsupported core nr %d (max is %d)", core_nr, nr_cores - 1);	
	return ret;
    }

    core->private_data = pdata = &ft_data[core_nr];
    core->cleanup      = cleanup;

    /* find hardware */
    pci = pci_find_device(MY_PCI_VENDOR_ID, MY_PCI_DEVICE_ID, NULL);
    if (pci == NULL) {
        ERR("failed to find PCI device (%04x:%04x) containing Peppercon PWM/tach core",
            MY_PCI_VENDOR_ID, MY_PCI_DEVICE_ID);
        goto bail;
    }

    pdata->io_base_phys = pci_resource_start(pci, PCI_BAR_NO) + IO_BASE_OFFS;

    if ((r = check_mem_region(pdata->io_base_phys, IO_SIZE))) {
	ERR("IO space %08lx-%08lx already in use (err %d)",
	    pdata->io_base_phys, pdata->io_base_phys + IO_SIZE - 1, r);
	goto bail;
    }
    
    request_mem_region(pdata->io_base_phys, IO_SIZE, PWMTACH_PEPPERCON_NAME);

    pdata->io_base_virt = ioremap_nocache(pdata->io_base_phys, IO_SIZE);
    if (pdata->io_base_virt == NULL) {
	ERR("failed to map PWM/tach I/O space %08lx-%08lx to memory",
	    pdata->io_base_phys, pdata->io_base_phys + IO_SIZE - 1);
	goto bail;
    }

#ifdef __powerpc__
    /* check if io range is (machine check save) accessible */
    {
        u32 val;
        int acc_err;
        val = mcs_in_le32((u32*)&pdata->io_base_virt[PWMTACH_REG_PRESCL], 
                          &acc_err);
        if (acc_err != 0) {
            ERR("failed to read PWM/TACHO registers");
            goto bail;
        }
        mcs_out_le32(val, (u32*)&pdata->io_base_virt[PWMTACH_REG_PRESCL], 
                     &acc_err);
        if (acc_err != 0) {
            ERR("failed to write PWM/TACHO registers");
            goto bail;
        }
    }
#endif

    core->caps = PWMTACH_CAPS_HAVE_PWM | PWMTACH_CAPS_HAVE_TACHO;
    core->pwm_count = PWM_COUNT;
    core->tacho_count = TACHO_COUNT;
    strcpy(core->name, PWMTACH_PEPPERCON_NAME);
    
    core->tach_get_rpm = tach_get_rpm;
    core->pwm_set_freq = pwm_set_freq;
    core->pwm_get_freq = pwm_get_freq;
    core->pwm_set_duty = pwm_set_duty;
    core->pwm_get_duty = pwm_get_duty;

    core->tach_activate_channel = NULL;
    
    /* set tach prescaler */
    prescl_reg = readl(&pdata->io_base_virt[PWMTACH_REG_PRESCL]);
    prescl_reg &= 0x0000FFFF;
    prescl_reg |= ((TACH_DIV_DFL + 1) / 2 - 1) << 16;
    writel(prescl_reg, &pdata->io_base_virt[PWMTACH_REG_PRESCL]);    

    ret = SUCCESS;
    INFO("PWM/Tacho core '%s' (%d) initialized", core->name, core_nr);

 bail:
    return ret;
}

/* ------------------------------------------------------------------------- *
 * core operations
 * ------------------------------------------------------------------------- */

static void
cleanup(pwmtach_core_t *core)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    
    if (pdata == NULL) {	
	return; // we were never really initialized
    }
    
    release_mem_region(pdata->io_base_phys, IO_SIZE);
}

static int
tach_get_rpm(struct pwmtach_core_s *core, uint8_t index,
		     uint8_t pulses_per_rev, uint16_t *revs_per_minute)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    bd_t * bd = (bd_t *)__res;
#else
    bd_t * bd = (bd_t *)&__res;
#endif
    u32 divisor, tach_value;

    if (revs_per_minute == NULL) return -1;
    
    tach_value = readl(&pdata->io_base_virt[PWMTACH_REG_TACH_BASE + index]);
    if (tach_value < 65536) {
	divisor = tach_value * pulses_per_rev * TACH_DIV_DFL;
	*revs_per_minute = (bd->bi_pci_busfreq * CLOCK_SCALER + divisor / 2) / divisor * 60;
    } else {
	*revs_per_minute = 0;
    }
    return 0;
}

static int
pwm_set_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int freq)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    bd_t * bd = (bd_t *)__res;
#else
    bd_t * bd = (bd_t *)&__res;
#endif
    u_int wanted_scaled_freq, wanted_prescale, wanted_max_duty;
    u_int real_scaled_freq, real_prescale;
    u32 prescl_reg;
    
    /* sanity checks */
    if (freq == 0) {
	DBG("pwm frequency must not be zero!");
	return -1;
    }

    /* calculate parameters */
    wanted_max_duty = 100;
    wanted_scaled_freq = freq * wanted_max_duty; /* duty cycle is doubled inside core */
    wanted_prescale = (bd->bi_pci_busfreq * CLOCK_SCALER + wanted_scaled_freq / 2) / wanted_scaled_freq;
    real_prescale = (wanted_prescale / 2) * 2; /* make it even (hardware restriction);
						  always round down to reduce error propagation */    
    
    if (real_prescale) {
	real_scaled_freq = (bd->bi_pci_busfreq * CLOCK_SCALER + real_prescale / 2) / real_prescale;
	pdata->real_max_duty = (real_scaled_freq + freq / 2) / freq;
    }
   
    /* sanity checks */
    if (real_prescale == 0
	|| (real_prescale / 2 - 1) > 65535
	|| pdata->real_max_duty < 2
	|| pdata->real_max_duty > 255) {
	DBG("sanity checks failed: real_prescale=%u, real_max_duty=%u", real_prescale, pdata->real_max_duty);
	return -EINVAL;
    }
    
    /* write pwm prescaler into hardware */
    prescl_reg = readl(&pdata->io_base_virt[PWMTACH_REG_PRESCL]);
    prescl_reg &= 0xFFFF0000;
    prescl_reg |= real_prescale / 2 - 1;
    writel(prescl_reg, &pdata->io_base_virt[PWMTACH_REG_PRESCL]);

    return 0;
}

static int
pwm_get_freq(struct pwmtach_core_s *core, uint8_t index, unsigned int *freq)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    bd_t * bd = (bd_t *)__res;
#else
    bd_t * bd = (bd_t *)&__res;
#endif
    u_int real_prescale;
    u32 prescl_reg;

    prescl_reg = readl(&pdata->io_base_virt[PWMTACH_REG_PRESCL]);

    real_prescale = ((prescl_reg & 0x0000FFFF) + 1) * 2;
    
    if (freq) *freq = (bd->bi_pci_busfreq * CLOCK_SCALER+ (pdata->real_max_duty * real_prescale) / 2) /
	(pdata->real_max_duty * real_prescale);

    return 0;
}

static int
pwm_set_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t duty_cycle)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    u_int real_duty, wanted_max_duty;
    
    /* calculate real duty cycle */
    wanted_max_duty = 100;
    real_duty = ((u_int)duty_cycle * pdata->real_max_duty + wanted_max_duty / 2) / wanted_max_duty;
    /*
     * NOTE: hardware has some specials:
     *       - the signal level duration written to register is increased
     *         by one internally and so we reduce it by one in software.
     *       - for a signal level duration of 0 you have to wrote zero
     *         to the register
     *       - a signal duration of 1 is not possible - so we map it to 2
     */
    if (real_duty < 2) ++real_duty;
    if ((pdata->real_max_duty - real_duty) < 2) --real_duty;
    
    /* write real duty into hardware */
    writel(((real_duty - 1) << 8) | (pdata->real_max_duty - real_duty - 1),
	   &pdata->io_base_virt[PWMTACH_REG_PWM_BASE + index]);

    return 0;
}

static int
pwm_get_duty(struct pwmtach_core_s *core, uint8_t index, uint8_t *duty_cycle)
{
    ft_data_t *pdata = (ft_data_t*) core->private_data;
    u_int pwm_reg_val, real_duty, wanted_max_duty;
    
    /* read real duty cycle from hardware */
    pwm_reg_val = readl(&pdata->io_base_virt[PWMTACH_REG_PWM_BASE + index]);
    real_duty = (pwm_reg_val >> 8) & 0xFF;
    pdata->real_max_duty = (pwm_reg_val & 0xFF) + real_duty;
    
    /* calculate real duty cycle */
    wanted_max_duty = 100;
    if (duty_cycle) *duty_cycle = (real_duty * wanted_max_duty + pdata->real_max_duty / 2) / pdata->real_max_duty;

    return 0;
}

/* ------------------------------------------------------------------------- *
 * internal/helper functions
 * ------------------------------------------------------------------------- */



