/***************************************************************************
 *   Copyright (C) 2004 by Ralf Guenther                                   *
 *   rgue@peppercon.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>

#include "esb2_tpt.h"

#ifndef u8
#define u8 unsigned char
#endif
#ifndef u32
#define u32 unsigned int
#endif
#ifndef wait_queue_head_t
#define wait_queue_head_t void*
#endif
#ifndef irqreturn_t
#define irqreturn_t void
#endif
//#include "../fmlcore.h"

/*****************************/
/* buffer offsets...       */
/*****************************/
// TODO: rgue: hase to be changed for KIRA100 R02!
#define REG_OFFSET  0x0
#define BUF0_OFFSET 0x100 
#define BUF1_OFFSET 0x140
#define BUF2_OFFSET 0x180
#define BUF3_OFFSET 0x1c0

/* slave buffers (2K) */
#define SBUF_WR_OFFSET 0x400
#define SBUF_RD_OFFSET 0x600
#define SBUF_SIZE      (0x200 * 4) // 2048 bytes

/*****************************/
/* register offsets...       */
/*****************************/
#define MGR_ADDR 0x0          /* general register */
#define MCR0_ADDR 0x1         /* command register */
#define MIACK_ADDR 0x2        /* interrupt acknowledge register */
#define MGSR_ADDR 0x3         /* status register */
#define MPRER_ADDR 0x4        /* devider register */
#define MTOUT_ADDR 0x5        /* FML timeout value in system clock cycles! */
#define MID1_ADDR 0x6         /* FML-Core ID1 register */
#define MID2_ADDR 0x7         /* FML-Core ID2 register */
#define SRBC_ADDR 0x8         /* slave read byte count */
#define SWBC_ADDR 0x9         /* slave write byte count */
#define SAR_ADDR  0xa         /* slave addr */
#define CORESTAT_ADDR 0xb     /* debug: core engine state */
#define BYTESTAT_ADDR 0xc     /* debug: byte engine state */
#define BITSTAT_ADDR 0xd      /* debug: bit engine state */

/*****************************/
/* define register values... */
/*****************************/

/* two ID registers to find the core */
#define FML_ID1 0x464d4c2d /* "FML-" */
#define FML_ID2 0x436f7265 /* "Core" */

/* values for the MGR-Register */
#define MGO  (0x1 << 0) // enable core
#define MRST (0x1 << 1) // reset core (auto cleared by hardware)
#define MS   (0x1 << 2) // 0..slave, 1..master
#define MALLIEN (0x1 << 4) // enable global interrupts
#define MFMLIEN (0x1 << 5) // enable FML line interrupts
#define MCOREIEN (0x1 << 6) // enable core interrupts

/* values for the MCR0-Register */
#define MSTC (0x1<<0)       // Master Start Command
#define MSPM (0x1<<1)       // Master Stop Command
#define MNC  (0x1<<2)       // Master NACK Command
#define MRDC (0x1<<4)       // Master Read Command
#define MWRC (0x1<<5)       // Master Write Command
#define MPC  (0x1<<6)       // Master PEC Command
#define BTCNT_EN (0x1<<11)
#define MBUFSEL_MASK (0x3<< 12)
#define MBUFSEL_SHIFT 12
#define MBC_MASK (0xf<16)
#define MBC_SHIFT 16		 
		 
/* values of the MIACK-Register */
#define MCOREIACK (0x1)
#define MFMLIACK (0x2)

/* values of the MGSR-Register */
#define MERR             (0x1 << 0)
#define MAB              (0x1 << 1)
#define MOP              (0x1 << 7)
#define MCORE_IRQ        (0x1 << 16)
#define MFMLSL_IRQ       (0x1 << 17)
#define MOFLW            (0x1 << 19)
#define GET_MBWD(a)      (( a & 0xff00 ) >> 8)
#define GET_RD_BNTCNT(a)      (( a & 0xff000000 ) >> 24) 


static int find_core(int sock)
{
    if (esb2_tpt_ioctl_get_reg(sock, MID1_ADDR) != FML_ID1
     || esb2_tpt_ioctl_get_reg(sock, MID2_ADDR) != FML_ID2) {
        return -1;
    }
    return 0;
}

static void dump_regs(int fd, const char* txt)
{
    printf("FML registers - %s\n", txt);
    printf("MGR   = 0x%08x  MCR0  = 0x%08x  MIACK = 0x%08x  MGSR  = 0x%08x\n",
        esb2_tpt_ioctl_get_reg(fd, MGR_ADDR), esb2_tpt_ioctl_get_reg(fd, MCR0_ADDR),
        esb2_tpt_ioctl_get_reg(fd, MIACK_ADDR), esb2_tpt_ioctl_get_reg(fd, MGSR_ADDR));

    printf("MPRER = 0x%08x  MTOUT = 0x%08x  SRBC  = 0x%08x  SWBC  = 0x%08x\n",
        esb2_tpt_ioctl_get_reg(fd, MPRER_ADDR), esb2_tpt_ioctl_get_reg(fd, MTOUT_ADDR),
        esb2_tpt_ioctl_get_reg(fd, SRBC_ADDR), esb2_tpt_ioctl_get_reg(fd, SWBC_ADDR));

    printf("SADDR = 0x%08x  CORE  = 0x%08x  BYTE  = 0x%08x  BIT   = 0x%08x\n",
        esb2_tpt_ioctl_get_reg(fd, SAR_ADDR), esb2_tpt_ioctl_get_reg(fd, CORESTAT_ADDR),
        esb2_tpt_ioctl_get_reg(fd, BYTESTAT_ADDR), esb2_tpt_ioctl_get_reg(fd, BITSTAT_ADDR));

}


unsigned char fml_read_req[] = {
    (2 << 6) | (0 << 4) | 5,
    0,
    255,
};

#define swap32(v) (((v & 0x000000ff) << 24) | ((v & 0x0000ff00) << 8) | ((v & 0x00ff0000) >> 8) | ((v & 0xff000000) >> 24))

int main(int argc, char *argv[])
{
    int fd;
    argc = argc; argv = argv;

    printf("Opening device...\n");
    fd = open("/dev/esb2_tpt", O_RDWR);
    if (fd == -1) {
        printf("ERROR: open failed (%s)\n", strerror(errno));
        exit(-1);
    }

    dump_regs(fd, "init");

    printf("Checking FML core...\n");
    if (find_core(fd)) {
        printf("FML core not found (%s)\n", strerror(errno));
        goto exit;
    }
    printf("ok.\n");

/*
    printf("Checking FML write...\n");
    if (write(fd, fml_read_req, sizeof(fml_read_req)) < 0) {
        printf("FML write failed (%s)\n", strerror(errno));
        dump_regs(fd, "write error");
        goto exit;
    }
    printf("ok.\n");
*/
    while (1) {
        char b[1000];

        fd_set set;
        FD_ZERO(&set);
        FD_SET(fd, &set);
        FD_SET(STDIN_FILENO, &set);

        printf("Waiting for incoming data... ([ENTER] to quit)\n");

        //struct timeval tv = { .tv_sec = 1, .tv_usec = 0, };
        if (select (FD_SETSIZE, &set, NULL, NULL, NULL/*&tv*/) <= 0) {
            // idle
            // ...
            continue;
        }

        if (FD_ISSET(STDIN_FILENO, &set))
        {
            int i, j, c = read(STDIN_FILENO, b, sizeof(b));
            if (c <= 1) break;
            b[c] = '\0';
            if (b[0] == '/') {
                if (strncmp(b + 1, "huge", strlen("huge")) == 0) {
                    const int SEND = 8;
                    const int RECV = 8;
                    const int LOOPS = 10;
                    unsigned char *blk;
                    int s, es = 0;

                    unsigned char go = 0xAB;
                    if (write(fd, &go, 1) != 1) {
                        printf("Failed to give start signal (err=%d)\n", errno);
                        continue;
                    }
                    printf("The start signal (0xAB) has been given!");

                    for (j = 0; j < LOOPS; j++) {

                        blk = malloc(SEND);
                        for (i = 0; i < SEND; i++) blk[i] = i + j;
                        printf("%.3d Sending...", j);
                        c = write(fd, blk, SEND);
                        printf("%d bytes sent\n", c);
                        free(blk);

                        blk = malloc(RECV);
                        for (s = 0; s < RECV; s += c) {
                            int e = 0;
                            printf("%.3d Receiving...", j);
                            c = read(fd, blk + s, RECV - s);
                            if (c < 0) { printf("err %d %s\n", errno, errno == EAGAIN ? "(EAGAIN)" : ""); continue; }
                            for (i = s; i < s + c; i++)
                                if (blk[i] != ((i + j) & 0xff)) e++;
                            printf("%d bytes received %.2x %.2x %.2x (%d errors)\n", c, blk[s], blk[s + 1], blk[s + 2], e);
                            es += e;

                            for (i = s; i < s + c; i++)
                                if (blk[i] != ((i + j) & 0xff))
                                    printf("error at %d: is=%.2x expected=%.2x\n", i, blk[i], (i + j) & 0xff);
                        }
                        free(blk);
                    }
                    printf("Finished (%d errors)\n", es);
                } else {
                    if (b[1] != '?') printf("Unknown command '%s'", b + 1);
                    printf("Valid commands are:\n");
                    printf("  /? ... print this help info\n");
                    printf("  /huge ... sends a huge data block of 200000 bytes\n");
                }
            } else {
                printf("Sending %d bytes:", c);
                for (i = 0; i < c; i++) printf("%.2x ", b[i]);
                printf("(");
                for (i = 0; i < c; i++) printf("%c", isgraph(b[i]) ? b[i] : '.');
                printf(")\n");
                write(fd, b, c);
            }
        }
        if (FD_ISSET(fd, &set)) {
            int i, c = read(fd, b, sizeof(b));
            printf("Received %d bytes:", c);
            for (i = 0; i < c; i++) printf("%.2x ", b[i]);
            printf("(");
            for (i = 0; i < c; i++) printf("%c", isgraph(b[i]) ? b[i] : '.');
            printf(")\n");
        }
    }


exit:
    dump_regs(fd, "exit");
    close(fd);
    return 0;
}
