#ifndef __FUSB220_M_H
#define __FUSB220_M_H

/*
 * Faraday USB220 ("FTUSB220") USB Device Controller driver
 *
 * Copyright (C) 2006-2006 Peppercon AG
 *      by Christian Krause
 * Copyright (C) 2004-2005 John
 *      by John Chiang
 * Copyright (C) 2004 Faraday corp. 
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#define TRUE			1
#define FALSE			0

#define MASK_F0			0xF0

// Bit defines
#define BIT0			0x00000001
#define BIT1			0x00000002
#define BIT2			0x00000004
#define BIT3			0x00000008
#define BIT4			0x00000010
#define BIT5			0x00000020
#define BIT6			0x00000040
#define BIT7			0x00000080

#define BIT8			0x00000100
#define BIT9			0x00000200
#define BIT10			0x00000400
#define BIT11			0x00000800
#define BIT12			0x00001000
#define BIT13			0x00002000
#define BIT14			0x00004000
#define BIT15			0x00008000	
	
#define BIT16			0x00010000
#define BIT17			0x00020000
#define BIT18			0x00040000
#define BIT19			0x00080000
#define BIT20			0x00100000
#define BIT21			0x00200000
#define BIT22			0x00400000
#define BIT23			0x00800000	
	
#define BIT24			0x01000000
#define BIT25			0x02000000
#define BIT26			0x04000000
#define BIT27			0x08000000
#define BIT28			0x10000000
#define BIT29			0x20000000
#define BIT30			0x40000000
#define BIT31			0x80000000	
	
#define mLowByte(u16)	((u8)(u16 & 0xff))
#define mHighByte(u16)	((u8)((u16 & 0xff00) >> 8))

// DMA defines
#define USE_DMA
#ifdef USE_DMA
#define DMA_ADDR_INVALID	(~(dma_addr_t)0)
#define AHB_DMA_MAX_LEN		8192
#include <asm/arch/ahb_dma.h>
#endif

// Buffer allocation for each EP
#define USB_EPX_BUFSIZ	4096

// reset value for EP
#define	MAX_EP0_SIZE	64

// number of maximum used endpoints
#define FUSB220_MAX_EP				14
#define FUSB220_MAX_FIFO			14
#define FUSB220_DMA_IS_IDLE_NOW			(FUSB220_MAX_EP+1)

// PHY Test Mode Selector Register, 13.6.2.2.1
#define TEST_J				0x02
#define TEST_K				0x04
#define TEST_SE0_NAK			0x08
#define TEST_PKT			0x10

// Block Size define, 13.6.2.7.2, FIFO Configuration Register, FIFO 0 - FIFO 13
#define BLK512BYTE		1
#define BLK1024BYTE		2
// FIFO 14, 15
#define BLK64BYTE		1
#define BLK128BYTE		2

// Block toggle number define, 13.6.2.7.2, FIFO Configuration Register
#define SINGLE_BLK		1
#define DOUBLE_BLK		2
#define TRIBLE_BLK		3

// Endpoint transfer type, 13.6.2.7.2, FIFO Configuration Register
#define TF_TYPE_CONTROL			0
#define TF_TYPE_ISOCHRONOUS		1
#define TF_TYPE_BULK			2
#define TF_TYPE_INTERRUPT		3

// FIFO Configuration and Status Register, 13.6.2.7.1
#define DIRECTION_IN	1
#define DIRECTION_OUT	0

// FIFO number define
#define FIFO0	0x0
#define FIFO14  0xe
#define FIFO15  0xf

// Endpoint number define
#define EP0        0x0
#define EP1        0x1
#define EP14       0xe
#define EP15       0xf

// DMA channel
#define USB_AHBDAC				3	

#define EP_IS_BULK_IN(x) ((x->ep_type == TF_TYPE_BULK) && (x->is_in))
#define EP_IS_BULK_OUT(x) ((x->ep_type == TF_TYPE_BULK) && (!x->is_in))
#define EP_IS_INTR_IN(x) ((x->ep_type == TF_TYPE_INTERRUPT) && (x->is_in))
#define EP_IS_INTR_OUT(x) ((x->ep_type == TF_TYPE_INTERRUPT) && (!x->is_in))

struct FTC_ep {
    struct usb_ep	 ep;
    struct FTC_udc	*dev;
    unsigned long 	 irqs;
    
    uint8_t num;
#ifdef USE_DMA
    uint8_t dma;
#endif
    uint8_t is_in;
    uint8_t stopped;
    uint8_t is_high;
    uint8_t ep_type;
    
    /* analogous to a host-side qh */
    struct list_head queue;
    const struct usb_endpoint_descriptor *desc;
};

struct FTC_request {
    struct usb_request	req;
    struct list_head	queue;
#ifdef USE_DMA
    u32 		u32DMACurrTxLen;
    dma_addr_t		CurDmaStartAddr;
    unsigned		mapped:1;
#endif
};

enum ep0state {
    EP0_DISCONNECT,	 /* no host */
    EP0_IDLE,		 /* between STATUS ack and SETUP report */
    EP0_IN,		 /* data stage */
    EP0_OUT,		 /* data stage */
    EP0_STATUS,		 /* status stage */
    EP0_STALL,		 /* data or status stages */
    EP0_SUSPEND,	 /* usb suspend */
};

#ifdef USE_DMA
typedef enum {
    PIO_Mode,
    AHB_DMA,
    APB_DMA
} DMA_mode;
#endif

struct FTC_udc {
    struct usb_gadget			gadget;
    spinlock_t			      	lock;
    struct FTC_ep		      ep[FUSB220_MAX_EP+1];
    struct usb_gadget_driver 	*driver;
    
    struct usb_request 		*EP0req;
    
    enum ep0state   ep0state;
    uint8_t got_irq;
    uint8_t got_region;
    uint8_t req_config;
    uint8_t configured;
    uint8_t enabled;

    struct usb_ctrlrequest ControlCmd;
    
    u8 u8UsbConfigValue;
    u8 u8UsbInterfaceValue;
    u8 u8UsbInterfaceAlternateSetting;
    
    u8 bUsbBufferEmpty;
    
    u8 u16TxRxCounter;
    
    u8 usb_interrupt_level1;
    u8 usb_interrupt_level1_Save;
    u8 usb_interrupt_level1_Mask;
    
    u8 *pu8DescriptorEX;
    
#ifdef USE_DMA
    DMA_mode Dma_Status;
    ahb_dma_channel_data_t  *ahb_dma;   
    u8 EPUseDMA;   //EP for DMA
    struct  FTC_request *ReqForDMA;
#endif

    unsigned long			irqs;
    u8 u8LineCount;

    uint32_t kira_rev;
};


#define DBG_OFF 	0x00
#define DBG_CTRL 	0x01
#define DBG_BULK 	0x02
#define DBG_ISO		0x04
#define DBG_INT		0x08
#define DBG_FUNC	0x10
#define DBG_TMP		0x20

#define xprintk(dev,level,fmt,args...) printk(level "%s : " fmt , driver_name , ## args)
#define wprintk(level,fmt,args...) printk(level "%s : " fmt , driver_name , ## args)

#if DEBUGLEVEL == D_BLABLA
# define DEBUG
# define VERBOSE
# define USB_DBG 	(DBG_CTRL|DBG_BULK|DBG_ISO|DBG_INT|DBG_FUNC|DBG_TMP)
#else
# undef DEBUG
# undef VERBOSE
# define USB_DBG 	DBG_OFF
#endif

#ifdef DEBUG
#define DBG(dev,fmt,args...) xprintk(dev , KERN_DEBUG , fmt , ## args)
#else
#define DBG(dev,fmt,args...) do { } while (0)
#endif /* DEBUG */

#ifdef VERBOSE
#define VDBG DBG
#else
#define VDBG(dev,fmt,args...) do { } while (0)
#endif	/* VERBOSE */

#define ERROR(dev,fmt,args...) xprintk(dev , KERN_ERR , fmt , ## args)
#define WARN(dev,fmt,args...) xprintk(dev , KERN_WARNING , fmt , ## args)
#define INFO(dev,fmt,args...) xprintk(dev , KERN_INFO , fmt , ## args)

#if (USB_DBG & DBG_TMP)  
#define DBG_TEMP(fmt,args...) wprintk(KERN_INFO , fmt , ## args)
#else
#define DBG_TEMP(fmt,args...)
#endif

#if (USB_DBG & DBG_FUNC)  
#define DBG_FUNCC(fmt,args...) wprintk(KERN_INFO , fmt , ## args)
#else
#define DBG_FUNCC(fmt,args...)
#endif

#if (USB_DBG & DBG_CTRL)  
#define DBG_CTRLL(fmt,args...) wprintk(KERN_INFO , fmt , ## args)
#else
#define DBG_CTRLL(fmt,args...)
#endif

#if (USB_DBG & DBG_BULK)  
#define DBG_BULKK(fmt,args...) wprintk(KERN_INFO , fmt , ## args)
#else
#define DBG_BULKK(fmt,args...)
#endif

#ifndef container_of
#define	container_of	list_entry
#endif

#ifndef	IRQ_NONE
typedef void irqreturn_t;
#define IRQ_NONE
#define IRQ_HANDLED
#define IRQ_RETVAL(x)
#endif

#define Cmd_Service_Fail	 	0
#define Cmd_Already_Service 		1
#define Cmd_Let_Gadget_To_Service 	2

#define EP_bBLKSIZE(x)         ((x->num < EP14)? \
				((x->ep.maxpacket <= 512)?BLK512BYTE:BLK1024BYTE) : \
				((x->ep.maxpacket <= 64)?BLK64BYTE:BLK128BYTE))
// we use bulk endpoints always double buffered, others not
#define EP_bBLKNO(x)	       (((x)->ep_type == TF_TYPE_BULK && (x)->is_in)?DOUBLE_BLK:SINGLE_BLK)
#define EP_MAXPACKET(x)        (x->ep.maxpacket)
#define EP_bDIRECTION(x)       ((x->is_in)?DIRECTION_IN:DIRECTION_OUT)
#define EP_bTYPE(x)            ((x)->ep_type)

uint8_t EP_FIFO_START(struct FTC_ep *ep);

// build value for EPx MAP Register:
//           EPxMAP[7:4]             EPxMAP[3:0]
//           FIFO_NUMBER_OUT_EP[3:0] FIFO_NUMBER_IN_EP[3:0]
//  in case: 1111                    fifo_start
// out case: fifo_start              1111
// 1111 is the reset value for the respective nibble (see 13.6.2.6.1)
#define EP_Map(ep) ( (ep)->is_in? (EP_FIFO_START(ep) | 0xF0) : (EP_FIFO_START(ep)<<4 | 0x0F) )
#define EP_FIFO_MAP(ep) ((EP_bDIRECTION(ep) << 4) | ep->num)
#define EP_FIFO_CONFIG(ep) (0x80 | ((EP_bBLKSIZE(ep) - 1) << 4) | ((EP_bBLKNO(ep)-1) << 2) | EP_bTYPE(ep))

static int FTC_fifo_status(struct usb_ep *_ep);
#define FUSB220_BASE_ADDRESS			(IO_ADDRESS(CPE_USBDEV_BASE))
#define FUSB220_FIFO_BASE(bOffset)		(IO_ADDRESS(CPE_USBDEV_BASE)+0xC0+(bOffset<<2)) 

#define bFUSBPort(bOffset)			*((volatile u8 *) ( FUSB220_BASE_ADDRESS | (u32)(bOffset)))
#define wFUSBPort(bOffset)			*((volatile u16 *) ( FUSB220_BASE_ADDRESS | (u32)(bOffset)))
#define dwFUSBPort(bOffset)			*((volatile u32 *) ( FUSB220_BASE_ADDRESS | (u32)(bOffset)))
	
#define DMAbFUSBPort(bOffset)			*((volatile u8 *) ( CPE_USBDEV_BASE | (u32)(bOffset)))
#define DMAwFUSBPort(bOffset)			*((volatile u16 *) ( CPE_USBDEV_BASE | (u32)(bOffset)))
#define DMAdwFUSBPort(bOffset)			*((volatile u32 *) ( CPE_USBDEV_BASE | (u32)(bOffset)))


// Main Control Register, 13.6.2.1.1
#define mUsbRmWkupST()				(bFUSBPort(0x00) & (u8)BIT0)
#define mUsbRmWkupClr()				(bFUSBPort(0x00) &= ~(u8)BIT0)
#define mUsbRmWkupSet()				(bFUSBPort(0x00) |= (u8)BIT0)

#define mUsbHbfFlush()				(bFUSBPort(0x00) |= (u8)BIT1)

#define mUsbGlobIntEnSet()			(bFUSBPort(0x00) |= (u8)BIT2)

#define mUsbGoSuspend()				(bFUSBPort(0x00) |=  (u8)BIT3)

#define mUsbSoftRstSet()			(bFUSBPort(0x00) |=  (u8)BIT4)
#define mUsbSoftRstClr()			(bFUSBPort(0x00) &= ~(u8)BIT4)

#define mUsbChipEnSet()				(bFUSBPort(0x00) |= (u8)BIT5)

#define mUsbHighSpeedST()			(bFUSBPort(0x00) & (u8)BIT6)

#define mUsbAhbRst()				(bFUSBPort(0x00) |= (u8)BIT7)	


// Device Address Register, 13.6.2.1.2
// sets address and resets AFT_CONF if it was already set
// AFT_CONF will be set after SetConfiguration with mUSBCFgSet
#define mUsbDevAddrSet(Value)			(bFUSBPort(0x01) = (Value & 0x7f))
#define mUsbCfgST()				(bFUSBPort(0x01) & (u8)BIT7)
#define mUsbCfgSet()				(bFUSBPort(0x01) |= (u8)BIT7)
#define mUsbCfgClr()				(bFUSBPort(0x01) &= ~(u8)BIT7)

// Test Register, 13.6.2.1.3
#define mUsbClrAllFIFOSet()			(bFUSBPort(0x02) |= (u8)BIT0)
#define mUsbClrAllFIFOClr()			(bFUSBPort(0x02) &= ~(u8)BIT0)
#define mUsbTstForceFsEn()			(bFUSBPort(0x02) |= (u8)BIT6)
#define mUsbTstForceFsDis()			(bFUSBPort(0x02) &= ~(u8)BIT6)
#define mUsbTstHalfSpeedEn()			(bFUSBPort(0x02) |= (u8)BIT7)
#define mUsbTstHalfSpeedDis()			(bFUSBPort(0x02) &= ~(u8)BIT7)


// SFO Freame Number Register Byte 0, 13.6.2.1.4
#define mUsbFrameNoLow()			(bFUSBPort(0x04))
// SFO Freame Number Register Byte 1, 13.6.2.1.5
#define mUsbFrameNoHigh()			(bFUSBPort(0x05) & 0x7)
#define mUsbFrameNoMicro()                      ((bFUSBPort(0x05) & 0x38) >> 3)

// SOF Mask Timer Register Byte 0, 13.6.2.1.6
#define mUsbSOFMaskLow()			(bFUSBPort(0x06))
#define mUsbSOFMaskHigh()			(bFUSBPort(0x07))

// PHY Test Mode Selector, 13.6.2.2.1
#define mUsbTsMdWr(item)			(bFUSBPort(0x08) = item)
#define mUsbUnPLGClr()				(bFUSBPort(0x08) &= ~(u8)BIT0)
#define mUsbUnPLGSet()                                 (bFUSBPort(0x08) |= (u8)BIT0)

// Vendor Specific IO Control, 13.6.2.2.2
// offset 0x09, not used

// Vendor Specific IO Control, 13.6.2.2.3
// offset 0x0a, not used

// CX Configuration and Status Register, 13.6.2.2.4
#define mUsbEP0DoneSet()			(bFUSBPort(0x0B) |= (u8)BIT0)
#define mUsbTsPkDoneSet()			(bFUSBPort(0x0B) |= (u8)BIT1)
#define mUsbEP0StallSet()			(bFUSBPort(0x0B) |= (u8)BIT2)
#define mUsbEP0ClearFIFO()			(bFUSBPort(0x0B) |= (u8)BIT3)
#define mUSbEP0Full()				(bFUSBPort(0x0B) &  (u8)BIT4)
#define mUsbEP0Empty()				(bFUSBPort(0x0B) &  (u8)BIT5)

// Endpoint 0 Data Port, 13.6.2.2.5
#define mUsbEP0DataRdByte0()			(bFUSBPort(0x0C))
#define mUsbEP0DataRdWord()			(wFUSBPort(0x0C))	
#define mUsbEP0DataRdDWord()			(dwFUSBPort(0x0C))

#define mUsbEP0DataWrByte0(data)		(bFUSBPort(0x0C) = data)
#define mUsbEP0DataWrWord(data)			(wFUSBPort(0x0C) = (u16)data)
#define mUsbEP0DataWrDWord(data)		(dwFUSBPort(0x0C) = (u32)data)

// Interrupt Group Mask, 13.6.2.3.1
#define mUsbIntGroupMaskRd()			(bFUSBPort(0x10))

// Interrupt Mask Regsiter 0, 13.6.2.3.2
#define mUsbIntByte0MaskRd()			(bFUSBPort(0x11))
#define mUsbIntEP0SetupDis()			(bFUSBPort(0x11) |= (u8)BIT0)
#define mUsbIntEP0InDis()			(bFUSBPort(0x11) |= (u8)BIT1)
#define mUsbIntEP0OutDis()			(bFUSBPort(0x11) |= (u8)BIT2)
#define mUsbIntEP0EndDis()			(bFUSBPort(0x11) |= (u8)BIT3)
#define mUsbIntEP0FailDis()			(bFUSBPort(0x11) |= (u8)BIT4)
#define mUsbIntRdBufErrDis()			(bFUSBPort(0x11) |= (u8)BIT5)

#define mUsbIntEP0SetupEn()			(bFUSBPort(0x11) &= ~((u8)BIT0))
#define mUsbIntEP0InEn()			(bFUSBPort(0x11) &= ~((u8)BIT1))
#define mUsbIntEP0OutEn()			(bFUSBPort(0x11) &= ~((u8)BIT2))
#define mUsbIntEP0EndEn()			(bFUSBPort(0x11) &= ~((u8)BIT3))
#define mUsbIntEP0FailEn()			(bFUSBPort(0x11) &= ~((u8)BIT4))
#define mUsbIntRdBufErrEn()			(bFUSBPort(0x11) &= ~((u8)BIT5))

// INterrupt Mask Register 1 - 7, 13.6.2.3.3 ... 13.6.2.3.10
#define mUsbIntFIFO0_3OUTDis()			(bFUSBPort(0x12) = 0xFF)
#define mUsbIntFIFO4_7OUTDis()			(bFUSBPort(0x13) = 0xFF)
#define mUsbIntFIFO8_9OUTDis()			(bFUSBPort(0x14) = 0xFF)
#define mUsbIntFIFO0_7INDis()			(bFUSBPort(0x16) = 0xFF)
#define mUsbIntFIFO8_9INDis()			(bFUSBPort(0x17) = 0xFF)

#define mUsbIntByte1MaskRd()			(bFUSBPort(0x12))
#define mUsbIntByte2MaskRd()			(bFUSBPort(0x13))
#define mUsbIntByte3MaskRd()			(bFUSBPort(0x14))
#define mUsbIntByte4MaskRd()			(bFUSBPort(0x15))
#define mUsbIntByte5MaskRd()			(bFUSBPort(0x16))
#define mUsbIntByte6MaskRd()			(bFUSBPort(0x17))
#define mUsbIntByte7MaskRd()			(bFUSBPort(0x18))

#define mUsbIntFIFOOUTEn(off,val)		(bFUSBPort(0x12 + off) &= ~(val))
#define mUsbIntFIFOOUTDis(off,val)		(bFUSBPort(0x12 + off) |=  (val))

#define mUsbIntFIFOINEn(off,val)		(bFUSBPort(0x16 + off) &= ~(val))
#define mUsbIntFIFOINDis(off,val)		(bFUSBPort(0x16 + off) |=  (val))

#define mUsbIntBufEmptyDis()			(bFUSBPort(0x18) |= BIT0)

// Receive Zero-length Data Packet Byte 0, 1, 13.6.2.3.10, 13.6.2.3.11
#define mUsbIntRX0Byte0Rd()			(bFUSBPort(0x19))
#define mUsbIntRX0Byte1Rd()			(bFUSBPort(0x1A))
#define mUsbIntRX0Byte0Clr()			(bFUSBPort(0x19) = 0)
#define mUsbIntRX0Byte1Clr()			(bFUSBPort(0x1A) = 0)

// FIFO Empty Byte 0, 1, 13.6.2.3.12, 13.6.2.3.13
#define mUsbFIFOEmpty0Rd()			(bFUSBPort(0x1C))
#define mUsbFIFOEmpty1Rd()			(bFUSBPort(0x1D))

// Interrupt Source Register, 13.6.2.4
#define mUsbIntGroupRegRd()			(bFUSBPort(0x20))
#define mUsbIntRdBufErrClr()			(bFUSBPort(0x21) &= ~((u8)BIT5))

#define mUsbIntByte0Rd()			(bFUSBPort(0x21))
#define mUsbIntByte1Rd()			(bFUSBPort(0x22))
#define mUsbIntByte2Rd()			(bFUSBPort(0x23))
#define mUsbIntByte3Rd()			(bFUSBPort(0x24))
#define mUsbIntByte4Rd()			(bFUSBPort(0x25))
#define mUsbIntByte5Rd()			(bFUSBPort(0x26))
#define mUsbIntByte6Rd()			(bFUSBPort(0x27))
#define mUsbIntByte7Rd()			(bFUSBPort(0x28))

#define mUsbIntBusRstClr()			(bFUSBPort(0x28) = 0x00)
#define mUsbIntSuspClr()			(bFUSBPort(0x28) = 0x00)
#define mUsbIntResmClr()			(bFUSBPort(0x28) = 0x00)

#define mUsbIntBufEmptyRd()			(bFUSBPort(0x28) & BIT0)

#define mUsbIntIsoSeqErrClr()			(bFUSBPort(0x28) &= ~BIT4)
#define mUsbIntIsoSeqAbortClr()			(bFUSBPort(0x28) &= ~BIT5)
#define mUsbIntTX0ByteClr()			(bFUSBPort(0x28) &= ~BIT6)
#define mUsbIntRX0ByteClr()			(bFUSBPort(0x28) &= ~BIT7)

// Isochronoujs Sequential Error Register, 13.6.2.4.10, 13.6.2.4.11
#define mUsbIntIsoSeqErr0Rd()			(bFUSBPort(0x29))
#define mUsbIntIsoSeqErr1Rd()			(bFUSBPort(0x2A))

#define mUsbIntIsoSeqErr0Clr()			(bFUSBPort(0x29) = 0)
#define mUsbIntIsoSeqErr1Clr()			(bFUSBPort(0x2A) = 0)

// Isochronous Sequential Abort Register, 13.6.2.4.12, 13.6.2.4.13
#define mUsbIntIsoSeqAbort0Rd()			(bFUSBPort(0x2B))
#define mUsbIntIsoSeqAbort1Rd()			(bFUSBPort(0x2C))
#define mUsbIntIsoSeqAbort0Clr()		(bFUSBPort(0x2B) = 0)
#define mUsbIntIsoSeqAbort1Clr()		(bFUSBPort(0x2C) = 0)

// Transferred Zero-length Register, 13.6.2.4.14, 13.6.2.4.15
#define mUsbIntTX0Byte0Rd()			(bFUSBPort(0x2D))
#define mUsbIntTX0Byte1Rd()			(bFUSBPort(0x2E))
#define mUsbIntTX0Byte0Clr()			(bFUSBPort(0x2D) = 0)
#define mUsbIntTX0Byte1Clr()			(bFUSBPort(0x2E) = 0)

// Idle Counter, 13.6.2.5.1
#define mUsbIdleCnt(time)			(bFUSBPort(0x2F) = (time & 0x7))

// Endpoint x Map Register, 13.6.2.6.1
#define mUsbEPMap(EPn, MAP)			(bFUSBPort(0x30 + (EPn-1)) = MAP)
#define mUsbEPMapRd(EPn)			(bFUSBPort(0x30 + (EPn-1)))

// HB Data Byte Count, 13.6.2.6.2
#define mUsbHbfCountRd()			(bFUSBPort(0x3F) & (0x1F))

// IN Endpoint x MaxPacketSize Low, 13.6.2.6.3
#define mUsbEPMxPtSzLow(EPn, dir, size)		(bFUSBPort(0x40 + ((1-dir) * 0x20) + ((EPn - 1) << 1)) = (u8)(size))

// IN/OUT Endpoint x MaxPacketSize High, 13.6.2.6.4, 13.6.2.6.5
#define mUsbEPMxPtSzHigh(EPn, dir, size)	(bFUSBPort(0x41 + ((1-dir) * 0x20) + ((EPn - 1) << 1)) = (u8)(size >> 8))

#define mUsbEPinMxPtSz(EPn)			((((bFUSBPort(0x41 + ((EPn - 1) << 1))) & 0x07) << 8) | (bFUSBPort(0x40 + ((EPn - 1) << 1))))
#define mUsbEPinStallST(EPn)			((bFUSBPort(0x41 + ((EPn - 1) << 1)) & BIT3) >> 3)
#define mUsbEPinStallClr(EPn)			(bFUSBPort(0x41 + ((EPn - 1) << 1)) &= ~BIT3)
#define mUsbEPinStallSet(EPn)			(bFUSBPort(0x41 + ((EPn - 1) << 1)) |=  BIT3)
#define mUsbEPinRsTgClr(EPn)			(bFUSBPort(0x41 + ((EPn - 1) << 1)) &= ~BIT4)
#define mUsbEPinRsTgSet(EPn)			(bFUSBPort(0x41 + ((EPn - 1) << 1)) |=  BIT4)
#define mUsbEPinHighBandSet(EPn, dir , size )	(bFUSBPort(0x41 + ((EPn - 1) << 1)) &= ~(BIT6 |BIT5));  (bFUSBPort(0x41 + ((EPn - 1) << 1)) |= ((((u8)(size >> 11)+1) << 5)*(dir)) )
	
#define mUsbEPoutMxPtSz(EPn)			((((bFUSBPort(0x61 + ((EPn - 1) << 1))) & 0x07) << 8) | (bFUSBPort(0x60 + ((EPn - 1) << 1))))
#define mUsbEPoutStallST(EPn)			((bFUSBPort(0x61 + ((EPn - 1) << 1)) & BIT3) >> 3)
#define mUsbEPoutStallClr(EPn)			(bFUSBPort(0x61 + ((EPn - 1) << 1)) &= ~BIT3)
#define mUsbEPoutStallSet(EPn)			(bFUSBPort(0x61 + ((EPn - 1) << 1)) |=  BIT3)
#define mUsbEPoutRsTgClr(EPn)			(bFUSBPort(0x61 + ((EPn - 1) << 1)) &= ~BIT4)
#define mUsbEPoutRsTgSet(EPn)			(bFUSBPort(0x61 + ((EPn - 1) << 1)) |=  BIT4)

// DMA mode Enable Register, 13.6.2.6.7, 13.6.2.6.8
#define mUsbFIFODMALowEn(val)			(bFUSBPort(0x7e) |= (val))
#define mUsbFIFODMALowDis(val)			(bFUSBPort(0x7e) &= ~(val))
	
#define mUsbFIFODMAHighEn(val)			(bFUSBPort(0x7f) |= (val))
#define mUsbFIFODMAHighDis(val)			(bFUSBPort(0x7f) &= ~(val))

// FIFOx Map Register, 13.6.2.7.1
#define mUsbFIFOMap(FIFOn, MAP)			(bFUSBPort(0x80 + FIFOn) = MAP)
#define mUsbFIFOMapRd(FIFOn)			(bFUSBPort(0x80 + FIFOn))

// FIFOx Configuration Register, 13.6.2.7.2
#define mUsbFIFOConfig(FIFOn, CONFIG)		(bFUSBPort(0x90 + FIFOn) = CONFIG)
#define mUsbFIFOConfigRd(FIFOn)			(bFUSBPort(0x90 + FIFOn))

// FIFOx Instruction Register, 13.6.2.7.3, 13.6.2.7.4
#define mUsbFIFOOutByteCount(fifo_num)		(u16)(((((u16)bFUSBPort(0xA0+fifo_num))&((u16)0x07))<<8) | ((u16)bFUSBPort(0xB0+fifo_num)))
#define mUsbFIFODone(fifo_num)			(bFUSBPort(0xA0+fifo_num) |= ((u8)BIT3))
#define mUsbFIFOReset(fifo_num)			(bFUSBPort(0xA0+fifo_num) |=  BIT4)
#define mUsbFIFOResetOK(fifo_num)		(bFUSBPort(0xA0+fifo_num) &= ~BIT4)

// Data Port Register, 13.6.2.8
#define mUsbRdByte0(FIFOn)			(bFUSBPort(0xC0 + (FIFOn<<2)))
#define mUsbRdWord(FIFOn)			(wFUSBPort(0xC0 + (FIFOn<<2)))
#define mUsbRdDWord(FIFOn)			(dwFUSBPort(0xC0 + (FIFOn<<2)))

#define mDMAUsbRdDWord(FIFOn)			(DMAdwFUSBPort(0xC0 + (FIFOn<<2)))
#define mUsbWrByte0(FIFOn, value)		(bFUSBPort(0xC0 + (FIFOn<<2)) = value)
#define mUsbWrWord(FIFOn, value)		(wFUSBPort(0xC0 + (FIFOn<<2)) = value)
#define mUsbWrDWord(FIFOn, value)		(dwFUSBPort(0xC0 + (FIFOn<<2)) = value)

#endif /* __FUSB220_M_H  */
