#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>


#include "../lara_common.h"
#undef DRV_NAME


#include "hal_isp1181.h"
#include "chap9.h"
#include "pp_usb.h"

extern volatile ControlData_t g_ControlData;
extern volatile control_state_t g_control_state;
extern usb_driver_t g_ud;
volatile usb_device_state_t g_DeviceState = PP_USB_DEVSTAT_ERROR;
volatile uint8_t g_DeviceSuspended = 1;
volatile uint8_t g_DeviceConfig = 0;

void Chap9_busreset(void) {
    g_DeviceState = PP_USB_DEVSTAT_DEFAULT;
    g_DeviceConfig = 0;
}

void Chap9_GetStatus(void) {
    uint8_t bRecipient = g_ControlData.DeviceRequest.bmRequestType & USB_RECIPIENT_MASK;
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;
    uint16_t wIndex = g_ControlData.DeviceRequest.wIndex;
    
    UD("************* +++++++++++++++++++++++ ***********%s, Rec %x, Ind %x\n", __FUNCTION__, bRecipient, wIndex);
    
    if(!(dir)) {
	Chap9_StallEP0InControlWrite();
    }

    if(g_ControlData.DeviceRequest.wValue == 0 && g_ControlData.DeviceRequest.wLength == 2 ) {
	uint8_t Buffer[2];
	switch(bRecipient) {
	  case USB_RECIPIENT_DEVICE:
	      /* Get Status Request to Device should return */
	      /* Remote Wakeup and Self Powered Status */
	      Buffer[0] = 0x01;
	      Buffer[1] = 0x00;
	      Chap9_SingleTransmitEP0(Buffer, 2);
	      break;
	  case USB_RECIPIENT_INTERFACE:
	      /* Get Status Request to Interface should return */
	      /* Zero, Zero (Reserved for future use) */
	      Buffer[0] = 0x00;
	      Buffer[1] = 0x00;
	      Chap9_SingleTransmitEP0(Buffer, 2);
	      break;
	  case USB_RECIPIENT_ENDPOINT:
	      {
		  uint8_t index;
		  uint8_t status;

		  index = (wIndex & 0x0f) + 1;
		  
		  status = hal_GetEndpointStatusWOClear(index);
		  /* Get Status Request to Endpoint should return */
		  /* Halt Status in D0 for Interrupt and Bulk */
		  switch (g_DeviceState) {
		    case PP_USB_DEVSTAT_DEFAULT:
			printk("Illegal get_status request when in Default state\n");
			Chap9_StallEP0InControlRead();
			break;
		    case PP_USB_DEVSTAT_ADDRESS:
			if ((index == EP0OUT) ||
			    (index == EP0IN)) {
			    if (status & ISP1181B_EP_STATUS_EPSTAL) {
				Buffer[0] = 0x01;
			    } else {
				Buffer[0] = 0x00;
			    }
			    Buffer[1] = 0x00;
			    Chap9_SingleTransmitEP0(Buffer, 2);
			} else {
			    printk("Illegal get status to endpoint: %02x when in Adress state\n", index);
			    Chap9_StallEP0InControlRead();
			}
			break;
		    case PP_USB_DEVSTAT_CONFIGURED:
			if (g_ud.eps[index].used) {
			    if (status & ISP1181B_EP_STATUS_EPSTAL) {
				UD("get_status: ep[%02x]: %02x stalled\n", index, status);
				Buffer[0] = 0x01;
			    } else {
				UD("get_status: ep[%02x]: %02x not stalled\n", index, status);
				Buffer[0] = 0x00;
			    }
			    Buffer[1] = 0x00;
			    Chap9_SingleTransmitEP0(Buffer, 2);
			} else {
			    printk("Illegal endpoint requested: %x\n", index);
			    Chap9_StallEP0InControlRead();
			}
			break;
		    default:
			printk("Illegal Device State - Programming Error\n");
			Chap9_StallEP0InControlRead();
			break;
		  }
		  break;
		default:
		    Chap9_StallEP0InControlRead();
	      }
	}
    } else {
	Chap9_StallEP0InControlRead();
    }
}

void Chap9_ClearFeature(void) {
    uint8_t bRecipient = g_ControlData.DeviceRequest.bmRequestType & USB_RECIPIENT_MASK;
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;
    uint8_t bRequest = g_ControlData.DeviceRequest.bRequest;
    uint16_t wIndex = g_ControlData.DeviceRequest.wIndex;
    uint16_t wValue = g_ControlData.DeviceRequest.wValue;
    
    UD("%s\n", __FUNCTION__);
    
    if(dir) {
	Chap9_StallEP0InControlRead();
    }
    
    if(g_ControlData.DeviceRequest.wLength == 0 ) {
	switch(bRecipient) {
	  case USB_RECIPIENT_DEVICE:
	      /* We don't support DEVICE_REMOTE_WAKEUP or TEST_MODE */
	      printk("standard device request wV: %x - set/clear feature - not supported\n", wValue);
	      Chap9_StallEP0InControlWrite();
	      break;
	  case USB_RECIPIENT_INTERFACE:
	      /* there is no feature selector for interfaces */
	      printk("*** standard interface request - set/clear feature - no feature\n");
	      Chap9_StallEP0InControlWrite();
	      break;
	  case USB_RECIPIENT_ENDPOINT:
	      /* Halt(Stall) feature required to be implemented on all Interrupt and */
	      /* Bulk Endpoints. It is not required nor recommended on the Default Pipe */
	      if (wValue != 0) {
		  Chap9_StallEP0InControlWrite();
	      } else {
		  u_int8_t index;
		  /* this function is called for both cases: set/clear feature */
		  u_int8_t action = (bRequest == SET_FEATURE) ? 1 : 0;
		  
		  index = (wIndex & 0x0f) + 1;
		  
		  if ((index == EP0OUT) ||
		      (index == EP0IN) ||
		      ((EP1 <= index) && (index <= EP14))) {
		      // valid endpoint
		      /* here could be some special handling, if the endpoint should not
			 be unstalled */
		      UD("stall/unstall: %d, index: %x\n", action, index);
		      if (action) {
			  // stall
			  hal_SetEndpointStatus(index, action);
			  Chap9_SingleTransmitEP0(0, 0);
		      } else if ((g_ud.eps[index].allow_unstall_cb == NULL) ||
				 (g_ud.eps[index].allow_unstall_cb(index))
				 ) { //unstall
			  hal_SetEndpointStatus(index, action);
			  Chap9_SingleTransmitEP0(0, 0);
			  if (g_ud.eps[index].unstall_cb) {
			      g_ud.eps[index].unstall_cb(index);
			  }
		      } else {
			  Chap9_SingleTransmitEP0(0, 0);
		      }
		  } else {
		      printk("Invalid Endpoint Request\n");
		      Chap9_StallEP0InControlWrite();
		  }
	      }
	      break;
	  default:
	      Chap9_StallEP0InControlWrite();
	      break;
	}
    } else {
	Chap9_StallEP0InControlWrite();
    }
}

void Chap9_SetFeature(void) {
    Chap9_ClearFeature();
}

void Chap9_SetAddress(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;
    uint8_t address = g_ControlData.DeviceRequest.wValue;
    uint8_t DeviceAddress;

    UD("%s\n", __FUNCTION__);
    
    if(dir) {
	Chap9_StallEP0InControlRead();
    }
    
    UD("*** set address, actual state: %d, to address: %x\n", g_DeviceState, address);
    switch (g_DeviceState) {
      case PP_USB_DEVSTAT_DEFAULT:
	  if (address != 0) {
	      g_DeviceState = PP_USB_DEVSTAT_ADDRESS;
	      DeviceAddress = address | 0x80;
	      hal_SetAddressEnable(DeviceAddress, 1);
	  } else {
	      // remains in Default state
	  }
	  break;
      case PP_USB_DEVSTAT_ADDRESS:
	  if (address == 0) {
	      g_DeviceState = PP_USB_DEVSTAT_DEFAULT;
	      DeviceAddress = address | 0x80;
	      hal_SetAddressEnable(DeviceAddress, 1);
	      } else {
	      DeviceAddress = address | 0x80;
	      hal_SetAddressEnable(DeviceAddress, 1);
	  }
	  break;
      case PP_USB_DEVSTAT_CONFIGURED:
      default:
	  printk("Illegal Set Address Request - do it anyway\n");
	  g_DeviceState = PP_USB_DEVSTAT_ADDRESS;
	  DeviceAddress = address | 0x80;
	  hal_SetAddressEnable(DeviceAddress, 1);

    }
    Chap9_SingleTransmitEP0(0, 0);
}
	
void Chap9_GetConfiguration(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;

    UD("%s\n", __FUNCTION__);
    
    if(!(dir)) {
	Chap9_StallEP0InControlWrite();
    }

    if(g_ControlData.DeviceRequest.wValue == 0 &&
       g_ControlData.DeviceRequest.wLength == 1 &&
       g_ControlData.DeviceRequest.wIndex == 0 ) {
	u_int8_t config;
	//pp_log("*** standard device request - get configuration| state: %d\n", DeviceState);
	switch (g_DeviceState) {
	  case PP_USB_DEVSTAT_DEFAULT:
	      config = 0;
	      //pp_log("Device behavior not specified when GetConfiguration while in Default State\n");
	      break;
	  case PP_USB_DEVSTAT_ADDRESS:
	      config = 0;
	      break;
	  case PP_USB_DEVSTAT_CONFIGURED:
	      config = g_DeviceConfig;
	      break;
	  default:
	      //pp_log("Illegal Device State - Programming Error\n");
	      config = 0;
	}
	Chap9_SingleTransmitEP0(&config, 1);
    } else {
	Chap9_StallEP0InControlRead();
    }
}


void Chap9_SetConfiguration(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;
    uint16_t wValue =g_ControlData.DeviceRequest.wValue;

    UD("%s\n", __FUNCTION__);
    
    if(dir) {
	Chap9_StallEP0InControlWrite();
    }

    endpoint_init();
		   
    if(g_ControlData.DeviceRequest.wLength == 0 &&
       g_ControlData.DeviceRequest.wIndex == 0) {
	u_int8_t config = wValue & 0xFF;
	//UD("*** set configuration: %d\n", DeviceState);
	
	switch (g_DeviceState) {
	  case PP_USB_DEVSTAT_DEFAULT:
	      printk("Device behavior not specified when SetConfiguration in Default State, do it anyway\n");
	      //Chap9_StallEP0InControlWrite();
	      //break;
	  case PP_USB_DEVSTAT_ADDRESS:
	      if (config == 0) {
		  // remain in ADDRESSED state
		  Chap9_SingleTransmitEP0(0, 0);
	      } else if (config == 1) { /* we have only one configuration */
		  // select this configuration
		  g_DeviceConfig = config;
		  g_DeviceState = PP_USB_DEVSTAT_CONFIGURED;
		  g_DeviceSuspended = 0;
		  Chap9_SingleTransmitEP0(0, 0);
		  propchange_fire(PP_PROP_USB_DEV_STATE_CHANGED, 0);
	      } else {
		  // request error
		  printk("Illegal Configuration requested\n");
		  Chap9_StallEP0InControlWrite();
	      }
	      break;
	  case PP_USB_DEVSTAT_CONFIGURED:
	      if (config == 0) {
		  // enteres ADRESSED state
		  g_DeviceState = PP_USB_DEVSTAT_ADDRESS;
		  Chap9_SingleTransmitEP0(0, 0);
	      } else if (config == 1) {
		  // select this config
		  g_DeviceConfig = config;
		  Chap9_SingleTransmitEP0(0, 0);
	      } else {
		  // request error
		  printk("Illegal Configuration requested\n");
		  Chap9_StallEP0InControlWrite();
	      }
	      break;
	  default:
	      printk("Illegal Device State - Programming Error\n");
	      Chap9_StallEP0InControlWrite();
	}
    } else {
	Chap9_StallEP0InControlWrite();
    }
				     
}


void Chap9_GetInterface(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;

    UD("%s\n", __FUNCTION__);
    
    if(!(dir)) {
	Chap9_StallEP0InControlWrite();
    }

    if(g_ControlData.DeviceRequest.wValue == 0 &&
       g_ControlData.DeviceRequest.wLength == 1) {
	uint8_t alt_setting;
	alt_setting = 0; /* Alternative Setting */
	Chap9_SingleTransmitEP0(&alt_setting, 1);
    } else {
	Chap9_StallEP0InControlRead();
    }
}

void Chap9_SetInterface(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;

    UD("%s\n", __FUNCTION__);

    if(dir) {
	Chap9_StallEP0InControlRead();
    }

    // we don't support alternative settings
    Chap9_SingleTransmitEP0(0,0);
}


void GetDescriptor(uint16_t wValue,
		   uint16_t wIndex,
		   uint16_t wLength)
{
    uint8_t length;
    UD("%s: wV: %x, wI: %x, wL: %x\n", __FUNCTION__, wValue, wIndex, wLength);
    switch((wValue & 0xFF00) >> 8) {

      case TYPE_DEVICE_DESCRIPTOR:
	  UD("device descriptor\n");
	  memcpy(&length, g_ud.DeviceDescriptor, 1);
	  Chap9_BurstTransmitEP0(g_ud.DeviceDescriptor, length);
	  break;
      case TYPE_CONFIGURATION_DESCRIPTOR:
	  Chap9_BurstTransmitEP0(g_ud.ConfigDescriptor,
				 le16_to_cpu(((PUSB_CONFIGURATION_DESCRIPTOR)g_ud.ConfigDescriptor)->wTotalLength));
	  break;
      case TYPE_STRING_DESCRIPTOR:
	  {
	      int index = wValue & 0xff;
	      if (index >= MAX_STRING_DESCRIPTORS) {
		  Chap9_SingleTransmitEP0(NULL, 0);
	      } else {
		  if (g_ud.StringDescriptors[index]) {
		      memcpy(&length, g_ud.StringDescriptors[index], 1);
		      Chap9_BurstTransmitEP0(g_ud.StringDescriptors[index], length);
		  } else {
		      Chap9_SingleTransmitEP0(NULL, 0);
		  }
	      }
	  }
	  break;
      default:
	  if (g_ud.non_std_descriptors) {
	      g_ud.non_std_descriptors(wValue, wIndex, wLength);
	  } else {
	      printk("unknown non_std_desc\n");
	      Chap9_StallEP0InControlRead();
	  }
    }
}

void Chap9_GetDescriptor(void) {
    uint8_t dir = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK;

    UD("%s\n", __FUNCTION__);
    
    if(!(dir)) {
	Chap9_StallEP0InControlWrite();
    }

    UD("GetDescriptor: wV: %x, wI: %x, wL: %x\n", g_ControlData.DeviceRequest.wValue,
       g_ControlData.DeviceRequest.wIndex,
       g_ControlData.DeviceRequest.wLength);
    GetDescriptor(g_ControlData.DeviceRequest.wValue,
		  g_ControlData.DeviceRequest.wIndex,
		  g_ControlData.DeviceRequest.wLength);
}


void Chap9_StallEP0(void) {
    uint8_t dir;

    dir = g_ControlData.DeviceRequest.bmRequestType & USB_ENDPOINT_DIRECTION_MASK;
    
    if(!dir && g_ControlData.DeviceRequest.wLength != 0) {
	hal_StallEP0InControlRead();
    } else {
	hal_StallEP0InControlWrite();
    }
    g_control_state = STALL;
}


void Chap9_StallEP0InControlWrite(void) {
    hal_StallEP0InControlWrite();
    g_control_state = STALL;
}

void Chap9_StallEP0InControlRead(void) {
    hal_StallEP0InControlRead();
    g_control_state = STALL;
}

void Chap9_SingleTransmitEP0(const uint8_t* buf, const uint16_t len) {
    hal_SingleTransmitEP0(buf, len);
    g_ControlData.wLength = g_ControlData.wCount = len;
    g_control_state = HANDSHAKE;
}


void Chap9_BurstTransmitEP0(const uint8_t * pRomData, const uint16_t len) {
    g_ControlData.wCount = 0;
    if(g_ControlData.wLength > len) {
	g_ControlData.wLength = len;
    }
    
    g_ControlData.pData = (uint8_t*)pRomData;
    
    if(g_ControlData.wLength >= ISP1181B_EP0IN_FIFO_SIZE) {
	hal_WriteEndpoint(EP0IN, g_ControlData.pData, ISP1181B_EP0IN_FIFO_SIZE);
	g_ControlData.wCount += ISP1181B_EP0IN_FIFO_SIZE;
	g_control_state = DATAIN;
    } else {
	hal_WriteEndpoint(EP0IN, g_ControlData.pData, g_ControlData.wLength);
	g_ControlData.wCount += g_ControlData.wLength;
	g_control_state = HANDSHAKE;
    }
}
