
/* this is the implementation of the HID mouse and keyboard 
   Reference: Universal Serial Bus
   Device Class Definition
   for Human Interface
   Devices (HID)
   Version 1.11
   September 7/27/01
	      
   all sections references to the above specification
*/

/* global includes */
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>

/* firmware includes */

/* local includes */
#include "hid.h"
#include "peppercon_usb.h"
#include "pp_usb.h"
#include "kernel_usb.h"
#include "../lara_common.h"

/* we send only reports when an event occurs, to enable synchronous sending
   of reports, define HID_IDLE and  create an timer: problem, maximal frequence
   of 250 Hz cannot achieved in User space (and hardly in kernel), if it is not
   absolutely necessary, don't do it!
*/
   
#define HID_IDLE

extern usb_driver_t g_ud;
extern lara_usb_dev_t dev;

uint8_t led = 0;

#ifdef HID_IDLE
uint8_t mouse_idle = 0; /* 7.2.4 */
uint8_t keyboard_idle = 125; /* 7.2.4. - 500ms recommended / 4 ms */
volatile uint8_t keyboard_send_report_timeout = 0;
volatile uint8_t mouse_send_report_timeout = 0;
#endif

#define MAX_MOUSE_REPORT 6 /* descriptor dependend */
#define MAX_KBD_REPORT 8 /* descriptor dependend */
#define INT_TIMEOUT 5 /* in 10ms steps */

uint8_t keyboard_protocol = PROTO_REPORT;
unsigned char* keyboard_report[MAX_KBD_REPORT];
uint8_t keyboard_report_length = 0;
unsigned char* mouse_report[MAX_MOUSE_REPORT];
uint8_t mouse_report_length = 0;

static void keyboard_send_report(uint8_t ep);
static void mouse_send_report(uint8_t ep);
static void hid_ep_cb(uint8_t ep);

static wait_queue_head_t hid_wq;

extern int current_mouse_report_length;
extern uint8_t g_setup_dataout[MAX_CONTROLDATA_OUT_SIZE];

void hid_init(void) {
    memset(&keyboard_report, 0, MAX_KBD_REPORT);
    memset(&mouse_report, 0, MAX_MOUSE_REPORT);
    init_waitqueue_head (&hid_wq);
    g_ud.eps[HID_MOUSE_INT_IN_ENDPOINT].isr_int_in_cb = hid_ep_cb;
    g_ud.eps[HID_KEYBOARD_INT_IN_ENDPOINT].isr_int_in_cb = hid_ep_cb;
}

void hid_mouse_handle_classrequest(uint8_t bmRequestType,
				   uint8_t bRequest,
				   uint16_t wValue,
				   uint16_t wIndex,
				   uint16_t wLength) {
    switch (bRequest) {
      case HID_REQ_GET_REPORT: /* 7.2.1 */
	  if (((wValue & 0xff00) >> 8)== 0x01)
	      mouse_send_report(EP0IN);
	  break;
      case HID_REQ_GET_IDLE: /* 7.2.3 */
#ifdef HID_IDLE
	  {
	      char buffer[1];
	      buffer[0] = mouse_idle;
	      Chap9_SingleTransmitEP0(buffer, 1);
	  }
	  break;
#endif
      case HID_REQ_SET_IDLE: /* 7.2.4 */
#ifdef HID_IDLE
	  mouse_idle = (uint8_t) ((wValue & 0xff00) >> 8);
	  Chap9_SingleTransmitEP0(0, 0);
	  break;
#endif
      case HID_REQ_GET_PROTOCOL: /* not supported because the mouse isn't in the boot subclass */
      case HID_REQ_SET_PROTOCOL: /* 7.2.6 */
      case HID_REQ_SET_REPORT: /* 7.2.2, optional Appendix G */
      default:
	  Chap9_StallEP0();
	  break;
    }
}

void
hid_keyboard_handle_classrequest(uint8_t bmRequestType,
				 uint8_t bRequest,
				 uint16_t wValue,
				 uint16_t wIndex,
				 uint16_t wLength) {
    switch (bRequest) {
      case HID_REQ_GET_REPORT:
	  if (((wValue & 0xff00) >> 8) == 0x01)
	      keyboard_send_report(EP0IN);
	  break;

      case HID_REQ_SET_REPORT:
	  {
	      led = g_setup_dataout[0];
	      printk("led: %02x\n", led);
              propchange_fire(PP_PROP_KBD_LED_STATE_CHANGED, 0);
	      Chap9_SingleTransmitEP0(0, 0);
	  }
	  break;
      case HID_REQ_GET_PROTOCOL:
	  {
	      uint8_t proto;
	      proto = keyboard_protocol;
	      Chap9_SingleTransmitEP0(&proto, 1);
	  }
	  break;
      case HID_REQ_SET_PROTOCOL: /* 7.2.6 */
	  keyboard_protocol = (uint8_t) (wValue & 0xff);
	  Chap9_SingleTransmitEP0(NULL, 0);
	  break;
      case HID_REQ_GET_IDLE:
#ifdef HID_IDLE
	  {
	      char buffer[1];
	      buffer[0] = keyboard_idle;
	      Chap9_SingleTransmitEP0(buffer, 1);
	  }
	  break;
#endif
      case HID_REQ_SET_IDLE:
#ifdef HID_IDLE
      	  keyboard_idle = (uint8_t) ((wValue & 0xff00) >> 8);
	  Chap9_SingleTransmitEP0(0, 0);
	  break;
#endif
      default:
	  Chap9_StallEP0();
	  break;
    }
}

void hid_mouse_set_pointer(const uint8_t *data, const int length) {
    if (length > MAX_MOUSE_REPORT || length != current_mouse_report_length) {
	printk("wrong data length: %d (%d expected)\n", length, current_mouse_report_length);
	return;
    }
    memcpy(&mouse_report, data, length);
    mouse_report_length = length;
    mouse_send_report(HID_MOUSE_INT_IN_ENDPOINT);
}

void hid_keyboard_set_report(const uint8_t* data, const int length) {
    /* keyboard reports are always 8 bytes long! */
    if (length  != MAX_KBD_REPORT) {
	printk("wrong data length: %d (%d expected)\n", length, MAX_KBD_REPORT);
	return;
    }
    /* only send a new report, when the data has changed,
       otherwise win2k produces bug #470, because
       it seems that it resets the counter before start repeating to zero,
       when a new USB event is sent */
    if (!memcmp(keyboard_report, data, length)) {
	/* old and new report are equal - don't sent it */
	return;
    }
    memcpy(&keyboard_report, data, length);
    keyboard_report_length = length;
    keyboard_send_report(HID_KEYBOARD_INT_IN_ENDPOINT);
}

void hid_send_report(uint8_t ep, const uint8_t* data, uint16_t len, uint8_t last_packet) {
    long time_left;
    DECLARE_WAITQUEUE (wait, current);
    UD("hid_send_report\n");

    if (g_DeviceState != PP_USB_DEVSTAT_CONFIGURED) {
	UD("**  **  device not configured\n");
	return;
    }
    
    if (last_packet) {
	/* resend the last packet */
	data = g_ud.eps[ep].last_packet;
	len = g_ud.eps[ep].last_length;
    }

    g_ud.eps[ep].waiting = 1;

    /* wait for free slot */
    while (g_ud.eps[ep].status != EVENT_IDLE) {
	/* sleep here until delivery was successful or timeout */
	/* wait a little bit longer than the timeout of the last request after sending the data */
	time_left = (INT_TIMEOUT) + 2;
	UD("start waiting\n");
	do {
	    current->state = TASK_INTERRUPTIBLE;
	    add_wait_queue(&hid_wq, &wait);
	    spin_unlock_irq(&dev.chip_access_lock);
	    time_left = schedule_timeout(time_left);
	    remove_wait_queue(&hid_wq, &wait);
	    spin_lock_irq(&dev.chip_access_lock);
	    UD("time_left: %lx\n", time_left);
	} while (time_left && g_ud.eps[ep].status != EVENT_IDLE);
	
	if (time_left == 0 && g_ud.eps[ep].status != EVENT_IDLE) {
	    /* only if the data are less than the buffer size */
	    if (len <= 64) {
		g_ud.eps[ep].timedout = 1; /* schedule resend */
		g_ud.eps[ep].last_length = len;
		memcpy(g_ud.eps[ep].last_packet, data, len);
	    }
	    g_ud.eps[ep].waiting = 0;
	    g_ud.eps[ep].status = EVENT_IDLE;
	    UD("timedout\n");
	    return;
	}
    }
    g_ud.eps[ep].waiting = 0;
    g_ud.eps[ep].timedout = 0;
    UD("hid_send_report - send data: ep: 0x%0x, len: %d, data: %p\n", ep, len, data);
    WriteEndpoint(ep, data, len);
    g_ud.eps[ep].status = EVENT_QUEUED;
    UD("hid_send_report - sent data - wait for delivery\n");
    if (!last_packet) {
	// not call within an interrupt
	time_left = INT_TIMEOUT;
	do {
	    UD("wait for reply\n");
	    current->state = TASK_INTERRUPTIBLE;
	    add_wait_queue(&hid_wq, &wait);
	    spin_unlock_irq(&dev.chip_access_lock);
	    time_left = schedule_timeout(time_left);
	    remove_wait_queue(&hid_wq, &wait);
	    spin_lock_irq(&dev.chip_access_lock);
	} while (time_left && g_ud.eps[ep].status != EVENT_IDLE);
	if (time_left == 0 && g_ud.eps[ep].status != EVENT_IDLE) {
	    UD("hid_send_report Timeout getting reply\n");
	}
	UD("hid_send_report done: time_left: %ld\n", time_left);
    }

}

void hid_ep_cb(uint8_t ep) {
    UD("hid_ep_cb\n");
    g_ud.eps[ep].status = EVENT_IDLE;
    /* signal */
    wake_up_interruptible(&hid_wq);
    if (g_ud.eps[ep].timedout && !g_ud.eps[ep].waiting) {
	/* there was a timeout and no other event is waiting
          so just resent the last packet */
	hid_send_report(ep, NULL, 0, 1);
    }
}

void mouse_send_report(uint8_t ep) {
    /* write the packet only synced, when we send to to hid endpoint (and not to EP0IN),
       otherwise we are "synced" in the main loop */
    if (ep == EP0IN) {
	Chap9_BurstTransmitEP0((uint8_t*)&mouse_report, mouse_report_length);
    } else {
	/* here we are synced via irq_lock */
	hid_send_report(ep, (uint8_t*) &mouse_report, mouse_report_length, 0);
    }
#ifdef HID_IDLE
    //    timer_start();
    mouse_send_report_timeout = 0;
#endif
}

void keyboard_send_report(uint8_t ep)
{
    if (ep == EP0IN) {
	Chap9_BurstTransmitEP0((uint8_t*)&keyboard_report, keyboard_report_length);
    } else {
	hid_send_report(ep, (uint8_t*) &keyboard_report, keyboard_report_length, 0);
    }
#ifdef HID_IDLE
    //       timer_start();
    keyboard_send_report_timeout = 0;
#endif
}

#ifdef HID_IDLE
void
hid_main_loop_callback(void) {
    if (keyboard_idle) { // only if keyboard_idle > 0 (keyboard_idle == 0 means, that there are no synchron reports */
	keyboard_send_report_timeout++;
	if (keyboard_send_report_timeout >= keyboard_idle) {
	    keyboard_send_report(HID_KEYBOARD_INT_IN_ENDPOINT);
	}
    }
    
    if (mouse_idle) {
	mouse_send_report_timeout++;
	if (mouse_send_report_timeout >= mouse_idle ) {
	    mouse_send_report(HID_MOUSE_INT_IN_ENDPOINT);
	}
    }
}
#endif
