/* this is the implementation of the Bulk-Only Transport specification of a USB Mass Storage Device
   Reference: Universal Serial Bus
              Mass Storage Class
	      Bulk-Only Transport
	      Revision: 1.0
	      September 31, 1999
  all sections references to the above specification
*/

#include "mass-storage-bo.h"
#include "pp_usb.h"
#include "chap9.h"
#include "scsi.h"
#include "hal_isp1181.h"

static int allocated = 0;
static bot_t bots[PP_FEAT_USB_MASS_STORAGE_NO];

extern usb_driver_t g_ud;

void TPBulk_CSWHandler(void);
static void dataout_handler(uint8_t ep);
static void ms_bo_cbw_handler(bot_t * bot);
static void csw_handler(bot_t * bot);
static void datain_handler(bot_t * bot);
static void state_changed(bot_t * bot);
static unsigned char is_cbw_valid(bot_t * bot, uint8_t len);

static bot_t * get_bot_from_ep(uint8_t ep) {
    int i;
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
    	if (bots[i].ep_in == ep || bots[i].ep_out == ep) {
    	    return &bots[i];
    	}
    }
    return NULL;
}

/* initialization */
static void ms_bo_init_by_index(int i) {
    g_ud.file_req[i].cmd = PP_USB_NOTHING;
    
    bots[i].index = i;
    bots[i].ep_in = EP_MS_BULK_IN(i);
    bots[i].ep_out = EP_MS_BULK_OUT(i);
    
    bots[i].file_transfer = 0;

    g_ud.file_req[i].length = 0;
    g_ud.file_req[i].block_length = 0;
    g_ud.file_req[i].block_address = 0;
    
    g_ud.eps[EP_MS_BULK_OUT(i)].unstall_cb = ms_bo_unstall_ep_out;
    g_ud.eps[EP_MS_BULK_OUT(i)].ioCount = 0;
    g_ud.eps[EP_MS_BULK_OUT(i)].status = 1;
    g_ud.eps[EP_MS_BULK_OUT(i)].main_bulk_out_cb = ms_bo_data_out_interrupt;
    g_ud.eps[EP_MS_BULK_OUT(i)].ioCount = 0;
    g_ud.eps[EP_MS_BULK_OUT(i)].ioSize = sizeof(bots[i].cbw);
    g_ud.eps[EP_MS_BULK_OUT(i)].bufSize = 64;
    g_ud.eps[EP_MS_BULK_OUT(i)].pData = (uint8_t*)&bots[i].cbw;
    g_ud.eps[EP_MS_BULK_OUT(i)].reset = 1;
    
    g_ud.eps[EP_MS_BULK_OUT(i)].allow_unstall_cb = ms_bo_allow_unstall_ep;
    
    g_ud.eps[EP_MS_BULK_IN(i)].unstall_cb = ms_bo_unstall_ep_in;
    g_ud.eps[EP_MS_BULK_IN(i)].ioCount = 0;
    g_ud.eps[EP_MS_BULK_IN(i)].ioSize = 0;
    g_ud.eps[EP_MS_BULK_IN(i)].status = 0;
    g_ud.eps[EP_MS_BULK_IN(i)].bufSize = 0;
    g_ud.eps[EP_MS_BULK_IN(i)].pData = NULL;
    g_ud.eps[EP_MS_BULK_IN(i)].main_bulk_in_cb = ms_bo_data_in_interrupt;
    
    g_ud.eps[EP_MS_BULK_IN(i)].allow_unstall_cb = ms_bo_allow_unstall_ep;

    /* de-stalling of endpoint is done by the host (5.3.4) */
    bots[i].BOTFSMstate = USBFSM4BOT_IDLE;
    scsi_reset(bots[i].scsi);
    state_changed(&bots[i]);
}

void ms_bo_init() {
    int i;
    
    // initialize structures
    for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	if (!allocated) bots[i].scsi = scsi_new(i);
    	ms_bo_init_by_index(i);
    }
    allocated = 1;
}

void ms_bo_exit()
{
    int i;
    if (allocated) {
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    scsi_free(bots[i].scsi);
	}
    }
    allocated = 0;
}

void ms_bo_set_file(int ms_index, uint32_t lba, uint32_t blocklen, usb_device_type_t type, uint8_t readonly)
{
    scsi_set_file(bots[ms_index].scsi, lba, blocklen, type, readonly);
}

static void fill_csw(bot_t * bot, transfer_phase_t  data_phase, uint32_t data_length, uint8_t status) {

    uint8_t HostDevCase = CASECMDFAIL;
    bot->csw.dCSWSignature = cpu_to_le32(CSW_SIGNATURE);
    bot->csw.dCSWTag = bot->cbw.dCBWTag; /* only copied - so don't care endianess */
    /* warning check this */
    bot->csw.dCSWDataResidue = cpu_to_le32(le32_to_cpu(bot->cbw.dCBWDataTransferLength) - data_length);
    bot->DummyRead = 0;
    bot->BOTXfer_wResidue = data_length;
    
    UD("fill_csw: hl: %08x, dl: %08lx, dresidue: %08x, \n",
       le32_to_cpu(bot->cbw.dCBWDataTransferLength),
       (unsigned long) data_length,
       le32_to_cpu(bot->csw.dCSWDataResidue));

    /* set the next state default:
       - host doesn't expect data
       - host expects data
       - host sends data
    */

    switch(data_phase) {
      case PHASE_DEVICE_NONE:
	  bot->BOTFSMstate = USBFSM4BOT_CSW;
	  if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) == 0) {
	      HostDevCase = CASE1;
	  } else if (bot->cbw.bmCBWFlags & CBW_FLAG_IN) {
	      HostDevCase = CASE4;
	  } else {
	      HostDevCase = CASE9;
	  }
	  break;
      case PHASE_DEVICE_IN:
	  bot->BOTFSMstate = USBFSM4BOT_DATAIN;
	  if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) == 0) {
	      HostDevCase = CASE2;
	      bot->BOTFSMstate = USBFSM4BOT_CSW;
	  } else if (bot->cbw.bmCBWFlags & CBW_FLAG_IN) {
	      if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) == data_length) {
		  HostDevCase = CASE6;

	      } else if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) > data_length) {
		  HostDevCase = CASE5;
	      } else {
		  HostDevCase = CASE7;
		  bot->csw.dCSWDataResidue = 0;
		  bot->BOTXfer_wResidue = le32_to_cpu(bot->cbw.dCBWDataTransferLength);
	      }
	  } else {
	      HostDevCase = CASE10;
	      bot->DummyRead = 1;
	      bot->BOTFSMstate = USBFSM4BOT_DATAOUT;
	      bot->csw.dCSWDataResidue = 0;
	      bot->BOTXfer_wResidue = le32_to_cpu(bot->cbw.dCBWDataTransferLength);
	      bot->complete_residue = le32_to_cpu(bot->cbw.dCBWDataTransferLength);
		  
	  }
	  break;
      case PHASE_DEVICE_OUT:
	  bot->BOTFSMstate = USBFSM4BOT_DATAOUT;
	  if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) == 0) {
	      HostDevCase = CASE3;
	      bot->BOTFSMstate = USBFSM4BOT_CSW;
	  } else if (bot->cbw.bmCBWFlags & CBW_FLAG_IN) {
	      HostDevCase = CASE8;
	      bot->BOTFSMstate = USBFSM4BOT_CSW;
	  } else {
	      if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) > data_length) {
		  HostDevCase = CASE11;
	      } else if (le32_to_cpu(bot->cbw.dCBWDataTransferLength) == data_length) {
		  HostDevCase = CASE12;
	      } else {
		  HostDevCase = CASE13;
	      }
	  }
	  bot->complete_residue = min(data_length, le32_to_cpu(bot->cbw.dCBWDataTransferLength));
	  
	  break;
      default:
	  printk("should not happened\n");
    }

    UD("case: %d\n", HostDevCase);
    
    bot->BOTBF_StallAtBulkIn = 0;
    bot->BOTBF_StallAtBulkOut = 0;

    switch(HostDevCase) {
      case CASEOK:
      case CASE1:     /* Hn=Dn*/
      case CASE6:     /* Hi=Di*/
	  bot->csw.bCSWStatus = CSW_GOOD;
	  break;
      case CASE12:    /* Ho=Do*/
	  bot->csw.bCSWStatus = CSW_GOOD;
	  break;
      case CASE2:     /* Hn<Di*/
      case CASE3:     /* Hn<Do*/
	  bot->BOTBF_StallAtBulkIn = 1; // may or may-not
	  bot->csw.bCSWStatus = CSW_PHASE_ERROR;
	  break;
      case CASE4:     /* Hi>Dn*/
      case CASE5:     /* Hi>Di*/
	  bot->BOTBF_StallAtBulkIn = 1;
	  bot->csw.bCSWStatus = CSW_GOOD;
	  break;
      case CASE7:     /* Hi<Di*/
       case CASE8:     /* Hi<>Do */
	  bot->BOTBF_StallAtBulkIn = 1; // may or may-not
	  bot->csw.bCSWStatus = CSW_PHASE_ERROR;
	  break;
      case CASE9:     /* Ho>Dn*/
      case CASE11:    /* Ho>Do*/
	  bot->BOTBF_StallAtBulkOut = 1; // may or may-not
	  bot->csw.bCSWStatus = CSW_FAIL;//CSW_GOOD or CSW_FAIL
	  break;
      case CASE10:    /* Ho<>Di */
	  //	  bot->BOTBF_StallAtBulkOut = 1;
	  bot->csw.bCSWStatus = CSW_PHASE_ERROR;
	  break;
      case CASE13:    /* Ho<Do*/
	  //bot->BOTBF_StallAtBulkIn = 1;// may or may-not
	  bot->BOTBF_StallAtBulkOut = 1;// may or may-not
	  bot->csw.bCSWStatus = CSW_PHASE_ERROR;
	  break;
      case CASECMDFAIL:
	  bot->BOTBF_StallAtBulkIn = 1;
	  bot->BOTBF_StallAtBulkOut = 1;// may or may-not
	  bot->csw.bCSWStatus = CSW_FAIL;
	  break;
      default:
	  break;
    }

    if (!status) {
	bot->csw.bCSWStatus = CSW_FAIL;
    }

    UD("fill_csw done: stallBI: %x, stallBO: %x, Csw_state: %x, next BOT state: %x, case: %d\n",
       bot->BOTBF_StallAtBulkIn, bot->BOTBF_StallAtBulkOut,
       bot->csw.bCSWStatus,
       bot->BOTFSMstate,
       HostDevCase);
}

/* class specific requests */
void ms_bo_handle_classrequest(uint8_t bmRequestType, uint8_t bRequest,
			       uint16_t wValue, uint16_t wIndex, uint16_t wLength) {
    UD("ms_bo_handle_classreq\n");
    switch (bRequest) {
      case BO_MASS_STORAGE_RESET:
	  {
	      int is_for_us = 0;
	      
	      if (wValue == 0 && wLength == 0) {
	          int i;
	          for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	              if (wIndex == i) {
	                  is_for_us = 1;
	              }
	          }
	      }
	      
	      if (!is_for_us) {
		  Chap9_StallEP0();
		  printk("*** broken mass storage reset - don't reset\n");
	      } else {
		  UD("*** mass storage reset\n");
		  ms_bo_init_by_index(wIndex);
		  Chap9_SingleTransmitEP0(NULL, 0);
	      }
	  }
	  break;
      case BO_MASS_STORAGE_GET_MAX_LUN:
	  {
	      int is_for_us = 0;

	      UD("** mass storage get_max_ung: value: %x, index: %x, len: %x\n", wValue, wIndex, wLength);

	      if (wValue == 0 && wLength == 1) {
	          int i;
	          for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	              if (wIndex == i) {
	                  is_for_us = 1;
	              }
	          }
	      }

	      if (!is_for_us) {
		  Chap9_StallEP0();
		  printk("broken getmaxlun request - stalling\n");
		  //		  Chap9_StallEP0InControlRead();
	      } else {
		  unsigned char Buffer[] = { 0x00 };
		  // we don't support multiple LUNs, just send "zero"
		  UD("ok getmaxlun request - sending one byte\n");
		  Chap9_SingleTransmitEP0(Buffer, 1);
	      }
	  }
	  break;
      default:
	  printk("unknow mass storage bulk only class request: reqtype: %x, req: %x, val: %hx, wI: %x, wL: %x\n", bmRequestType, bRequest, wValue, wIndex, wLength);
	  Chap9_StallEP0();
	  break;
    }
}

static void ms_bo_cbw_handler(bot_t * bot) {
    transfer_phase_t data_phase;
    uint32_t data_length;
    uint8_t status;

    UD("--MASS-- %s\n", __FUNCTION__);
    bot->cbw_len = g_ud.eps[bot->ep_out].ioCount;
    if (bot->cbw_len == 0) return;
    
    if(is_cbw_valid(bot, bot->cbw_len)) {
	UD("cbw valid\n");
	/* this calls the block oriented protocol, regardless what it is */
	status = scsi_handler(bot, (uint8_t*)&bot->cbw.CBWCB, bot->cbw.bCBWCBLength,
			      le32_to_cpu(bot->cbw.dCBWDataTransferLength),
			      (bot->cbw.dCBWDataTransferLength > 0)? ((bot->cbw.bmCBWFlags & CBW_FLAG_IN)? PHASE_DEVICE_IN : PHASE_DEVICE_OUT) : PHASE_DEVICE_NONE,
			      NULL, &data_phase, &data_length);
	UD("scsi handler returns: data_phase: %x, data_length: %x, status: %x\n", data_phase, data_length, status);
	if (status == 4) {
	    // read/write - wait for user space
	    fill_csw(bot, data_phase, data_length, status);
	    bot->complete_residue = (data_length < le32_to_cpu(bot->cbw.dCBWDataTransferLength)) ? data_length : le32_to_cpu(bot->cbw.dCBWDataTransferLength);

	    UD("initial complete_resi: %x\n", bot->complete_residue);
	    bot->complete_length = bot->complete_residue;
	    bot->BOTFSMstate = USBFSM4BOT_WAITFORFILE;
	    bot->file_transfer = 1;
	    state_changed(bot);
	    pp_usb_wakeup_file_process();
	    return;
	} else {
	    bot->file_transfer = 0;
	}
	UD("ms_bo_cbw_handler, file_transfer: %x\n", bot->file_transfer);
	fill_csw(bot, data_phase, data_length, status);
	/* sets the correct BOTFSMstate (DATAIN, DATAOUT, CSW)
	   and fills csw */
    } else {
	printk("invalid cbw\n");
	bot->BOTFSMstate = USBFSM4BOT_RESETRECOVERY;
    }
    state_changed(bot);
}


void ms_bo_response(int ms_index, file_response_t* response) {
    transfer_phase_t data_phase;
    uint32_t data_length;
    uint32_t maxbuf = MAX_BULKIN_BUFFER;
    bot_t * bot = &bots[ms_index];
    
    // write case
    if (bot->BOTFSMstate == USBFSM4BOT_WAITFORFILEWRITE) {
	if (response->response == FILE_DATA_OK) {
	    copy_to_user(&response->data[response->length - bot->complete_residue],
			 bot->pDeviceDataBulk,
			 g_ud.eps[bot->ep_out].ioSize);
	    bot->complete_residue -= g_ud.eps[bot->ep_out].ioSize;
	    UD("got %d bytes, await %d\n", g_ud.eps[bot->ep_out].ioSize, bot->complete_residue);
	    if (bot->complete_residue <= 0) {
		/* stupid assumption: if file_req.cmd = SETUP_PROTO_REQ, then we are
		   in "getting-setup-proto-request"-mode and inform the user space
		   otherwise we just get some data from the OS otherwise */
		if (g_ud.file_req[ms_index].cmd == PP_USB_SETUP_PROTO_REQ) {
		    g_ud.file_req[ms_index].cmd = PP_USB_SETUP_PROTO_REQ_DONE;
		    g_ud.file_req[ms_index].length = bot->complete_length;
		    pp_usb_wakeup_file_process();
		}
		if (g_ud.file_req[ms_index].cmd == PP_USB_WRITE) {
		    g_ud.file_req[ms_index].cmd = PP_USB_WRITE_DONE;
		    pp_usb_wakeup_file_process();
		}
		bot->BOTFSMstate = USBFSM4BOT_CSW;
	    } else {
		bot->BOTFSMstate = USBFSM4BOT_DATAOUT;
	    }
	    state_changed(bot);
	}
    } else if (bot->BOTFSMstate == USBFSM4BOT_WAITFORFILE) {
	// read case
	if (response->response == FILE_DATA_OK || response->response == FILE_DATA_OK_SHORT) {
	    UD("ms_bo_resp: resplne: %x, completeresidue: %x, maxbuf: %x\n",
	       response->length, bot->complete_residue, maxbuf);
	    if (response->response == FILE_DATA_OK_SHORT) {
		// only some bytes left
		bot->complete_residue = min(bot->complete_residue, response->length);
	    }
	    data_length = min(response->length, bot->complete_residue);
	    data_phase = PHASE_DEVICE_IN;
	    /* copy only a necessary junk */
	    fill_csw(bot, data_phase, response->length, 1);
	    copy_from_user(bot->pDeviceDataBulk,
			   &response->data[response->length - bot->complete_residue],
			   min(maxbuf, bot->complete_residue));
	} else {
	    data_phase = PHASE_DEVICE_NONE;
	    data_length = 0;
	    fill_csw(bot, data_phase, data_length, 0);
	}
	//	printk("ret from user: compl_res=%d\n", bot->complete_residue);
	bot->BOTXfer_wResidue = min(maxbuf, bot->complete_residue);
	bot->BOTXfer_pData = bot->pDeviceDataBulk;
	state_changed(bot);
    } else {
    	g_ud.file_req[bot->index].cmd = PP_USB_NOTHING;
    }
}

void ms_bo_unstall_ep_in(uint8_t ep) {
    bot_t * bot = get_bot_from_ep(ep);
    
    UD("--MASS-- %s\n", __FUNCTION__);
    bot->BOTBF_StallAtBulkIn = 0;
    if (bot->BOTFSMstate == USBFSM4BOT_CSWSTALL) {
	bot->BOTFSMstate = USBFSM4BOT_CSWSEND;
	state_changed(bot);
    }
}

void ms_bo_unstall_ep_out(uint8_t ep) {
    bot_t * bot = get_bot_from_ep(ep);
    
    UD("--MASS-- %s\n", __FUNCTION__);
    bot->BOTBF_StallAtBulkOut = 0;
}

uint8_t ms_bo_allow_unstall_ep(uint8_t ep) {
    bot_t * bot = get_bot_from_ep(ep);
    
    return (bot->BOTFSMstate != USBFSM4BOT_RESETRECOVERY);
}
       
static void state_changed(bot_t * bot) {
    uint32_t maxbuf = MAX_BULKIN_BUFFER;
 again:
    UD("next state: %x\n", bot->BOTFSMstate);
    switch (bot->BOTFSMstate) {
      case USBFSM4BOT_WAITFORFILE:
	  // just wait
	  break;
      case USBFSM4BOT_WAITFORFILEWRITE:
	  // just wait
	  break;
      case USBFSM4BOT_IDLE:
	  g_ud.file_req[bot->index].cmd = PP_USB_NOTHING;
	  break;
      case USBFSM4BOT_RESETRECOVERY:
	  bot->BOTBF_StallAtBulkIn = 1;
	  bot->BOTBF_StallAtBulkOut = 1;
	  hal_SetEndpointStatus(bot->ep_out, 1);
	  hal_SetEndpointStatus(bot->ep_in, 1);
	  break;
      case USBFSM4BOT_DATAIN:
	  /* start to send the data */
	  datain_handler(bot);
	  break;
      case USBFSM4BOT_DATAINSENT:
	  /* the last csw data was sent, wait for the corresponding interrupt */
	  break;
      case USBFSM4BOT_DATAOUT:
	  g_ud.eps[bot->ep_out].ioCount = 0;
	  g_ud.eps[bot->ep_out].status = 1;
	  g_ud.eps[bot->ep_out].ioSize = min(maxbuf, bot->complete_residue);
	  g_ud.eps[bot->ep_out].bufSize = 64;
	  g_ud.eps[bot->ep_out].pData = bot->pDeviceDataBulk;
	  pp_usb_wakeup_main_loop();

	  /* do nothing - wait for an interrupt, that the host has sent something */
	  break;
      case USBFSM4BOT_CSW:
	  g_ud.eps[bot->ep_out].ioCount = 0;
	  g_ud.eps[bot->ep_out].ioSize = sizeof(bot->cbw);
	  g_ud.eps[bot->ep_out].status = 1;
	  g_ud.eps[bot->ep_out].bufSize = 64;
	  g_ud.eps[bot->ep_out].pData = (uint8_t*)&bot->cbw;

	  bot->BOTXfer_wResidue = sizeof(bot->csw);
	  bot->BOTXfer_pData = (uint8_t*)&bot->csw;

	  /* stalls the endpoints if necessary */
	  bot->BOTFSMstate = USBFSM4BOT_CSWSEND;

	  if(bot->BOTBF_StallAtBulkOut) {
	      hal_SetEndpointStatus(bot->ep_out, 1);
	  }

	  if(bot->BOTBF_StallAtBulkIn) {
	      hal_SetEndpointStatus(bot->ep_in, 1);
	      bot->BOTFSMstate = USBFSM4BOT_CSWSTALL; 
	  }
	  goto again; /* use goto instead of recursive calling to save stack space */
	  break;
      case USBFSM4BOT_CSWSEND:
	  /* starts to send the first bytes out */
	  csw_handler(bot);
	  break;
      case USBFSM4BOT_CSWSTALL:
	  /* do nothing - this state is unset in ms_bo_unstall_ep_in() */
	  break;
      case USBFSM4BOT_CSWSENT:
	  /* the last csw data was sent, wait for the corresponding interrupt */
	  break;
    }
}

static unsigned char is_cbw_valid(bot_t * bot, uint8_t len) {
    UD("%s: len: %d, LUN: %d, cdbLength: %d, CBWFlags: %x\n", __FUNCTION__, len, bot->cbw.bCBWLUN, bot->cbw.bCBWCBLength, bot->cbw.bmCBWFlags);
    if( len == 31
	&& le32_to_cpu(bot->cbw.dCBWSignature) == CBW_SIGNATURE 
        && bot->cbw.bCBWLUN <= 1 
        && bot->cbw.bCBWCBLength <= MAX_CBLEN )
	return 1;
    else
	return 0;
}

static void csw_handler(bot_t * bot) {
    uint32_t FlexByte;
    UD("cswh:%d\n", bot->BOTXfer_wResidue);

    if (bot->BOTXfer_wResidue >= 64) {
	WriteEndpoint(bot->ep_in, bot->BOTXfer_pData, 64);
	FlexByte = 64;
    } else {
	WriteEndpoint(bot->ep_in, bot->BOTXfer_pData, bot->BOTXfer_wResidue);
	FlexByte = bot->BOTXfer_wResidue;
    }
    
    bot->BOTXfer_pData += FlexByte;
    bot->BOTXfer_wResidue -= FlexByte;
    if(bot->BOTXfer_wResidue == 0) {
	// complete CSW sent */
	bot->BOTFSMstate = USBFSM4BOT_CSWSENT;
	UD("sent");
	state_changed(bot);
    }
    UD("done");
}


static void dataout_handler(uint8_t ep) {
    bot_t * bot = get_bot_from_ep(ep);
    
    UD("%s, sizeof(bot->cbw): %d, io_count: %d, io_size: %d\n", __FUNCTION__, sizeof(bot->cbw),
       g_ud.eps[ep].ioCount, g_ud.eps[ep].ioSize);
       
    if (g_ud.eps[ep].ioCount >= g_ud.eps[ep].ioSize) {
	g_ud.eps[bot->ep_out].status = 0;

	if (bot->DummyRead) {
	    // dummy read
	    bot->complete_residue -= g_ud.eps[bot->ep_out].ioSize;
	    UD("got %d bytes, await %d\n", g_ud.eps[bot->ep_out].ioSize, bot->complete_residue);
	    if (bot->complete_residue <= 0) {
		bot->BOTFSMstate = USBFSM4BOT_CSW;
	    } else {
		bot->BOTFSMstate = USBFSM4BOT_DATAOUT;
	    }
	    state_changed(bot);
	} else {
	    // read all data
	    bot->BOTFSMstate = USBFSM4BOT_WAITFORFILEWRITE;
	    state_changed(bot);
	    pp_usb_wakeup_file_process();
	}
    }
}

static void datain_handler(bot_t * bot) {
    uint32_t FlexByte;
    
    //UD("di %x\n", bot->BOTXfer_wResidue);
    if (bot->BOTXfer_wResidue >= 64) {
	WriteEndpoint(bot->ep_in, bot->BOTXfer_pData, 64);
	FlexByte = 64;
    } else {
	WriteEndpoint(bot->ep_in, bot->BOTXfer_pData, bot->BOTXfer_wResidue);
	FlexByte = bot->BOTXfer_wResidue;
    }

    bot->BOTXfer_pData += FlexByte;
    bot->BOTXfer_wResidue -= FlexByte;
    bot->complete_residue -= FlexByte;
    
    if(bot->BOTXfer_wResidue == 0) {
	UD("datain: wRes = %d\n", bot->BOTXfer_wResidue);
	if (bot->complete_residue == 0 || bot->file_transfer == 0) {
	    UD("complete == 0 or file_trans == 0)");
	    bot->BOTFSMstate = USBFSM4BOT_DATAINSENT;
	    state_changed(bot);
	} else {
	    bot->BOTFSMstate = USBFSM4BOT_SENTWAITFORFILE;
	    state_changed(bot);
	}
    }
}

void ms_bo_data_in_interrupt(uint8_t ep) {
    /* the device has send some data successfully to the host */
    bot_t * bot = get_bot_from_ep(ep);

    if (bot->BOTFSMstate == USBFSM4BOT_CSWSEND) {
	csw_handler(bot);
    } else if (bot->BOTFSMstate == USBFSM4BOT_CSWSENT) {
	bot->BOTFSMstate = USBFSM4BOT_IDLE;
	state_changed(bot);
    } else if (bot->BOTFSMstate == USBFSM4BOT_DATAIN) {
	datain_handler(bot);
    } else if (bot->BOTFSMstate == USBFSM4BOT_DATAINSENT) {
	/* the last one */
	bot->BOTFSMstate = USBFSM4BOT_CSW;
	state_changed(bot);
    } else if (bot->BOTFSMstate == USBFSM4BOT_SENTWAITFORFILE) {
	bot->BOTFSMstate = USBFSM4BOT_WAITFORFILE;
	state_changed(bot);
	pp_usb_wakeup_file_process();
    }
}

void ms_bo_data_out_interrupt(uint8_t ep) {
    bot_t * bot = get_bot_from_ep(ep);
    
    UD("%s --- data_out_intr BOTstate = %d\n", __FUNCTION__, bot->BOTFSMstate);
    /* the host has send some data successfully to the device */
    if (bot->BOTFSMstate == USBFSM4BOT_IDLE) {
	UD("sizeof(bot->cbw): %d, io_count: %d\n", sizeof(bot->cbw), g_ud.eps[ep].ioCount);
	// works only for endpoint sizes >=64
	//if (g_ud.eps[ep].ioCount >= sizeof(bot->cbw)) {
	ms_bo_cbw_handler(bot);
    } else if (bot->BOTFSMstate == USBFSM4BOT_DATAOUT) {
	dataout_handler(ep);
    } else {
	/* the host want to send something but we are in a different state
	   because we don't know, what to do, we stall both
	   endpoints and wait for a reset recovery

	   if the host works correctly this should not happen
	*/
	bot->BOTFSMstate = USBFSM4BOT_RESETRECOVERY;
	UD("got unexpected data -> going into reset recovery state");
	
	state_changed(bot);
	return;
    }
}
