/* linux includes */
#include <linux/version.h>
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
# include <asm/ppc405_pcimcs.h>
#else
# include <syslib/ppc405_pcimcs.h>
#endif
#include <linux/rtc.h>
#include <asm/time.h>
#include <asm/io.h>
#include <asm/irq.h>

#include "../lara_common.h"
#undef DRV_NAME

/* local includes */
#include "pp_usb.h"
#include "chap9.h"
#include "hal_isp1181.h"
#include "peppercon_usb.h"
#include "ripcbelkin_usb.h"
#ifdef PP_FEAT_MASS_STORAGE
#include "mass-storage-bo.h"
#include "scsi.h"
#endif /* PP_FEAT_MASS_STORAGE */

#define DRV_NAME			"pp_usb"

lara_usb_dev_t dev = {
    name:			DRV_NAME,
    isr_registered:		0,
    chip_access_lock:		SPIN_LOCK_UNLOCKED,
    req_wq:			{},
    int_wq:                     {}, 
    wakeup:                     0,
    int_occured:                0,
    usb_thread:                 NULL,
    usb_notify:                 NULL,
    rpc_queue_lock:             SPIN_LOCK_UNLOCKED,
    rpc_queue_elem_cnt:         0,
    rpc_queue:                  {},
};

usb_driver_t g_ud;
extern uint8_t led;

#define MAXRINGBUF (ISP1181B_MAX_EP*2)
static uint8_t ep_int_ringbuf[MAXRINGBUF];
static uint8_t ep_int_next;
static uint8_t ep_int_last;
static uint8_t dummy_buffer[64];
static uint8_t add_mod(uint8_t a1, uint8_t a2, uint8_t maxsize) {
    uint16_t a;
    a = a1+a2;
    return a % maxsize;
}

static void ringbuf_init(void) {
    u_long flags;
    spin_lock_irqsave(&dev.chip_access_lock, flags);
    ep_int_next = 0;
    ep_int_last = 0;
    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
}

static void ringbuf_enqueue(uint8_t ep) {
    u_long flags;
    spin_lock_irqsave(&dev.chip_access_lock, flags);
    if (add_mod(ep_int_next, 1, MAXRINGBUF) != ep_int_last) {
	ep_int_ringbuf[ep_int_next] = ep;
	ep_int_next = add_mod(ep_int_next, 1, MAXRINGBUF);
    } else {
	printk("ring buffer full, event discarded\n");
    }
    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
}

static uint8_t ringbuf_getfirst(void) {
    uint8_t ep;
    
    if (ep_int_next != ep_int_last) {
	ep = ep_int_ringbuf[ep_int_last];
    } else {
	ep = EPNONE;
    }

    return ep;
}

static uint8_t ringbuf_delfirst(void) {
    uint8_t ep;
    
    if (ep_int_next != ep_int_last) {
	ep = ep_int_ringbuf[ep_int_last];
	ep_int_last = add_mod(ep_int_last, 1, MAXRINGBUF);
    } else {
	ep = EPNONE;
    }

    return ep;
}

usb_device_t usb_device_new = usb_device_none;
usb_device_t usb_device = usb_device_none;
usb_device_mouse_type_t usb_device_mouse_type_new = peppercon_mouse_type_none;
usb_device_mouse_type_t usb_device_mouse_type = peppercon_mouse_type_none;
#ifdef PP_FEAT_MASS_STORAGE
int enable_mass_storage_new[PP_FEAT_USB_MASS_STORAGE_NO];
int enable_mass_storage[PP_FEAT_USB_MASS_STORAGE_NO];
usb_device_type_t mass_storage_types[PP_FEAT_USB_MASS_STORAGE_NO];
#endif /* PP_FEAT_MASS_STORAGE */

int reenumerate_request = 0;

static uint8_t inform_user_space = 0;

void (*StandardDeviceRequest[])(void) = {
        Chap9_GetStatus,
        Chap9_ClearFeature,
        Chap9_StallEP0,
        Chap9_SetFeature,
        Chap9_StallEP0,
        Chap9_SetAddress,
        Chap9_GetDescriptor,
        Chap9_StallEP0,
        Chap9_GetConfiguration,
        Chap9_SetConfiguration,
        Chap9_GetInterface,
        Chap9_SetInterface,
        Chap9_StallEP0
};

int init_module(void);
void cleanup_module(void);
int pp_usb_init(void);
static int usb_thread(void * arg);

uint8_t g_setup_dataout[MAX_CONTROLDATA_OUT_SIZE];

volatile ControlData_t g_ControlData;

static void pp_usb_cleanup(void);
static int pp_usb_ioctl(struct inode *, struct file *, uint, ulong);
static int pp_usb_open(struct inode *, struct file *);
static int pp_usb_release(struct inode *, struct file *);
static unsigned int pp_usb_poll(struct file * file, poll_table * wait);

volatile int g_bus_reset;
volatile control_state_t g_control_state;
extern volatile uint8_t g_DeviceSuspended;

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations lara_usb_ops = {
    owner:   THIS_MODULE,
    ioctl:   pp_usb_ioctl,
    open:    pp_usb_open,
    release: pp_usb_release,
    poll:    pp_usb_poll,
};

#ifdef INTERRUPT_DEBUG
static volatile u32* gpio_regs;
#endif

uint8_t read_buffer[130];

#ifdef MODULE
MODULE_AUTHOR("info@peppercon.de");
MODULE_DESCRIPTION("Peppercon's USB device driver");
MODULE_LICENSE("GPL");

int init_module(void) {
    int rv;
    if (register_chrdev(LARA_USB_MAJOR, dev.name, &lara_usb_ops) < 0) {
	printk("%s: failed to register driver (register_chrdev)\n", dev.name);
	return -ENODEV;
    }
    rv = pp_usb_init();
    if (rv != SUCCESS) {
	cleanup_module();
    }
    return rv;
}

void cleanup_module(void) {
    int r;

    pp_usb_cleanup();

    if ((r = unregister_chrdev(LARA_USB_MAJOR, dev.name)) < 0) {
	printk("%s: failed to unregister driver (%d)\n", dev.name, r);
    } else {
	printk("%s: driver unregistered\n", dev.name);
    }
}
#endif	/* MODULE */


static struct semaphore inform_user_space_mutex = __MUTEX_INITIALIZER(inform_user_space_mutex);

void pp_usb_wakeup_file_process(void) {
    inform_user_space = 1;
    wake_up_interruptible(&dev.int_wq);
}

void pp_usb_wakeup_main_loop(void) {
    dev.wakeup = 1;
    wake_up_interruptible(&dev.req_wq);
}

static unsigned int pp_usb_poll(struct file * file, poll_table * wait) {
    unsigned int r = 0;
    
    poll_wait(file, &dev.int_wq, wait);

    if (down_interruptible(&inform_user_space_mutex)) {
    	printk("%s(): cannot get semaphore.\n", __FUNCTION__);
    	return -ERESTARTSYS;
    }
    if (inform_user_space || dev.rpc_queue_elem_cnt > 0) {
	inform_user_space = 0;
	r = POLLIN | POLLRDNORM;
    }
    up(&inform_user_space_mutex);

    return r;
}


void endpoint_init(void) {
    uint8_t mode;

    /* configure default EPs */
    mode = 0x83;
    hal_SetEndpointConfig(EP0OUT, mode);
    mode = 0xc3;
    hal_SetEndpointConfig(EP0IN, mode);
    /* configure device EPs */
    g_ud.config_endpoints();
}

void usb_connect(void) {
    uint8_t mode;
    g_control_state = IDLE;
    mode =
	ISP1181B_MODE_INTENA |
	ISP1181B_MODE_SOFTCT;
    UD("ep init\n");
    endpoint_init();

    hal_SetMode(mode);
}

/* ------------------------------------------------------------------------- *
 * driver (de-)initialization
 * ------------------------------------------------------------------------- */

static int sanity_check_driver(usb_driver_t* ud) {
    /* bMaxPacketSize <= chip FIFO size && size in (8,16,32,64) USB spec, 9.6.1*/
    if ((ud->usb_max_packet_size > ISP1181B_EP0OUT_FIFO_SIZE) ||
	(ud->usb_max_packet_size > ISP1181B_EP0IN_FIFO_SIZE) ||
	! ((ud->usb_max_packet_size == 8) ||
	   (ud->usb_max_packet_size == 16) ||
	   (ud->usb_max_packet_size == 32) ||
	   (ud->usb_max_packet_size == 64))) {
	printk("usblib: bMaxPacketSize failure: %d\n", ud->usb_max_packet_size);
	return 1;
    }
    /* the driver must set the config_endpoint function */
    if (ud->config_endpoints == NULL) {
	printk("usblib: config_endpoints not set\n");
	return 1;
    }
    if (!ud->initialized) return 1;
    return 0;
}

void USB_Init(void) {
    uint32_t interrupts, reg_ints;
    int i;

    /* Sets Interrupt Enable Register */
    interrupts = 0;
    for (i = EP1; i <= EP14; i++) {
	if (g_ud.eps[i].used) {
	    interrupts |= ISP1181B_INT_IEP(i);
	}
    }
    reg_ints = (
		ISP1181B_INT_IERESM |
		ISP1181B_INT_IESUSP |
		ISP1181B_INT_IERST |
		ISP1181B_INT_IEP0IN |
		ISP1181B_INT_IEP0OUT |
		interrupts
		);
    
    hal_SetIntEnable(reg_ints);
    if (g_ud.usb_init) {
	g_ud.usb_init(); /* init the device specific stuff, e.g. bot_init, hid_init, ... */
    }
}



static void change_usb_device(void) {
    int i;
    
    usb_device = usb_device_new;
    usb_device_mouse_type = usb_device_mouse_type_new;
#ifdef PP_FEAT_MASS_STORAGE
    memcpy(enable_mass_storage, enable_mass_storage_new, sizeof(enable_mass_storage));
#endif /* PP_FEAT_MASS_STORAGE */
    reenumerate_request = 0;
    /* our device has changed */
    g_ud.eps[EP0OUT].used = 1;
    g_ud.eps[EP0IN].used = 1;
    for (i = EP1; i <= EP14; i++) {
	g_ud.eps[i].used = 0;
    }
    for (i = EP0OUT; i <= EP14; i++) {
	g_ud.eps[i].unstall_cb = NULL;
	g_ud.eps[i].allow_unstall_cb = NULL;
	g_ud.eps[i].waiting = 0;
	g_ud.eps[i].timedout = 0;
	g_ud.eps[i].isr_int_in_cb = NULL;
	g_ud.eps[i].main_bulk_in_cb = NULL;
	g_ud.eps[i].main_bulk_out_cb = NULL;
	g_ud.eps[i].last_length = 0;
    }
 
    UD("reset dev\n");
    hal_ResetDevice();
   
    if (usb_device == usb_device_peppercon) {
	printk("** configure for peppercon\n");
#ifdef PP_FEAT_MASS_STORAGE
	peppercon_usb_init(&g_ud, usb_device_mouse_type, enable_mass_storage, mass_storage_types); /* for this time we are the normal peppercon device */
#else
	peppercon_usb_init(&g_ud, usb_device_mouse_type, NULL, NULL);
#endif /* PP_FEAT_MASS_STORAGE */
    } else if (usb_device == usb_device_ripcbelkin) {
	printk("** configure for RIPC\n");
#ifdef PP_FEAT_MASS_STORAGE
	ripcbelkin_usb_init(&g_ud, usb_device_mouse_type, enable_mass_storage);
#else
	ripcbelkin_usb_init(&g_ud, usb_device_mouse_type, NULL);
#endif /* PP_FEAT_MASS_STORAGE */
    } else {
	usb_device = usb_device_none;
	usb_device_mouse_type = peppercon_mouse_type_none;
	usb_device_new = usb_device_none;
	usb_device_mouse_type_new = peppercon_mouse_type_none;
	return;
    }
    
    /* do some sanity checks against our filled out driver structure */
    if (sanity_check_driver(&g_ud)) {
	usb_device = usb_device_none;
	usb_device_mouse_type = peppercon_mouse_type_none;
	usb_device_new = usb_device_none;
	usb_device_mouse_type_new = peppercon_mouse_type_none;
	printk("no mouse and no mass storage, just deactive USB\n");
	return;
    }

    ringbuf_init();
    
    UD("usb_init\n");
    USB_Init();
    /* this disconnects from the Bus, because the Mode Register
       (and therewith SoftConnect) is cleare
       according to USB spec 1.1, 7.1.7.1 the hub recognizes the disconnect
       after 2.5 us, we have on the belkin switch wait 100ms, this needs re-evaluated
       argl, after some testing with W2K, it seems that they only 
       detect the usb reconnect, if we wait 100ms,
       but we want to be sure, that the disconnect is realized, so
       we wait finally 200 ms
       */
    mdelay(200);
    UD("connect\n");
    usb_connect();
}


int __init pp_usb_init(void)
{
    int r;

#ifdef PP_FEAT_MASS_STORAGE
    memset(enable_mass_storage, 0, sizeof(enable_mass_storage));;
    memset(enable_mass_storage_new, 0, sizeof(enable_mass_storage_new));;

    {
	int i;
	for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
	    mass_storage_types[i] = PP_USB_IMAGE_TYPE_NONE;
	}
    }
#endif /* PP_FEAT_MASS_STORAGE */
    
    DECLARE_MUTEX_LOCKED(mtx);

    /* init wait queues */
    init_waitqueue_head (&dev.req_wq);
    init_waitqueue_head (&dev.int_wq);
    /* init rpc queue */
    INIT_LIST_HEAD(&dev.rpc_queue);

    if (hal_init() == SUCCESS) {
	dev.usb_notify = &mtx;
	r = kernel_thread(usb_thread, NULL, 0);
	if (r < 0) {
	    printk("%s: failed to create state engine thread (exit code: %d)\n",
		   dev.name, r);
	    goto bailout;
	}
	down(&mtx);
	dev.usb_notify = NULL;

	return SUCCESS;
    }
    
 bailout:
    printk("%s: kernel module didn't load successfully!\n", dev.name);
    return -ENODEV;
}

static void pp_usb_cleanup(void)
{
    struct list_head * entry;
    u_long flags;
    
    DECLARE_MUTEX_LOCKED(mtx);

    /* terminate usb thread */
    if (dev.usb_thread != NULL) {
	dev.usb_notify = &mtx;
	dev.rmmod = 1;
	dev.wakeup = 1;
	wake_up_interruptible(&dev.req_wq);
	down(&mtx);
	dev.usb_notify = NULL;
    }

    hal_cleanup();
#ifdef PP_FEAT_MASS_STORAGE
    ms_bo_exit();
#endif
    /* free rpc queue */
    spin_lock_irqsave(&dev.rpc_queue_lock, flags);
    list_for_each(entry, &dev.rpc_queue) {
	rpc_queue_entry_t * rpc_entry = list_entry(entry, rpc_queue_entry_t, listnode);
	entry = entry->prev;
	list_del(&rpc_entry->listnode);
	kfree(rpc_entry);
    }
    dev.rpc_queue_elem_cnt = 0;
    spin_unlock_irqrestore(&dev.rpc_queue_lock, flags);
}

void call_user_space(usb_usrspace_data_t * rpc) {
    rpc_queue_entry_t * rpc_entry;
    u_long flags;

    spin_lock_irqsave(&dev.rpc_queue_lock, flags);
    if (dev.rpc_queue_elem_cnt >= 20) {
	spin_unlock_irqrestore(&dev.rpc_queue_lock, flags);
	printk("pp_usb: call_user_space(): queue full\n");
	return;
    }
    spin_unlock_irqrestore(&dev.rpc_queue_lock, flags);

    // allocate queue element
    if ((rpc_entry = (rpc_queue_entry_t *)kmalloc(sizeof(rpc_queue_entry_t), GFP_ATOMIC)) == NULL) {
	printk("pp_usb: call_user_space(): kmalloc(): out of memory\n");
	return;
    }

    // copy data into queue element
    memcpy(&rpc_entry->rpc, rpc, sizeof(usb_usrspace_data_t));

    // insert element into queue
    spin_lock_irqsave(&dev.rpc_queue_lock, flags);
    list_add_tail(&rpc_entry->listnode, &dev.rpc_queue);
    dev.rpc_queue_elem_cnt++;
    spin_unlock_irqrestore(&dev.rpc_queue_lock, flags);

    /* tell the watchers about new data */
    wake_up_interruptible(&dev.int_wq);
}



void isr_ep0in(void) {
    uint8_t status;
    uint32_t to_write;
    
    UD("%s\n", __FUNCTION__);

    status = hal_GetEndpointStatus(EP0IN); /* clears interrupt, too */

    to_write = g_ControlData.wLength - g_ControlData.wCount;
    switch (g_control_state) {
      case HANDSHAKE:
	  break;
      case DATAIN:
	  if(to_write >= ISP1181B_EP0IN_FIFO_SIZE) {
	      hal_WriteEndpoint(EP0IN, g_ControlData.pData + g_ControlData.wCount, ISP1181B_EP0IN_FIFO_SIZE);
	      g_ControlData.wCount += ISP1181B_EP0IN_FIFO_SIZE;
	  } else if( to_write != 0) {
	      hal_WriteEndpoint(EP0IN, g_ControlData.pData + g_ControlData.wCount, to_write);
	      g_ControlData.wCount += to_write;
	      g_control_state = HANDSHAKE;
	  } else if (to_write == 0) {
	      hal_SingleTransmitEP0(NULL, 0);
	      g_control_state = HANDSHAKE;
	  }
	  break;
        default:
	    g_control_state = STALL;
	    hal_StallEP0InControlRead();
	    break;
    }
}

void isr_ep0out(void) {
    uint8_t status;
    
    UD("%s\n", __FUNCTION__);

    status = hal_GetEndpointStatus(EP0OUT); /* clears interrupt, too */
    if (!(status & ISP1181B_EP_STATUS_EPFULL0)) {
	printk("Control Out Status Wrong: %02x\n", status);
	return;
    }
    if (status & ISP1181B_EP_STATUS_SETUPT) {
	/* setup packet - will be handle in main loop */
	g_control_state = SETUPPROC;
	UD("setup-proc\n");
    } else {
	/* Data Out Packet */
	switch (g_control_state) {
	  case DATAOUT:
	      {
		  uint32_t i;
		  i = hal_ReadEndpoint(EP0OUT, g_ControlData.pData + g_ControlData.wCount, ISP1181B_EP0OUT_FIFO_SIZE);
		  g_ControlData.wCount += i;
		  if( i != ISP1181B_EP0OUT_FIFO_SIZE || g_ControlData.wCount >= g_ControlData.wLength) {
		      g_control_state = REQUESTPROC;
		  }
	      }
	      break;
	  case HANDSHAKE:
	      g_control_state = IDLE;
	      break;
	  default:
	      g_control_state = STALL;
	      hal_StallEP0InControlWrite();
	      break;
	}
    }
}

void isr_bulk_out (uint8_t ep) {
    uint8_t status;

    status = hal_GetEndpointStatus(ep); /* clears interrupt, too */
    //UD("isr bo: %x\n", status);
    ringbuf_enqueue(ep);
}

void isr_bulk_in(uint8_t ep) {
    uint8_t status;
    
    status = hal_GetEndpointStatus(ep); /* clears interrupt, too */
    //UD("isr bi: %x\n", status);
    ringbuf_enqueue(ep);
}

void isr_int_out(uint8_t ep) {
    uint8_t status;
    status = hal_GetEndpointStatus(ep); /* clears interrupt, too */
}

void isr_int_in(uint8_t ep) {
    uint8_t status;
    status = hal_GetEndpointStatus(ep); /* clears interrupt, too */
    //    UD("%s: ep: %x\n", __FUNCTION__, ep);
    if (g_ud.eps[ep].isr_int_in_cb) {
	//	UD("calling is_int_in_cb\n");
	g_ud.eps[ep].isr_int_in_cb(ep);
    }
}

irqreturn_t lara_usb_isr(int irq, void * dev_id, struct pt_regs * regs) {
    uint32_t interrupts;
    u_long flags;
    int i;
    
    spin_lock_irqsave(&dev.chip_access_lock, flags);
    interrupts = hal_ReadInterruptRegister();
    UD("isr: %08lx\n", (unsigned long)interrupts);

    // #warning the sun does suspend during enumeration so this caused an error here
    if ((interrupts & ~ISP1181B_INT_BUSSTATE) == 0) {
	    /* IRQ is not for us */
	    goto bail;
    }    
    
    if (interrupts & ISP1181B_INT_IESUSP) {
	g_DeviceSuspended = 1;
	propchange_fire(PP_PROP_USB_DEV_STATE_CHANGED, 0);
    }

    /* we ignore suspend request so we don't leave the suspend state when rsm irq is called,
       we leave our virtual suspend state when the device is enumerated again */
#if 0
    if (interrupts & ISP1181B_INT_IERESM) {
	g_DeviceSuspended = 0;
	propchange_fire(PP_PROP_USB_DEV_STATE_CHANGED, 0);
    }
#endif
    
    if (interrupts & ISP1181B_INT_IERST) {
	g_bus_reset = 1;
	goto bail;
    }
    
    if (interrupts & ISP1181B_INT_IEP0IN) {
	isr_ep0in();
    }
    
    if (interrupts & ISP1181B_INT_IEP0OUT) {
	isr_ep0out();
    }
    
    for (i = ISP1181B_INT_N_IEP1; i <= ISP1181B_INT_N_IEP14; i++) {
	if (interrupts & ISP1181B_INT_N_TO_MASK(i)) {
	    uint8_t ep;
	    ep = i-ISP1181B_INT_N_IEP1+EP1;
	    //UD("isr: %x, %08lx, ep: %x, type: %x\n", i, (unsigned long)ISP1181B_INT_N_TO_MASK(i), ep, g_ud.eps[ep].ep_type);
	  switch (g_ud.eps[ep].ep_type) {
	    case BULK_IN:
		isr_bulk_in(ep);
		break;
	    case BULK_OUT:
		isr_bulk_out(ep);
		break;
	    case INTERRUPT_IN:
		isr_int_in(ep);
		break;
	    case INTERRUPT_OUT:
		isr_int_out(ep);
		break;
	  }
	    //	    if (ud.ep_int_functions[i-ISP1181B_INT_N_IEP1+EP1]) {
	    //		ud.ep_int_functions[i-ISP1181B_INT_N_IEP1+EP1](i-ISP1181B_INT_N_IEP1+EP1);
	}
    }

    dev.wakeup = 1;
    wake_up_interruptible(&dev.req_wq);

bail:
    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
    /* assume we handled the interrupt */
    return IRQ_HANDLED;
}


void debug_show_packet(const char *packet, const unsigned int size) {
    unsigned int i;

    for (i = 0; i < size; i++) {
        if (!(i % 16)) printk("%04x: ", i);
        printk("0x%02x ", ((uint8_t*) packet)[i]);
        if (!((i+1) % 16)) printk("\n");
    }
    printk("\n");
}


static int pp_usb_ioctl(struct inode * inode, struct file * file, uint cmd, ulong arg) {
    uint8_t data[64];
    u_long flags;
    
    switch (cmd) {
#ifdef PP_FEAT_MASS_STORAGE
      case LARAUSBENABLESETUPPROTO:
	  {
	      uint8_t enable_sp;
	      if (copy_from_user(&enable_sp, (uint8_t*)arg, sizeof(uint8_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      scsi_setup_proto_enabled = enable_sp;
	      return SUCCESS;
	      break;
	  }
      case LARAUSBSETFILE:
	  {
	      file_info_t file_info;
	      
	      if (copy_from_user(&file_info, (file_info_t*)arg, sizeof(file_info_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }

	      if (file_info.dev_no >= PP_FEAT_USB_MASS_STORAGE_NO) {
	          printk("wrong mass storage device number.\n");
	          return -ENODEV;
	      }

	      ms_bo_set_file(file_info.dev_no, file_info.lba, file_info.blocklen, file_info.image_type, file_info.readonly);
	      mass_storage_types[file_info.dev_no] = file_info.image_type;
	    
	      return SUCCESS;
	      break;
	  }
      case LARAUSBGETREQUEST:
          {
	      file_request_control_t request;
	      if (copy_from_user(&request, (char*)arg, sizeof(file_request_control_t))) {
	          printk("cannot copy from user\n");
	          return -EFAULT;
	      }
	      
	      if (request.dev_no >= PP_FEAT_USB_MASS_STORAGE_NO) {
	          printk("wrong mass storage device number.\n");
	          return -ENODEV;
	      }
	      
	      memcpy(&request.request, &g_ud.file_req[request.dev_no], sizeof(file_request_t));

	      if (copy_to_user((char*)arg, (uint8_t*)&request, sizeof(file_request_control_t))) {
	          printk("cannot copy to user\n");
	          return -EFAULT;
	      }
	      return SUCCESS;
	      break;
	  }
      case LARAUSBSETRESPONSE:
      	  {
	      file_response_t file_resp;

	      if (copy_from_user(&file_resp, (file_response_t*)arg, sizeof(file_response_t))) {
	          printk("cannot copy from user\n");
	          return -EFAULT;
	      }

	      if (file_resp.dev_no >= PP_FEAT_USB_MASS_STORAGE_NO) {
	          printk("wrong mass storage device number.\n");
	          return -ENODEV;
	      }
	      
	      UD("info from user space: response: %d, data: %p, len: %x (bytes)\n",
	         file_resp.response, file_resp.data, file_resp.length);
	      //debug_show_packet(file_resp.data, 16);
	      spin_lock_irqsave(&dev.chip_access_lock, flags);

	      ms_bo_response(file_resp.dev_no, &file_resp);
	      spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	      return SUCCESS;
	      break;
	  }
#endif /* PP_FEAT_MASS_STORAGE */
      case LARAUSBGETBELKIN:
	  {
	      rpc_queue_entry_t * rpc_entry = NULL;
	      
	      /* get from queue */
	      spin_lock_irqsave(&dev.rpc_queue_lock, flags);
	      if (!list_empty(&dev.rpc_queue)) {
		  rpc_entry = list_entry(dev.rpc_queue.next, rpc_queue_entry_t, listnode);
		  list_del(&rpc_entry->listnode);
		  dev.rpc_queue_elem_cnt--;
	      }
	      spin_unlock_irqrestore(&dev.rpc_queue_lock, flags);
             
	      if (rpc_entry) {
		  if (copy_to_user((char*)arg, (char*)&rpc_entry->rpc, sizeof(usb_usrspace_data_t))) {
		      kfree(rpc_entry);
		      printk("cannot copy to user\n");
		      return -EFAULT;
		  }
		  kfree(rpc_entry);
		  return SUCCESS;
	      } else {
		  return -EAGAIN;
	      }
	  }
	  break;
      case LARAUSBGETDEVICESTATE:
	  {
	      /* because the suspended state is ortogonal the the other states,
		 we must compose it with 2 variables */
	      usb_device_state_t d;
	      d = g_DeviceSuspended ? (PP_USB_DEVSTAT_SUSPENDED) : g_DeviceState;
	      if (copy_to_user((char *)arg, (uint8_t*)&d, sizeof(usb_device_state_t))) {
		  printk("cannot copy to user\n");
		  return -EFAULT;
	      }
	  }
	  return SUCCESS;
	  break;
      case LARAUSBSETDEVICE:
	  {
	      lara_usb_device_t d;
	      
	      if (copy_from_user(&d, (lara_usb_device_t*) arg, sizeof(lara_usb_device_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	      if ((d.dev != usb_device_unchanged &&
		   d.dev != usb_device_peppercon &&
		   d.dev != usb_device_ripcbelkin)) {
		  return -EINVAL;
	      }
	      UD("Set Device: dev: %d: mouse %d, msx: %d", d.dev, d.mouse_type, d.mass_storage);
	      if (d.dev != usb_device_unchanged) {
		  usb_device_new = d.dev;
	      }
	      if (d.mouse_type != peppercon_mouse_type_unchanged) {
		  usb_device_mouse_type_new = d.mouse_type;
	      }
#ifdef PP_FEAT_MASS_STORAGE
	      {
		  int i;
		  for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
		      if (d.mass_storage[i] != usb_device_ms_unchanged) {
			  enable_mass_storage_new[i] = d.mass_storage[i] == usb_device_ms_enabled;
		      }
		  }
	      }
#endif /* PP_FEAT_MASS_STORAGE */
	      dev.wakeup = 1;
	      wake_up_interruptible(&dev.req_wq);
	  }
	  return SUCCESS;
	  break;
      case LARAUSBRESET:
	  UD("Reset Device\n");
	  reenumerate_request = 1;
	  dev.wakeup = 1;
	  wake_up_interruptible(&dev.req_wq);
	  return SUCCESS;
	  break;
      case LARAUSBSENDDATA:
	  {
	      lara_usb_command_t command;
	      if (copy_from_user(&command, (lara_usb_command_t*) arg, sizeof(lara_usb_command_t))) {
		  printk("cannot copy from user\n");
		  return -EFAULT;
	      }
	  
	      /* this is only a sanity check, so that we support up to 64 bytes transfers,
		 but the user space has to beware not to send more than 32+2 bytes for an
		 interrupt or bulk endpoint */
	      
	      if (command.length > (64+2)) {
		  printk("command_length too long: %d\n", command.length);
		  return -EFAULT;
	      }
	      
	      if (copy_from_user((char*) data, &((char*)arg)[sizeof(command)], command.length)) {
		  printk("cannot copy from user cmd\n");
		  return -EFAULT;
	      }
	      
	      spin_lock_irqsave(&dev.chip_access_lock, flags);
	      if (g_ud.initialized && g_ud.send_data != NULL) {
                  g_ud.send_data(command.channel, command.type, data, command.length);
	      }
	      spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	      return SUCCESS;
	  }
	  break;
      case LARAUSBGETLEDSTATE:
          {
              if (copy_to_user((char*)arg, (uint8_t*)&led, sizeof(uint8_t))) {
                  printk("cannot copy led state to user\n");
                  return -EFAULT;
              }
              return SUCCESS;
          }
          break;
      default:
	  return -EINVAL;
    }
    return -EINVAL;
}

static int pp_usb_open(struct inode * inode, struct file * file) {
    return SUCCESS;
}

static int pp_usb_release(struct inode * inode, struct file * file) {
    return SUCCESS;
}

void ML_AcknowledgeSETUP(void) {
    hal_AcknowledgeSetup();
    hal_ClearEndpoint(EP0OUT);
}

void request_handler(void) {
    uint8_t type = g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_TYPE_MASK;
    uint8_t req = g_ControlData.DeviceRequest.bRequest & USB_REQUEST_MASK;
    uint8_t bRecipient = g_ControlData.DeviceRequest.bmRequestType & USB_RECIPIENT_MASK;
    uint16_t wIndex = g_ControlData.DeviceRequest.wIndex;

    UD("%s: dir: %01x, type: %02x, rec: %04x, req: %02x, len: %x, wValue: %x, wIndex: %x\n", __FUNCTION__,
       g_ControlData.DeviceRequest.bmRequestType & USB_REQUEST_DIR_MASK, type,
       bRecipient, req,
       g_ControlData.DeviceRequest.wLength, g_ControlData.DeviceRequest.wValue, wIndex);

    if ((type == USB_STANDARD_REQUEST) && (req < MAX_STANDARD_REQUEST)) {
	(*StandardDeviceRequest[req])();
    } else if ((type == USB_CLASS_REQUEST) && (bRecipient == USB_RECIPIENT_INTERFACE)) {
	if (wIndex >= MAX_INTERFACES) {
	    Chap9_StallEP0();
	    UD("class interface request out of range: %d (max: %d)\n",
	       wIndex, MAX_INTERFACES);
	} else {
	    if (g_ud.class_interface_request[wIndex]) {
		g_ud.class_interface_request[wIndex](g_ControlData.DeviceRequest.bmRequestType,
						     g_ControlData.DeviceRequest.bRequest,
						     g_ControlData.DeviceRequest.wValue,
						     g_ControlData.DeviceRequest.wIndex,
						     g_ControlData.DeviceRequest.wLength);
		UD("class interface request for index: %x\n", wIndex);
	    } else {
		printk("handler not found\n");
		Chap9_StallEP0();
	    }
	}
    } else if ((type == USB_VENDOR_REQUEST) && (bRecipient == USB_RECIPIENT_DEVICE)) {
	if (g_ud.vendor_device_request) {
	    g_ud.vendor_device_request(g_ControlData.DeviceRequest.bmRequestType,
				       g_ControlData.DeviceRequest.bRequest,
				       g_ControlData.DeviceRequest.wValue,
				       g_ControlData.DeviceRequest.wIndex,
				       g_ControlData.DeviceRequest.wLength);
	} else {
	    UD("unknown vendor device request\n");
	    Chap9_StallEP0();
	}
    } else {
	Chap9_StallEP0();
	printk("bad request\n");
    }
}

void setup_handler(void) {
    uint32_t len;
    
    g_ControlData.wLength = 0;
    g_ControlData.wCount = 0;
    len = hal_ReadEndpointWOClear(EP0OUT, (uint8_t *)(&(g_ControlData.DeviceRequest)), sizeof(g_ControlData.DeviceRequest));
    if( len == sizeof(device_request_t) ) {
	g_ControlData.DeviceRequest.wValue = le16_to_cpu(g_ControlData.DeviceRequest.wValue);
	g_ControlData.DeviceRequest.wIndex = le16_to_cpu(g_ControlData.DeviceRequest.wIndex);
	g_ControlData.DeviceRequest.wLength = le16_to_cpu(g_ControlData.DeviceRequest.wLength);

	g_ControlData.wLength = g_ControlData.DeviceRequest.wLength;
	g_ControlData.wCount = 0;

	if (g_ControlData.DeviceRequest.bmRequestType & USB_ENDPOINT_DIRECTION_MASK) {
	    ML_AcknowledgeSETUP();
	    g_control_state = REQUESTPROC;
	} else {
	    if (g_ControlData.DeviceRequest.wLength == 0) {
		ML_AcknowledgeSETUP();
		g_control_state = REQUESTPROC;
	    } else {
		if(g_ControlData.DeviceRequest.wLength <= MAX_CONTROLDATA_OUT_SIZE) {
		    g_ControlData.pData = g_setup_dataout;
		    g_control_state = DATAOUT;
		    ML_AcknowledgeSETUP();
		} else {
		    ML_AcknowledgeSETUP();
		    hal_StallEP0InControlWrite();
		    g_control_state = STALL;
		}
	    }
	}
    } else {
	printk("should not happen - setup handler wrong setup request\n");
	ML_AcknowledgeSETUP();
	Chap9_StallEP0();
    }
}

void WriteEndpoint(uint8_t ep, const uint8_t * buf, uint16_t len) {
    if (g_DeviceState != PP_USB_DEVSTAT_CONFIGURED) {
	printk("state is %d, but %d expected\n", g_DeviceState, PP_USB_DEVSTAT_CONFIGURED);
	return;
    }
    hal_WriteEndpoint(ep, buf, len);
}

#define FINISH_IF_RMMOD		do { if (dev.rmmod) goto finish; } while(0)

static int usb_thread(void * arg) {
    u_long flags;
    
#ifdef __SMP__
    lock_kernel();
#endif

# if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    daemonize();
    strcpy(current->comm, "usb");
#else
    daemonize("usb");
#endif

    dev.usb_thread = current;

#ifdef __SMP__
    unlock_kernel();
#endif

    UD("%s: usb main loop started\n", dev.name);

    if (dev.usb_notify != NULL) {
	up(dev.usb_notify);
    }

    while(1) {
	wait_event_interruptible(dev.req_wq, dev.wakeup || dev.rmmod);

#if 0
#warning just a test
	{ uint8_t status;
	status = hal_GetEndpointStatusWOClear(EP2);
	printk("status (EP2): %x\n", status);

	status = hal_GetEndpointError(EP2);
	printk("status (EP2): %x\n", status);

	}
#endif
	dev.wakeup = 0;

	{
	    int do_change_device = 0;
	    
	    if (usb_device_new != usb_device || usb_device_mouse_type_new != usb_device_mouse_type
	        || reenumerate_request) {
	    	do_change_device = 1;
	    }
	    
#ifdef PP_FEAT_MASS_STORAGE
	    {
		int i;
		for (i = 0; i < PP_FEAT_USB_MASS_STORAGE_NO; i++) {
		    if (enable_mass_storage_new[i] != enable_mass_storage[i]) {
			do_change_device = 1;
		    }
		}
	    }
#endif /* PP_FEAT_MASS_STORAGE */
	    
	    if (do_change_device) {
		UD("change_usb_dev\n");
	    	change_usb_device();
	    }
	}

	spin_lock_irqsave(&dev.chip_access_lock, flags);

	if (g_bus_reset) {
	    //	    spin_lock_irqsave(&dev.chip_access_lock, flags);
	    UD("USB Bus reset\n");
	    g_bus_reset = 0;
	    Chap9_busreset();
#ifdef PP_FEAT_MASS_STORAGE
	    ms_bo_init();
#endif /* PP_FEAT_MASS_STORAGE */
	    //	    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	}

	if (g_control_state == SETUPPROC) {
	    //	    spin_lock_irqsave(&dev.chip_access_lock, flags);
	    setup_handler();
	    //	    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	}

	if (g_control_state == REQUESTPROC) {
	    //	    spin_lock_irqsave(&dev.chip_access_lock, flags);
	    request_handler();
	    //	    spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	}

	{
	    uint8_t ep;
	    //spin_lock_irqsave(&dev.chip_access_lock, flags);

	    while ((ep = ringbuf_getfirst()) != EPNONE) {
		/* there is an event in the queue */
		if (g_ud.eps[ep].ep_type == BULK_IN) {
		    /* locking here? */
		    if (g_ud.eps[ep].main_bulk_in_cb != NULL) {
			g_ud.eps[ep].main_bulk_in_cb(ep);
		    }
		    ringbuf_delfirst();
		} else if (g_ud.eps[ep].ep_type == BULK_OUT) {
		    uint32_t len;
		    uint32_t status;
		    //		    int rd = 0;
		    status = hal_GetEndpointStatusWOClear(ep);

		    if (status & ISP1181B_EP_STATUS_EPFULL0 || status & ISP1181B_EP_STATUS_EPFULL1) {
			if (g_ud.eps[ep].status) {
			    if ((g_ud.eps[ep].ioCount < g_ud.eps[ep].ioSize) && g_ud.eps[ep].reset != 1) {
				int rd;
				if((g_ud.eps[ep].ioSize-g_ud.eps[ep].ioCount) > g_ud.eps[ep].bufSize) {
				    len = g_ud.eps[ep].bufSize;
				} else {
				    len = g_ud.eps[ep].ioSize - g_ud.eps[ep].ioCount;
				}
				/* this clears ep, too */
				rd = hal_ReadEndpoint(ep, g_ud.eps[ep].pData + g_ud.eps[ep].ioCount, len);
				//if (len != 0x1f)
				    //printk("got data on %x: %x/%x, will read %x: \n", ep, g_ud.eps[ep].ioCount, g_ud.eps[ep].ioSize, rd);
				g_ud.eps[ep].ioCount += rd;
			    } else {
				/* we don't wait for data or got an reset - clear the ep for now */
				printk(" we don't wait for data - should not happen: ioCount[%x]: %x, iosize: %x, epconf: %x\n", ep, g_ud.eps[ep].ioCount, g_ud.eps[ep].ioSize, status);
				hal_ReadEndpoint(ep, dummy_buffer, 64);
				//				hal_ClearEndpoint(ep);
			    }
			    ringbuf_delfirst();
			} else {
			    printk("status: %02x, but not ready\n", status);
			    break;
			}
		    } else {
			printk("main loop got event, but EP buffer not full - status: %02x\n", status);
			ringbuf_delfirst();
		    }
			
		    if (g_ud.eps[ep].main_bulk_out_cb != NULL) {
			g_ud.eps[ep].main_bulk_out_cb(ep);
		    }
		    //		    if (rd) hal_ClearEndpoint(ep);
		} else {
		    /* at the moment there should only bulk events in the queue */
		    printk("should not happen\n");
		    ringbuf_delfirst();
		}
#if 0
		spin_unlock_irqrestore(&dev.chip_access_lock, flags);
		schedule();
		spin_lock_irqsave(&dev.chip_access_lock, flags);
#endif
	    }
#ifdef PP_FEAT_MASS_STORAGE
	    {
	    	int ms;
	    	for (ms = 0; ms < PP_FEAT_USB_MASS_STORAGE_NO; ms++) {
	    	    g_ud.eps[EP_MS_BULK_OUT(ms)].reset = 0;
	    	}
	    }
#endif /* PP_FEAT_MASS_STORAGE */
	}
	spin_unlock_irqrestore(&dev.chip_access_lock, flags);
	FINISH_IF_RMMOD;
    }
 finish:
    dev.usb_thread = NULL;
    if (dev.usb_notify != NULL) {
	up(dev.usb_notify);
    }
	
    UD("%s: state engine terminated\n", dev.name);

    return 0;
}

#undef FINISH_IF_RMMOD
