#include "pphid.h"
#include "FTC_zero.h"
#include "hid.h"

#undef HID_DEBUG

#ifdef HID_DEBUG
# define HD(...) printk(__VA_ARGS__)
#else
# define HD(...)
#endif

#define INT_TIMEOUT 5 /* in 10ms steps */

struct _pphid_t {
    wait_queue_head_t hid_wq;
    unsigned char *keyboard_report[MAX_KBD_REPORT];
    unsigned char *mouse_report[MAX_MOUSE_REPORT];
    uint8_t mouse_idle;
    uint8_t mouse_report_length;
    uint8_t mouse_protocol;
    int current_mouse_report_length;
    usb_device_mouse_type_t mouse_type;
    uint8_t keyboard_idle;
    uint8_t keyboard_report_length;
    uint8_t keyboard_protocol;
};

void pphid_set_mouse_report_length(pphid_t *hid, int i)
{
    hid->current_mouse_report_length = i;
    hid->mouse_report_length = i;
}

void pphid_set_mouse_type(pphid_t *hid, usb_device_mouse_type_t mouse_type)
{
    hid->mouse_type = mouse_type;
}

pphid_t *pphid_init(void)
{
    pphid_t *hid = kmalloc(sizeof(pphid_t), GFP_KERNEL);
    memset(hid->keyboard_report, 0, MAX_KBD_REPORT);
    memset(hid->mouse_report, 0, MAX_MOUSE_REPORT);
    hid->mouse_idle = 0;      /* 7.2.4 */
    hid->keyboard_idle = 125; /* 7.2.4: 500ms (recommended) / 4ms */
    hid->mouse_protocol = PROTO_REPORT;
    hid->current_mouse_report_length = 4;
    hid->mouse_type = peppercon_mouse_type_none;
    hid->mouse_report_length = 4;
    hid->keyboard_protocol = PROTO_REPORT;
    hid->keyboard_report_length = 8;
    init_waitqueue_head(&hid->hid_wq);
    return hid;
}

void pphid_deinit(pphid_t *hid)
{
    if (hid) kfree(hid);
}

int pphid_mouse_handle_classrequest(struct FTC_zero_dev *ZeroDev, const struct usb_ctrlrequest *ctrl)
{
    pphid_t *hid = ZeroDev->hid;
    struct usb_request	*req = ZeroDev->ep0req;
    int value = -EOPNOTSUPP;

    switch (ctrl->bRequest) {
	case HID_REQ_GET_REPORT: /* 7.2.1 */
	    if (((ctrl->wValue & 0xff00) >> 8)== 0x01) {
		value = min((int)ctrl->wLength, (int)hid->mouse_report_length);
		memcpy(req->buf, hid->mouse_report, value);
	    }
	    break;
	case HID_REQ_GET_IDLE: /* 7.2.3 */
	    {
		uint8_t buffer[1];
		buffer[0] = hid->mouse_idle;
		value = 1;
		memcpy(req->buf, buffer, value);
	    }
	    break;
	case HID_REQ_SET_IDLE: /* 7.2.4 */
	    hid->mouse_idle = (uint8_t) ((ctrl->wValue & 0xff00) >> 8);
	    value = 0;
	    break;
	case HID_REQ_GET_PROTOCOL: /* 7.2.5 */
	    if (hid->mouse_type == peppercon_mouse_type_relative) {
		uint8_t proto;
		proto = hid->mouse_protocol;
		value = 1;
		memcpy(req->buf, &proto, value);
	    }
	    break;
	case HID_REQ_SET_PROTOCOL: /* 7.2.6 */
	    if (hid->mouse_type == peppercon_mouse_type_relative) {
		hid->mouse_protocol = (uint8_t) (ctrl->wValue & 0xff);
		value = 0;
	    }
	    break;
	case HID_REQ_SET_REPORT: /* 7.2.2, optional Appendix G */
	default:
	    ;
    }
    return value;
}

int pphid_keyboard_handle_classrequest(struct FTC_zero_dev *ZeroDev, const struct usb_ctrlrequest *ctrl)
{
    pphid_t *hid = ZeroDev->hid;
    struct usb_request	*req = ZeroDev->ep0req;
    int value = -EOPNOTSUPP;

    switch (ctrl->bRequest) {
	case HID_REQ_GET_REPORT:
	    if (((ctrl->wValue & 0xff00) >> 8)== 0x01) {
		value = min((int)ctrl->wLength, (int)hid->keyboard_report_length);
		memcpy(req->buf, hid->keyboard_report, value);
	    }
	    break;
	case HID_REQ_SET_REPORT:
	    {
		uint8_t led;
		led = ((uint8_t*)req->buf)[0];
		HD("led: %02x\n", led);
		value = 0;
	    }
	    break;
	case HID_REQ_GET_PROTOCOL:
	    {
		uint8_t proto;
		proto = hid->keyboard_protocol;
		value = 1;
		memcpy(req->buf, &proto, value);
	    }
	    break;
	case HID_REQ_SET_PROTOCOL: /* 7.2.6 */
	    hid->keyboard_protocol = (uint8_t) (ctrl->wValue & 0xff);
	    value = 0;
	    break;
	case HID_REQ_GET_IDLE:
	    {
		char buffer[1];
		buffer[0] = hid->keyboard_idle;
		value = 1;
		memcpy(req->buf, buffer, value);
	    }
	    break;
	case HID_REQ_SET_IDLE:
	    hid->keyboard_idle = (uint8_t) ((ctrl->wValue & 0xff00) >> 8);
	    value = 0;
	    break;
	default:
	    ;
    }
    return value;
}

static int send_report(pphid_t *hid, struct usb_ep* ep, const uint8_t *fdata, const int flength, uint8_t last_packet)
{
    ep_driver_t *ep_driver = (ep_driver_t *)ep->driver_data;
    struct zero_buffhd	*bh;
    struct FTC_zero_dev *ZeroDev = ep_driver->dev;
    hid_class_ep_data_t * hiddata;
    long time_left;
    DECLARE_WAITQUEUE (wait, current);
    int length = flength;
    const uint8_t* data = fdata;
    
/* FIXME (miba) return if ep_driver==NULL, temporary */
    if (ep_driver == NULL) return FALSE;
    if (ZeroDev->config == 0) return FALSE;
    hiddata = (hid_class_ep_data_t *)ep_driver->class_ep_data;
    if (hiddata == NULL) return FALSE;
    bh = ep_driver->bufhd;

    if (last_packet) {
	data = hiddata->last_packet;
	length = hiddata->last_len;
    }

    hiddata->waiting = 1;
    
    while (bh->state != BUF_STATE_EMPTY) {
	/* sleep here until delivery was successful or timeout */
	/* wait a little bit longer than the timeout of the last request
	   after sending the data
	*/

	time_left = (INT_TIMEOUT) + 2;
	HD("start waiting\n");
	do {
	    current->state = TASK_INTERRUPTIBLE;
	    add_wait_queue(&hid->hid_wq, &wait);
	    spin_unlock(&ZeroDev->lock);
	    time_left = schedule_timeout(time_left);
	    remove_wait_queue(&hid->hid_wq, &wait);
	    spin_lock(&ZeroDev->lock);
	    HD("time left: %lx\n", time_left);
	} while (time_left && bh->state != BUF_STATE_EMPTY);
	if (time_left == 0 && bh->state != BUF_STATE_EMPTY) {
	    /* only if the data are less than the buffer size */
	    if (length <= 64) {
		hiddata->last_len = length;
		hiddata->timedout = 1; /* schedule resend */
		memcpy(hiddata->last_packet, data, length);
	    }
	    hiddata->waiting = 0;
	    bh->state = BUF_STATE_EMPTY;
	    HD("timedout\n");
	    return FALSE;
	}
    }

    hiddata->waiting = 0;
    hiddata->timedout = 0;

    HD("send_report - send data: ep: %s, len: %d, data: %p\n", ep->name, length, data);
    
    memcpy(bh->buf, data, min((u16)ep->maxpacket, (u16)length));
    bh->inreq->length = min((u16)ep->maxpacket, (u16)length);
    start_transfer(ZeroDev, ep, bh->inreq,
		   &bh->inreq_busy, &bh->state);
    if (!last_packet) {
	// not call within an interrupt
	time_left = INT_TIMEOUT;
	while (time_left && bh->state != BUF_STATE_EMPTY) {
	    HD("wait for reply\n");
	    current->state = TASK_INTERRUPTIBLE;
	    add_wait_queue(&hid->hid_wq, &wait);
	    spin_unlock(&ZeroDev->lock);
	    time_left = schedule_timeout(time_left);
	    remove_wait_queue(&hid->hid_wq, &wait);
	    spin_lock(&ZeroDev->lock);
	}
	if (time_left == 0 && bh->state != BUF_STATE_EMPTY) {
	    HD("send_report Timeout getting reply\n");
	}
	HD("send_report done. time_left: %ld HZ = %d\n", time_left, HZ);
    }
    
    return TRUE;
}

static void mouse_send_report(pphid_t *hid, struct usb_ep* ep)
{
    /* write the packet only synced, when we send to to hid endpoint (and not to EP0IN),
       otherwise we are "synced" in the main loop */
    send_report(hid, ep, (uint8_t*)hid->mouse_report, hid->mouse_report_length, 0);
}

static void keyboard_send_report(pphid_t *hid, struct usb_ep* ep)
{
    send_report(hid, ep, (uint8_t*)hid->keyboard_report, hid->keyboard_report_length, 0);
}

void pphid_ep_cb(struct usb_ep * ep)
{
    ep_driver_t * ep_driver = (ep_driver_t*) ep->driver_data;
    pphid_t *hid = ep_driver->dev->hid;
    hid_class_ep_data_t * hiddata = (hid_class_ep_data_t *)ep_driver->class_ep_data;

    HD("pphid_ep_cb\n");
    /* signal */
    wake_up_interruptible(&hid->hid_wq);

    if (hiddata->timedout && !hiddata->waiting) {
	/* there was a timeout and no other event is waiting
	   so just resend the last packet */
	send_report(hid, ep, NULL, 0, 1);
    }
}

void pphid_mouse_set_pointer(pphid_t *hid, struct usb_ep *ep, const uint8_t *data, int length)
{
    if (length > MAX_MOUSE_REPORT || length != hid->current_mouse_report_length) {
	printk("wrong data length: %d (%d expected)\n", length, hid->current_mouse_report_length);
	return;
    }
    memcpy(hid->mouse_report, data, length);
    hid->mouse_report_length = length;
    mouse_send_report(hid, ep);
}

void pphid_keyboard_set_report(pphid_t *hid, struct usb_ep *ep, const uint8_t *data, int length)
{
    /* keyboard reports are always 8 bytes long! */
    if (length  != MAX_KBD_REPORT) {
	printk("wrong data length: %d (%d expected)\n", length, MAX_KBD_REPORT);
	return;
    }
    /* only send a new report, when the data has changed,
       otherwise win2k produces bug #470, because
       it seems that it resets the counter before start repeating to zero,
       when a new USB event is sent */
    if (!memcmp(hid->keyboard_report, data, length)) {
	/* old and new report are equal - don't sent it */
	return;
    }
    memcpy(hid->keyboard_report, data, length);
    hid->keyboard_report_length = length;
    keyboard_send_report(hid, ep);
}

