/* linux includes */
#include <linux/config.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/version.h>

/* local includes */
#include "ripcbelkin_usb.h"
#include "pp_usb.h"
#include "hal_isp1181.h"
#include "kernel_usb.h"

extern usb_driver_t g_ud;
extern volatile ControlData_t g_ControlData;
extern lara_usb_dev_t dev;

#define RIPCBELKIN_USB_MAX_PACKET_SIZE 64

#define INT_TIMEOUT 1 /* 1 second */

/* callback functions */
static void ripcbelkin_usb_config_endpoints(void);
static void belkinripc_handle_vendor_device_request(u_int8_t bmRequestType,
						    u_int8_t bRequest,
						    u_int16_t wValue,
						    u_int16_t wIndex,
						    u_int16_t wLength);
static int belkinripc_get_data_length(u_int8_t bmRequestType,
				       u_int8_t bRequest,
				       u_int16_t wValue,
				       u_int16_t wIndex,
				       u_int16_t wLength);
static int belkinripc_send_data(const uint8_t channel, const uint8_t type, const char* data, const int length);

/* descriptors */
static const USB_DEVICE_DESCRIPTOR  ripcbelkinDeviceDescriptor;
static const LANGID_DESCRIPTOR  LANGID_Descriptor;
static const MANUFACTURER_DESCRIPTOR  Manufacturer_Descriptor;
static const PRODUCT_DESCRIPTOR  Product_Descriptor;
static const SERIAL_DESCRIPTOR  Serial_Descriptor;
static const RIPCBELKIN_USB_CONFIG_DATA  ripcbelkinConfigurationDescriptor;

static wait_queue_head_t ripc_wq;
static void ripc_ep_cb(uint8_t ep);

void ripc_init(void) {
    init_waitqueue_head (&ripc_wq);
    g_ud.eps[BELKIN_RIPC_MOUSE_EP_1].isr_int_in_cb = ripc_ep_cb;
    g_ud.eps[BELKIN_RIPC_MOUSE_EP_0].isr_int_in_cb = ripc_ep_cb;
    g_ud.eps[BELKIN_RIPC_KEYBOARD_EP_0].isr_int_in_cb = ripc_ep_cb;
    g_ud.eps[BELKIN_RIPC_KEYBOARD_EP_1].isr_int_in_cb = ripc_ep_cb;
}



int ripcbelkin_usb_init(usb_driver_t *ud, usb_device_mouse_type_t mouse_type, int * enable_mass_storage) {
    int i;
    /* some default stuff */
    for (i = 0; i < MAX_INTERFACES; i++) {
	ud->class_interface_request[i] = NULL;
    }
    for (i = EP1; i <= EP14; i++) {
	    ud->ep_int_functions[i] = NULL;
    }
    for (i = EP1; i <= EP14; i++) {
	ud->eps[i].used = 0;
    }
    ud->vendor_device_request = NULL;
    
    ud->send_data = belkinripc_send_data;
    ud->usb_max_packet_size = RIPCBELKIN_USB_MAX_PACKET_SIZE;
    ud->ConfigDescriptor = (char*)&ripcbelkinConfigurationDescriptor;
    ud->DeviceDescriptor = (char*)&ripcbelkinDeviceDescriptor;
    ud->StringDescriptors[0] = (char*)&LANGID_Descriptor;
    ud->get_data_length = belkinripc_get_data_length;
    for (i = 1; i < MAX_STRING_DESCRIPTORS; i++) {
	ud->StringDescriptors[i] = NULL;
    }
    ud->config_endpoints = ripcbelkin_usb_config_endpoints; /* configures endpoints 1 to 14 */
    ud->usb_init = ripc_init;

    ud->non_std_descriptors = NULL;
    
    ud->eps[BELKIN_RIPC_MOUSE_EP_0].used = 1;
    ud->eps[BELKIN_RIPC_MOUSE_EP_0].ep_type = INTERRUPT_IN;
   
    ud->eps[BELKIN_RIPC_KEYBOARD_EP_0].used = 1;
    ud->eps[BELKIN_RIPC_KEYBOARD_EP_0].ep_type = INTERRUPT_IN;

    ud->eps[BELKIN_RIPC_MOUSE_EP_1].used = 1;
    ud->eps[BELKIN_RIPC_MOUSE_EP_1].ep_type = INTERRUPT_IN;
   
    ud->eps[BELKIN_RIPC_KEYBOARD_EP_1].used = 1;
    ud->eps[BELKIN_RIPC_KEYBOARD_EP_1].ep_type = INTERRUPT_IN;

    ud->vendor_device_request = belkinripc_handle_vendor_device_request;

    ud->initialized = 1;

    return 0;
}

static int belkinripc_get_data_length(uint8_t bmRequestType,
				      uint8_t bRequest,
				      uint16_t wValue,
				      uint16_t wIndex,
				      uint16_t wLength) {
    
    uint8_t type = bmRequestType & USB_REQUEST_TYPE_MASK;
    uint8_t bRecipient = bmRequestType & USB_RECIPIENT_MASK;
    
    if (((type == USB_VENDOR_REQUEST) && (bRecipient == USB_RECIPIENT_DEVICE)) &&
	(bRequest == VRC_CONSOLE_BEEP)) {
	return 0;
    } else {
	return wLength;
    }
}

void ripc_send_report(uint8_t ep, const uint8_t* data, uint16_t len, uint8_t last_packet) {
    long time_left;
    DECLARE_WAITQUEUE (wait, current);
    UD("ripc_send_report\n"); 
    
    if (g_DeviceState != PP_USB_DEVSTAT_CONFIGURED) {
        return;
    }
    
    if (last_packet) {
        /* resend the last packet */
        data = g_ud.eps[ep].last_packet;
        len = g_ud.eps[ep].last_length;
    }
    
    g_ud.eps[ep].waiting = 1;

    /* wait for free slot */
    while (g_ud.eps[ep].status != EVENT_IDLE) {
        /* sleep here until delivery was successful or timeout */
        time_left = INT_TIMEOUT * HZ;
        UD("start waiting\n");
        do {
            current->state = TASK_INTERRUPTIBLE;
            add_wait_queue(&ripc_wq, &wait);
            spin_unlock_irq(&dev.chip_access_lock);
            time_left = schedule_timeout(time_left);
            remove_wait_queue(&ripc_wq, &wait);
            spin_lock_irq(&dev.chip_access_lock);
            UD("time_left: %lx\n", time_left);
        } while (time_left && g_ud.eps[ep].status != EVENT_IDLE);

        if (time_left == 0 && g_ud.eps[ep].status != EVENT_IDLE) {
            /* only if the data are less than the buffer size */ 
            if (len <= 64) {
                g_ud.eps[ep].timedout = 1; /* schedule resend */
                g_ud.eps[ep].last_length = len;
                memcpy(g_ud.eps[ep].last_packet, data, len);
            }
            g_ud.eps[ep].waiting = 0;
            g_ud.eps[ep].status = EVENT_IDLE;
            UD("timedout\n");
            return;
        }
    }
    g_ud.eps[ep].waiting = 0;
    g_ud.eps[ep].timedout = 0;
    UD("ripc_send_report - send data\n");
    WriteEndpoint(ep, data, len);
    g_ud.eps[ep].status = EVENT_QUEUED;
    UD("ripc_send_report - sent data - wait for delivery\n");
    if (!last_packet) {
        // not call within an interrupt
        time_left = INT_TIMEOUT * HZ;
        do {
            UD("wait for reply\n");
            current->state = TASK_INTERRUPTIBLE;
            add_wait_queue(&ripc_wq, &wait);
            spin_unlock_irq(&dev.chip_access_lock);
            time_left = schedule_timeout(time_left);
            remove_wait_queue(&ripc_wq, &wait);
            spin_lock_irq(&dev.chip_access_lock);
        } while (time_left && g_ud.eps[ep].status != EVENT_IDLE);
    }
    UD("ripc_send_report done\n");
}

void ripc_ep_cb(uint8_t ep) {
    UD("ripc_ep_cb\n");
    g_ud.eps[ep].status = EVENT_IDLE;
    /* signal */
    wake_up_interruptible(&ripc_wq);
    if (g_ud.eps[ep].timedout && !g_ud.eps[ep].waiting) {
        /* there was a timeout and no other event is waiting
          so just resent the last packet */
        ripc_send_report(ep, NULL, 0, 1);
    }
}

static int belkinripc_send_data(const uint8_t channel, const uint8_t type, const char* data, const int length) {
    uint8_t ep;

    switch (channel) {
      case 0:
	  if (type == PP_KM_INPUT_KEYBOARD) {
	      ep = BELKIN_RIPC_KEYBOARD_EP_0;
	  } else if (type == PP_KM_INPUT_MOUSE) {
	      ep = BELKIN_RIPC_MOUSE_EP_0;
	  } else {
	      return -1;
	  }
	  break;
      case 1:
	  if (type == PP_KM_INPUT_KEYBOARD) {
	      ep = BELKIN_RIPC_KEYBOARD_EP_1;
	  } else if (type == PP_KM_INPUT_MOUSE) {
	      ep = BELKIN_RIPC_MOUSE_EP_1;
	  } else {
	      return -1;
	  }
	  break;
      default:
	  return -1;
    }
	
    ripc_send_report(ep, (uint8_t*)data, length, 0);
    return 0;
}

static void ripcbelkin_usb_config_endpoints(void) {
    uint8_t i;
    uint8_t mode;

    for (i = EP1; i <= EP4; i++) {
	mode =
	    ISP1181B_EP_CONFIG_FIFOEN |
	    ISP1181B_EP_CONFIG_EPIN |
	    (0x1 & 0xf);
	hal_SetEndpointConfig(i, mode);
    }
    
    for (i = EP5; i <= EP14; i++) {
	mode = 0;
	hal_SetEndpointConfig(i, mode);
    }
}

void
belkinripc_handle_vendor_device_request(
					u_int8_t bmRequestType,
					u_int8_t bRequest,
					u_int16_t wValue,
					u_int16_t wIndex,
					u_int16_t wLength) {
    
    static host_change_t hc;
    usb_usrspace_data_t rpc;
    printk("-- USB --belkinripc handle vendor request: %x\n", bRequest); 
    switch (bRequest) {
      case VRC_CONSOLE_HOST_CHANGE: {
	  WriteEndpoint(EP0IN, NULL, 0);
	  memcpy(&hc, g_ControlData.pData, sizeof(hc));
	  rpc.function = PP_KVM_PORT_CHANGED;
	  rpc.arguments.pp_kvm_port_changed_data.channel = wValue;
	  rpc.arguments.pp_kvm_port_changed_data.old_bank = hc.old_bank;
	  rpc.arguments.pp_kvm_port_changed_data.old_host = hc.old_host;
	  rpc.arguments.pp_kvm_port_changed_data.new_bank = hc.new_bank;
	  rpc.arguments.pp_kvm_port_changed_data.new_host = hc.new_host;
	  rpc.arguments.pp_kvm_port_changed_data.control = hc.control;
	  call_user_space(&rpc);
	  break;
      }
      case VRC_CONSOLE_DEVICE_STATUS:
	  WriteEndpoint(EP0IN, NULL, 0);
	  rpc.function = PP_KVM_DEVICE_STATUS;
	  rpc.arguments.pp_kvm_device_status_data.channel = wValue;
	  rpc.arguments.pp_kvm_device_status_data.status = wIndex;
	  call_user_space(&rpc);
	  break;
      case VRC_CONSOLE_BEEP:
	  WriteEndpoint(EP0IN, NULL, 0);
	  rpc.function = PP_KVM_CONSOLE_BEEP;
	  rpc.arguments.pp_kvm_console_beep_data.channel = wValue;
	  rpc.arguments.pp_kvm_console_beep_data.pitch = wIndex;
	  rpc.arguments.pp_kvm_console_beep_data.time = wLength;
	  call_user_space(&rpc);
	  break;
      case VRC_CONSOLE_AUTOSCAN:
	  WriteEndpoint(EP0IN, NULL, 0);
	  rpc.function = PP_KVM_AUTOSCAN;
	  rpc.arguments.pp_kvm_autoscan_data.channel = wValue;
	  rpc.arguments.pp_kvm_autoscan_data.status = wIndex;
	  call_user_space(&rpc);
	  break;
      case VRC_IP_ADDRESS:
	  {
	      ip_and_mask_t im;
	      uint32_t ip, mask;
	      
	      WriteEndpoint(EP0IN, NULL, 0);
	      memcpy(&im, g_ControlData.pData, sizeof(im));
	      ip = ((uint32_t)im.ip0 << 24) |
		  ((uint32_t)im.ip1 << 16) |
		  ((uint32_t)im.ip2 << 8) |
		  ((uint32_t)im.ip3);
	      mask = ((uint32_t)im.mask0 << 24) |
		  ((uint32_t)im.mask1 << 16) |
		  ((uint32_t)im.mask2 << 8) |
		  ((uint32_t)im.mask3);
	      rpc.function = PP_KVM_IP_ADDRESS;
	      rpc.arguments.pp_kvm_ip_address_data.ip = ip;
	      rpc.arguments.pp_kvm_ip_address_data.mask = mask;
              rpc.arguments.pp_kvm_ip_address_data.valid = KVM_IP_ADDRESS_VALID_IP | KVM_IP_ADDRESS_VALID_MASK;
	      call_user_space(&rpc);
	  }
	  break;
      case VRC_CONSOLE_LED_STATUS:
	  WriteEndpoint(EP0IN, NULL, 0);
	  rpc.function = PP_KVM_LED_CHANGE;
	  rpc.arguments.pp_kvm_led_change_data.channel = wValue;
	  rpc.arguments.pp_kvm_led_change_data.scroll_lock = (wIndex & 0x01)?1:0;
	  rpc.arguments.pp_kvm_led_change_data.num_lock = (wIndex & 0x02)?1:0;
	  rpc.arguments.pp_kvm_led_change_data.caps_lock = (wIndex & 0x04)?1:0;
	  call_user_space(&rpc);
	  break;
      case VRC_OSD_ON:
	  {
	      osd_info_t oi;
	      uint16_t x,y,w,h;
	      
	      WriteEndpoint(EP0IN, NULL, 0);
	      memcpy(&oi, g_ControlData.pData, sizeof(oi));
	      x = le16_to_cpu(oi.x);
	      y = le16_to_cpu(oi.y);
	      w = le16_to_cpu(oi.w);
	      h = le16_to_cpu(oi.h);
	      printk("USB: osd on: x: %u, y: %u, w: %u, h: %u\n",x,y,w,h);
	  }
	  break;
      case VRC_OSD_OFF:
	  WriteEndpoint(EP0IN, NULL, 0);
	  printk("USB: osd off on console: %d\n", wValue);
	  break;
      case VRC_UNIT_ADDED:
	  WriteEndpoint(EP0IN, NULL, 0);
	  rpc.function = PP_KVM_SET_UNIT_HOSTS;
	  rpc.arguments.pp_kvm_set_unit_hosts_data.unit = wValue;
	  rpc.arguments.pp_kvm_set_unit_hosts_data.hosts = wIndex;
	  call_user_space(&rpc);
	  break;
      case VRC_UNIT_REMOVED:
         WriteEndpoint(EP0IN, NULL, 0);
         rpc.function = PP_KVM_SET_UNIT_HOSTS;
         rpc.arguments.pp_kvm_set_unit_hosts_data.unit = wValue;
         rpc.arguments.pp_kvm_set_unit_hosts_data.hosts = -1;
         call_user_space(&rpc);
         break;
      case VRC_GATEWAY_ADDRESS:
         {
             gateway_t im;
             uint32_t gw;
             
             WriteEndpoint(EP0IN, NULL, 0);
             memcpy(&im, g_ControlData.pData, sizeof(im));
             gw = ((uint32_t)im.gw0 << 24) |
                 ((uint32_t)im.gw1 << 16) |
                 ((uint32_t)im.gw2 << 8) |
                 ((uint32_t)im.gw3);
             rpc.function = PP_KVM_IP_ADDRESS;
             rpc.arguments.pp_kvm_ip_address_data.gw = gw;
             rpc.arguments.pp_kvm_ip_address_data.valid = KVM_IP_ADDRESS_VALID_GW;
             call_user_space(&rpc);
         }
         break;

      default:
	  Chap9_StallEP0();
	  break;
    }
}

#pragma pack(1)
static const USB_DEVICE_DESCRIPTOR  ripcbelkinDeviceDescriptor = {
    sizeof(USB_DEVICE_DESCRIPTOR), /* bLength */
    TYPE_DEVICE_DESCRIPTOR,        /* bDescriptorType */
    __constant_cpu_to_le16(0x0110),     /* bcdUSB USB Version 1.1 */
    0xff,                          /* bDeviceClass */
    0,                             /* bDeviceSubclass */
    0,                             /* bDeviceProtocol */
    RIPCBELKIN_USB_MAX_PACKET_SIZE,/* bMaxPacketSize 8 Bytes */
    __constant_cpu_to_le16(0x050d),     /* idVendor */
    __constant_cpu_to_le16(0xff07),     /* idProduct */
    __constant_cpu_to_le16(0x0002),     /* bcdDevice */
    0,                             /* iManufacturer String Index */
    0,                             /* iProduct String Index */
    0,                             /* iSerialNumber String Index */
    1                              /* bNumberConfigurations */
};

static const RIPCBELKIN_USB_CONFIG_DATA  ripcbelkinConfigurationDescriptor = {
    {  /* configuration descriptor */
	sizeof(USB_CONFIGURATION_DESCRIPTOR),              /* bLength */
	TYPE_CONFIGURATION_DESCRIPTOR,                    /* bDescriptorType */
	__constant_cpu_to_le16(sizeof(RIPCBELKIN_USB_CONFIG_DATA)),       /* wTotalLength */
	1,                                                /* bNumInterfaces */
	1,                                                /* bConfigurationValue */
	0,                                                /* iConfiguration String Index */
	0xC0,                                             /* bmAttributes Bus Powered, No Remote Wakeup */
	0x00                                              /* bMaxPower, 100mA */ 
    },
    {  /* interface descriptor */
	sizeof(USB_INTERFACE_DESCRIPTOR), /* bLength */
	TYPE_INTERFACE_DESCRIPTOR,        /* bDescriptorType */
	0,                                /* bInterface Number */
	0,                                /* bAlternateSetting */
	2,                                /* bNumEndpoints */
	0x00,                             /* bInterfaceClass (Vendor specific) */
	0x00,                             /* bInterfaceSubClass: No Subclass */
	0x00,                             /* bInterfaceProtocol: Mouse */
	0                                 /* iInterface String Index */
    },
    {   /* endpoint descriptor */
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x81,                             /* bEndpoint Address EP1 IN */
	0x03,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0008),        /* wMaxPacketSize */
	0x08                              /* bInterval */
    },
    {   /* endpoint descriptor */
	sizeof(USB_ENDPOINT_DESCRIPTOR),  /* bLength */
	TYPE_ENDPOINT_DESCRIPTOR,         /* bDescriptorType */
	0x82,                             /* bEndpoint Address EP1 IN */
	0x03,                             /* bmAttributes - Interrupt */
	__constant_cpu_to_le16(0x0008),        /* wMaxPacketSize */
	0x08                              /* bInterval */
    },
};

static const LANGID_DESCRIPTOR  LANGID_Descriptor = { /* LANGID String Descriptor 0 */
    sizeof(LANGID_DESCRIPTOR),                 /* bLength */
    TYPE_STRING_DESCRIPTOR,                    /* bDescriptorType */
    __constant_cpu_to_le16(0x0409)             /* LANGID US English */
};

#pragma pack()

