#include <linux/sched.h>
#include <linux/types.h>
#include <linux/random.h>
#include <linux/socket.h>
#include <linux/net.h>
#include <linux/un.h>
#include <net/sock.h>

#include <pp_ipc_socket.h>
#include "scsi_ipmi.h"

#define MAX_SESSIONS 10

typedef struct {
    unsigned long id;
    struct socket *sock;
    unsigned long timestamp;
    int timeout;
} session_t;

session_t sessions[MAX_SESSIONS];

static struct sockaddr_un eric_sa;

static inline session_t *get_session(int id)
{
    int i;
    for (i = 0; i < MAX_SESSIONS; i++) {
	if (sessions[i].sock && sessions[i].id == id) {
	    return &sessions[i];
	}
    }
    return NULL;
}

static inline int expired(session_t *session)
{
    return (jiffies > session->timestamp + session->timeout * HZ);
}

int scsi_ipmi_init(void)
{
    memset(sessions, 0, sizeof(sessions));
    eric_sa.sun_family = AF_UNIX;
    strcpy(eric_sa.sun_path, "/tmp/eric_control.socket");
    return 0;
}

unsigned int scsi_ipmi_session_new(int timeout)
{
    int i;
    session_t *sess = sessions;
    int optval = 1;
    for (i = 0; i < MAX_SESSIONS; i++) {
	if (!sessions[i].sock || expired(&sessions[i])) {
	    sess = &sessions[i];
	    break;
	}
	if (sessions[i].timestamp < sess->timestamp) {
	    sess = &sessions[i];
	}
    }
    get_random_bytes(&sess->id, sizeof(sess->id));
    if (sess->sock) sock_release(sess->sock);
    sock_create(AF_UNIX, SOCK_DGRAM, 0, &sess->sock);
    sock_setsockopt(sess->sock, SOL_SOCKET, SO_PASSCRED, (char *)&optval, sizeof(optval));
    sess->timestamp = jiffies;
    sess->timeout = timeout;
    return sess->id;
}

void scsi_ipmi_session_close(unsigned int id)
{
    session_t *sess = get_session(id);
    if (sess) {
	sock_release(sess->sock);
	sess->sock = NULL;
    }
}

int scsi_ipmi_send_cmd(unsigned int id, int length, unsigned char *data)
{
    pp_ipc_req_type_t request_type = PP_IPC_REQ_IPMI;
    struct iovec iov[2];
    struct msghdr msg;
    int r;

    session_t *sess = get_session(id);
    if (!sess) return -EINVAL;

    iov[0].iov_base = &request_type;
    iov[0].iov_len = sizeof(request_type);
    iov[1].iov_base = data;
    iov[1].iov_len = length;

    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &eric_sa;
    msg.msg_namelen = sizeof(eric_sa);
    msg.msg_iov = iov;
    msg.msg_iovlen = 2;
    r = sock_sendmsg(sess->sock, &msg, sizeof(request_type) + length);
    if (r < 0) return r;
    return r - sizeof(request_type);
}

int scsi_ipmi_get_response(unsigned int id, int length, unsigned char *data)
{
    pp_ipc_rsp_status_t status;
    struct iovec iov[2];
    struct msghdr msg;
    int r;

    session_t *sess = get_session(id);
    if (!sess) return -EINVAL;

    iov[0].iov_base = &status;
    iov[0].iov_len = sizeof(status);
    iov[1].iov_base = data;
    iov[1].iov_len = length;

    memset(&msg, 0, sizeof(msg));
    msg.msg_iov = iov;
    msg.msg_iovlen = 2;
    r = sock_recvmsg(sess->sock, &msg, sizeof(status) + length, 0);
    if (r < 0) return r;
    return r - sizeof(status);
}

