#include <fcntl.h>
#include <stdio.h>
#include <pthread.h>
#include <unistd.h>
#include <openssl/bio.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <pp/base.h>
#include <pp/intl.h>
#include <liberic_config.h>
#include <liberic_cert.h>

#include "CertManager.h"
#include "req.h"

static const char * eric_cert_errors[] = {
    N_("The operation performed successfully."),
    N_("An internal error occurred."),
    N_("There is no CSR pending."),
    N_("The certificate is invalid or does not match our key."),
    N_("The provided file is not a valid certificate.<BR>\n You have to upload a valid signature for your pending CSR."),
};

static int initialized = 0;

PP_SYM_HIDDEN pthread_mutex_t cert_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

int
eric_cert_init(void)
{
    initialized = 1;
    return 0;
}

void
eric_cert_cleanup(void)
{
    initialized = 0;
}

const char *
eric_cert_get_error_string(const int error)
{
    if (error < (int)(sizeof(eric_cert_errors) / sizeof(*eric_cert_errors))) {
	return eric_cert_errors[error];
    } else {
	return N_("An unknown error occurred.");
    }
}

void
eric_cert_install(const void * cert_data, size_t cert_data_size, int * error)
{
    SSL_CTX * ssl_ctx = NULL;
    BIO * out = NULL;

    *error = ERIC_CERT_ERR_INTERNAL_ERROR;

    MUTEX_LOCK(&cert_mtx);

    if ((out = BIO_new_file(ERIC_CERT_NEWCERT_FILEPATH, "w+")) == NULL) {
#ifndef NDEBUG
	pp_log("BIO_new_file() failed\n");
#endif
	unlink(ERIC_CERT_NEWCERT_FILEPATH);
	goto bail;
    }

    if (BIO_write(out, cert_data, cert_data_size) != (ssize_t)cert_data_size) {
#ifndef NDEBUG
	pp_log_err("BIO_write() failed");
#endif
	unlink(ERIC_CERT_NEWCERT_FILEPATH);
	goto bail;
    }

    BIO_free(out);
    out = NULL;

    if ((ssl_ctx = SSL_CTX_new(SSLv23_server_method())) == NULL) {
#ifndef NDEBUG
	pp_log("SSL_CTX_new() failed\n");
#endif
	unlink(ERIC_CERT_NEWCERT_FILEPATH);
	goto bail;
    }
    
    if (SSL_CTX_use_certificate_file(ssl_ctx, ERIC_CERT_NEWCERT_FILEPATH,
				     SSL_FILETYPE_PEM) <= 0) {
#ifndef NDEBUG
	pp_log("SSL_CTX_use_certificate_file() failed\n");
#endif
        // this usually happens when the certificate file has the wrong format
        *error = ERIC_CERT_ERR_CERT_FORMAT_WRONG;
	unlink(ERIC_CERT_NEWCERT_FILEPATH);
	goto bail;
    }

    if (SSL_CTX_use_PrivateKey_file(ssl_ctx, ERIC_CERT_NEWKEY_FILEPATH,
				    SSL_FILETYPE_PEM) <= 0) {
#ifndef NDEBUG
	pp_log("SSL_CTX_use_PrivateKey_file() failed\n");
#endif
	*error = ERIC_CERT_ERR_CERT_NOT_MATCHING;
	unlink(ERIC_CERT_NEWCERT_FILEPATH);
	goto bail;
    }

    SSL_CTX_free(ssl_ctx);
    ssl_ctx = NULL;

    if (rename(ERIC_CERT_NEWCERT_FILEPATH, ERIC_CERT_USRCERT_FILEPATH) == -1) {
#ifndef NDEBUG
	pp_log_err("rename(CERT) failed\n");
#endif
	goto bail;
    }

    if (rename(ERIC_CERT_NEWKEY_FILEPATH, ERIC_CERT_USRKEY_FILEPATH) == -1) {
#ifndef NDEBUG
	pp_log_err("rename(KEY) failed\n");
#endif
	goto bail;
    }

    if (unlink(ERIC_CERT_CSR_FILEPATH) == -1) {
#ifndef NDEBUG
	pp_log_err("unlink(CSR) failed\n");
#endif
	goto bail;
    }

    /* flush to flash */
    eric_config_flush();

    *error = ERIC_CERT_ERR_NO_ERROR;

 bail:
    MUTEX_UNLOCK(&cert_mtx);

    if (out) { BIO_free(out); }
    if (ssl_ctx) { SSL_CTX_free(ssl_ctx); }
}

void
eric_cert_get_csr(void * csr_data, size_t * csr_data_size_p, int * error)
{
    BIO * in = NULL;
    ssize_t n;

    *error = ERIC_CERT_ERR_INTERNAL_ERROR;

    MUTEX_LOCK(&cert_mtx);

    if ((in = BIO_new(BIO_s_file())) == NULL) {
	goto bail;
    }

    /* open the file with the CSR */
    /* strdupa() to avoid warning because of const */
    if (BIO_read_filename(in, strdupa(ERIC_CERT_CSR_FILEPATH)) <= 0) {
	*error = ERIC_CERT_ERR_NO_PENDING_CERT;
	goto bail;
    }

    if ((n = BIO_read(in, csr_data, *csr_data_size_p)) < 0) {
	goto bail;
    }

    *csr_data_size_p = n;

    *error = ERIC_CERT_ERR_NO_ERROR;

 bail:
    MUTEX_UNLOCK(&cert_mtx);
    if (in) { BIO_free(in); }
}

void
eric_cert_get_csr_info(char * info_buf, size_t info_buf_size, int * error)
{
    BIO *in = NULL;
    BIO *out = NULL;
    X509_REQ *req = NULL;
    char * info = NULL;
    size_t out_size, size;
	
    *error = ERIC_CERT_ERR_INTERNAL_ERROR;

    MUTEX_LOCK(&cert_mtx);

    if ((in = BIO_new(BIO_s_file())) == NULL) {
	goto bail;
    }

    if ((out = BIO_new(BIO_s_mem())) == NULL) {
	goto bail;
    }

    /* open the file with the CSR */
    /* strdupa() to avoid warning because of const */
    if (BIO_read_filename(in, strdupa(ERIC_CERT_CSR_FILEPATH)) <= 0) {
	*error = ERIC_CERT_ERR_NO_PENDING_CERT;
	goto bail;
    }

    /* extract CSR information */
    if ((req = PEM_read_bio_X509_REQ(in, NULL, NULL, NULL)) == NULL) {
	goto bail;
    }

    X509_NAME_print_ex(out, X509_REQ_get_subject_name(req), 0,
		       XN_FLAG_MULTILINE);

    out_size = BIO_get_mem_data(out, &info);
    size = out_size < info_buf_size ? out_size : info_buf_size;
    memcpy(info_buf, info, size);
    info_buf[size] = '\0';

    *error = ERIC_CERT_ERR_NO_ERROR;

 bail:
    MUTEX_UNLOCK(&cert_mtx);
    if (req) { X509_REQ_free(req); }
    if (out) { BIO_free(out); }
    if (in) { BIO_free(in); }
}

void
eric_cert_delete_key(int * error)
{
    int ret, myerror = 0;
    
    *error = ERIC_CERT_ERR_INTERNAL_ERROR;

    MUTEX_LOCK(&cert_mtx);

    ret = unlink(ERIC_CERT_NEWKEY_FILEPATH);
    if (ret && errno != ENOENT) {
	myerror += 1;
    }
    ret = unlink(ERIC_CERT_CSR_FILEPATH);
    if (ret && errno != ENOENT) {
	myerror += 1;
    }
    if (myerror) {
	goto bail;
    }

    /* flush to flash */
    eric_config_flush();

    *error = ERIC_CERT_ERR_NO_ERROR;

 bail:
    MUTEX_UNLOCK(&cert_mtx);
}

void
eric_cert_create_key(const char * c, const char * st, const char * l,
		     const char * o, const char * ou, const char * cn,
		     const char * email, const char * challenge,
		     unsigned int bits, int * error)
{
    char bits_buf[9];
    FILE * cfg_in_fp = NULL;
    FILE * cfg_out_fp = NULL;
    const char * argv[] = {
	"req", 
	"-newkey", bits_buf,
	"-keyout", ERIC_CERT_NEWKEY_FILEPATH,
	"-out", ERIC_CERT_CSR_FILEPATH, 
	"-config", ERIC_CERT_CFG_FILEPATH,
	NULL
    };
    int argc = 9;
    
    *error = ERIC_CERT_ERR_INTERNAL_ERROR;
    
    /* check input */
    if (bits < ERIC_CERT_MIN_KEY_LENGTH || bits > ERIC_CERT_MAX_KEY_LENGTH) {
	return;
    }

    snprintf(bits_buf, sizeof(bits_buf), "rsa:%u", bits);

    MUTEX_LOCK(&cert_mtx);

    /* FIXME: check for pending CSR and return with error if found! */

    if ((cfg_in_fp = fopen(ERIC_CERT_CFGHDR_FILEPATH, "r")) == NULL) {
	goto bail;
    }
    if ((cfg_out_fp = fopen(ERIC_CERT_CFG_FILEPATH, "w+")) == NULL) {
	goto bail;
    }

    /* copy the openssl config header */
    while (!feof(cfg_in_fp) && !ferror(cfg_in_fp) && !ferror(cfg_out_fp)) {
	char buf[1024];
	int n = fread(buf, 1, sizeof(buf), cfg_in_fp);
	fwrite(buf, 1, n, cfg_out_fp);
    }
    fclose(cfg_in_fp);
    cfg_in_fp = NULL;

    /* append our stuff to the openssl config file */
    fprintf(cfg_out_fp, "[ req_distinguished_name ]\n");
    if (c && *c != '\0') {
	fprintf(cfg_out_fp, "countryName = %s\n", c);
    }
    if (st && *st != '\0') {
	fprintf(cfg_out_fp, "stateOrProvinceName = %s\n", st);
    }
    if (l && *l != '\0') {
	fprintf(cfg_out_fp, "localityName = %s\n", l);
    }
    if (o && *o != '\0') {
	fprintf(cfg_out_fp, "0.organizationName = %s\n", o);
    }
    if (ou && *ou != '\0') {
	fprintf(cfg_out_fp, "organizationalUnitName = %s\n", ou);
    }
    if (cn && *cn != '\0') {
	fprintf(cfg_out_fp, "commonName = %s\n", cn);
    }
    if (email && *email != '\0') {
	fprintf(cfg_out_fp, "emailAddress = %s\n", email);
    }
    fprintf(cfg_out_fp, "\n[ req_attributes ]\n");
    if (challenge && *challenge != '\0') {
	fprintf(cfg_out_fp, "challengePassword = %s\n", challenge);
    }
    fclose(cfg_out_fp);
    cfg_out_fp = NULL;
    
    /* generate the key and the CSR */
    if (req_wrapper(argc, argv) != 0) { goto bail; }

    /* flush to flash */
    eric_config_flush();

    *error = ERIC_CERT_ERR_NO_ERROR;

 bail:
    MUTEX_UNLOCK(&cert_mtx);

    if (cfg_in_fp) { fclose(cfg_in_fp); }
    if (cfg_out_fp) { fclose(cfg_out_fp); }
}
