#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <pthread.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_pthread.h>
#include <liberic_net.h>
#include <liberic_webs.h>
#include <pp/bio.h>
#include "net.h"
#include "reconf.h"

#ifdef PP_FEAT_ESB2_TPT
#include "esb2_tpt_net.h"
#endif

#define POLL_TIMEOUT_MS	1000

static int initialized = 0;
static int listener_started = 0;
static pthread_t listener_thread;

static u_int overall_port_count = 0;
static pp_hash_i_t * acceptor_hash_by_proto = NULL;
static pp_hash_i_t * acceptor_hash_by_fd = NULL;
static pthread_mutex_t acceptor_hash_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;

static void * listener_thread_func(void * arg);
static void free_acceptor(void * _acceptor);
static void configure_net(struct pollfd ** fds_p, size_t * fd_count_p);
static void unconfigure_net(void);

extern void demuxer_handle_connection(pp_net_conn_data_t * conn_data);

int
eric_net_init(void)
{
    if (!initialized) {
	if ((acceptor_hash_by_proto = pp_hash_create_i(17)) == NULL) {
	    pp_log("%s(): pp_hash_create_i() failed\n", ___F);
	    return PP_ERR;
	}

	if ((acceptor_hash_by_fd = pp_hash_create_i(31)) == NULL) {
	    pp_log("%s(): pp_hash_create_i() failed\n", ___F);
	    pp_hash_delete_i(acceptor_hash_by_proto);
	    return PP_ERR;
	}

	net_reconf_init();

#ifdef PP_FEAT_WLAN
	start_wlan_watcher();
#endif

#ifdef PP_FEAT_ESB2_TPT
        /* start Intel's ESB2 TPT via FML listener */
	esb2_tpt_init();
#endif

#ifdef PP_FEAT_ESB2_FNI
	esb2_fni_init();
#endif

	initialized = 1;
    }

    return PP_SUC;
}

void
eric_net_cleanup(void)
{
#ifdef PP_FEAT_ESB2_TPT
    esb2_tpt_cleanup();
#endif

#ifdef PP_FEAT_ESB2_FNI
    esb2_fni_cleanup();
#endif

#ifdef PP_FEAT_WLAN
    stop_wlan_watcher();
#endif
    net_reconf_cleanup();

    pp_hash_delete_i(acceptor_hash_by_fd);
    pp_hash_delete_i(acceptor_hash_by_proto);
    if (listener_started) pthread_join(listener_thread, NULL);
    initialized = 0;
}

#ifdef PP_FEAT_ESB2_TPT
void eric_net_esb2_tpt_cleanup(void) {
    esb2_tpt_cleanup();
}
#endif

int
eric_net_start_listener(void)
{
    int ret = PP_ERR;

    if (initialized) {
	listener_started = 1;
	if (eric_pthread_create(&listener_thread, 0, 65536,
				listener_thread_func, NULL) == 0) {
	    ret = PP_SUC;
	} else {
	    listener_started = 0;
	    pp_log("%s(): Cannot create listener-thread.\n", ___F);
	}
    }

    return ret;
}

int
eric_net_register_proto(pp_net_protocol_type_t type, const char * name, u_int port_count,
			get_port_func_t get_port, init_bio_func_t init_bio, conn_handler_func_t handler)
{
    acceptor_t * acceptor;
    int ret = PP_ERR;

    assert(name);
    assert(initialized);

    MUTEX_LOCK(&acceptor_hash_mtx);
    if (port_count > 0) {
	if ((acceptor = malloc(sizeof(acceptor_t))) != NULL) {
	    acceptor->protocol_type = type;
	    acceptor->name = name;
	    acceptor->bios = calloc(port_count, sizeof(BIO*));
	    acceptor->bio_count = port_count;
	    acceptor->get_port = get_port;
	    acceptor->init_bio = init_bio;
	    acceptor->handler = handler;
	    if (PP_SUCCED(pp_hash_set_entry_i(acceptor_hash_by_proto, type, acceptor, free_acceptor))) {
		overall_port_count += port_count;
		ret = PP_SUC;
	    } else {
		free_acceptor(acceptor);
	    }
	}
    }
    MUTEX_UNLOCK(&acceptor_hash_mtx);
    return ret;
}

void
eric_net_unregister_proto(pp_net_protocol_type_t type)
{
    MUTEX_LOCK(&acceptor_hash_mtx);
    pp_hash_delete_entry_i(acceptor_hash_by_proto, type);
    MUTEX_UNLOCK(&acceptor_hash_mtx);
}

u_int
eric_net_get_registered_port_count(pp_net_protocol_type_t type)
{
    acceptor_t * acceptor;
    u_int port_count;
    MUTEX_LOCK(&acceptor_hash_mtx);
    acceptor = (acceptor_t *)pp_hash_get_entry_i(acceptor_hash_by_proto, type);
    port_count = acceptor ? acceptor->bio_count : 0;
    MUTEX_UNLOCK(&acceptor_hash_mtx);
    return port_count;
}

BIO *
eric_net_init_bio_standard(u_short port)
{
    char buf[8];
    BIO* in_bio;

    snprintf(buf, sizeof(buf), "*:%u", port);
    if ((in_bio = pp_bio_new_accept(buf, AF_INET)) == NULL) {
	pp_log_err("%s(): Cannot create accept BIO on port %u.", ___F, port);
	return NULL;
    }

    /* set accept sockets to non blocking mode and allow socket rebinding */
    /* BIO_set_nbio_accept(in_bio, 1) creates warning; workaround: */
    char a[2] = "a";
    BIO_ctrl(in_bio, BIO_C_SET_ACCEPT, 1, a);
    BIO_set_bind_mode(in_bio, BIO_BIND_REUSEADDR);

    /*
     * The first call will setup the accept socket, and the second
     * will get a socket.  In this loop, the first actual accept
     * will occur in the BIO_read() function.
     */
    if (BIO_do_accept(in_bio) <= 0) {
	pp_log_err("%s(): Cannot create accept socket on port %u.", ___F, port);
	BIO_free_all(in_bio);
	return NULL;
    }

    return in_bio;
}

static void *
listener_thread_func(void * arg UNUSED)
{
    pthread_t mpx_thread;
    unsigned int i;
    int sfd, n, one = 1;
    struct timeval tv;
    BIO* conn_bio;
    pp_net_conn_data_t * conn_data;
    struct pollfd * fds;
    u_int fd_count;
    
    tv.tv_sec = PP_NET_IO_TIMEOUT;
    tv.tv_usec = 0;

    // initially setup acceptor fds
    configure_net(&fds, &fd_count);

    for (;;) {

	if (reconf_listener) {
	    MUTEX_LOCK(&reconf_listener_mtx);
	    MUTEX_LOCK(&acceptor_hash_mtx);
    	    unconfigure_net();
	    configure_net(&fds, &fd_count);
	    reconf_listener = 0;
	    MUTEX_UNLOCK(&acceptor_hash_mtx);
	    MUTEX_UNLOCK(&reconf_listener_mtx);
	    continue;
	}

	if ((n = poll(fds, fd_count, POLL_TIMEOUT_MS)) == 0
	    || (n < 0 && (errno == EINTR || errno == EAGAIN))) {
	    continue;
	} else if (n < 0) {
	    pp_log_err("%s(): poll()", ___F);
	    break;
	}

	/* ---- accept connections -------------------------------------- */

	for (i = 0; i < fd_count; ++i) {
	    if (fds[i].revents & POLLIN) {

		acceptor_t * acceptor = pp_hash_get_entry_i(acceptor_hash_by_fd, fds[i].fd);
		BIO * bio = NULL;
		u_int port_offset;

		if (acceptor == NULL) continue;

		for (port_offset = 0; port_offset < acceptor->bio_count; ++port_offset) {
		    BIO * _bio = acceptor->bios[port_offset];
		    if (_bio && fds[i].fd == BIO_get_fd(_bio, NULL)) {
			bio = _bio;
			break;
		    }
		}

		if (bio == NULL) continue;

		if (BIO_do_accept(bio) <= 0) {
		    if (!BIO_should_retry(bio)) {
			/* some strange error occured - it's better we exit */
			exit(1);
		    }
		}
		if (NULL == (conn_bio = BIO_pop(bio))) continue;

		/* ---- set socket timeout and keep-alive -------------- */

		if ((sfd = BIO_get_fd(conn_bio, NULL)) == -1) {
		    pp_log("%s(): %s BIO not initialized.\n", ___F,
			   acceptor->name);
		    eric_net_close(conn_bio, NOWAIT);
		    continue;
		}

		/* set keep-alive */
		if (setsockopt(sfd, SOL_SOCKET, SO_KEEPALIVE,
			       (void *)&one, sizeof(one)) == -1) {
		     pp_log_err("%s(): %s BIO: Couldn't set keep-alive", ___F,
			        acceptor->name);
		}

		/* set send timeout */
		if (setsockopt(sfd, SOL_SOCKET,
			       SO_SNDTIMEO, (void *)&tv, sizeof(tv)) == -1) {
		     pp_log_err("%s(): %s BIO: Couldn't set send timeout", ___F,
			       acceptor->name);
		}

		/* set receive timeout */
		if (setsockopt(sfd, SOL_SOCKET, SO_RCVTIMEO,
			       (void *)&tv, sizeof(tv)) == -1) {
		     pp_log_err("%s(): %s BIO: Couldn't set receive timeout", ___F,
			       acceptor->name);
		}

		/* ---- start handler thread ------------------------- */

		if ((conn_data = malloc(sizeof(pp_net_conn_data_t))) != NULL) {
		    conn_data->protocol_type = acceptor->protocol_type;
		    conn_data->bio = conn_bio;
		    conn_data->port_offset = port_offset;
		    if (eric_pthread_create(&mpx_thread, 1, 128 * 1024,
					    (void*(*)(void*))acceptor->handler,
					    conn_data) != 0) {
			pp_log("%s(): %s BIO: Cannot create multiplexer thread.\n",
			       ___F, acceptor->name);
			eric_net_close(conn_bio, NOWAIT);
			free(conn_data);
		    }
		}
	    }
	}
    }
    /* stop listening */
    unconfigure_net();

    pthread_exit(NULL);
}

static void
free_acceptor(void * _acceptor)
{
    acceptor_t * acceptor = (acceptor_t *)_acceptor;
    u_int i;

    if (acceptor) {
	if (acceptor->bios) {
	    for (i = 0; i < acceptor->bio_count; ++i) {
		BIO * bio = acceptor->bios[i];
		if (bio != NULL) {
		    int fd = BIO_get_fd(bio, NULL);
		    if (fd >= 0) {
			pp_hash_delete_entry_i(acceptor_hash_by_fd, fd);
		    }
		    BIO_free_all(bio);
		}
	    }
	    free(acceptor->bios);
	}
	overall_port_count -= acceptor->bio_count;
	free(acceptor);
    }
}

static void
configure_net(struct pollfd ** fds_p, size_t * fd_count_p)
{
    struct pollfd clear_poll_fd = { -1, POLLIN, 0 };
    struct pollfd * fds;
    u_int i = 0, port_offset;
    int fd;
    u_short port;
    acceptor_t * acceptor;
    
    if (PP_FAILED(init_ssl())) {
	/* FIXME */
    }
    
    MUTEX_LOCK(&acceptor_hash_mtx);

    fds = (struct pollfd *)malloc(overall_port_count * sizeof(struct pollfd));

    /* start listening on all configured ports */
    for (acceptor = (acceptor_t *)pp_hash_get_first_entry(acceptor_hash_by_proto);
	 acceptor != NULL;
         acceptor = (acceptor_t *)pp_hash_get_next_entry(acceptor_hash_by_proto)) {
	for (port_offset = 0; port_offset < acceptor->bio_count; ++port_offset, ++i) {
	    fds[i] = clear_poll_fd;

	    /* check if port is enabled, otherwise: that's it */
	    port = acceptor->get_port ? acceptor->get_port() : 0;
	    if (port > 0) {
		BIO * bio;
		port += port_offset;
		bio = acceptor->init_bio ? acceptor->init_bio(port) : NULL;
		if (bio != NULL) {
		    fd = BIO_get_fd(bio, NULL);
		    if (fd >= 0) {
			if (PP_SUCCED(pp_hash_set_entry_i(acceptor_hash_by_fd, fd, acceptor, NULL))) {
			    fds[i].fd = fd;
			} else {
			    pp_log("%s(): pp_hash_set_entry_i() failed (ignored)\n", ___F);
			    BIO_free_all(bio);
			    bio = NULL;
			}
		    } else {
			pp_log("%s(): %s Accept BIO (port %u) not initialized (ignored)\n",
			       ___F, acceptor->name, port);
			BIO_free_all(bio);
			bio = NULL;
		    }
		} else {
		    pp_log("%s(): %s init_bio(%d) failed (ignored)\n", ___F, acceptor->name, port);
		}
		acceptor->bios[port_offset] = bio;
	    }
	}
    }

    *fds_p = fds;
    *fd_count_p = overall_port_count;

    MUTEX_UNLOCK(&acceptor_hash_mtx);
}

static void
unconfigure_net(void)
{
    acceptor_t * acceptor;
    u_int port_offset;

    MUTEX_LOCK(&acceptor_hash_mtx);

    for (acceptor = (acceptor_t *)pp_hash_get_first_entry(acceptor_hash_by_proto);
	 acceptor != NULL;
         acceptor = (acceptor_t *)pp_hash_get_next_entry(acceptor_hash_by_proto)) {
	for (port_offset = 0; port_offset < acceptor->bio_count; ++port_offset) {
	    if (acceptor->bios[port_offset]) {
		BIO_free_all(acceptor->bios[port_offset]);
		acceptor->bios[port_offset] = NULL;
	    }
	}
    }
    pp_hash_clear_i(acceptor_hash_by_fd);

    MUTEX_UNLOCK(&acceptor_hash_mtx);

    uninit_ssl();
}

int
eric_net_close(BIO * bio, eric_net_close_mode_t close_mode)
{
    static char buf[1024];
    struct timeval tv;
    int one = 1;
    int fd;

    if (!bio || (fd = BIO_get_fd(bio, NULL)) == -1) return -1;

#ifdef PP_FEAT_ESB2_TPT
    if (BIO_method_type(bio) == BIO_TYPE_PP_ESB2_TPT) {
        // nothing special to do, fd must stay open
        BIO_free_all(bio);
        return 0;
    }
#endif

    switch (close_mode) {
      case WAIT:
	  /* set receive timeout */
	  tv.tv_sec = PP_NET_IO_TIMEOUT;
	  tv.tv_usec = 0;
	  if (setsockopt(fd, SOL_SOCKET,
			 SO_RCVTIMEO, (void *)&tv, sizeof(tv)) == -1) {
	      pp_log_err("%s(): Couldn't set receive timeout", ___F);
	  }
	  shutdown(fd, SHUT_WR); /* shutdown write direction */
	  while (read(fd, buf, sizeof(buf)) > 0); /* wait for EOF or timeout */
	  BIO_free_all(bio);
	  return 0;
      case NOWAIT:
	  if (ioctl(fd, FIONBIO, &one) == 0) { /* switch to non-blocking I/O */
	      while (read(fd, buf, sizeof(buf)) > 0); /* discard input */
	  }
	  BIO_free_all(bio);
	  return 0;
      default:
	  pp_log("%s(): Unknown close mode (%d)\n", ___F, close_mode);
	  return -1;
    }
}

int
eric_net_read_exact(BIO * bio, void * buf, size_t len, int no_timeout)
{
    ssize_t n;

    if (bio == NULL || buf == NULL) {
	return -1;
    }

    while (len > 0) {
	if ((n = eric_net_read(bio, buf, len, no_timeout)) > 0) {
	    buf += n;
	    len -= n;
	} else {
	    return -1;
	}
    }
    return 0;
}

int
eric_net_write_exact(BIO * bio, const void * buf, size_t len, int no_timeout)
{
    ssize_t n;

    if (bio == NULL || buf == NULL) {
	return -1;
    }

    while (len > 0) {
	if ((n = eric_net_write(bio, buf, len, no_timeout)) > 0) {
	    buf += n;
	    len -= n;
	} else {
	    return -1;
	}
    }
    return 0;
}

ssize_t
eric_net_read(BIO * bio, void * buf, size_t len, int no_timeout)
{
    ssize_t n;

    if (bio == NULL || buf == NULL) {
	return -1;
    }

    while (1) {
	if ((n = BIO_read(bio, buf, len)) > 0) {
	    return n;
	} else if (n == 0) {
	    // pp_log("%s(): client gone\n", ___F);
	    return -1;
	} else if (errno == EAGAIN) {
	    if (no_timeout) {
		continue;
	    }
	    pp_log("%s(): client timed out\n", ___F);
	    return -1;
	} else if (errno == EINTR) {
	    continue;
	} else {
	    pp_log_err("%s(): read() failed", ___F);
	    return -1;
	}
    }
}

ssize_t
eric_net_write(BIO * bio, const void * buf, size_t len, int no_timeout)
{
    ssize_t n;
#ifdef PP_FEAT_ESB2_TPT
    int retries = 0;
#endif

    if (bio == NULL || buf == NULL) {
	return -1;
    }

    while (1) {
	if ((n = BIO_write(bio, buf, len)) > 0) {
	    return n;
	} else if (n == 0 || errno == EPIPE) {
	    //pp_log("%s(): client gone\n", ___F);
	    return -1;
	} else if (errno == EAGAIN) {
	    if (no_timeout) {
		continue;
	    }
	    pp_log("%s(): client timed out\n", ___F);
	    return -1;
	} else if (errno == EINTR) {
	    continue;
	} else {
#ifdef PP_FEAT_ESB2_TPT
            if (BIO_method_type(bio) == BIO_TYPE_PP_ESB2_TPT && ++retries < 3) {
                // try to work around BMC error
                pp_log("%s(): ASMI write error... retry\n", ___F);
                usleep(100000);
                continue;
            }
#endif
	    pp_log_err("%s(): write() failed", ___F);
	    return -1;
	}
    }
}

BIO *
eric_net_open_connection(const char * server, const char * port, unsigned int to_sec)
{
    BIO * bio;
    char host_port[255+1+5+1];

    snprintf(host_port, sizeof(host_port), "%s:%s", server, port);
    if ((bio = pp_bio_new_connect(host_port, AF_INET,
				  to_sec)) == NULL) {
        return NULL;
    }

    if (BIO_do_connect(bio) != 1) {
        eric_net_close_connection(bio);
        return NULL;
    }

    return bio;
}

void
eric_net_close_connection(BIO * bio)
{
    eric_net_close(bio, NOWAIT);
}
