#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <pp/base.h>
#include <pp/um.h>
#include <liberic_net.h>
#include <liberic_notify.h>
#include "reconf.h"

volatile int reconf_listener = 0;
pthread_mutex_t reconf_listener_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

static const char * proto_options[] = {
    "network.https_port",
    "network.http_port",
#ifdef PP_FEAT_TELNET
    "network.telnet_port",
    "network.telnet_enabled",
#endif
#ifdef PP_FEAT_SSH
    "network.ssh_port",
    "network.ssh_enabled",
#endif
#ifdef PP_FEAT_TERMINAL_SERVER
    "network.termsrv_baseport",
    "network.termsrv_enabled",
#endif
    "security.force_https",
    "security.rc_ssl",
    NULL /* must be present and the last */
};

static const char * fw_options[] = {
    "security.ip_fw.enabled",
    "security.ip_fw.default_policy",
    "security.ip_fw.ruleset",
    NULL /* must be present and the last */
};

static const char * eth_options[] = {
    "network.eth_speed",
    "network.eth_duplex",
    NULL /* must be present and the last */
};

static const char * main_options[] = {
    "network.ip_auto_config_proto",
    "network.mac",
    "network.ipaddr",
    "network.netmask",
    "network.gateway",
    "network.dns_ip_1",
    "network.dns_ip_2",
    "network.traffic_rate",
// FIXME: create extra handler for "time", it takes ages to reconfigure the whole net
    "time",
    "wlan.essid",
// FIXME    "wlan.mode",
    "wlan.security_type",
    "wlan.security_type.WEP.enabled",
    "wlan.security_type.WEP.key",
    "wlan.security_type.WPA.enabled",
// FIXME!    "wlan.security_type.WPA.type",
    "wlan.security_type.WPA.eap_peap_mschapv2.enabled",
    "wlan.security_type.WPA.eap_peap_mschapv2.identity",
    "wlan.security_type.WPA.eap_peap_mschapv2.password",
    "wlan.security_type.WPA.psk.enabled",
    "wlan.security_type.WPA.psk.key",
    NULL /* must be present and the last */
};

static void reconf_listener_core(void);
static int net_reconf_proto_ch(pp_cfg_chg_ctx_t * ctx);
static int net_reconf_fw_ch(pp_cfg_chg_ctx_t * ctx);
static int net_reconf_eth_ch(pp_cfg_chg_ctx_t * ctx);
static int net_reconf_main_ch(pp_cfg_chg_ctx_t * ctx);

#if defined(PRODUCT_ASMIDC) && defined(OEM_INTEL)
// on Intel ESB2 platforms, the ASMI module's network settings have to be
// announced to the external ESB2 BMC as well (LAN3)

#include <pp/ipmi.h>
#include <pp/xdefs.h>

#define ESB2_RMM2_LAN_CHAN 3 // Intel's BMC stores our settings in chan 3 (=LAN3)!

static void report_net_cfg_to_bmc(void)
{
    char *str;
    pp_ipmi_parameter_t ipmi_params;
    pp_ipmi_return_t ipmi_ret;
    int ipmi_err;

    memset(&ipmi_params, 0, sizeof(ipmi_params));
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    // MAC
    if (PP_FAILED(pp_cfg_get(&str, "network.mac"))) {
        pp_log("Error: failed to read MAC address from cfg\n");
    } else {
        ipmi_params.data.lanp_set_mac_address.chan = ESB2_RMM2_LAN_CHAN;
        ipmi_params.data.lanp_set_mac_address.value = str;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, 
                          PP_IPMI_LANP_SUBCMD_SET_MAC_ADDRESS, 
                          &ipmi_params, &ipmi_ret, &ipmi_err, NULL))) {
            pp_log("Error: failed to send MAC address to external BMC: %s\n", 
                pp_ipmi_get_error_string(ipmi_err));
        }

        pp_ipmi_cleanup_ret(&ipmi_ret);
        free(str);
    }

    // IP Source
    if (PP_FAILED(pp_cfg_get(&str, "network.ip_auto_config_proto"))) {
        pp_log("Error: failed to read IP source from cfg\n");
    } else {
        ipmi_params.data.lanp_set_ip_source.chan = ESB2_RMM2_LAN_CHAN;

        if (strcmp(str, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_DHCP_STR) == 0) {
            ipmi_params.data.lanp_set_ip_source.source = 
                PP_IPMI_LANP_IP_ADDR_SOURCE_DHCP;
        } else if (strcmp(str, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_BOOTP_STR) == 0) {
            ipmi_params.data.lanp_set_ip_source.source = 
                PP_IPMI_LANP_IP_ADDR_SOURCE_OTHER; // here: BOOTP
        } else if (strcmp(str, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_NONE_STR) == 0) {
            ipmi_params.data.lanp_set_ip_source.source = 
                PP_IPMI_LANP_IP_ADDR_SOURCE_STATIC;
        } else {
            ipmi_params.data.lanp_set_ip_source.source = 
                PP_IPMI_LANP_IP_ADDR_SOURCE_UNSPECIFIED;
        }

        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, 
                        PP_IPMI_LANP_SUBCMD_SET_IP_SOURCE, 
                        &ipmi_params, &ipmi_ret, &ipmi_err, NULL))) {
            pp_log("Error: failed to send IP source to external BMC: %s\n", 
                pp_ipmi_get_error_string(ipmi_err));
        }

        pp_ipmi_cleanup_ret(&ipmi_ret);
        free(str);
    }

    // IP
    if (PP_FAILED(pp_cfg_get(&str, "network.ipaddr"))) {
        pp_log("Error: failed to read IP address from cfg\n");
    } else {
        ipmi_params.data.lanp_set_mac_address.chan = ESB2_RMM2_LAN_CHAN;
        ipmi_params.data.lanp_set_mac_address.value = str;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, 
                        PP_IPMI_LANP_SUBCMD_SET_IP_ADDRESS, 
                        &ipmi_params, &ipmi_ret, &ipmi_err, NULL))) {
            pp_log("Error: failed to send IP address to external BMC: %s\n", 
                pp_ipmi_get_error_string(ipmi_err));
        }

        pp_ipmi_cleanup_ret(&ipmi_ret);
        free(str);
    }

    // Netmask
    if (PP_FAILED(pp_cfg_get(&str, "network.netmask"))) {
        pp_log("Error: failed to read netmask from cfg\n");
    } else {
        ipmi_params.data.lanp_set_mac_address.chan = ESB2_RMM2_LAN_CHAN;
        ipmi_params.data.lanp_set_mac_address.value = str;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, 
                        PP_IPMI_LANP_SUBCMD_SET_NETMASK, 
                        &ipmi_params, &ipmi_ret, &ipmi_err, NULL))) {
            pp_log("Error: failed to send netmask to external BMC: %s\n", 
                pp_ipmi_get_error_string(ipmi_err));
        }

        pp_ipmi_cleanup_ret(&ipmi_ret);
        free(str);
    }

    // Gateway
    if (PP_FAILED(pp_cfg_get(&str, "network.gateway"))) {
        pp_log("Error: failed to read gateway from cfg\n");
    } else {
        // no gw: set to 0 to clear it at BMC
        if (strlen(str) == 0) {
            free(str);
            str = strdup("0.0.0.0");
        }
        ipmi_params.data.lanp_set_mac_address.chan = ESB2_RMM2_LAN_CHAN;
        ipmi_params.data.lanp_set_mac_address.value = str;
        if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_LANP, 
                        PP_IPMI_LANP_SUBCMD_SET_DEF_GW_IP, 
                        &ipmi_params, &ipmi_ret, &ipmi_err, NULL))) {
            pp_log("Error: failed to send gateway to external BMC: %s\n", 
                pp_ipmi_get_error_string(ipmi_err));
        }

        pp_ipmi_cleanup_ret(&ipmi_ret);
        free(str);
    }

    pp_log("External BMC updated.\n");
}
#endif

void
net_reconf_init(void)
{
    u_int i;

    /* add change handlers for settings */
    for (i = 0; proto_options[i] != NULL; ++i) {
	pp_cfg_add_post_tx_change_listener(net_reconf_proto_ch, proto_options[i]);
    }
    for (i = 0; fw_options[i] != NULL; ++i) {
	pp_cfg_add_post_tx_change_listener(net_reconf_fw_ch, fw_options[i]);
    }
    for (i = 0; eth_options[i] != NULL; ++i) {
	pp_cfg_add_post_tx_change_listener(net_reconf_eth_ch, eth_options[i]);
    }
    for (i = 0; main_options[i] != NULL; ++i) {
	pp_cfg_add_post_tx_change_listener(net_reconf_main_ch, main_options[i]);
    }

#if defined(PRODUCT_ASMIDC) && defined(OEM_INTEL)
    report_net_cfg_to_bmc();
#endif
}

void
net_reconf_cleanup(void)
{
    u_int i;

    /* remove change handlers for settings */
    for (i = 0; proto_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(net_reconf_proto_ch, proto_options[i]);
    }
    for (i = 0; fw_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(net_reconf_fw_ch, fw_options[i]);
    }
    for (i = 0; eth_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(net_reconf_eth_ch, eth_options[i]);
    }
    for (i = 0; main_options[i] != NULL; ++i) {
	pp_cfg_rem_change_listener(net_reconf_main_ch, main_options[i]);
    }
}

static void
reconf_listener_core(void)
{
    MUTEX_LOCK(&reconf_listener_mtx);
    reconf_listener = 1;
    MUTEX_UNLOCK(&reconf_listener_mtx);
}   

void
eric_net_reconf_listener(void)
{
    reconf_listener_core();
    while (reconf_listener) { usleep(200000); }
}

int
eric_net_listener_reconfiguring(void)
{
    return reconf_listener;
}

static int
net_reconf_proto_ch(pp_cfg_chg_ctx_t * ctx UNUSED)
{
    pp_log("Reconfigure NET subsystem (proto)\n");
    reconf_listener_core();
    /*
     * Do not wait for reconf_listener to get ready (causes deadlock with
     * cfg_get), query eric_net_listener_reconfiguring() instead
     */
    pp_log("NET subsystem (proto) reconfigured\n");
    return PP_SUC;
}

static int
net_reconf_fw_ch(pp_cfg_chg_ctx_t * ctx UNUSED)
{
    pp_log("Reconfigure NET subsystem (firewall)\n");
    pp_system("/etc/rc.firewall > /dev/null 2>&1");
    pp_log("NET subsystem (firewall) reconfigured\n");
    return PP_SUC;
}

static int
net_reconf_eth_ch(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    char *speed_str, *duplex_str, *if_str, *state_str;
    pp_eth_speed_t eth_speed;
    pp_eth_duplex_t eth_duplex;

    pp_log("Reconfigure NET subsystem (speed/duplex)\n");
    
    pp_cfg_get(&if_str, "network.eth_interface");
    pp_cfg_get(&speed_str, "network.eth_speed");
    pp_cfg_get(&duplex_str, "network.eth_duplex");

    if (!pp_strcmp_safe(speed_str, "10")) {
	eth_speed = PP_ETH_SPEED_10;
    } else if (!pp_strcmp_safe(speed_str, "100")) {
	eth_speed = PP_ETH_SPEED_100;
    } else {
	eth_speed = PP_ETH_SPEED_AUTO;
    }

    if (!pp_strcmp_safe(duplex_str, "half")) {
	eth_duplex = PP_ETH_DUPLEX_HALF;
    } else if (!pp_strcmp_safe(duplex_str, "full")) {
	eth_duplex = PP_ETH_DUPLEX_FULL;
    } else {
	eth_duplex = PP_ETH_DUPLEX_AUTO;
    }

    if (if_str && !strncmp(if_str, "eth", strlen("eth"))) {
	pp_eth_set_parameters(if_str, eth_speed, eth_duplex);
	// HACK: autonegotiation takes a bit; currently we do not know when it's
	// done so we sleep some time
	sleep(3);
    }
    if ((state_str = pp_eth_get_link_state(if_str)) != NULL) {
	printf("LAN interface: %s\n", state_str);
    }
    free(if_str);
    free(speed_str);
    free(duplex_str);
    free(state_str);

    pp_log("NET subsystem (speed/duplex) reconfigured\n");

    return PP_SUC;
}

static int
net_reconf_main_ch(pp_cfg_chg_ctx_t * ctx UNUSED)
{
    pp_log("Reconfigure NET subsystem (main)\n");

    eric_notify_nfs_unmount();

    pp_system("/etc/rc.net > /dev/null 2>&1");

    eric_notify_nfs_mount();

    reconf_listener_core();
    /*
     * Do not wait for reconf_listener to get ready (causes deadlock with
     * cfg_get), query eric_net_listener_reconfiguring() instead
     */

#if defined(PRODUCT_ASMIDC) && defined(OEM_INTEL)
    report_net_cfg_to_bmc();
#endif

    pp_log("NET subsystem (main) reconfigured\n");
    
    return PP_SUC;
}
