#include <pthread.h>
#include <sys/stat.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rsa.h>
#include <openssl/x509v3.h>
#include <pp/base.h>
#include <pp/bio.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <liberic_pthread.h>
#include <liberic_cert.h>
#include <liberic_net.h>
#include "net.h"
#include "reconf.h"

#ifdef PP_FEAT_WLAN
# include "wlan.h"
#endif

static pthread_mutex_t * crypto_locks;
SSL_CTX * ssl_ctx = NULL;

static void ssl_auth_init(void);
static int ssl_auth_verify(int preverify_ok, X509_STORE_CTX *ctx);
static int ssl_auth_check_for_client_certs(void);
static char * ssl_auth_find_user_by_hash(char * serial);
static u_long crypto_id_callback(void);
static void crypto_locking_callback(int mode, int type, const char * file, int line);
static RSA * ssl_tmp_rsa_callback(SSL * ssl, int is_export, int key_length);

int
init_ssl(void)
{
    static int init_once = 1;
    int use_default;
    struct stat st;
    int i;

    /* this gets executed only once */
    if (init_once) {
	crypto_locks = OPENSSL_malloc(CRYPTO_num_locks() *
				      sizeof(pthread_mutex_t));
	for (i = 0; i < CRYPTO_num_locks(); i++) {
	    pthread_mutex_init(&crypto_locks[i], NULL);
	}

	CRYPTO_set_id_callback(crypto_id_callback);
	CRYPTO_set_locking_callback(crypto_locking_callback);

	SSL_load_error_strings();
	SSL_library_init();
	X509V3_add_standard_extensions();
	init_once = 0;
    }

    /* Important!  Enable SSL versions 2 and 3 and TLSv1 */
    if ((ssl_ctx = SSL_CTX_new(SSLv23_server_method())) == NULL) {
	pp_log_err("%s(): Unable to create SSL context", ___F);
	return -1;
    }

    /* adjust some SSL Context variables */
    SSL_CTX_set_quiet_shutdown(ssl_ctx, 1);
    SSL_CTX_set_options(ssl_ctx, SSL_OP_ALL);
    SSL_CTX_set_options(ssl_ctx, SSL_OP_NO_SSLv2);
    SSL_CTX_set_options(ssl_ctx, SSL_OP_SINGLE_DH_USE);
    SSL_CTX_set_session_cache_mode(ssl_ctx, SSL_SESS_CACHE_SERVER);
    SSL_CTX_sess_set_cache_size(ssl_ctx, 128);
    SSL_CTX_set_session_id_context(ssl_ctx,"peppercon",9);
    SSL_CTX_set_tmp_rsa_callback(ssl_ctx, ssl_tmp_rsa_callback);

    /* make settings for SSL Authentication */
    ssl_auth_init();

    /* set ciphers */
    /*"ALL:!ADH:!EXPORT56:!IDEA:RC4+RSA:+HIGH:+MEDIUM:+LOW:-SSLv2:+EXP"*/
    if (!SSL_CTX_set_cipher_list(ssl_ctx, "ALL:!ADH:!EXPORT56:!IDEA:RC4+RSA:+HIGH:+MEDIUM:+LOW:-SSLv2:+EXP")) {
	pp_log("%s(): Unable to set cipher suite.\n", ___F);
	return -1;
    }

#undef SHOW_CIPHER_LIST
#ifdef SHOW_CIPHER_LIST
    {
        SSL * ssl = SSL_new(ssl_ctx);
	if (ssl) {
	    STACK_OF(SSL_CIPHER) * sk = SSL_get_ciphers(ssl);
	    char buf[512];
	    int i;
	    for (i = 0; i < sk_SSL_CIPHER_num(sk); ++i) {
		printf("%s", SSL_CIPHER_description(sk_SSL_CIPHER_value(sk,i), buf,sizeof(buf)));
	    }
	    SSL_free(ssl);
	}
    }
#endif /* SHOW_CIPHER_LIST */

    /****
     * set the certificate file for the SSL context
     */

    /* use default if ERIC_CERT_USRCERT_FILEPATH or ERIC_CERT_USRKEY_FILEPATH doesn't exist */
    use_default = stat(ERIC_CERT_USRCERT_FILEPATH, &st) != 0 || stat(ERIC_CERT_USRKEY_FILEPATH, &st) != 0;

 again:
    if (use_default
	&& SSL_CTX_use_certificate_file(ssl_ctx, ERIC_CERT_DEFCERT_FILEPATH, SSL_FILETYPE_PEM) <= 0) {
	pp_log_err("%s(): Unable to set SSL cert file '%s'", ___F, ERIC_CERT_DEFCERT_FILEPATH);
	return -1;
    } else if (!use_default
	       && SSL_CTX_use_certificate_chain_file(ssl_ctx, ERIC_CERT_USRCERT_FILEPATH) <= 0) {
	pp_log("%s(): Unable to set SSL cert file '%s'. Trying default...\n",
	       ___F, ERIC_CERT_USRCERT_FILEPATH);
	use_default = 1;
	goto again;
    }

    /* set the private key file for the SSL context */
    if (use_default &&
	SSL_CTX_use_PrivateKey_file(ssl_ctx, ERIC_CERT_DEFKEY_FILEPATH,
				    SSL_FILETYPE_PEM) <= 0) {
	pp_log_err("%s(): Unable to set SSL key file '%s'",
		   ___F, ERIC_CERT_DEFKEY_FILEPATH);
	return -1;
    } else if (!use_default
	       && SSL_CTX_use_PrivateKey_file(ssl_ctx, ERIC_CERT_USRKEY_FILEPATH, SSL_FILETYPE_PEM) <= 0) {
	pp_log("%s(): Unable to set SSL key file '%s'. Trying default ...\n",
	       ___F, ERIC_CERT_USRKEY_FILEPATH);
	use_default = 1;
	goto again;
    }

    /*
     * This seems to be done in SSL_CTX_use_PrivateKey_file() already.
     * To be sure we call it directly again.
     */
    if (!SSL_CTX_check_private_key(ssl_ctx)) {
	pp_log("%s(): Checking private key file FAILED! Trying default...\n",
		 ___F);
	if (!use_default) {
	    use_default = 1;
	    goto again;
	}
	return -1;
    }

    return 0;
}

void
uninit_ssl(void)
{
    if (ssl_ctx) {
	SSL_CTX_free(ssl_ctx);
	ssl_ctx = NULL;
    }
}

BIO* 
eric_net_init_bio_https(u_short port)
{
    BIO* ssl_bio;
    BIO* rfb_check_bio;
    BIO* https_in_bio;
    char buf[8];

    if ((ssl_bio = BIO_new_ssl(ssl_ctx, 0)) == NULL) {
	pp_log_err("%s(): Cannot create SSL BIO", ___F);
	return NULL;
    }

    /* add our special rfb check BIO to the ssl_bio */
    rfb_check_bio = pp_bio_new_rfb_check(pp_rfb_check_cb);
    BIO_push(ssl_bio, rfb_check_bio);

    snprintf(buf, sizeof(buf), "*:%u", port);

    if ((https_in_bio = pp_bio_new_accept(buf, AF_INET)) == NULL) {
	pp_log_err("%s(): Cannot create accept BIO on port %u", ___F, port);
	return NULL;
    }

    /* set accept sockets to non blocking mode and allow socket rebinding */
    /* BIO_set_nbio_accept(https_in_bio, 1) creates warning; workaround: */
    char a[2] = "a";
    BIO_ctrl(https_in_bio, BIO_C_SET_ACCEPT, 1, a);
    BIO_set_bind_mode(https_in_bio, BIO_BIND_REUSEADDR);

    /*
     * This means that when a new connection is accepted on
     * 'in', the sslBIO will be 'duplicated' and have the
     * new socket BIO push into it.  Basically it means the
     * SSL BIO will be automatically setup
     */
    BIO_set_accept_bios(https_in_bio, ssl_bio);

    /*
     * The first call will setup the accept socket, and the second
     * will get a socket.  In this loop, the first actual accept
     * will occur in the BIO_read() function.
     */
    if (BIO_do_accept(https_in_bio) <= 0) {
	pp_log_err("%s(): Cannot create accept socket on port %u", ___F, port);
	BIO_free_all(https_in_bio);
	return NULL;
    }

    return https_in_bio;
}

eric_session_int_id_t 
eric_net_ssl_auth_check(BIO * bio, const char ** error)
{
    SSL *ssl;
    char ip_str[INET6_ADDRSTRLEN];
    int err;
    eric_session_int_id_t s;
    X509 *cert;
    char id[ERIC_ID_LENGTH+1];
    unsigned long issuer_serial;
    char hash[9];
    char *user;
    
    pp_bio_get_peer_name(bio, ip_str, sizeof(ip_str));
    
    if (!BIO_get_ssl(bio, &ssl)) return 0;
    
    if ((cert = SSL_get_peer_certificate(ssl)) != NULL) {
	issuer_serial = X509_issuer_and_serial_hash(cert);
	snprintf(id, sizeof(id), "%lX@%s", issuer_serial, ip_str);
	snprintf(hash, sizeof(hash), "%lX", issuer_serial);
	X509_free(cert);
	if (SSL_get_verify_result(ssl) != X509_V_OK) {
	    switch (SSL_get_verify_result(ssl)) {
	      case X509_V_ERR_CERT_HAS_EXPIRED:
		  *error = "The certificate has expired.";
		  break;
	      case X509_V_ERR_CERT_NOT_YET_VALID:
		  *error = "The certificate is not yet valid.";
		  break;
	      case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT:
		  *error = "The issuer of the certificate is unknown.";
		  break;
	      default:
		  *error = "The certificate is rejected.";
		  break;
	    }
	    return 0;
	}
	if ((s = eric_session_get_by_id(id)) != 0) return s;

	user = ssl_auth_find_user_by_hash(hash);
	if (user == NULL) {
	    *error = "No user to that certificate";
	    return 0;
        }
	s = eric_session_create_with_id(user, ip_str, "", id, 0, &err);
	eric_session_get_by_id(id);
	return s;
    }
    
    return 0;
}

void
eric_net_ssl_auth_logout(BIO *bio)
{
    SSL *ssl;
    SSL_SESSION *session;
    
    if (!BIO_get_ssl(bio, &ssl)) return;
    if ((session = SSL_get_session(ssl)) == NULL) return;
    SSL_SESSION_set_timeout(session, 0);
}

static void 
ssl_auth_init(void)
{
    if (ssl_auth_check_for_client_certs()) {
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_PEER, ssl_auth_verify);
	SSL_CTX_load_verify_locations(ssl_ctx, NULL, "/etc/e-RIC/ssl/ca");
	SSL_CTX_get_cert_store(ssl_ctx)->cache = 0;
    } else {
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_NONE, NULL);
    }
}

static int 
ssl_auth_verify(int preverify_ok UNUSED, X509_STORE_CTX *ctx UNUSED)
{
    return 1;
}

static int
ssl_auth_check_for_client_certs(void)
{
    vector_t* users;
    int ret = 0;
    
    if (NULL != (users = pp_um_get_all_users())) {
	u_int i;
	int auth_cert;
	for (i = 0; i < vector_size(users); ++i) {
	    const char * user = (char*)vector_get(users, i);
	    if (PP_SUCCED(pp_cfg_is_enabled_nodflt(&auth_cert, "user[%s].auth_cert", user)) && auth_cert) {
		ret = 1;
		break;
	    }
	}
	vector_delete(users);
    }
    return ret;
}

static char *
ssl_auth_find_user_by_hash(char * serial)
{
    int auth_cert, smatch;
    unsigned int i;
    vector_t* users;
    char *ret = NULL, *opt;

    if (NULL != (users = pp_um_get_all_users())) {
	for (i = 0; i < vector_size(users); ++i) {
	    const char * user = vector_get(users, i);
	    if (PP_SUCCED(pp_cfg_is_enabled_nodflt(&auth_cert, "user[%s].auth_cert", user))
		&& auth_cert
		&& PP_SUCCED(pp_cfg_get_nodflt(&opt, "user[%s].cert_hash", user))) {
		smatch = !pp_strcmp_safe(opt, serial);
		free(opt);
		if (smatch) {
		    ret = strdup(user);
		    break;
		}
	    }
	}
	vector_delete(users);
    }
    return ret;
}

static u_long
crypto_id_callback(void)
{
    return (u_long)pthread_self();
}

static void
crypto_locking_callback(int mode, int type, const char * file UNUSED,
			int line UNUSED)
{
    if (mode & CRYPTO_LOCK) {
	pthread_mutex_lock(&crypto_locks[type]);
    } else {
	pthread_mutex_unlock(&crypto_locks[type]);
    }
}

static RSA *
ssl_tmp_rsa_callback(SSL * ssl UNUSED, int is_export UNUSED, int key_length)
{
    static pthread_mutex_t rsa_tmp_mtx = PTHREAD_MUTEX_INITIALIZER;
    static RSA * rsa_tmp_512  = NULL;
    static RSA * rsa_tmp_1024 = NULL;
    RSA * rsa_tmp;
    BIGNUM *bn;

    MUTEX_LOCK(&rsa_tmp_mtx);
    if (key_length == 512) {
	if (rsa_tmp_512 == NULL) {
	    if (((bn = BN_new()) == NULL)
		|| !BN_set_word(bn, RSA_F4)
		|| ((rsa_tmp_512 = RSA_new()) == NULL)
		|| !RSA_generate_key_ex(rsa_tmp_512, 512, bn, NULL)) {
		if (rsa_tmp_512) RSA_free(rsa_tmp_512);
		rsa_tmp_512 = NULL;
	    }
	    if (bn) BN_free(bn);
	}
	rsa_tmp = rsa_tmp_512;
    } else {
	if (rsa_tmp_1024 == NULL) {
	    if (((bn = BN_new()) == NULL)
		|| !BN_set_word(bn, RSA_F4)
		|| ((rsa_tmp_1024 = RSA_new()) == NULL)
		|| !RSA_generate_key_ex(rsa_tmp_1024, 1024, bn, NULL)) {
		if (rsa_tmp_1024) RSA_free(rsa_tmp_1024);
		rsa_tmp_1024 = NULL;
	    }
	    if (bn) BN_free(bn);
	}
	rsa_tmp = rsa_tmp_1024;
    }
    MUTEX_UNLOCK(&rsa_tmp_mtx);

    return rsa_tmp;
}
