#include <fcntl.h>
#include <pthread.h>
#include <stdio.h>
#include <unistd.h>
#include <syslog.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_notify.h>

#include "rsyslog.h"

#define MAX_LENGTH_SYSLOG 1024
#define MAX_LENGTH_COMMAND 255

static int initialized = 0;
static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;

int
syslog_log_init(void)
{
    char command[MAX_LENGTH_COMMAND + 1];
    int rsyslog_enabled=0;
    char * rsyslog_server = NULL;
    int status;

    if (initialized) return PP_SUC;

    pp_cfg_is_enabled(&rsyslog_enabled, "log.syslog.enabled");
    if (!rsyslog_enabled) return PP_SUC;

    if (PP_FAILED(pp_cfg_get_nodflt(&rsyslog_server, "log.syslog.server"))){
        pp_log("%s(): Missing remote server for syslog.\n", ___F);
        goto bail;
    }

    status = pp_system("killall syslogd");
    if (WEXITSTATUS(status) != 0) {
	pp_log_err("%s(): pp_system(killall syslogd) failed (exit != 0).", ___F);
	goto bail;
    }

    snprintf(command, sizeof(command), "syslogd -R %s", rsyslog_server);
    status = pp_system(command);
    if (WEXITSTATUS(status) != 0) {
	pp_log_err("%s(): pp_system(%s) failed (exit != 0).", ___F, command);
        goto bail;
    }
    
    initialized = 1;
    return PP_SUC;

bail:
    return PP_ERR;
}

void
syslog_log_cleanup(void)
{
    MUTEX_LOCK(&lock);
    initialized = 0;
    pp_system("killall syslogd");
    MUTEX_UNLOCK(&lock);
}


int
syslog_log_event(time_t date, const char * event, const char * desc)
{
    char logbuf[MAX_LENGTH_SYSLOG];

    MUTEX_LOCK(&lock);

    if(initialized) {
	snprintf(logbuf, MAX_LENGTH_SYSLOG, "%lu\t%s\t%s\n", date, event, desc);
	syslog(LOG_USER | LOG_INFO, logbuf);
    }

    MUTEX_UNLOCK(&lock);

    return PP_SUC;
}

