#include <fcntl.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <sys/vfs.h>
#include <pp/base.h>
#include <pp/um.h>
#include <liberic_pthread.h>
#include <liberic_session.h>
#include <liberic_webs.h>
#include <openssl/md5.h>
#include <liberic_notify.h>
#ifdef PP_FEAT_RPCCFG
#include <pp/rpc_ripc.h>
#endif
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
#include <ctype.h> // isupper, tolower
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

#ifdef OEM_TANGTOP
#define MAX_SESSIONS	15
#else    
#define MAX_SESSIONS	25
#endif
#define SESSION_TIMEOUT_HEARTBEAT   300

typedef struct {
    struct list_head listnode;
    eric_session_close_cb_t cb;
} close_cb_list_entry_t;

// FIXME: review locking in this module, might need new concept once if we want to be able to initiate session closes from outside.

/* ------------------------------------------------------------------------- *
 * global definitions
 * ------------------------------------------------------------------------- */

struct _eric_session_t {
    unsigned int int_id;

    int ref_count;
    int radius_sid;
    char id[ERIC_ID_LENGTH + 1];
    char last_applet_id[ERIC_ID_LENGTH + 1];
    time_t creation_time;
    time_t last_act_time;
    char user[ERIC_MAX_USER_LENGTH + 1];
    char nickname[ERIC_MAX_USER_LENGTH + 1];
    int  gid;
    char ip_str[INET6_ADDRSTRLEN];

    int do_close;			/* if set, session should be closed */
    int closed;				/* session closed marker */
    int do_not_timeout;			/* disable session timeout - only used for firmware update */
    struct list_head close_cb_list;	/* list of callbacks called on close */
    pthread_mutex_t close_cb_list_mtx;	/* mutex for close callback list */
};

typedef struct _eric_session_t eric_session_t;

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */

static int initialized = 0;

static pthread_mutex_t sessions_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static pp_hash_t * sessions_by_id = NULL;
static pp_hash_i_t * sessions_by_session = NULL;
static int auto_logout_enabled = 1;
static u_int session_timeout;
static u_int session_count = 0;
static u_int max_sessions;
static pthread_t reaper_thread;
static pthread_mutex_t reaper_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t reaper_cond = PTHREAD_COND_INITIALIZER;
static int reaper_die = 0;
#if defined(PP_FEAT_AUTH_RADIUS)
static pthread_t radius_thread;
#endif

static eric_session_encoder_desc_t *encoder_hook = NULL;

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */
#if defined(PP_FEAT_AUTH_RADIUS)
static void * radius(void* arg);
#endif
static eric_session_t * session_create_with_id(const char * user, const char * ip_str, const char * nickname,
					       const char* id, int radius_sid, int* error);
static void session_release(eric_session_t * s);
static void session_close(eric_session_t * s);
static void * reaper(void * arg);
static void reaper_cb(eric_session_int_id_t, va_list args);
static int create_id(char * id);
static void session_free(void * _s);
static int    get_auto_logout_enabled(void);
static unsigned int get_session_timeout(void);
static int    idle_timeout_ch(pp_cfg_chg_ctx_t *ctx UNUSED);
static void   set_max_session_count(void);
static u_int session_get_idle_time(eric_session_t * s);

/* ------------------------------------------------------------------------- *
 * initialization / cleanup
 * ------------------------------------------------------------------------- */

int
eric_session_init()
{
    const char * fn = ___F;

    if (initialized) {
	return 0;
    }

    if ((sessions_by_id = pp_hash_create(100)) == NULL) {
	pp_log("%s(): pp_hash_create() failed\n", fn);
	return -1;
    }

    if ((sessions_by_session = pp_hash_create_i(100)) == NULL) {
	pp_log("%s(): pp_hash_create() failed\n", fn);
	pp_hash_delete(sessions_by_id);
	return -1;
    }

    auto_logout_enabled = get_auto_logout_enabled();
    session_timeout = auto_logout_enabled ? get_session_timeout() : SESSION_TIMEOUT_HEARTBEAT;

    if (eric_pthread_create(&reaper_thread, 0, 65536, reaper, NULL) != 0) {
	pp_log("%s(): Cannot create reaper-thread\n", fn);
	pp_hash_delete(sessions_by_id);
	pp_hash_delete_i(sessions_by_session);
	return -1;
    }

    pp_cfg_add_change_listener(idle_timeout_ch,"security.idle_timeout");
    set_max_session_count();

    initialized = 1;

    return 0;
}

void
eric_session_cleanup(void)
{
    MUTEX_LOCK(&reaper_mtx);
    reaper_die = 1;
    if (pthread_cond_signal(&reaper_cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
    }
    MUTEX_UNLOCK(&reaper_mtx);
    
    pthread_join(reaper_thread, NULL);

    if (sessions_by_id != NULL) {
	pp_hash_delete(sessions_by_id);
	sessions_by_id = NULL;
    }

    if (sessions_by_session != NULL) {
	pp_hash_delete(sessions_by_session);
	sessions_by_session = NULL;
    }

    initialized = 0;
}

void
set_max_session_count(void)
{
    max_sessions = MAX_SESSIONS;
}

u_int
eric_session_get_max_count(void)
{
    return max_sessions;
}

/* ------------------------------------------------------------------------- *
 * API routines
 * ------------------------------------------------------------------------- */


eric_session_int_id_t
eric_session_create(const char * user, const char * ip_str,
		    const char * nickname, int radius_sid, int* error)
{
    char id[ERIC_ID_LENGTH + 1];
    if (create_id(id) == -1) {
	pp_log("%s(): create_id() failed\n", ___F);
	return 0;
    }
    return eric_session_create_with_id(user, ip_str, nickname, id, radius_sid, error);
}

eric_session_int_id_t
eric_session_create_with_id(const char * user, const char * ip_str,
			    const char * nickname,
			    const char* id, int radius_sid, int* error)
{
    eric_session_t * s;

    assert(user);
    assert(ip_str);

    s = session_create_with_id(user, ip_str,nickname, id, radius_sid, error);
    return s ? s->int_id : 0;
}

eric_session_int_id_t
eric_session_get_by_id(const char * id)
{
    eric_session_t * s;

    assert(id);

    MUTEX_LOCK(&sessions_mtx);
    s = (eric_session_t *)pp_hash_get_entry(sessions_by_id, id);
    if (s != NULL) s->ref_count++;
    MUTEX_UNLOCK(&sessions_mtx);

    return s ? s->int_id : 0;
}

eric_session_int_id_t
eric_session_get_by_challenge_response(const char *challenge, const char * response)
{
    int ret = 0;
    const char *key;
    
    assert(challenge && response);
    
    MUTEX_LOCK(&sessions_mtx);
    
    for (key = pp_hash_get_first_key(sessions_by_id); key; key = pp_hash_get_next_key(sessions_by_id)) {
    	MD5_CTX md5_ctx;
    	eric_session_t * s = (eric_session_t *)pp_hash_get_entry(sessions_by_id, key);
    	char hash[MD5_DIGEST_LENGTH];
    	char hash_string[2 * MD5_DIGEST_LENGTH + 1];
    	
    	if (!s) {
    	    continue;
    	}
    	
    	MD5_Init(&md5_ctx);
    	MD5_Update(&md5_ctx, challenge, ERIC_ID_LENGTH);
    	MD5_Update(&md5_ctx, s->id, ERIC_ID_LENGTH);
    	MD5_Final(hash, &md5_ctx);
    	
    	pp_md5_hash_to_str(hash, hash_string);
    	
    	if (!strcasecmp(hash_string, response)) {
    	    ret = s->int_id;
    	    s->ref_count++;
    	    goto bail;
    	}
    }
    
 bail:
    MUTEX_UNLOCK(&sessions_mtx);
    
    return ret;
}

eric_session_int_id_t
eric_session_get_by_login(const char *login)
{
    int ret = 0;
    const char *key;
    
    assert(login);
    
    MUTEX_LOCK(&sessions_mtx);
    
    for (key = pp_hash_get_first_key(sessions_by_id); key; 
         key = pp_hash_get_next_key(sessions_by_id)) {
    	eric_session_t * s = (eric_session_t *)pp_hash_get_entry(sessions_by_id,
                                                                     key);
    	if (!s) {
    	    continue;
    	}
    	
    	if (!strcmp(login, s->user)) {
    	    ret = s->int_id;
    	    s->ref_count++;
    	    goto bail;
    	}
    }
    
 bail:
    MUTEX_UNLOCK(&sessions_mtx);
    
    return ret;
}

static void do_notify_logout(eric_session_t *s)
{
    char *user_name;
    char *ip_str;
    log_event_t event;
    time_t duration;

    user_name = strdupa(s->user);
    ip_str = strdupa(s->ip_str);
    duration = time(NULL) - s->creation_time;
	    
    event.eventdata.logintrapdata.loginname = user_name;
    event.eventdata.logintrapdata.remote_host = ip_str;
    event.eventdata.logintrapdata.session_duration = duration;
    event.trapoid = "Logout";

    eric_notify_post_event(&event, "auth", PP_NOTIFY_EVENT_LEGACY); 
}

#if defined(PP_FEAT_AUTH_RADIUS)
typedef struct _rad_req_data {
    char *name;
    int  sid;
} rad_req_data;
#endif

void
eric_session_release(eric_session_int_id_t id)
{
    eric_session_t * s;
    MUTEX_LOCK(&sessions_mtx);
    s = (eric_session_t *)pp_hash_get_entry_i(sessions_by_session, id);
    MUTEX_UNLOCK(&sessions_mtx);

    if (s) session_release(s);
}

void
eric_session_close(eric_session_int_id_t id)
{
    eric_session_t * s;
    MUTEX_LOCK(&sessions_mtx);
    s = (eric_session_t *)pp_hash_get_entry_i(sessions_by_session, id);
    MUTEX_UNLOCK(&sessions_mtx);

    if (s) session_close(s);
}

void
eric_session_for_all(void (*cb)(eric_session_int_id_t, va_list), ...)
{
    eric_session_t * s;
    va_list args;

    assert(cb);
  
    MUTEX_LOCK(&sessions_mtx);

    for (s = (eric_session_t *)pp_hash_get_first_entry_i(sessions_by_session);
	 s != NULL;
	 s = (eric_session_t *)pp_hash_get_next_entry_i(sessions_by_session)) {

	va_start(args, cb);
	cb(s->int_id, args);
	va_end(args);
    }

    MUTEX_UNLOCK(&sessions_mtx);
}

void
eric_session_touch(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    s = pp_hash_get_entry_i(sessions_by_session, id);
    if (s != NULL) {
    	s->last_act_time = time(NULL);
    }
}

const char *
eric_session_get_id(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    s = pp_hash_get_entry_i(sessions_by_session, id);
    return s ? s->id : NULL;
}

time_t
eric_session_get_creation_time(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    
    s = pp_hash_get_entry_i(sessions_by_session, id);
    
    if (s == NULL) return 0;
    
    return s->creation_time;
}

int
eric_session_get_radius_sid(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    
    s = pp_hash_get_entry_i(sessions_by_session, id);
    
    if (s == NULL) return 0;
    
    return s->radius_sid;
}

u_int
eric_session_get_idle_time(eric_session_int_id_t id)
{
    eric_session_t * s;
    u_int ret;

    assert(id != 0);
    MUTEX_LOCK(&sessions_mtx);
    s = pp_hash_get_entry_i(sessions_by_session, id);
    ret = session_get_idle_time(s);
    MUTEX_UNLOCK(&sessions_mtx);
    
    return ret;
}

const char *
eric_session_get_user(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    
    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return NULL;
    
    return s->user;
}

const char *
eric_session_get_nickname(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);

    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return NULL;

    return s->nickname;
}

int
eric_session_get_gid(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);

    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return -1;

    return s->gid;
}

int 
eric_session_has_permission(eric_session_int_id_t s,
                            const char* permission_name,
                            const char* permission_value)
{
    int gid;

    if ((gid = eric_session_get_gid(s)) == PP_ERR) {
	pp_log("eric_session_get_gid() failed\n");
	return PP_ERR;
    }

    return(pp_um_gid_has_permission(gid, permission_name, permission_value));
}

int
eric_session_get_permission(eric_session_int_id_t s,
                            const char* permission_name,
                            char** permission_value)
{
    int gid;

    if ((gid = eric_session_get_gid(s)) == PP_ERR) {
        return PP_ERR;
    }

    return(pp_um_gid_get_permission(gid, permission_name, permission_value));
}

void
eric_session_set_user(eric_session_int_id_t id, const char * user)
{
    eric_session_t * s;

    assert(id != 0);
    
    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return;
    
    snprintf(s->user, sizeof(s->user), "%s", user);
}

void
eric_session_set_gid(eric_session_int_id_t id, int gid)
{
    eric_session_t * s;

    assert(id != 0);

    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return;

    s->gid = gid;
}

const char *
eric_session_get_ip_str(eric_session_int_id_t id)
{
    eric_session_t * s;

    assert(id != 0);
    
    s = pp_hash_get_entry_i(sessions_by_session, id);

    if (s == NULL) return "0.0.0.0";
    
    return s->ip_str;
}

int eric_session_register_close_cb(eric_session_int_id_t sid,
				   eric_session_close_cb_t cb)
{
    eric_session_t* s;
    close_cb_list_entry_t * close_cb_list_entry;
    int ret = PP_ERR;

    assert(sid != 0);
    assert(cb != NULL);
    
    MUTEX_LOCK(&sessions_mtx);
    
    if (NULL == (s = pp_hash_get_entry_i(sessions_by_session, sid))) {
	goto bailout;
    }

    /* allocate & add new entry */
    MUTEX_LOCK(&s->close_cb_list_mtx);
    if (!s->closed) {
	/* check for duplicates - in this case do nothing */
	struct list_head *tmp;	    
	int duplicate = 0;
	list_for_each(tmp, &s->close_cb_list) {
	    close_cb_list_entry_t * e =
		list_entry(tmp, close_cb_list_entry_t, listnode);
	    if (e->cb == cb) {
		duplicate = 1;
		break;
	    }
	}
	if (!duplicate) {
	    close_cb_list_entry = malloc(sizeof(close_cb_list_entry_t));
	    close_cb_list_entry->cb = cb;
	    list_add(&close_cb_list_entry->listnode, &s->close_cb_list);
	}
	ret = PP_SUC;
    }
    MUTEX_UNLOCK(&s->close_cb_list_mtx);

 bailout:
    MUTEX_UNLOCK(&sessions_mtx);
    return ret;
}

unsigned int eric_session_get_count(void)
{
    unsigned int cnt;
    
    MUTEX_LOCK(&sessions_mtx);
    cnt = session_count;
    MUTEX_UNLOCK(&sessions_mtx);

    return cnt;
}

/* ------------------------------------------------------------------------- *
 * internal routines
 * ------------------------------------------------------------------------- */

static eric_session_t *
session_create_with_id(const char * user, const char * ip_str, const char * nickname,
		       const char* id, int radius_sid, int* error)
{
    eric_session_t * s = NULL;

    *error = ERIC_SESSION_ERR_INTERNAL_ERROR;

    assert(user && ip_str);

#if  0 /* permission has been checked in pp_um_user_authenticate_with_ip_str() */
#ifdef PP_FEAT_RPCCFG
    if (pp_rpc_am_i_master()) {
#endif
    if(PP_ERR == pp_um_user_is_allowed_from_ip_str(user, ip_str)) {
        /* TODO log something? */
        return 0;
    }
#ifdef PP_FEAT_RPCCFG
    }
#endif
#endif

    if (session_count >= max_sessions) {
	/* FIXME: no free slot */
	pp_log("%s(): max number of sessions reached\n", ___F);
	*error = ERIC_SESSION_ERR_MAXIMUM_REACHED;
	return 0;
    }

    if ((s = calloc(1, sizeof(eric_session_t))) == NULL) {
	pp_log_err("%s(): malloc()", ___F);
	return 0;
    }

    s->ref_count = 1;

    snprintf(s->id, sizeof(s->id), "%s", id);
    snprintf(s->user, sizeof(s->user), "%s", user);
    snprintf(s->nickname, sizeof(s->nickname), "%s", nickname);
    snprintf(s->ip_str, sizeof(s->ip_str), "%s", ip_str);

    s->creation_time = s->last_act_time = time(NULL);

    INIT_LIST_HEAD(&s->close_cb_list);
    MUTEX_CREATE_RECURSIVE(&s->close_cb_list_mtx);

    s->int_id = (unsigned int) s;
    s->radius_sid = radius_sid;
    s->gid = -1; // invalid group.
    
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
    /* convert user to lower case */
    {
        char *c;
        
        for(c = s->user; *c; ++c) {
            if(isupper(*c)) {
                *c = tolower(*c);
            }
        }
    }
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

    MUTEX_LOCK(&sessions_mtx);
    if (pp_hash_set_entry(sessions_by_id, s->id, s, NULL) == -1) {
	pp_log("%s(): pp_hash_set_entry() failed\n", ___F);
	session_free(s);
	s = NULL;
    } else {
	session_count++;
    }

    if (pp_hash_set_entry_i(sessions_by_session, s->int_id, s, NULL) == -1) {
	session_free(s);
	s = NULL;
    }

    *error = ERIC_SESSION_ERR_NO_ERROR;

    if (encoder_hook && encoder_hook->encoder_session_opened) {
    	encoder_hook->encoder_session_opened(s->int_id);
    }

    MUTEX_UNLOCK(&sessions_mtx);

    return s;
}

static void
session_release(eric_session_t * s)
{
    if (s == NULL) return;

    MUTEX_LOCK(&sessions_mtx);

    assert(s->ref_count > 0);

    if (--s->ref_count == 0) {
	do_notify_logout(s);

#if defined(PP_FEAT_AUTH_RADIUS)
	{
	    rad_req_data * rad_data = malloc(sizeof(rad_req_data));
	    rad_data->name = strdup(s->user);
	    rad_data->sid  = s->radius_sid;
	    if (eric_pthread_create(&radius_thread, 1, 65536, radius, (void*) rad_data) != 0) {
		pp_log("Cannot create radius-thread\n");		
	    }
	}
#endif
	
    	if (encoder_hook && encoder_hook->encoder_session_closed) {
    	    encoder_hook->encoder_session_closed(s->int_id);
    	}

	pp_hash_delete_entry_i(sessions_by_session, s->int_id);
	pp_hash_delete_entry(sessions_by_id, s->id);

	--session_count;
    } else {
	s = NULL;
    }

    if (s) {
	session_free(s);
    }

    MUTEX_UNLOCK(&sessions_mtx);
}

void
session_close(eric_session_t * s)
{
    if (s == NULL) return;

    MUTEX_LOCK(&s->close_cb_list_mtx);

    if (!s->closed) {
	/* set session to closed */
	s->closed = 1;

	/* remove from session id hash - so it cannot be requested anymore */
	pp_hash_delete_entry(sessions_by_id, s->id);

	/* call close callbacks */
	struct list_head *tmp;	    
	list_for_each(tmp, &s->close_cb_list) {
	    close_cb_list_entry_t * e =
		list_entry(tmp, close_cb_list_entry_t, listnode);
	    if (e->cb) e->cb(s->int_id);
	}
    }

    MUTEX_UNLOCK(&s->close_cb_list_mtx);

    session_release(s);
}

/* 
 * This funciton is called within a eric_session_for_all() context to determine
 * whether a session must be closed or not. If the session has timed out, it
 * will be added to the reap_sessions hash that is passed as argument.
 */
static void
reaper_cb(eric_session_int_id_t id, va_list args)
{
    pp_hash_i_t * reap_sessions = va_arg(args, pp_hash_i_t *);
    eric_session_t * s = (eric_session_t *)pp_hash_get_entry_i(sessions_by_session, id);
    int is_persistent_local_session = 0;

    if (s) {
	/* Choose sessions to close. */
	if (!s->closed
	    && (s->do_close
		|| (!is_persistent_local_session && !s->do_not_timeout
		    && (auto_logout_enabled && session_get_idle_time(s) >= session_timeout))))
	{
	    ++s->ref_count; /* hold a reference - released in session_close */
	    //pp_log("%s->ref_count inc, now %d\n", s->session_id, s->ref_count);
	    pp_hash_set_entry_i(reap_sessions, id, s, NULL);
	}
    }
}

/*
 * Reaper thread, periodically checks all sessions for timeouts.
 */
static void *
reaper(void * arg UNUSED)
{
    pp_hash_i_t * reap_sessions = pp_hash_create_i(20);
    eric_session_t * s;
    struct timeval now;
    struct timespec cond_timeout;
 
    MUTEX_LOCK(&reaper_mtx);

    while (!reaper_die) {
	gettimeofday(&now, NULL);
	cond_timeout.tv_sec = now.tv_sec + 20;
	cond_timeout.tv_nsec = now.tv_usec * 1000;
	pthread_cond_timedwait(&reaper_cond, &reaper_mtx, &cond_timeout);

	MUTEX_LOCK(&sessions_mtx);
	/* get a list of sessions to release */
	eric_session_for_all(reaper_cb, reap_sessions);
	MUTEX_UNLOCK(&sessions_mtx);

	/* close the sessions in the timeout list */
	for (s = pp_hash_get_first_entry_i(reap_sessions);
	     s != NULL;
	     s = pp_hash_get_next_entry_i(reap_sessions))
	{
	    //pp_log("reaping session %s\n", s->id);

	    /* close the session */
	    session_close(s);
	}

	pp_hash_clear_i(reap_sessions);

    }

    MUTEX_UNLOCK(&reaper_mtx);

    pp_hash_delete(reap_sessions);

    pthread_exit(NULL);
}

#if defined(PP_FEAT_AUTH_RADIUS)     
static void*
radius(void* arg)
{
    rad_req_data *rad_data;

    rad_data = (rad_req_data*) arg;
   
    pp_um_radius_acct_request(rad_data->name, rad_data->sid, RAD_STOP);
    
    free (rad_data->name);
    free (rad_data);
    
    pthread_exit(NULL);
}
#endif

static int
create_id(char * id)
{
    u_int32_t data[ERIC_ID_LENGTH / 8];
    FILE * fp;
    int i;

    assert(id);

    if ((fp = fopen("/dev/urandom", "r")) == NULL) {
	pp_log_err("%s(): opening /dev/urandom failed", ___F);
	return -1;
    }

    if (fread(data, sizeof(data), 1, fp) != 1) {
	pp_log_err("%s(): reading /dev/urandom failed", ___F);
	fclose(fp);
	return -1;
    }
	
    for (i = 0; i < (ERIC_ID_LENGTH / 8); i++) {
	snprintf(&id[i * 8], 9, "%08X", data[i]);
    }

    fclose(fp);

    return 0;
}

static void
session_free(void * _s)
{
    eric_session_t * s = (eric_session_t *)_s;

    assert(s);

    if (s != NULL) {
	/* free the close callback registry */
	struct list_head *tmp, *tmp2;
	list_for_each_safe(tmp, tmp2, &s->close_cb_list) {
	    close_cb_list_entry_t * e =
		list_entry(tmp, close_cb_list_entry_t, listnode);
	    list_del_init(&e->listnode);
	    free(e);
	}

	/* free the session structure */
	free(s);
    }
}

int eric_session_connect_encoder(eric_session_encoder_desc_t* encoder) {
    assert(encoder_hook == NULL);
    
    encoder_hook = encoder;
    return PP_SUC;
}

static int get_auto_logout_enabled(void) {
    int _auto_logout_enabled = 0;
    pp_cfg_is_enabled(&_auto_logout_enabled, "security.idle_timeout.enabled");
    return _auto_logout_enabled;
}

static unsigned int get_session_timeout(void) {
    unsigned int timeout_period = 0;
    pp_cfg_get_uint(&timeout_period, "security.idle_timeout.period");
    return timeout_period * 60;
}

static int idle_timeout_ch(pp_cfg_chg_ctx_t *ctx UNUSED)
{
    MUTEX_LOCK(&sessions_mtx);
    auto_logout_enabled = get_auto_logout_enabled();
    session_timeout = auto_logout_enabled ? get_session_timeout() : SESSION_TIMEOUT_HEARTBEAT;
    MUTEX_UNLOCK(&sessions_mtx);
    return PP_SUC;
}

static u_int
session_get_idle_time(eric_session_t * s)
{
    u_int last_act_time;
    if (s) {
	last_act_time = s->last_act_time;
	return (u_int)time(NULL) - last_act_time;
    } else {
	// recovery mechanism, should not happen
	return session_timeout + 1;
    }
}

