/*-----------------------------------------------------------------------------
  source file for MC68360 terminal emulation
  Thomas Hoffmann, TUChZ Sektion Rechnertechnik und Datenkommunikation 
  aug 14, 1996
  -----------------------------------------------------------------------------*/

#include <term_internal.h>

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "terminal.h"
#include "errors.h"

/*----------------------------------------------------------------------------
  kill spaces in beginning of a string by returning a pointer
  to the first non space character
  ----------------------------------------------------------------------------*/
static char* KillSpaces(char *buf)
{
    char* ptr = buf;
    while(isspace((int) *ptr)) ptr++;
    return(ptr);
}


/*----------------------------------------------------------------------------
  search for next space character and return a pointer to
  ----------------------------------------------------------------------------*/
static char* GetNextSpace(char *buf)
{
    char* ptr = buf;
    while((!isspace((int) *ptr)) && (*ptr != 0) ) ptr++;
    return(ptr);
}

/*----------------------------------------------------------------------------
  match a command (works expanding); 
  return an index to the matching	command or an error code
  -1 ... no match
  -2 ... ambiguous (more than one match)
 
  (command list ends with an empty command entry)
  ----------------------------------------------------------------------------*/
static int MatchCommand(struct command *List,char *cmd)
{
    int idx = -1,i = 0;
    int nummatches = 0;
    int kwlen;

    kwlen = strlen(cmd);
    while(List[i].cmd[0] != 0)
	{
	    if(kwlen <= (int)strlen(List[i].cmd))
		{
		    if(strncasecmp(List[i].cmd,cmd,kwlen) == 0)
			{
			    nummatches++;
			    idx = i;
			}
		}
	    i++;
	} 
 
    if(nummatches == 1) return(idx); 
    else if(nummatches > 1) return(-2);
    else return(-1);
}


/*----------------------------------------------------------------------------
  print an argument into a string
  ----------------------------------------------------------------------------*/
void term_print_argument(struct arg *arg,char *str)
{
    long number;

    *str = 0;
    if(arg != NULL)
	{
	    switch(arg->type)
		{
		  case t_string:
		      strcpy(str,arg->value.s);
		      break;

		  case t_number:
		      number = *arg->value.n;
		      sprintf(str,"%ld (0x%08lX)",number,number);
		      break;

		  default:
		      strcpy(str,"invalid argument type");
		      break;
		}
	}
}


/*----------------------------------------------------------------------------
  print subcommand list
  ----------------------------------------------------------------------------*/
static int term_print_commands(struct command *cmd, term_cl_t * clp)
{
    int idx = 0;
    while(cmd[idx].cmd[0] != 0)
	{
	    if((idx % 4) == 0) {
		eric_term_printf(clp, "\r\n  ");
	    }
	    eric_term_printf(clp, "%-16s ",cmd[idx].cmd);
	    idx++;
	}
    eric_term_printf(clp, "\r\n\n");
  
    return 0;
}


/*----------------------------------------------------------------------------
  match a command and its subcommands

  returns an offset to the first non-command argument 
  in the argument list and a pointer to the found command
  silent == 0 means to output error messages

  return values:
  =  0 ... command found
  = -1 ... command not found
  = -2 ... command ambiguous
  = -3 ... other errors
  ----------------------------------------------------------------------------*/
static int MatchCommandTree(struct command *CmdList,struct arg *arg,int *offset,struct command **cmd,int silent, term_cl_t * clp)
{
    int idx = 0;
    int i = 0;
    int error = 0;
    char str[64];
    (*offset) = 0;
    (*cmd) = NULL;
    clp->argv = clp->argptrfld;		       /* Re-Ini argv */
   
    while ((error == 0) && (CmdList != NULL) && (arg[i].type != t_endoflist)) {
	if (arg[i].type == t_string) {
	    if ((idx = MatchCommand(CmdList,arg[i].value.s)) >= 0) {
		(*offset)++;
		if ((*offset) > 1) { clp->argv++; clp->argc--; }  
		i++;
		(*cmd) = &CmdList[idx];
		CmdList = (*cmd)->subcmd;
	    } else {
		if (silent == 0) {
		    term_print_argument(&arg[i],str);
		    switch(idx) {
		      case -1:
			  eric_term_printf(clp,
				     "Command %s not found. "
				     "Available commands are:\r\n", str);
			  if (term_print_commands(CmdList, clp) < 0) {
			      return -1;
			  }
			  break;
		      case -2:
			  eric_term_printf(clp, "Command %s ambiguous. "
				     "Available commands are:\r\n", str);
			  if (term_print_commands(CmdList, clp) < 0) {
			      return -1;
			  }
			  break;
		    }
		}
		error = idx;
	    }
	} else {
	    if (silent == 0) {
		term_print_argument(&arg[i],str);
		eric_term_printf(clp, "ARG: %s : string argument expected.\r\n",str);
	    }
	    error = -3;
	}
    }
   
    return error;
}

/*----------------------------------------------------------------------------
  get argument from input line by filling a TArg structure
  return PTArg->type = t_endoflist if no further argument in string line
  and a pointer to the position after the read argument or NULL if
  an error occured
  ----------------------------------------------------------------------------*/
static char* GetArgument(struct arg *arg,char *line)
{
    char *ptr,*ptr1;
    int len;
    int error = 0;

    arg->type = t_endoflist;
    arg->value.s = NULL;

    ptr = KillSpaces(line);
    if(isdigit((int) *ptr))		         /* test if arg is a number */
	{
	    arg->type = t_number;
	    ptr1 = GetNextSpace(ptr);

	    if((arg->value.n = (long*)malloc(sizeof(long))) != NULL)
		{
		    if(ptr[1] == 'x')				/* test if number is hex number */
			{
			    if(sscanf(ptr + 2,"%lX",arg->value.n) == 0) error = -1;	/* read hex number */
			}
		    else
			{
			    if(sscanf(ptr,"%ld",arg->value.n) == 0) error = -1;		/* read dec number */
			}
		} else error = -1;
	}
    else if( *ptr == '"' )				/* test if arg is explizit a string */
	{
	    arg->type = t_string;
	    if((ptr1 = strchr(ptr + 1,'"')) != NULL)
		{
		    len = ptr1 - ptr - 1; 
		    if((arg->value.s = (char*)malloc(len + 1)) != NULL)
			{
			    strncpy(arg->value.s,ptr + 1,len);
			    arg->value.s[len] = 0;
			} else error = -1;
		    ptr1++;
		}
	}
    else if(*ptr != 0)				/* test if string end */
	{
	    arg->type = t_string;				/* take arg as string till next space */
	    ptr1=GetNextSpace(ptr);

	    len = ptr1 - ptr; 
	    if( (arg->value.s = (char*)malloc(len + 1)) != NULL)
		{
		    strncpy(arg->value.s,ptr,len);
		    arg->value.s[len] = 0;
		} else error = -1;
	}
    else
	{
	    ptr1 = ptr;
	}

    if(error == 0) return(ptr1);
    else
	{
	    arg->type = t_endoflist;
	    free(arg->value.s);
	    arg->value.s = NULL;  
	    return(NULL);
	}
}


/*----------------------------------------------------------------------------
  delete an argument list (free allocated memory)
  ----------------------------------------------------------------------------*/
void term_free_argumentlist(struct arg *arg)
{
    int i = 0;
    while(arg[i].type != t_endoflist)
	{
	    free(arg[i].value.s);
	    i++;
	}
    free(arg);
}

/*----------------------------------------------------------------------------
  build argv and argc from a command line copy
  ----------------------------------------------------------------------------*/
static void    Term_BuildArgs(char *ArgLine, char Sep2, term_cl_t * clp)
{
    char   *pStr;
    register char Ch, argfs=0;
    register char ix;
    
    pStr = ArgLine;
    clp->argv = clp->argptrfld;		       /* Re-Ini argv */
    clp->argc = 0;			       /* Re-Ini argc */
    clp->argv[0] = pStr;
    
    do {
	/*-- Find Beginning of Argument --*/
	while (*pStr == ' ')           /* drop preceding ' ' */
	    pStr++;
	clp->argv[(int)clp->argc] = pStr;
	if ((Ch = *pStr) == '\0')
	    return;                    /* no Parameters */

	/*-- Find End of Argument --*/
	while (((Ch = *pStr) != Sep2) && (Ch != ',') && (Ch != 0)) {
	    pStr++;
	}
	*pStr++ = '\0';
	if (!clp->argc++)
	    argfs = Ch;                /* store 1st Par. Separator */

	if (clp->argc == maxargs)
	    return;
    } while (Ch != 0);

    pStr--;                            /* Point to Last '0' */
    for (ix = clp->argc; ix < maxargs; ix++) { /* Init Unused Operands */
	clp->argv[(int)ix] = pStr;
    }
}

/*----------------------------------------------------------------------------
  UpCase a String
  ----------------------------------------------------------------------------*/
char *UpCaseStr(char *TarStr, const char *SrcStr)
{
    register char Ch,
	*targ = TarStr;
    while ((Ch = *SrcStr++) != '\0') {
	if ((Ch >= 'a') && (Ch <= 'z'))
	    Ch = Ch - ('a' - 'A');     /* UpCase */
	*targ++ = Ch;
    };
    return TarStr;
}

/*----------------------------------------------------------------------------
  transform complete argument line
  return a pointer to a TArg list and an error code 
  0 ... no error
  -1 ... allocation error (argument list will be deleted here)
  -2 ... argument format error
  ----------------------------------------------------------------------------*/
int term_transform_argumentline(struct arg **arg, char *line, term_cl_t * clp)
{
    int error = 0;
    int end = 0;
    int i = 0;
    char* ptr = line;

    if(((*arg) = malloc((maxargs + 1)*sizeof(struct arg))) != NULL)
	{
	    do
		{
		    if( (ptr = GetArgument(&(*arg)[i],ptr)) == NULL) error = -2;
		    if( (*arg)[i].type == t_endoflist) end = 1;			/* test if last argument */

		    i++;
		} while((error == 0) && (end == 0));
	} else error = -1;
 
    strcpy(clp->ArgStrLine, line);
    Term_BuildArgs(clp->ArgStrLine, ' ', clp);/* (Space also separates) */
    UpCaseStr(clp->argv[0], clp->argv[0]);
 
    return(error);
}

/*----------------------------------------------------------------------------
  insert a command into MainCmd list (and system call service routine)
  ----------------------------------------------------------------------------*/
long term_insert_command(struct command *Cmd, term_cl_t * clp)
{
    long error = NOERROR;
#define ERROR(x) EFILE_TERMINAL+EFUNC_TERMINSERTCOMMAND+(x);

    struct command *new;
    int i;

    if(MatchCommand(clp->MainCmd,Cmd->cmd) == -1)
	{
	    if((new = (struct command*)malloc((clp->NumCmds + 2)*sizeof(struct command))) != NULL)
		{
		    for(i = 0; i < clp->NumCmds; i++) new[i] = clp->MainCmd[i];
		    free(clp->MainCmd);

		    clp->MainCmd = new;
		    clp->MainCmd[clp->NumCmds] = *Cmd;
		    clp->MainCmd[clp->NumCmds+1] = CmdTerminate;
		    clp->NumCmds++;
		} else error=ERROR(ECODE_OUTOFMEMORY);
	} else error=ERROR(ECODE_ALREADYEXISTS);

    return(error);
#undef ERROR 
}

/*----------------------------------------------------------------------------
  remove a command from MainCmd list (and system call service routine)
  ----------------------------------------------------------------------------*/
long term_remove_command(struct command *Cmd, term_cl_t * clp)
{
    long error = NOERROR;
#define ERROR(x) EFILE_TERMINAL+EFUNC_TERMDELETECOMMAND+(x);

    struct command *new;
    int idx,ni,mi;

    if((idx = MatchCommand(clp->MainCmd,Cmd->cmd)) >= 0)
	{
	    if((new = (struct command*)malloc(clp->NumCmds * sizeof(struct command))) != NULL)
		{
		    for(mi = 0, ni = 0; mi < clp->NumCmds; mi++) if(mi != idx) new[ni++] = clp->MainCmd[mi];
		    free(clp->MainCmd);

		    clp->MainCmd = new;
		    clp->NumCmds--;
		    clp->MainCmd[clp->NumCmds] = CmdTerminate;
		} else error=ERROR(ECODE_OUTOFMEMORY);
	} else error=ERROR(ECODE_NOTEXISTS);

    return(error);
#undef ERROR 
}



/*----------------------------------------------------------------------------
  print help screen and help function
  ----------------------------------------------------------------------------*/
static int HelpFuncHelp(const char *cmd UNUSED, term_cl_t * clp)
{
    eric_term_printf(clp,
	       "\r\nUsage: %s [<cmd> [<subcmd> [<subcmd> ...]]]\r\n\n"
	       " A help screen for specified command is printed.\r\n"
	       " With no arguments given a table of all commands\r\n"
	       " is printed to the screen.\r\n\n", cmd);
  
    return 0;
}


static long HelpFunc(struct arg *arg, term_cl_t * clp)
{
    int error;
    int argoffset;
    struct command *command;
 
    if (arg->type == t_endoflist) {
	HelpFuncHelp("help", clp);
	eric_term_printf(clp, "The following commands are supported :\r\n");
	if (term_print_commands(clp->MainCmd, clp) < 0) {
	    return -1;
	}
    } else {
	if ((error = MatchCommandTree(clp->MainCmd, arg,
				      &argoffset, &command, 0, clp)) == 0) {
	    if (command->subcmd != NULL) {
		eric_term_printf(clp,
			   "A subcommand is required. "
			   "Available subcommands are:\r\n");
		if (term_print_commands(command->subcmd, clp) < 0) {
		    return -1;
		}
	    }
      
	    if (command->helpfunc != NULL) {
		command->helpfunc(command->cmd, clp); 
	    } else {
		eric_term_printf(clp, "Help function not available.\r\n");
	    }
	}
    }
    return 0;
}


/*----------------------------------------------------------------------------
  exit terminal function
  ----------------------------------------------------------------------------*/
static long ExitTerminal(struct arg *arg UNUSED, term_cl_t * clp UNUSED)
{
    pthread_exit(NULL);
}


static int ExitTerminalHelp(const char *cmd UNUSED, term_cl_t * clp)
{
    eric_term_printf(clp,
	       "\r\nUsage: %s\r\n\n  Exit the command terminal.\r\n\n", cmd);

    return 0;
}


/*----------------------------------------------------------------------------
  main terminal functions
  ----------------------------------------------------------------------------*/

struct command HelpCmd = {"help",NULL,HelpFunc,HelpFuncHelp};
struct command ExitCmd = {"quit",NULL,ExitTerminal,ExitTerminalHelp};

void term_init(term_cl_t * clp)
{
    /* ----- init command list ----- */
    clp->NumCmds = 0;
    clp->MainCmd = (struct command*)malloc(sizeof(struct command));
    *clp->MainCmd = CmdTerminate;
	
    /* make public terminal functions available by system calls - not needed*/
    /*   term_insert_scsr(); */
	
    term_insert_command(&HelpCmd, clp);
    term_insert_command(&ExitCmd, clp);

    eric_term_printf(clp, "\x1b[20l"); /* disable linefeed mode */
}

void term_deinit(term_cl_t * clp) {
    remove_eric_cmds(clp);

#if defined(PP_FEAT_DIAGNOS)
    remove_diag_cmds(clp);
#endif // #if defined(PP_FEAT_DIAGNOS)

    term_remove_command(&HelpCmd, clp);
    term_remove_command(&HelpCmd, clp);
    free(clp->MainCmd);
}

int esh_parse_and_execute_cmd(term_cl_t * clp)
{
    long error;
    struct command *command;
    int argoffset;
    struct arg *arg;
    int argerr;
    long cmderr = 0;

    if ((argerr = term_transform_argumentline(&arg, clp->i_buf, clp)) == 0) {
	if ((error = MatchCommandTree(clp->MainCmd, arg,
				      &argoffset, &command, 0, clp)) == 0) {
	    if (command != NULL) {
		if (command->cmdfunc != NULL) {
		    cmderr = command->cmdfunc(&arg[argoffset], clp); 
		} else {
		    if (command->subcmd != NULL) {
			eric_term_printf(clp, "Subcommand expected.\r\n");
		    } else {
			eric_term_printf(clp, "Command function not available.\r\n");
		    }
		}
	    } else {
		eric_term_printf(clp, "Command expected.\r\n");
	    }
	    if (cmderr != 0) {
		eric_term_printf(clp, "Command error code : 0x%08lX\r\n", cmderr);
		if (cmderr < 0) { return cmderr; }
	    }
	}
	term_free_argumentlist(arg);
    }
    /* FIXME */
    return 0;
}
