/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "telnetd.h"

#include <openssl/bio.h>
#include <pp/base.h>
#include <pp/um.h>
#include <liberic_net.h>
#include <liberic_notify.h>
#include <term_internal.h>
#include "../eshell/app/keycodes.h"

#if defined(PP_FEAT_CLP)
#include <pp/clp.h>
#endif

const char * login_prompt = "Login: ";
const char * passwd_prompt = "Password: ";
const char * shell_prompt = "eSH> ";

static unsigned char doopt[] = { IAC, DO, '%', 'c', 0 };
static unsigned char dont[] = { IAC, DONT, '%', 'c', 0 };
unsigned char	will[] = { IAC, WILL, '%', 'c', 0 };
unsigned char	wont[] = { IAC, WONT, '%', 'c', 0 };

/*
 * Buffer for sub-options, and macros
 * for suboptions buffer manipulations
 */

#define	SB_CLEAR()	clp->subpointer = clp->subbuffer;
#define	SB_TERM()	{ clp->subend = clp->subpointer; SB_CLEAR(); }
#define	SB_ACCUM(c)	if (clp->subpointer < (clp->subbuffer + sizeof(clp->subbuffer)-1)) { \
	*clp->subpointer++ = (c);					\
    }
#define	SB_GET()	((*clp->subpointer++)&0xff)
#define	SB_EOF()	(clp->subpointer >= clp->subend)
#define	SB_LEN()	(clp->subend - clp->subpointer)



/*
 * State for recv fsm
 */
#define	TS_DATA		0	/* base state */
#define	TS_IAC		1	/* look for double IAC's */
#define	TS_CR		2	/* CR-LF ->'s CR */
#define	TS_SB		3	/* throw away begin's... */
#define	TS_SE		4	/* ...end's (suboption negotiation) */
#define	TS_WILL		5	/* will option negotiation */
#define	TS_WONT		6	/* wont " */
#define	TS_DO		7	/* do " */
#define	TS_DONT		8	/* dont " */

void
switch_to_sub_state(term_cl_t * clp, sub_state_t sub_state, const char * prompt)
{
    clp->sub_state = sub_state;
    memset(clp->i_buf, 0, sizeof(clp->i_buf));
    clp->i_buf_end = clp->i_buf_cur = clp->i_buf;
    clp->prompt = prompt;
    if (*prompt != '\0') {
	eric_term_printf(clp, "\r%s", prompt);
    }
}

/*
 * insert_char() - inserts a character at the current position in
 *		   the input buffer
 *
 *   clp	- the context
 *   c		- the character to insert
 *
 * ret value:	0 - did nothing (buffer full)
 *		1 - character appended
 *		2 - character inserted
 */
static int
insert_char(term_cl_t * clp, char c)
{
    int ret = 0;

    if ((clp->i_buf + sizeof(clp->i_buf) - clp->i_buf_end) > 1) {
	ret = 1;
	if (clp->i_buf_end > clp->i_buf_cur) {
	    memmove(clp->i_buf_cur + 1, clp->i_buf_cur,
		    clp->i_buf_end - clp->i_buf_cur);
	    ret++;
	}
	*clp->i_buf_cur++ = c;
	clp->i_buf_end++;
    }

    return ret;
}

/*
 * delete_char() - deletes the character before the current position in
 *		   the input buffer
 *
 *   clp	- the context
 *
 * ret value:	0 - did nothing (at line start)
 *		1 - character deleted
 */
static int
delete_char(term_cl_t * clp)
{
    int ret = 0;

    if (clp->i_buf_cur > clp->i_buf) {
	memmove(clp->i_buf_cur - 1, clp->i_buf_cur,
		clp->i_buf_end - clp->i_buf_cur);
	clp->i_buf_cur--;
	*--clp->i_buf_end = '\0';
	ret = 1;
    }

    return ret;
}

static int
handle_input_char(term_cl_t * clp, char c)
{
    switch (clp->esc_state) {
      case ESC_STATE_ESC:
	  if (clp->sub_state == SUB_STATE_TERMINAL) {
	      switch (c) {
		case c_ESC:
		    clp->esc_state = ESC_STATE_NORMAL;
		    serial_write_byte(clp, c);
		    break;
		case 'e':
		    clp->esc_state = ESC_STATE_HOTSEQ;
		    clp->hstype = HOTKEY_SEQ_EXIT;
		    clp->hsidx = 1;
		    break;
		case 'b':
		    clp->esc_state = ESC_STATE_HOTSEQ;
		    clp->hstype = HOTKEY_SEQ_BREAK;
		    clp->hsidx = 1;
		    break;
		default:
		    serial_write_byte(clp, c_ESC);
		    serial_write_byte(clp, c);
		    clp->esc_state = ESC_STATE_NORMAL;
	      }
	  } 
	  else {
	      switch (c) {
		case '[': clp->esc_state = ESC_STATE_SQUARE; break;
		case 'O': clp->esc_state = ESC_STATE_GET_PARAMS; break;
		default:  clp->esc_state = ESC_STATE_NORMAL; break;
	      }
	  }
	  break;
      case ESC_STATE_HOTSEQ:
	  if (clp->hsidx == 1 && clp->hstype == HOTKEY_SEQ_EXIT && c == 'x') {
	      clp->hsidx++;
	  } else if (clp->hsidx == 2 && clp->hstype == HOTKEY_SEQ_EXIT && c == 'i') {
	      clp->hsidx++;
	  } else if (clp->hsidx == 3 && clp->hstype == HOTKEY_SEQ_EXIT && c == 't') {
	      if (clp->sub_state == SUB_STATE_TERMINAL) {
		  eric_term_printf(clp, "\r\nTerminal mode left.\r\n");
		  serial_detach_client(clp);
		  switch_to_sub_state(clp, SUB_STATE_SHELL, shell_prompt);
	      }
	      clp->esc_state = ESC_STATE_NORMAL;
	  } else if (clp->hsidx == 1 && clp->hstype == HOTKEY_SEQ_BREAK && c == 'r') {
	      clp->hsidx++;
	  } else if (clp->hsidx == 2 && clp->hstype == HOTKEY_SEQ_BREAK && c == 'k') {
	      serial_write_break(clp, 0);
	      clp->esc_state = ESC_STATE_NORMAL;
	  } else {
	      clp->esc_state = ESC_STATE_NORMAL;
	  }
	  break;
      case ESC_STATE_SQUARE:
	  for (clp->npar = 0 ; clp->npar < MAX_ESC_PARAMS ; clp->npar++) {
	      clp->esc_params[clp->npar] = 0;
	  }
	  clp->npar = 0;
	  clp->esc_state = ESC_STATE_GET_PARAMS;
	  if (c == '[') { /* Function key */
	      clp->esc_state = ESC_STATE_FUNC_KEY;
	      break;
	  }
	  if (c == '?') { break; }
	  /* no break here! */
      case ESC_STATE_GET_PARAMS:
	  if (c == ';' && clp->npar < MAX_ESC_PARAMS-1) {
	      clp->npar++;
	      break;
	  } else if (c >= '0' && c <= '9') {
	      clp->esc_params[clp->npar] *= 10;
	      clp->esc_params[clp->npar] += c - '0';
	      break;
	  } else {
	      clp->esc_state = ESC_STATE_GOT_PARAMS;
	  }
	  /* no break here! */
      case ESC_STATE_GOT_PARAMS:
	  switch (c) {
	    case 'D':		/* handle cursor left */
		if (clp->i_buf_cur > clp->i_buf) {
		    clp->i_buf_cur--;
		    if (clp->sub_state != SUB_STATE_PASSWD) {
			eric_term_printf(clp, "\x1b[D");
		    }
		}
		break;
	    case 'C':		/* handle cursor right */
		if (clp->i_buf_cur < clp->i_buf_end) {
		    clp->i_buf_cur++;
		    if (clp->sub_state != SUB_STATE_PASSWD) {
			eric_term_printf(clp, "\x1b[C");
		    }
		}
		break;
	  }
	  clp->esc_state = ESC_STATE_NORMAL;
	  break;
      case ESC_STATE_FUNC_KEY:
	  clp->esc_state = ESC_STATE_NORMAL;
	  break;
      case ESC_STATE_NORMAL:
	  switch (c) {
	    case '\0':
	    case c_LF:
		break; /* ignore NUL and LF */
	    case c_RETURN:
		switch (clp->sub_state) {
		  case SUB_STATE_TERMINAL:
		  case SUB_STATE_TERMINAL_FIXED:
		      /*
		       *  We are in terminal mode -> send raw character
		       */
		      serial_write_string(clp, "\r");
		      break;
#ifdef PP_FEAT_CLP
		  case SUB_STATE_CLP:
		      *clp->i_buf_end++ = '\0';
		      eric_term_printf(clp, "\r\n");
		      /* invoke the clp command */
		      if(PP_CLP_QUIT == pp_clp_invoke(clp->clp_session, clp->i_buf)){
			  /* switch back to the esh mode */
			  pp_clp_close(clp->clp_session);
			  clp->clp_session = NULL;
			  switch_to_sub_state(clp, SUB_STATE_SHELL, shell_prompt);
			  break;
		      }
		      else return 1;
#endif /* PP_FEAT_CLP */

		  default:
		      *clp->i_buf_end++ = '\0';
		      eric_term_printf(clp, "\r\n");
		      return 1; /* this means we have seen the RETURN key */
		}
		break;
	    case c_ESC:	/* handle escape sequences */
		if (clp->sub_state == SUB_STATE_TERMINAL_FIXED) {
		    serial_write_byte(clp, c);
		} else if (clp->esc_state == ESC_STATE_NORMAL) {
		    clp->esc_state = ESC_STATE_ESC;
		} else {
		    clp->esc_state = ESC_STATE_NORMAL;
		}
		break;
	    default:
		switch (clp->sub_state) {
		  case SUB_STATE_TERMINAL:
		  case SUB_STATE_TERMINAL_FIXED:
		      /*
		       *  We are in terminal mode -> send raw character
		       */
		      serial_write_byte(clp, c);
		      break;
		  default:
		      /*
		       *  We are NOT in terminal mode
		       */
		      if (c == c_BS || c == c_DEL) {
			  /* handle backspace and delete */
			  if (delete_char(clp) &&
			      clp->sub_state != SUB_STATE_PASSWD) {
			      
			      eric_term_printf(clp, "\r%s%s \x1b[%dD", clp->prompt,
					 clp->i_buf,
					 clp->i_buf_end - clp->i_buf_cur + 1);
			  }
		      } else if (isprint(c)) {
			  /* handle printable characters */
			  switch (insert_char(clp, c)) {
			    case 1:
				if (clp->sub_state != SUB_STATE_PASSWD) {
				    net_putchar(clp, c);
				}
				break;
			    case 2:
				if (clp->sub_state != SUB_STATE_PASSWD) {
				    eric_term_printf(clp, "%c%s\x1b[%dD", c,
					       clp->i_buf_cur,
					       clp->i_buf_end - clp->i_buf_cur);
				}
				break;
			  }
		      }
		}
	  }
	  break;
      default:
	  break;
    }
    
    return 0;
}

void telrcv(term_cl_t * clp) {
    log_event_t event;
    char c;
    int error;
    int server_sid = 0;
    int gid;

    while (clp->ncc > 0) {
	c = *clp->netip++ & 0377;
	clp->ncc--;

	switch (clp->state) {
	  case TS_DATA:
	      if (c == IAC) {
		  clp->state = TS_IAC;
		  break;
	      }
	      /*
	       * We now map \r\n ==> \r for pragmatic reasons.
	       * Many client implementations send \r\n when
	       * the user hits the CarriageReturn key.
	       *
	       * We USED to map \r\n ==> \n, since \r\n says
	       * that we want to be in column 1 of the next
	       * printable line, and \n is the standard
	       * unix way of saying that (\r is only good
	       * if CRMOD is set, which it normally is).
	       */
	      switch (clp->sub_state) {
		case SUB_STATE_LOGIN:
		    if (handle_input_char(clp, c)) {
			snprintf(clp->login, sizeof(clp->login), "%s", clp->i_buf);
			switch_to_sub_state(clp, SUB_STATE_PASSWD, passwd_prompt);
		    }
		    break;
		case SUB_STATE_PASSWD:
		    if (handle_input_char(clp, c)) {
			snprintf(clp->passwd, sizeof(clp->passwd), "%s", clp->i_buf);

			event.eventdata.logintrapdata.loginname = clp->login;
			event.eventdata.logintrapdata.remote_host = clp->client_ip;

			if (PP_ERR == pp_um_user_authenticate_with_ip_str(
                                                 clp->login, clp->passwd,
                                                 PP_UM_AUTH_NO_FLAGS,
                                                 clp->client_ip, &server_sid,
                                                 &gid)) {
			    memset(clp->passwd, 0, sizeof(clp->passwd));			    
			    eric_notify_post_event(&event, "Loginfailed", PP_NOTIFY_EVENT_LEGACY);
			    eric_term_printf(clp, "Login failed. ");
			    if (errno == PP_UM_ERR_USER_IS_BLOCKED) eric_term_printf(clp, "User is blocked.");
			    eric_term_printf(clp, "\r\n\r\n");		    
			    if (++clp->login_tries >= MAX_LOGIN_TRIES) {
				eric_term_printf(clp,
					   "Too many failed login trials. "
					   "Disconnecting...\r\n\r\n");
				pthread_exit(NULL);
			    }
			    sleep(1);
			    switch_to_sub_state(clp, SUB_STATE_LOGIN, login_prompt);
			} else if (PP_ERR == pp_um_user_has_permission(
                                             clp->login, "telnet",
                                             pp_acl_raasip_yes_str)) {
			    eric_term_printf(clp, "Permission denied.\r\n\r\n");
			    sleep(1);
			    switch_to_sub_state(clp, SUB_STATE_LOGIN, login_prompt);
			} else {
			    memset(clp->passwd, 0, sizeof(clp->passwd));
			    
			    eric_notify_post_event(&event, "Loginsuccess", PP_NOTIFY_EVENT_LEGACY);
			   
			    clp->session = eric_session_create(clp->login, clp->client_ip, "", server_sid, &error);
			    if (!clp->session) {
				if (error == ERIC_SESSION_ERR_MAXIMUM_REACHED) {
				    eric_term_printf(clp, "Maximum number of %d open session reached.\r\n\r\n",
					       eric_session_get_max_count());
				}
				/* FIXME: eric_session_create() failed */
				pthread_exit(NULL);
			    }

			    eric_session_set_gid(clp->session, gid);

			    if (clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TERMSRV) {
				if ((clp->serial_fd = serial_attach_client(clp)) >= 0) {
				    switch_to_sub_state(clp, SUB_STATE_TERMINAL_FIXED, "");
				} else {
				    eric_term_printf(clp, "\r\nThe selected serial port is not available.\r\n\r\n");
				    pthread_exit(NULL);
				}
			    } else {
				switch_to_sub_state(clp, SUB_STATE_SHELL, shell_prompt);
			    }
			}
		    }
		    break;
		case SUB_STATE_SHELL:
		    if (handle_input_char(clp, c)) {
			esh_parse_and_execute_cmd(clp);
			switch_to_sub_state(clp, clp->sub_state, clp->prompt);
		    }
		    break;
		case SUB_STATE_TERMINAL:
		case SUB_STATE_TERMINAL_FIXED:
		    handle_input_char(clp, c);
		    break;
#ifdef PP_FEAT_CLP
		case SUB_STATE_CLP:
			/* handle the input (ignore the ESC sequences)
				and call the sub functions 
			*/

			if (handle_input_char(clp, c)) {
				//esh_parse_and_execute_cmd(clp);
				/* the clp parser is called inner handle_input_char */
				switch_to_sub_state(clp, clp->sub_state, clp->clp_session->prompt);
			}
			break;
#endif /* PP_FEAT_CLP */

		default:
		    break;
	      }
	      break;

	  case TS_IAC:
	  gotiac:
	      switch (c) {
		 
		case BREAK:
		    serial_write_break(clp, 0);
		    break;
		case IP:
		case AYT:
		case AO:
		case EC:
		case EL:
		case DM:
		case EOR:
		case xEOF:
		case SUSP:
		case ABORT:
		case IAC:
		    break;
		  
		    /*
		     * Begin option subnegotiation...
		     */
		case SB:
		    clp->state = TS_SB;
		    SB_CLEAR();
		    continue;

		case WILL:
		    clp->state = TS_WILL;
		    continue;

		case WONT:
		    clp->state = TS_WONT;
		    continue;

		case DO:
		    clp->state = TS_DO;
		    continue;
		  
		case DONT:
		    clp->state = TS_DONT;
		    continue;

	      }
	      clp->state = TS_DATA;
	      break;

	  case TS_SB:
	      if (c == IAC) {
		  clp->state = TS_SE;
	      } 
	      else {
		  SB_ACCUM(c);
	      }
	      break;
	     
	  case TS_SE:
	      if (c != SE) {
		  if (c != IAC) {
		      /*
		       * bad form of suboption negotiation.
		       * handle it in such a way as to avoid
		       * damage to local state.  Parse
		       * suboption buffer found so far,
		       * then treat remaining stream as
		       * another command sequence.
		       */
		     
		      /* for DIAGNOSTICS */
		      SB_ACCUM(IAC);
		      SB_ACCUM(c);
		      clp->subpointer -= 2;
		     
		      SB_TERM();
		      suboption(clp);
		      clp->state = TS_IAC;
		      goto gotiac;
		  }
		  SB_ACCUM(c);
		  clp->state = TS_SB;
	      }
	      else {
		  /* for DIAGNOSTICS */
		  SB_ACCUM(IAC);
		  SB_ACCUM(SE);
		  clp->subpointer -= 2;
		 
		  SB_TERM();
		  suboption(clp);	/* handle sub-option */
		  clp->state = TS_DATA;
	      }
	      break;
	     
	  case TS_WILL:
	      willoption(c, clp);
	      clp->state = TS_DATA;
	      continue;

	  case TS_WONT:
	      wontoption(c, clp);
	      clp->state = TS_DATA;
	      continue;

	  case TS_DO:
	      dooption(c, clp);
	      clp->state = TS_DATA;
	      continue;
	     
	  case TS_DONT:
	      dontoption(c, clp);
	      clp->state = TS_DATA;
	      continue;
	     
	  default:
	      pp_log("telnet protocol panic in state %d\n", clp->state);
	      pthread_exit(NULL);
	}
    }
}

/*
 * The will/wont/do/dont state machines are based on Dave Borman's
 * Telnet option processing state machine.
 *
 * These correspond to the following states:
 *	my_state = the last negotiated state
 *	want_state = what I want the state to go to
 *	want_resp = how many requests I have sent
 * All state defaults are negative, and resp defaults to 0.
 *
 * When initiating a request to change state to new_state:
 * 
 * if ((want_resp == 0 && new_state == my_state) || want_state == new_state) {
 *	do nothing;
 * } else {
 *	want_state = new_state;
 *	send new_state;
 *	want_resp++;
 * }
 *
 * When receiving new_state:
 *
 * if (want_resp) {
 *	want_resp--;
 *	if (want_resp && (new_state == my_state))
 *		want_resp--;
 * }
 * if ((want_resp == 0) && (new_state != want_state)) {
 *	if (ok_to_switch_to new_state)
 *		want_state = new_state;
 *	else
 *		want_resp++;
 *	send want_state;
 * }
 * my_state = new_state;
 *
 * Note that new_state is implied in these functions by the function itself.
 * will and do imply positive new_state, wont and dont imply negative.
 *
 * Finally, there is one catch.  If we send a negative response to a
 * positive request, my_state will be the positive while want_state will
 * remain negative.  my_state will revert to negative when the negative
 * acknowlegment arrives from the peer.  Thus, my_state generally tells
 * us not only the last negotiated state, but also tells us what the peer
 * wants to be doing as well.  It is important to understand this difference
 * as we may wish to be processing data streams based on our desired state
 * (want_state) or based on what the peer thinks the state is (my_state).
 *
 * This all works fine because if the peer sends a positive request, the data
 * that we receive prior to negative acknowlegment will probably be affected
 * by the positive state, and we can process it as such (if we can; if we
 * can't then it really doesn't matter).  If it is that important, then the
 * peer probably should be buffering until this option state negotiation
 * is complete.
 *
 */


int send_do(int option, int init, term_cl_t * clp) {
    int retval = 0;
    
    if (init) {
	if ((clp->do_dont_resp[option] == 0 && his_state_is_will(option)) ||
	    his_want_state_is_will(option))
	    return 0;
	/*
	 * Special case for TELOPT_TM:  We send a DO, but pretend
	 * that we sent a DONT, so that we can send more DOs if
	 * we want to.
	 */
	if (option == TELOPT_TM)
	    set_his_want_state_wont(option);
	else
	    set_his_want_state_will(option);
	clp->do_dont_resp[option]++;
    }
    eric_term_printf(clp, (char *)doopt, option);
    
    DIAG(TD_OPTIONS, printoption("td: send do", option));
    return retval;
}



int send_dont(int option, int init, term_cl_t * clp) {
    int retval = 0;
    
    if (init) {
	if ((clp->do_dont_resp[option] == 0 && his_state_is_wont(option)) ||
	    his_want_state_is_wont(option))
	    return 0;
	set_his_want_state_wont(option);
	clp->do_dont_resp[option]++;
    }
    eric_term_printf(clp, (char *) dont, option);

    DIAG(TD_OPTIONS, printoption("td: send dont", option));
    return retval;
}



int send_wont(int option, int init, term_cl_t * clp) {
    int retval = 0;
    
    if (init) {
	if ((clp->will_wont_resp[option] == 0 && my_state_is_wont(option)) ||
	    my_want_state_is_wont(option))
	    return 0;
	set_my_want_state_wont(option);
	clp->will_wont_resp[option]++;
    }
    eric_term_printf(clp, (char *) wont, option);
    
    DIAG(TD_OPTIONS, printoption("td: send wont", option));
    return retval;
}



int send_will(int option, int init, term_cl_t * clp) {
    int retval = 0;
    
    if (init) {
	if ((clp->will_wont_resp[option] == 0 && my_state_is_will(option))||
	    my_want_state_is_will(option))
	    return 0;
	set_my_want_state_will(option);
	clp->will_wont_resp[option]++;
    }
    eric_term_printf(clp, (char *) will, option);

    DIAG(TD_OPTIONS, printoption("td: send will", option));
    return retval;
}


void willoption(int option, term_cl_t * clp) {
    int changeok = 0;
    void (*func)(void) = 0;
    
    /*
     * process input from peer.
     */
    
    DIAG(TD_OPTIONS, printoption("td: recv will", option));
    
    if (clp->do_dont_resp[option]) {
	clp->do_dont_resp[option]--;
	if (clp->do_dont_resp[option] && his_state_is_will(option))
	    clp->do_dont_resp[option]--;
    }
    if (clp->do_dont_resp[option] == 0) {
	if (his_want_state_is_wont(option)) {
	    switch (option) {
		
	      case TELOPT_BINARY:
		  changeok++;
		  break;
		
	      case TELOPT_ECHO:
		  /*
		   * See comments below for more info.
		   */
		  break;
		
	      case TELOPT_TM:
		  /*
		   * We never respond to a WILL TM, and
		   * we leave the state WONT.
		   */
		  return;

	      case TELOPT_LFLOW:
	      case TELOPT_TTYPE:
	      case TELOPT_SGA:
	      case TELOPT_NAWS:
	      case TELOPT_TSPEED:
	      case TELOPT_XDISPLOC:
	      case TELOPT_ENVIRON:
		  changeok++;
		  break;
		

	      default:
		  break;
	    }
	    if (changeok) {
		set_his_want_state_will(option);
		send_do(option, 0, clp);
	    } 
	    else {
		clp->do_dont_resp[option]++;
		send_dont(option, 0, clp);
	    }
	} 
	else {
	    /*
	     * Option processing that should happen when
	     * we receive conformation of a change in
	     * state that we had requested.
	     */
	    switch (option) {
	      case TELOPT_ECHO:
		  /*
		   * Egads, he responded "WILL ECHO".  Turn
		   * it off right now!
		   */
		  send_dont(option, 1, clp);
		  /*
		   * "WILL ECHO".  Kludge upon kludge!
		   * A 4.2 client is now echoing user input at
		   * the tty.  This is probably undesireable and
		   * it should be stopped.  The client will
		   * respond WONT TM to the DO TM that we send to
		   * check for kludge linemode.  When the WONT TM
		   * arrives, linemode will be turned off and a
		   * change propogated to the pty.  This change
		   * will cause us to process the new pty state
		   * in localstat(), which will notice that
		   * linemode is off and send a WILL ECHO
		   * so that we are properly in character mode and
		   * all is well.
		   */
		  break;

	    }
	}
    }
    set_his_state_will(option);
    if (func) (*func)();
}


void wontoption(int option, term_cl_t * clp) {
    /*
     * Process client input.
     */

    DIAG(TD_OPTIONS, printoption("td: recv wont", option));
    
    if (clp->do_dont_resp[option]) {
	clp->do_dont_resp[option]--;
	if (clp->do_dont_resp[option] && his_state_is_wont(option))
	    clp->do_dont_resp[option]--;
    }
    if (clp->do_dont_resp[option] == 0) {
	if (his_want_state_is_will(option)) {
	    /* it is always ok to change to negative state */
	    switch (option) {
	      case TELOPT_ECHO:
	      case TELOPT_BINARY:
		  break;
		
	      case TELOPT_TM:
		  /*
		   * If we get a WONT TM, and had sent a DO TM,
		   * don't respond with a DONT TM, just leave it
		   * as is.  Short circut the state machine to
		   * achive this.
		   */
		  set_his_want_state_wont(TELOPT_TM);
		  return;
		
	      case TELOPT_LFLOW:
		  break;
		
		  /*
		   * For options that we might spin waiting for
		   * sub-negotiation, if the client turns off the
		   * option rather than responding to the request,
		   * we have to treat it here as if we got a response
		   * to the sub-negotiation, (by updating the timers)
		   * so that we'll break out of the loop.
		   */
	      case TELOPT_TTYPE:
		  settimer(ttypesubopt);
		  break;

	      case TELOPT_TSPEED:
		  settimer(tspeedsubopt);
		  break;

	      case TELOPT_XDISPLOC:
		  settimer(xdisplocsubopt);
		  break;
		
	      case TELOPT_ENVIRON:
		  settimer(environsubopt);
		  break;

	      default:
		  break;
	    }
	    set_his_want_state_wont(option);
	    if (his_state_is_will(option)) send_dont(option, 0, clp);
	} 
	else {
	    switch (option) {
	      case TELOPT_TM:
		  break;
	      default:
		  break;
	    }
	}
    }
}  /* end of wontoption */


void dooption(int option, term_cl_t * clp) {
    int changeok = 0;

    /*
     * Process client input.
     */
    
    DIAG(TD_OPTIONS, printoption("td: recv do", option));
    
    if (clp->will_wont_resp[option]) {
	clp->will_wont_resp[option]--;
	if (clp->will_wont_resp[option] && my_state_is_will(option))
	    clp->will_wont_resp[option]--;
    }
    if ((clp->will_wont_resp[option] == 0) && (my_want_state_is_wont(option))) {
	switch (option) {
	  case TELOPT_ECHO:
	  case TELOPT_BINARY:
	  case TELOPT_SGA:
	  case TELOPT_STATUS:
	      changeok++;
	      break;
	    
	  case TELOPT_TM:
	      /*
	       * Special case for TM.  We send a WILL, but
	       * pretend we sent a WONT.
	       */
	      send_will(option, 0, clp);
	      set_my_want_state_wont(option);
	      set_my_state_wont(option);
	      return;
	    
	  case TELOPT_LOGOUT:
	      /*
	       * When we get a LOGOUT option, respond
	       * with a WILL LOGOUT, make sure that
	       * it gets written out to the network,
	       * and then just go away...
	       */
	      set_my_want_state_will(TELOPT_LOGOUT);
	      send_will(TELOPT_LOGOUT, 0, clp);
	      set_my_state_will(TELOPT_LOGOUT);
	      /* NOT REACHED */
	      break;

	  case TELOPT_LINEMODE:
	  case TELOPT_TTYPE:
	  case TELOPT_NAWS:
	  case TELOPT_TSPEED:
	  case TELOPT_LFLOW:
	  case TELOPT_XDISPLOC:
	  case TELOPT_ENVIRON:
	  default:
	      break;
	}
	if (changeok) {
	    set_my_want_state_will(option);
	    send_will(option, 0, clp);
	} 
	else {
	    clp->will_wont_resp[option]++;
	    send_wont(option, 0, clp);
	}
    }
    set_my_state_will(option);
}

void dontoption(int option, term_cl_t * clp) {
    /*
     * Process client input.
     */
    DIAG(TD_OPTIONS, printoption("td: recv dont", option));

    if (clp->will_wont_resp[option]) {
	clp->will_wont_resp[option]--;
	if (clp->will_wont_resp[option] && my_state_is_wont(option))
	    clp->will_wont_resp[option]--;
    }
    if ((clp->will_wont_resp[option] == 0) && (my_want_state_is_will(option))) {
	set_my_want_state_wont(option);
	if (my_state_is_will(option))
	    send_wont(option, 0, clp);
    }
    set_my_state_wont(option);
}

/*
 * suboption()
 *
 *	Look at the sub-option buffer, and try to be helpful to the other
 * side.
 *
 *	Currently we recognize:
 *
 *	Terminal type is
 *	Window size
 */
void suboption(term_cl_t * clp) {
    int subchar;

    DIAG(TD_OPTIONS, {printsub('<', clp->subpointer, SB_LEN()+2);});

    subchar = SB_GET();
    switch (subchar) {
      case TELOPT_TSPEED: {
	  int xspeed, rspeed;
	  if (his_state_is_wont(TELOPT_TSPEED))	/* Ignore if option disabled */
	      break;

	  settimer(tspeedsubopt);
	  if (SB_EOF() || SB_GET() != TELQUAL_IS) return;
	  xspeed = atoi((char *)clp->subpointer);

	  while (SB_GET() != ',' && !SB_EOF());
	  if (SB_EOF()) return;
	
	  rspeed = atoi((char *)clp->subpointer);
	  break;
      }

      case TELOPT_TTYPE: {		/* Yaaaay! */
	  if (his_state_is_wont(TELOPT_TTYPE))	/* Ignore if option disabled */
	      break;
	  settimer(ttypesubopt);
	
	  if (SB_EOF() || SB_GET() != TELQUAL_IS) {
	      return;		/* ??? XXX but, this is the most robust */
	  }

	  clp->terminaltype = clp->terminalname;

	  while ((clp->terminaltype < (clp->terminalname + sizeof (clp->terminalname) -1) ) &&
		 !SB_EOF()) 
	      {
		  int c;
		  c = SB_GET();
		  if (isupper(c)) {
		      c = tolower(c);
		  }
		  *clp->terminaltype++ = c;    /* accumulate name */
	      }
	  *clp->terminaltype = 0;
	  clp->terminaltype = clp->terminalname;
	  break;
      }

      case TELOPT_NAWS: {
	  int xwinsize, ywinsize;
	  if (his_state_is_wont(TELOPT_NAWS))	/* Ignore if option disabled */
	      break;

	  if (SB_EOF()) return;
	  xwinsize = SB_GET() << 8;
	  if (SB_EOF()) return;
	  xwinsize |= SB_GET();
	  if (SB_EOF()) return;
	  ywinsize = SB_GET() << 8;
	  if (SB_EOF()) return;
	  ywinsize |= SB_GET();
	  break;
      }

      case TELOPT_STATUS: {
	  int mode;

	  if (SB_EOF())
	      break;
	  mode = SB_GET();
	  switch (mode) {
	    case TELQUAL_SEND:
		if (my_state_is_will(TELOPT_STATUS))
		    send_status(clp);
		break;

	    case TELQUAL_IS:
		break;

	    default:
		break;
	  }
	  break;
      }  /* end of case TELOPT_STATUS */

      case TELOPT_XDISPLOC: {
	  if (SB_EOF() || SB_GET() != TELQUAL_IS)
	      return;
	  settimer(xdisplocsubopt);
	  clp->subpointer[SB_LEN()] = '\0';
	  (void)setenv("DISPLAY", (char *)clp->subpointer, 1);
	  break;
      }  /* end of case TELOPT_XDISPLOC */

      case TELOPT_ENVIRON: {
	  register int c;
	  register unsigned char *cp, *varp, *valp;

	  if (SB_EOF())
	      return;
	  c = SB_GET();
	  if (c == TELQUAL_IS)
	      settimer(environsubopt);
	  else if (c != TELQUAL_INFO)
	      return;

	  while (!SB_EOF() && SB_GET() != ENV_VAR)
	      ;

	  if (SB_EOF())
	      return;

	  cp = varp = (char *)clp->subpointer;
	  valp = 0;

	  while (!SB_EOF()) {
	      switch (c = SB_GET()) {
		case ENV_VALUE:
		    *cp = '\0';
		    cp = valp = (char *)clp->subpointer;
		    break;
		
		case ENV_VAR:
		    *cp = '\0';
		    cp = varp = (char *)clp->subpointer;
		    valp = 0;
		    break;
		
		case ENV_ESC:
		    if (SB_EOF())
			break;
		    c = SB_GET();
		    /* FALL THROUGH */
		default:
		    /* I think this test is correct... */
		    if (cp < clp->subbuffer+sizeof(clp->subbuffer)-1) *cp++ = c;
		    break;
	      }
	  }
	  *cp = '\0';
	  break;
      }  /* end of case TELOPT_ENVIRON */

      default:
	  break;
    }  /* end of switch */

}  /* end of suboption */

#define	ADD(c)	 *ncp++ = c;
#define	ADD_DATA(c) { *ncp++ = c; if (c == SE) *ncp++ = c; }

void send_status(term_cl_t * clp) {
    unsigned char statusbuf[256];
    register unsigned char *ncp;
    register unsigned char i;
    
    ncp = statusbuf;
    
    ADD(IAC);
    ADD(SB);
    ADD(TELOPT_STATUS);
    ADD(TELQUAL_IS);
    
    /*
     * We check the want_state rather than the current state,
     * because if we received a DO/WILL for an option that we
     * don't support, and the other side didn't send a DONT/WONT
     * in response to our WONT/DONT, then the "state" will be
     * WILL/DO, and the "want_state" will be WONT/DONT.  We
     * need to go by the latter.
     */
    for (i = 0; i < NTELOPTS; i++) {
	if (my_want_state_is_will(i)) {
	    ADD(WILL);
	    ADD_DATA(i);
	    if (i == IAC) ADD(IAC);
	}
	if (his_want_state_is_will(i)) {
	    ADD(DO);
	    ADD_DATA(i);
	    if (i == IAC) ADD(IAC);
	}
    }

    if (his_want_state_is_will(TELOPT_LFLOW)) {
	ADD(SB);
	ADD(TELOPT_LFLOW);
	ADD(clp->flowmode);
	ADD(SE);
    }

    ADD(IAC);
    ADD(SE);

    net_write(clp, statusbuf, ncp - statusbuf, 0);

    DIAG(TD_OPTIONS, {printsub('>', statusbuf, ncp - statusbuf);});
}
