/*
 * Copyright (c) 1989 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <netdb.h>
/*#include <termcap.h>*/
#include <netinet/in.h>
/* #include <netinet/ip.h> */ /* Don't think this is used at all here */
#include <arpa/inet.h>
#include <assert.h>
#include "telnetd.h"
#include <paths.h>
#include <sys/poll.h>

#include <openssl/bio.h>
#include <pp/base.h>
#include <liberic_misc.h>
#include <liberic_net.h>
#include <term_internal.h>

/* In Linux, this is an enum */
#if defined(__linux__) || defined(IPPROTO_IP)
#define HAS_IPPROTO_IP
#endif

static int getterminaltype(term_cl_t * clp);
static int terminaltypeok(const char *s, term_cl_t * clp);
static int _gettermname(term_cl_t * clp);
static int init_telnet(term_cl_t * clp);
static void telnet_mainloop(term_cl_t * clp);

/*
 * scan input lines.
 */
int
start_telnet(term_cl_t * clp)
{
    int ret = PP_ERR;

    if (clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TELNET) {
	/*
	 * get terminal type.
	 */
	if (getterminaltype(clp) < 0) {
	    pp_log("unable to get telnet client terminal type\n");
	    goto bail;
	}
	setenv("TERM", clp->terminaltype ? clp->terminaltype : "network", 1);
	pp_log("Telnet terminaltype: %s\n", clp->terminaltype);
    }

    /*
     * Start up the login process on the slave side of the terminal
     */

    if (esh_init(clp) < 0) {
	pp_log("failure in terminal server\n");
	goto bail;
    }
	
    if (init_telnet(clp) < 0) {
	pp_log("unable to init telnet server\n");
	goto bail;
    }

    if (esh_cleanup(clp) < 0) {
	pp_log("failure in terminal server\n");
	goto bail;
    }

    ret = PP_SUC;

 bail:
    return ret;
}

/*
 * getterminaltype
 *
 *	Ask the other end to send along its terminal type and speed.
 * Output is the variable terminaltype filled in.
 */
static int
getterminaltype(term_cl_t * clp)
{
    int retval = 0;

    settimer(baseline);

    if(send_do(TELOPT_TTYPE, 1, clp) < 0) {
    	return -1;
    }
    if(send_do(TELOPT_TSPEED, 1, clp) < 0) {
    	return -1;
    }
    if(send_do(TELOPT_XDISPLOC, 1, clp) < 0) {
    	return -1;
    }
    if(send_do(TELOPT_ENVIRON, 1, clp) < 0) {
    	return -1;
    }
    
    while (
	   his_will_wont_is_changing(TELOPT_TTYPE) ||
	   his_will_wont_is_changing(TELOPT_TSPEED) ||
	   his_will_wont_is_changing(TELOPT_XDISPLOC) ||
	   his_will_wont_is_changing(TELOPT_ENVIRON)) {
	if(ttloop(clp) < 0) {
            return -1;
        }
    }

    if (his_state_is_will(TELOPT_TSPEED)) {
	eric_term_printf(clp, "%c%c%c%c%c%c",
		   IAC, SB, TELOPT_TSPEED, TELQUAL_SEND, IAC, SE);
    }
    if (his_state_is_will(TELOPT_XDISPLOC)) {
	eric_term_printf(clp, "%c%c%c%c%c%c",
		   IAC, SB, TELOPT_XDISPLOC, TELQUAL_SEND, IAC, SE);
    }
    if (his_state_is_will(TELOPT_ENVIRON)) {
	eric_term_printf(clp, "%c%c%c%c%c%c",
		   IAC, SB, TELOPT_ENVIRON, TELQUAL_SEND, IAC, SE);
    }
    if (his_state_is_will(TELOPT_TTYPE)) {
        eric_term_printf(clp, "%c%c%c%c%c%c",
		   IAC, SB, TELOPT_TTYPE, TELQUAL_SEND, IAC, SE);
    }
    if (his_state_is_will(TELOPT_TSPEED)) {
	while (sequenceIs(tspeedsubopt, baseline))
            if(ttloop(clp) < 0) {
                return -1;
            }
    }
    if (his_state_is_will(TELOPT_XDISPLOC)) {
	while (sequenceIs(xdisplocsubopt, baseline))
            if(ttloop(clp) < 0) {
                return -1;
            }
    }
    if (his_state_is_will(TELOPT_ENVIRON)) {
	while (sequenceIs(environsubopt, baseline))
            if(ttloop(clp) < 0) {
                return -1;
            }
    }
    if (his_state_is_will(TELOPT_TTYPE)) {
	char first[256], last[256];

	while (sequenceIs(ttypesubopt, baseline))
            if(ttloop(clp) < 0) {
                return -1;
            }

	/*
	 * If the other side has already disabled the option, then
	 * we have to just go with what we (might) have already gotten.
	 */
	if (his_state_is_will(TELOPT_TTYPE) && !terminaltypeok(clp->terminaltype, clp)) {
	    /*
	     * Due to state.c, terminaltype points to a static char[41].
	     * Therefore, this assert cannot fail, and therefore, strings
	     * arising from "terminaltype" can be safely strcpy'd into
	     * first[] or last[].
	     */
	    assert(strlen(clp->terminaltype) < sizeof(first));

	    strcpy(first, clp->terminaltype);

	    for(;;) {
		/*
		 * Save the unknown name, and request the next name.
		 */
		strcpy(last, clp->terminaltype);

		if(_gettermname(clp) < 0) {
		    return -1;
		}
		assert(strlen(clp->terminaltype) < sizeof(first));

		if (terminaltypeok(clp->terminaltype, clp))
		    break;

		if (!strcmp(last, clp->terminaltype) ||
		    his_state_is_wont(TELOPT_TTYPE)) {
		    /*
		     * We've hit the end.  If this is the same as
		     * the first name, just go with it.
		     */
		    if (!strcmp(first, clp->terminaltype))
			break;
		    /*
		     * Get the terminal name one more time, so that
		     * RFC1091 compliant telnets will cycle back to
		     * the start of the list.
		     */
		    if(_gettermname(clp) < 0) {
		    	return -1;
		    }
		    assert(strlen(clp->terminaltype) < sizeof(first));

		    if (strcmp(first, clp->terminaltype)) {
			/*
			 * first[] came from terminaltype, so it must fit
			 * back in.
			 */
			strcpy(clp->terminaltype, first);
		    }
		    break;
		}
	    }
	}
    }
    return(retval);
}

static int
_gettermname(term_cl_t * clp)
{
    /*
     * If the client turned off the option,
     * we can't send another request, so we
     * just return.
     */
    if (his_state_is_wont(TELOPT_TTYPE))
	return 0;

    settimer(baseline);
    eric_term_printf(clp, "%c%c%c%c%c%c",
	       IAC, SB, TELOPT_TTYPE, TELQUAL_SEND, IAC, SE);
    while (sequenceIs(ttypesubopt, baseline))
	if(ttloop(clp) < 0) {
	    return -1;
	}
    return 0;
}

static int
terminaltypeok(const char *s, term_cl_t * clp)
{
    /* char buf[2048]; */

    if (clp->terminaltype == NULL)
	return(1);

    /*
     * Fix from Chris Evans: if it has a / in it, termcap will
     * treat it as a filename. Oops.
     */
    if (strchr(s, '/')) {
	return 0;
    }

    /*
     * If it's absurdly long, accept it without asking termcap.
     *
     * This means that it won't get seen again until after login,
     * at which point exploiting buffer problems in termcap doesn't
     * gain one anything.
     *
     * It's possible this limit ought to be raised to 128, but nothing
     * in my termcap is more than 64, 64 is _plenty_ for most, and while
     * buffers aren't likely to be smaller than 64, they might be 80 and
     * thus less than 128.
     */
    if (strlen(s) > 63) {
       return 0;
    }

    /*
     * tgetent() will return 1 if the type is known, and
     * 0 if it is not known.  If it returns -1, it couldn't
     * open the database.  But if we can't open the database,
     * it won't help to say we failed, because we won't be
     * able to verify anything else.  So, we treat -1 like 1.
     */

    /*
     * Don't do this - tgetent is not really trustworthy. Assume
     * the terminal type is one we know; terminal types are pretty
     * standard now. And if it isn't, it's unlikely we're going to
     * know anything else the remote telnet might send as an alias
     * for it.
     *
     * if (tgetent(buf, s) == 0)
     *    return(0);
     */
    return(1);
}

static int
init_telnet(term_cl_t * clp)
{
    int on = 1;
    /*
     * Initialize the slc mapping table.
     */
    clp->state = 0;

    if (clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TELNET
	|| clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TERMSRV) {
	/*
	 * Do some tests where it is desireable to wait for a response.
	 * Rather than doing them slowly, one at a time, do them all
	 * at once.
	 */
	if (my_state_is_wont(TELOPT_SGA)) send_will(TELOPT_SGA, 1, clp);

	/*
	 * Is the client side a 4.2 (NOT 4.3) system?  We need to know this
	 * because 4.2 clients are unable to deal with TCP urgent data.
	 *
	 * To find out, we send out a "DO ECHO".  If the remote system
	 * answers "WILL ECHO" it is probably a 4.2 client, and we note
	 * that fact ("WILL ECHO" ==> that the client will echo what
	 * WE, the server, sends it; it does NOT mean that the client will
	 * echo the terminal input).
	 */
	send_do(TELOPT_ECHO, 1, clp);
    
	/*
	 * Send along a couple of other options that we wish to negotiate.
	 */
	send_do(TELOPT_NAWS, 1, clp);
	send_will(TELOPT_STATUS, 1, clp);
	clp->flowmode = 1;  /* default flow control state */
	send_do(TELOPT_LFLOW, 1, clp);
    
	/*
	 * Spin, waiting for a response from the DO ECHO.  However,
	 * some REALLY DUMB telnets out there might not respond
	 * to the DO ECHO.  So, we spin looking for NAWS, (most dumb
	 * telnets so far seem to respond with WONT for a DO that
	 * they don't understand...) because by the time we get the
	 * response, it will already have processed the DO ECHO.
	 * Kludge upon kludge.
	 */
	while (his_will_wont_is_changing(TELOPT_NAWS)) ttloop(clp);
    
	/*
	 * But...
	 * The client might have sent a WILL NAWS as part of its
	 * startup code; if so, we'll be here before we get the
	 * response to the DO ECHO.  We'll make the assumption
	 * that any implementation that understands about NAWS
	 * is a modern enough implementation that it will respond
	 * to our DO ECHO request; hence we'll do another spin
	 * waiting for the ECHO option to settle down, which is
	 * what we wanted to do in the first place...
	 */
	if (his_want_state_is_will(TELOPT_ECHO) &&
	    his_state_is_will(TELOPT_NAWS)) {
	    while (his_will_wont_is_changing(TELOPT_ECHO)) ttloop(clp);
	}

	/*
	 * On the off chance that the telnet client is broken and does not
	 * respond to the DO ECHO we sent, (after all, we did send the
	 * DO NAWS negotiation after the DO ECHO, and we won't get here
	 * until a response to the DO NAWS comes back) simulate the
	 * receipt of a will echo.  This will also send a WONT ECHO
	 * to the client, since we assume that the client failed to
	 * respond because it believes that it is already in DO ECHO
	 * mode, which we do not want.
	 */
	if (his_want_state_is_will(TELOPT_ECHO)) willoption(TELOPT_ECHO, clp);
    
	/*
	 * Finally, to clean things up, we turn on our echo.  This
	 * will break stupid 4.2 telnets out of local terminal echo.
	 */
    
	if (my_state_is_wont(TELOPT_ECHO)) send_will(TELOPT_ECHO, 1, clp);
    }
    
    /*
     * Call telrcv() once to pick up anything received during
     * terminal type negotiation, 4.2/4.3 determination, and
     * linemode negotiation.
     */
    telrcv(clp);
    
    if (clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TELNET
	|| clp->conn_data->protocol_type == PP_NET_PROTOCOL_TYPE_TERMSRV) {

	char *boardname = NULL;
	
	if (setsockopt(clp->net_fd, SOL_SOCKET, SO_OOBINLINE, &on, sizeof(on)) < 0){
	    pp_log_err("%s(): setsockopt()", ___F);
	    return -1;
	}

	boardname = eric_misc_get_boardname();
	eric_term_printf(clp, "\r\n%s Terminal Server (c) 2000-2002\r\n\n", boardname);	
	switch_to_sub_state(clp, SUB_STATE_LOGIN, login_prompt);

	free(boardname);
    } else {
	switch_to_sub_state(clp, SUB_STATE_SHELL, shell_prompt);
    }

    telnet_mainloop(clp);

    return 0;
}

static void
telnet_mainloop(term_cl_t * clp)
{
    struct pollfd fds[2] = {
	{
	    .fd = clp->net_fd,
	    .events = POLLIN
	},
	{
	    .fd = clp->serial_fd,
	    .events = POLLIN
	}
    };
    u_int pollfd_cnt = sizeof(fds) / sizeof(struct pollfd);
    char wbuf[512];
    size_t wbuf_pos = 0;
    ssize_t n, wbuf_len = 0;    

    for (;;) {

	fds[1].fd = clp->serial_fd;

	if (poll(fds, pollfd_cnt, -1) < 0) {
	    if (errno == EINTR) continue;
	    pp_log_err("%s(): poll()", ___F);
	    break;
	}

	/* handle input data from network */
	if (fds[0].revents & POLLIN) {
	    n = read(fds[0].fd, clp->netibuf, sizeof(clp->netibuf));
	    if (n > 0) {
		clp->ncc = n;
		clp->netip = clp->netibuf;
		telrcv(clp);
	    } else if (n == 0) {
		break;
	    } else if (n < 0) {
		if (errno != EAGAIN && errno != EINTR) {
		    pp_log_err("%s(): read() from network", ___F);
		    break;
		}
		clp->ncc = 0;
	    }
	}

	/* handle input data from inter link socket */
	if (fds[1].revents & POLLIN) {
	    n = read(fds[1].fd, wbuf, sizeof(wbuf));
	    if (n > 0) {
		wbuf_len = n;
	    } else if (n == 0){
		break;
	    } else if (n < 0) {
		if (errno != EAGAIN && errno != EINTR) {
		    pp_log_err("%s(): read() from inter link socket", ___F);
		    break;
		}
		wbuf_len = 0;
	    }
	    wbuf_pos = 0;
	}

	/* handle output data to network port */
	if (fds[0].revents & POLLOUT || (wbuf_len > 0 && !(fds[0].events & POLLOUT))) {
	    n = write(fds[0].fd, &wbuf[wbuf_pos], wbuf_len);
	    if (n == 0) {
		break;
	    } else if (n != wbuf_len) {
		if (n > 0 || errno == EAGAIN || errno == EINTR) {
		    fds[0].events |= POLLOUT;
		    fds[1].events &= ~POLLIN;
		} else {
		    pp_log_err("%s(): write() to serial", ___F);
		    break;
		}
	    }
	    if (n > 0) wbuf_pos += n;
	    wbuf_len -= n;
	    if (wbuf_len == 0) {
		fds[0].events &= ~POLLOUT;
		fds[1].events |= POLLIN;
		wbuf_pos = 0;
	    }
	}
    }
}
