/*
 * asp.c -- Active Server Page Support
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 */

/******************************* Description *********************************/

/*
 * The ASP module processes ASP pages and executes embedded scripts. It 
 * support an open scripting architecture with in-built support for 
 * Ejscript(TM).
 */

/******************************** Includes ***********************************/

#include "wsIntrn.h"
#include "eric_util.h"

#include <pp/intl.h>

/********************************* Locals ************************************/

static sym_fd_t	websAspFunctions = -1;	/* Symbol table of functions */

/**************************** Forward Declarations ***************************/

static char *strtokcmp(char *s1, const char *s2);
static char *skipWhite(char *s);
static int websAspWrite(int ejid UNUSED, webs_t wp, int argc, char **argv);
static int websAspWritePure(int ejid UNUSED, webs_t wp, int argc, char **argv);
static int websAspWriteEsc(int ejid UNUSED, webs_t wp, int argc, char **argv);
static int websAspSkipStart(int ejid, webs_t wp, int argc, char **argv);
static int websAspSkipEnd(int ejid, webs_t wp, int argc, char **argv);

/************************************ Code ***********************************/
/*
 *	Create script spaces and commands
 */

int
websAspOpen()
{
    /*
     * Create the table for ASP functions
     */
    websAspFunctions = symOpen(WEBS_SYM_INIT * 2);

    /*
     * Create standard ASP commands
     */
    websAspDefine("write", websAspWrite);
    websAspDefine("writePure", websAspWritePure);
    websAspDefine("writeEsc", websAspWriteEsc);
    websAspDefine("skipStart", websAspSkipStart);
    websAspDefine("skipEnd", websAspSkipEnd);

    return 0;
}

/*
 * Close Asp symbol table.
 */

void
websAspClose()
{
    if (websAspFunctions != -1) {
	symClose(websAspFunctions);
	websAspFunctions = -1;
    }
}

/*
 * Process ASP requests and expand all scripting commands. We read the
 * entire ASP page into memory and then process. If you have really big 
 * documents, it is better to make them plain HTML files rather than ASPs.
 */

int
websAspRequest(webs_t wp, char *lpath)
{
    websStatType sbuf;
    char *buf, *lang, *result, *path, *ep, *cp, *nextp;
    char *last;
    int rc, engine, len, ejid;

    a_assert(wp);
    a_assert(lpath && *lpath);

    rc = -1;
    buf = NULL;
    engine = EMF_SCRIPT_EJSCRIPT;
    wp->flags |= WEBS_HEADER_DONE;
    path = wp->path;

    /*
     *	Create Ejscript instance in case it is needed
     */
    ejid = ejOpenEngine(wp->cgiVars, websAspFunctions, wp);
    if (ejid < 0) {
	websError(wp, 200, _("Can't create Ejscript engine"));
	goto done;
    }
    ejSetUserHandle(ejid, (int) wp);

    if (websPageStat(wp, lpath, path, &sbuf) < 0) {
	websError(wp, 200, _("Can't stat %s"), lpath);
	goto done;
    }

    /*
     *	Create a buffer to hold the ASP file in-memory
     */
    len = sbuf.size * sizeof(char);
    if ((buf = balloc(B_L, len + 1)) == NULL) {
	websError(wp, 200, _("Can't get memory"));
	goto done;
    }
    buf[len] = '\0';

    if (websPageReadData(wp, buf, len) != len) {
	websError(wp, 200, _("Cant read %s"), lpath);
	goto done;
    }
    websPageClose(wp);

    /*
     *	Scan for the next "<%"
     */
    last = buf;
    rc = 0;
    while (rc == 0 && *last && ((nextp = strstr(last, "<%")) != NULL)) {
	websWriteBlock(wp, last, (nextp - last));
	nextp = skipWhite(nextp + 2);

	/*
	 * Decode the language
	 */
	if ((lang = strtokcmp(nextp, "language")) != NULL) {
	    if ((cp = strtokcmp(lang, "=javascript")) != NULL) {
		engine = EMF_SCRIPT_EJSCRIPT;
	    } else {
		cp = nextp;
	    }
	    nextp = cp;
	}

	/*
	 * Find tailing bracket and then evaluate the script
	 */
	if ((ep = strstr(nextp, "%>")) != NULL) {

	    *ep = '\0';
	    last = ep + 2;
	    nextp = skipWhite(nextp);
	    /*
	     * Handle backquoted newlines
	     */
	    for (cp = nextp; *cp; ) {
		if (*cp == '\\' && (cp[1] == '\r' || cp[1] == '\n')) {
		    *cp++ = ' ';
		    while (*cp == '\r' || *cp == '\n') {
			*cp++ = ' ';
		    }
		} else {
		    cp++;
		}
	    }

	    /*
	     * Now call the relevant script engine. Output is done directly
	     * by the ASP script procedure by calling websWrite()
	     */
	    if (*nextp) {
		result = NULL;
		if (engine == EMF_SCRIPT_EJSCRIPT) {
		    rc = scriptEval(engine, nextp, &result, ejid);
		} else {
		    rc = scriptEval(engine, nextp, &result, (int) wp);
		}
		if (rc < 0) {
		    /*
		     * On an error, discard all output accumulated so far
		     * and store the error in the result buffer. Be careful
		     * if the user has called websError() already.
		     */
		    int skipOutput = wp->skipOutput;
		    wp->skipOutput = 0;
		    if (result) {
			websWrite(wp, "<h2><b>");
			websWrite(wp, _("ASP Error: %s"), result);
			websWrite(wp, "</b></h2>\n");
			websWrite(wp, "<pre>%s</pre>", nextp);
			bfree(B_L, result);
		    } else {
			websWrite(wp, "<h2><b>");
			websWrite(wp, _("ASP Error"));
			websWrite(wp, "</b></h2>\n%s\n", nextp);
		    }
		    websWrite(wp, "</body></html>\n");
		    wp->skipOutput = skipOutput;
		    rc = 0;
		    goto done;
		}
	    }

	} else {
	    websError(wp, 200, _("Unterminated script in %s: \n"), lpath);
	    rc = -1;
	    goto done;
	}
    }
    /*
     *	Output any trailing HTML page text
     */
    if (last && *last && rc == 0) {
	websWriteBlock(wp, last, strlen(last));
    }
    rc = 0;

    /*
     *	Common exit and cleanup
     */
 done:
    websPageClose(wp);
    if (ejid >= 0) {
	ejCloseEngine(ejid);
    }
    bfreeSafe(B_L, buf);
    return rc;
}

/*
 * Define an ASP Ejscript function. Bind an ASP name to a C procedure.
 */

int
websAspDefine(const char *name, int (*fn)(int ejid, webs_t wp, int argc, char **argv))
{
    return ejSetGlobalFunctionDirect(websAspFunctions, name, 
				     (int (*)(int, void*, int, char**)) fn);
}

/*
 * Asp write command. This implemements <% write("text"); %> command
 */

static int
websAspWrite(int ejid UNUSED, webs_t wp, int argc, char **argv)
{
    int i;

    a_assert(wp);
	
    for (i = 0; i < argc; ) {
	a_assert(argv);
	if (websWriteBlock(wp, argv[i], strlen(argv[i])) < 0) return -1;
	if (++i < argc) { if (websWriteBlock(wp, " ", 1) < 0) return -1; }
    }
    return 0;
}

/*
 * Asp writePure command. Same as write but without appending a whitespace.
 */

static int
websAspWritePure(int ejid UNUSED, webs_t wp, int argc, char **argv)
{
    int i;

    a_assert(wp);
	
    for (i = 0; i < argc; ) {
	a_assert(argv);
	if (websWriteBlock(wp, argv[i], strlen(argv[i])) < 0) return -1;
	i++;
    }
    return 0;
}

/*
 * Asp writeEsc command. Same as write but escapes some reserved characters.
 */

static int
websAspWriteEsc(int ejid UNUSED, webs_t wp, int argc, char **argv)
{
    char * esc_html;
    int i;

    a_assert(wp);
	
    for (i = 0; i < argc; ) {
	a_assert(argv);
	esc_html = escape_html(argv[i]);
	if (!esc_html || websWriteBlock(wp, esc_html, strlen(esc_html)) < 0) {
	    free(esc_html);
	    return -1;
	}
	free(esc_html);
	if (++i < argc) { if (websWriteBlock(wp, " ", 1) < 0) return -1; }
    }
    return 0;
}

/*
 * Asp skipStart command. Skip HTML output until skipEnd is called.
 */

static int
websAspSkipStart(int ejid UNUSED, webs_t wp, int argc UNUSED,
		 char **argv UNUSED)
{
    a_assert(wp);

    wp->skipOutput++;
    return 0;
}

/*
 * Asp skipEnd command. Enables HTML output again.
 */

static int
websAspSkipEnd(int ejid UNUSED, webs_t wp, int argc UNUSED,
	       char **argv UNUSED)
{
    a_assert(wp);

    if (wp->skipOutput > 0) { wp->skipOutput--; }
    return 0;
}

/*
 * strtokcmp -- Find s2 in s1. We skip leading white space in s1.
 * Return a pointer to the location in s1 after s2 ends.
 */

static char *
strtokcmp(char *s1, const char *s2)
{
    int len;

    s1 = skipWhite(s1);
    len = strlen(s2);
    for (len = strlen(s2); len > 0 && (tolower(*s1) == tolower(*s2)); len--) {
	if (*s2 == '\0') {
	    return s1;
	}
	s1++;
	s2++;
    }
    if (len == 0) {
	return s1;
    }
    return NULL;
}

/*
 * Skip white space
 */

static char *
skipWhite(char *s) 
{
    a_assert(s);

    if (s == NULL) {
	return s;
    }
    while (*s && isspace(*s)) {
	s++;
    }
    return s;
}
