#include <fcntl.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <pp/cfg.h>
#if defined(PP_FEAT_REMOTE_CONSOLE)
#include <pp/rfb.h>
#include <pp/vsc.h>
#endif /* PP_FEAT_REMOTE_CONSOLE */
#include <pp/intl.h>
#include <oem/eric_fw_oem.h>
#include "eric_base.h"
#include "eric_internal.h"
#include "wsIntrn.h"

typedef struct {
    const char* name;
    char* (*function)(int eid, webs_t wp, int argc, char **argv);
} internal_var_t;

typedef struct {
    const char* name;
    const char* def;
} special_color_default_t;

#if defined(PP_FEAT_REMOTE_CONSOLE)
static char* get_rfb_version(int eid, webs_t wp, int argc, char **argv);
static char* get_screenshot_info(int eid, webs_t wp, int argc, char **argv);
static char* get_video_settings_type(int eid, webs_t wp, int argc, char **argv);
#endif
static char* get_dyndns_default(int eid, webs_t wp, int argc, char **argv);
static char* get_vendor_homepage(int eid, webs_t wp, int argc, char **argv);
static char* get_special_color(int eid, webs_t wp, int argc, char **argv);
static char* get_redirect_timeout(int eid, webs_t wp, int argc, char **argv);

internal_var_t vars[] = {
#if defined(PP_FEAT_REMOTE_CONSOLE)
    { "rfb_version",	get_rfb_version },
    { "screenshot_info",get_screenshot_info }, 
    { "video_settings_type",	get_video_settings_type },
#endif
    { "dyndns_default",	get_dyndns_default },
    { "vendor_homepage",get_vendor_homepage },
    { "special_color",  get_special_color },
    { "redirect_timeout",  get_redirect_timeout },
    { NULL, NULL }	/* keep this */
};

 #define STYLE_FILE	"/lib/webpages/style.asp"
 #define STYLE_FILE_OEM	"/lib/webpages/oem/style.asp"

static char* get_color_from_style(char* name);
static char* get_color_default(char* name);

special_color_default_t special_color_defaults[] = {
    { "cornerbg", "#FFFFFF" },
    { "telnetbg", "#FFFFFF" },
    { "telnetfg", "#000000" },
    { "sshbg", "#FFFFFF" },
    { "sshfg", "#000000" },
    { NULL, NULL } /* keep this */
};

/* get internal variable for asp getInternalVar */
int
internal_var_asp(int eid, webs_t wp, int argc, char **argv)
{
    internal_var_t *var = vars;
    char* result;

    if (argc >= 1) {
	while (var && var->name) {
	    if (!strcmp(var->name, argv[0])) {
		result = var->function(eid, wp, argc, argv);
		ejSetResult(eid, (result) ? result : "");
		free(result);
		break;
	    }
	    var++;
	}
    } else {	
	char error_msg[256];
	snprintf(error_msg, 256, _("Unknown internal var: %s"), argv[0]);
	ejSetResult(eid, error_msg);
    }

    return 0;
}

#if defined(PP_FEAT_REMOTE_CONSOLE)
static char*
get_rfb_version(int eid UNUSED, webs_t wp UNUSED,
		int argc UNUSED, char **argv UNUSED)
{
    char* result;
    unsigned int major, minor;

    pp_rfb_get_proto_version(&major, &minor);

    if (asprintf(&result, "%02d.%02d", major, minor) == -1) {
	return 0;
    } else {
	return result;
    }
}
#endif

static char*
get_dyndns_default(int eid UNUSED, webs_t wp UNUSED,
		   int argc UNUSED, char **argv UNUSED)
{
    return strdup(DYNDNS_DEFAULT);
}

static char*
get_vendor_homepage(int eid UNUSED, webs_t wp UNUSED,
		    int argc UNUSED, char **argv UNUSED)
{
    char* vh;
    pp_cfg_get(&vh, "vendor_homepage");
    return vh;
}

static char*
get_redirect_timeout(int eid UNUSED, webs_t wp UNUSED,
		     int argc UNUSED, char **argv UNUSED)
{
    int to;
    char *to_s;

    pp_cfg_get_int(&to, "reset_redirect_time");

    asprintf(&to_s, "%d", to);
    return to_s;
}

static char*
get_special_color(int eid, webs_t wp UNUSED, int argc, char **argv)
{
    char* color_value;
    
    if (argc < 2) {
	ejSetResult(eid, "");
	return 0;
    }

    color_value = get_color_from_style(argv[1]);
    if (color_value == NULL) {
	color_value = get_color_default(argv[1]);
    }
    
    if (color_value) {
	//pp_log("Color for %s: %s\n", argv[1], color_value);
	return color_value;
    } else {
	return strdup("Unknown color");
    }
    
    return 0;
}

static char*
get_color_from_style(char* name)
{
    const char * fn = ___F;
    char *filebuf = NULL, *ret = NULL;
    struct stat filestat;
    int fd = -1, regret, size;
    char regexp_str[256];
    regex_t rgexp;
    regmatch_t rgmatch[2];
    
    // load (complete) stylesheet, try OEM directory first
    if ((fd = open(STYLE_FILE_OEM, O_RDONLY)) == -1) {
	if ((fd = open(STYLE_FILE, O_RDONLY)) == -1) {
	    pp_log_err("%s(): open()", fn);
	    goto bail;
	}
    }

    if (fstat(fd, &filestat) == -1) {
	pp_log_err("%s(): fstat()", fn);
	goto bail;
    }

    filebuf = malloc(filestat.st_size);
    if (read(fd, filebuf, filestat.st_size) != filestat.st_size) {
	pp_log_err("%s(): read()", fn);
	goto bail;
    }

    snprintf(regexp_str, sizeof(regexp_str),
	     "specialcolor_%s[ ]*?\\{[ ]*?color:[ ]*?(#[0-9a-fA-F]{6});[ ]*?\\}",
	     name);
  
    if ((regret = regcomp(&rgexp, regexp_str, REG_EXTENDED | REG_ICASE)) != 0) {
	pp_log("%s(): regcomp(): %d\n", fn, regret);
	goto bail;
    }

    if ((regret = regexec(&rgexp, filebuf, 2, rgmatch, REG_NOTBOL | REG_NOTEOL)) != 0) {
	if (regret == REG_NOMATCH) {
	    pp_log("%s(): regexec doesn't match\n", fn);
	} else {
	    pp_log_err("%s(): regexec()", fn);
	}
	goto bail_free;
    }

    if (rgmatch[1].rm_so == -1) {
	pp_log("%s(): regexp: no submatch\n", fn);
	goto bail_free;
    }

    // we found something, copy it
    size = rgmatch[1].rm_eo - rgmatch[1].rm_so;
    ret = malloc(size + 1);
    memcpy(ret, &filebuf[rgmatch[1].rm_so], size);
    ret[size] = '\0';
    
 bail_free:
    regfree(&rgexp);
    
 bail:
    if (filebuf) free(filebuf);
    if (fd != -1) close(fd);
    return ret;
}

static char*
get_color_default(char* name)
{
    special_color_default_t *entry = special_color_defaults;

    while(entry) {
	if (!strcmp(entry->name, name)) {
	    return strdup(entry->def);
	}
	entry++;
    }

    return NULL;
}

#if defined(PP_FEAT_REMOTE_CONSOLE)
static char*
get_screenshot_info(int eid UNUSED, webs_t wp, int argc, char **argv)
{
    char* ret = NULL;
    unsigned short val = 0;
    
    if (argc < 2) {
	goto bail;
    }
    if (!strcmp(argv[1], "width")) {
	val = wp->sd->screenshot_width;
    } else if (!strcmp(argv[1], "height")) {
	val = wp->sd->screenshot_height;	
    } else {
	goto bail;
    }
    
    if (asprintf(&ret, "%d", val) == -1) {
	ret = NULL;
    }

 bail:
    return ret;
}

static char*
get_video_settings_type(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    char *ret = NULL;
    
#ifdef PP_FEAT_VSC_PANEL_INPUT
    if (pp_vsc_have_display_enable(0) == 0) {
	ret = strdup("offset");
    }
#endif

    return ret;
}
#endif
