#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <pp/base.h>
#include <pp/intl.h>
#include <pp/cfg.h>
#include <pp/firmware.h>
#include <liberic_misc.h>
#include <liberic_net.h>
#include "eric_util.h"
#include "eric_forms.h"
#if defined(PP_FEAT_RPCCFG)
#include <pp/rpc_ripc.h>
#endif


static signed char map64[] = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1
};

static char alphabet64[] = {
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
	'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
	'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
	'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
	'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
	'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
	'w', 'x', 'y', 'z', '0', '1', '2', '3',
	'4', '5', '6', '7', '8', '9', '+', '/',
};

char* gen_response_msg_for_bulk(webs_t wp) {
    char* msg;
    const char* response_code;
    size_t size;
    int ret;
    const char* fmt = "<!-- response_code_begin %s response_code_end "
	"response_msg_begin %s response_msg_end  -->";
    
    if (wp->response_code == ERIC_RESPONSE_OK) {
	response_code = "ERIC_RESPONSE_OK";
    } else if (wp->response_code == ERIC_RESPONSE_WARNING) {
	response_code = "ERIC_RESPONSE_WARNING";
    } else if (wp->response_code == ERIC_RESPONSE_ERROR) {
	response_code = "ERIC_RESPONSE_ERROR";
    } else {
	response_code = "ERIC_RESPONSE_ERROR";
    }
    if (wp->response_msg == NULL) {
	wp->response_msg = "";
    }
    size = strlen(response_code) + strlen(fmt) + strlen(wp->response_msg) + 1;
    msg = (char*) malloc(size);
    if (msg == NULL) {
	return NULL;
    }
    ret = snprintf(msg, size, fmt, response_code, wp->response_msg);
    if (ret < 0) {
	return NULL;
    }
    return msg;
}

ssize_t
decodeBase64(char * data)
{
    u_long sh_buf;
    char * cp;
    char * op;
    int    c, i, j, shift;
    
    cp = op = data;
    while (*cp && *cp != '=') {
	/*
	 * Map 4 (6bit) input bytes and store in a single long (sh_buf)
	 */
	sh_buf = 0;
	shift = 18;
	for (i = 0; i < 4 && *cp && *cp != '='; cp++) {
	    if (*cp == '\r' || *cp == '\n') {
		continue;
	    }
	    if ((c = map64[*cp & 0xff]) == -1) {
		return -1;
	    }
	    sh_buf |= c << shift;
	    shift -= 6;
	    i++;
	}
	/*
	 * Interpret as 3 normal 8 bit bytes (fill in reverse order).
	 * Check for potential buffer overflow before filling.
	 */
	--i;
	for (j = 0; j < i; j++) {
	    *op++ = ((sh_buf >> (8 * (2 - j))) & 0xff);
	}
    }
    return op - data;
}

void eric_webs_encode64(char *outbuf, const char *s, int outlen)
__attribute__ ((weak, alias ("websEncode64")));

void
websEncode64(char *outbuf, const char *s, int outlen)
{
    unsigned long	shiftbuf;
    const char		*cp;
    char		*op;
    int			x, i, j, shift;

    op = outbuf;
    *op = '\0';
    cp = s;
    while (*cp) {
	/*
	 * Take three characters and create a 24 bit number in shiftbuf
	 */
	shiftbuf = 0;
	for (j = 2; j >= 0 && *cp; j--, cp++) {
	    shiftbuf |= ((*cp & 0xff) << (j * 8));
	}
	/*
	 * Now convert shiftbuf to 4 base64 letters.  The i,j magic calculates
	 * how many letters need to be output.
	 */
	shift = 18;
	for (i = ++j; i < 4 && op < &outbuf[outlen] ; i++) {
	    x = (shiftbuf >> shift) & 0x3f;
	    *op++ = alphabet64[(shiftbuf >> shift) & 0x3f];
	    shift -= 6;
	}
	/*
	 * Pad at the end with '='
	 */
	while (j-- > 0) {
	    *op++ = '=';
	}
	*op = '\0';
    }
}

char*
get_pp_cookie(webs_t wp, char* idbuf) {
    char * eric_session;

    if (wp->cookie != NULL) {
	eric_session = strstr(wp->cookie, PP_COOKIE_NAME);
    } else {
	eric_session = NULL;
    }
    if (eric_session == NULL || 
	sscanf(eric_session, PP_COOKIE_NAME "=%64[0-9A-Za-z]", idbuf) != 1) {
	return NULL;
    }
    return idbuf;
}

int
check_cookie(webs_t wp)
{
    char id[ERIC_ID_LENGTH + 1];
    const char * referer = websGetVar(wp, "HTTP_REFERER", "");
    int ret = 0;
    int auth = (strstr(wp->url, "/auth.asp") == wp->url) && strstr(referer, "/auth.asp");
    char * _url = NULL, * url_cut;
    int url_cut_len;

    if (NULL == get_pp_cookie(wp, id)) {
	if (auth) {
	    set_response(wp, ERIC_RESPONSE_OK,
			 _("%s Authentication failed. "
			 "Your browser must accept cookies!"),
			 "<!-- Authentication failed -->");
	}
    } else {
	int trial = 0;
    again:
	if ((wp->session = eric_session_get_by_id(id)) == 0) {
	    _url = strdup(wp->url);
	    url_cut = &_url[strspn(_url, "/")]; /* drop leading slashes */
	    url_cut[strcspn(url_cut, "?")] = '\0'; /* drop GET parameters */
	    url_cut_len = strlen(url_cut);
	    /* files that are unrestricted (*.jar is a MacOS java workaround!) */
	    if (!strcmp(url_cut, "style.asp") ||
		!strcmp(url_cut, "favicon.ico") ||
		!strcmp(url_cut, "life_id.html") ||
		(url_cut_len > 4 && (!strcmp(&url_cut[url_cut_len-4], ".gif") ||
				     !strcmp(&url_cut[url_cut_len-4], ".jpg") ||
				     !strcmp(&url_cut[url_cut_len-4], ".ico") ||
				     !strcmp(&url_cut[url_cut_len-4], ".jar")))) {
		free(_url);
	    } else if (trial++ < 10) {
		free(_url);
		usleep(100000);
		goto again;
	    }
	    if (form_was_submitted(wp) && auth) {
		set_response(wp, ERIC_RESPONSE_OK, _("%s Authentication failed."),
			     "<!-- Authentication failed -->");
	    }
	} else if ((wp->user = eric_session_get_user(wp->session)) == NULL) {
	    set_response(wp, ERIC_RESPONSE_OK, _(internal_error_msg));
	} else if (!(wp->sd = pp_hash_get_entry_i(webs_session_data, wp->session))) {
	    /*
	     * It seems this session was not created by ourself. This might happen
	     * if we are part of a cluster. To resolve this we create the missing
	     * webs session data structure.
	     */
	    if ((wp->sd = eric_webs_alloc_session_data()) != NULL) {
		pp_hash_set_entry_i(webs_session_data, wp->session, wp->sd,
				    eric_webs_free_session_data);
		eric_session_register_close_cb(wp->session, eric_webs_session_close_cb);
		ret = 1;
	    } else {
		set_response(wp, ERIC_RESPONSE_OK, _(internal_error_msg));
	    }
	} else {
	    ret = 1;
	}
    }

    return ret;
}

int
check_auth(webs_t wp)
{
    const char* error = "";
    
    if((wp->session = eric_net_ssl_auth_check(wp->bio, &error))) {
	if(!(wp->sd = pp_hash_get_entry_i(webs_session_data, wp->session))) {
	    if ((wp->sd = eric_webs_alloc_session_data()) != NULL) {
		pp_hash_set_entry_i(webs_session_data, wp->session, wp->sd,
				    eric_webs_free_session_data);
		eric_session_register_close_cb(wp->session, eric_webs_session_close_cb);
	    } else {
		set_response(wp, ERIC_RESPONSE_OK, _(internal_error_msg));
		return 0;
	    }
	}
	
	if ((wp->user = eric_session_get_user(wp->session))==NULL) {
	    set_response(wp, ERIC_RESPONSE_OK, _(internal_error_msg));
	    return 0;
	}
	
	return 1;
    }
    else {
	if (!check_cookie(wp)) {
	    set_response(wp, ERIC_RESPONSE_OK, _(error));
	    return 0;
	} else {
	    return 1;
	}
    }
    
}

int
translate_hotkey(webs_t wp, const char * hotkey, char ** keycode,
		 vector_t **keyvector, const char *label, int include_delis)
{
    int n_err = 0;
    char * error = NULL;
   
    if (keyvector == NULL || keycode != NULL) {
        char * _keycode;
        _keycode = eric_misc_get_key_appletparam(hotkey, &error, include_delis);
        if (_keycode == NULL) {
            *keycode = strdup("");
	    n_err = -1;
	} else {
	    *keycode = _keycode;
	}
	/* FIXME free _keycode ? */
    } else if (keyvector != NULL || keycode == NULL) {
        vector_t * _keyvector;
        _keyvector = eric_misc_key_get_keysequence(hotkey, &error);
	if (_keyvector == NULL) {
	    n_err = -1;
	    *keyvector = NULL;
	} else {
	    *keyvector = _keyvector;
        }
	/* FIXME free _keyvector ? */
    } else {
        n_err = -1;
    }
    
    if (n_err < 0) {	   
        set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Can't parse %s: '%s'<br>"
		       "Unknown key symbol '%s'!\n"), _(label), hotkey, error);    
    }
    free(error);
    return n_err;
}

void
set_response(webs_t wp, response_code_t code, const char * format, ...)
{
    if (format != NULL) {
	va_list args;
	va_start(args, format);
	vsnprintf(wp->msg_buf, sizeof(wp->msg_buf), format, args);
	wp->response_msg = wp->msg_buf;
	va_end(args);
    } else {
	wp->response_msg = NULL;
    }

    wp->response_code = code;
}

char *
escape_html(const char * html)
{
    int i, j;
    char * esc_html;

    if (!html) return NULL;

    /* first, count the number of extra characters */
    for (i = 0, j = 0; html[i] != '\0'; i++, j++) {
        if (html[i] == '<' || html[i] == '>') {
            j += 3;
	} else if (html[i] == '&') {
            j += 4;
	} else if (html[i] == '"') {
            j += 5;
	}
    }

    if (i == j) { return strdup(html); }

    esc_html = (char *)malloc(j + 1);
    if (esc_html != NULL) {
	for (i = 0, j = 0; html[i] != '\0'; i++, j++) {
	    if (html[i] == '<') {
		memcpy(&esc_html[j], "&lt;", 4);
		j += 3;
	    } else if (html[i] == '>') {
		memcpy(&esc_html[j], "&gt;", 4);
		j += 3;
	    } else if (html[i] == '&') {
		memcpy(&esc_html[j], "&amp;", 5);
		j += 4;
	    } else if (html[i] == '"') {
		memcpy(&esc_html[j], "&quot;", 6);
		j += 5;
	    } else {
		esc_html[j] = html[i];
	    }
	}
	esc_html[j] = '\0';
    }

    return esc_html;
}

char *
url_encode(const char * s)
{
    static unsigned char hexchars[] = "0123456789ABCDEF";
    size_t s_len = 0;
    unsigned int x, y;
    unsigned char * s_enc;

    if (s == NULL) { return NULL; }

    s_len = strlen(s);

    if ((s_enc = malloc(3 * s_len + 1)) != NULL) {
	for (x = 0, y = 0; s_len--; x++, y++) {
	    if (s[x] == ' ') {
		s_enc[y] = '+';
	    } else if ((s[x] < '0' && s[x] != '-' && s[x] != '.') ||
		       (s[x] < 'A' && s[x] > '9') ||
		       (s[x] > 'Z' && s[x] < 'a' && s[x] != '_') ||
		       (s[x] > 'z')) {
		s_enc[y++] = '%';
		s_enc[y++] = hexchars[(unsigned char) s[x] >> 4];
		s_enc[y] = hexchars[(unsigned char) s[x] & 15];
	    } else {
		s_enc[y] = (unsigned char) s[x];
	    }
	}
	s_enc[y] = '\0';
    }
    return ((char *)s_enc);
}

int
device_reset(webs_t wp)
{
#ifdef PP_FEAT_RPCCFG
    if (pp_rpc_am_i_master()) {
	/* reset slave */
	int clnt_res;
	if (PP_FAILED(pp_rpc_action(&clnt_res, PP_RPC_TARGET_SLAVE, PP_RPC_ACTION_RESET_DEVICE))
	    || PP_FAILED(clnt_res)) {
	    set_response(wp, ERIC_RESPONSE_ERROR, internal_error_msg);
	    return -1;
	}
    }
#endif /* PP_FEAT_RPCCFG */
    /* NOTE: Reset is done asynchronously after some seconds. */
    switch (pp_firmware_erla_reset_device(wp->session)) {
      case 0:
	  return 0;
      case -1:
	  set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	  return -1;
      case -2:
	  set_response(wp, ERIC_RESPONSE_ERROR,
		       _("There is a firmware upgrade running currently. "
			 "Try again later."));
	  return -1;
      default:
	  set_response(wp, ERIC_RESPONSE_ERROR, internal_error_msg);
	  return -1;
    }
}

