/*
 * main.c -- Main program for the GoAhead WebServer (LINUX version)
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 */

/******************************* Description *********************************/

/*
 * Main program for for the GoAhead WebServer. This is a demonstration
 * main program to initialize and configure the web server.
 */

/******************************** Includes ***********************************/

#include <pthread.h>
#include <signal.h>
#include <unistd.h> 
#include <sys/types.h>
#include <sys/wait.h>
#include <pp/base.h>
#include "uemf.h"
#include "wsIntrn.h"
#include "eric_base.h"
#include "webs.h"
#include "reconf.h"

/***************************** Forward Declarations **************************/

#ifdef B_STATS
static void printMemStats(int handle, char *fmt, ...);
static void memLeaks();
#endif

/********************************** Code *************************************/

int
eric_webs_init(void)
{
    if ((webs_session_data = pp_hash_create_i(100)) == NULL) {
	pp_log("%s(): pp_hash_create_i() failed\n", ___F);
	return PP_ERR;
    }

    /*
     * Initialize the memory allocator. Allow use of malloc and start 
     * with a 60K heap.  For each page request approx 8KB is allocated.
     * 60KB allows for several concurrent page requests.  If more space
     * is required, malloc will be used for the overflow.
     */
    bopen(NULL, (60 * 1024), B_USE_MALLOC);
    signal(SIGPIPE, SIG_IGN);

    /*
     *	Set the root web directory and the default page.
     */
    websSetDefaultDir("/lib/webpages");
    websSetDefaultPage("auth.asp");

    /*
     * Open the web server
     */
    if (PP_FAILED(websOpen())) return PP_ERR;

    /*
     * First create the URL handlers. Note: handlers are called in sorted
     * order with the longest path handler examined first. Here we define
     * the default web page handler.
     */
    websUrlHandlerDefine("", NULL, 0, websDefaultHandler,WEBS_HANDLER_LAST);

    /*
     * e-RIC or LARA support
     */
    if (eric_init() == -1) {
	eric_webs_cleanup();
	return PP_ERR;
    }

    if (PP_FAILED(reconf_init())) {
	pp_log("Initializing reconfiguration handlers failed\n");
        return PP_ERR;
    }	
    
    return PP_SUC;
}

void
eric_webs_cleanup(void)
{
    reconf_cleanup();

    eric_cleanup();
    websClose();
    websDefaultClose();
    
#ifdef B_STATS
    memLeaks();
#endif
    bclose();

    if (webs_session_data != NULL) {
	pp_hash_delete_i(webs_session_data);
	webs_session_data = NULL;
    }
}

#ifdef B_STATS

static void
memLeaks() 
{
    int fd;

    if ((fd = gopen("leak.txt", O_CREAT | O_TRUNC | O_WRONLY)) >= 0) {
	bstats(fd, printMemStats);
	close(fd);
    }
}

/*
 * Print memory usage / leaks
 */

static void
printMemStats(int handle, char *fmt, ...)
{
    va_list args;
    char buf[256];

    va_start(args, fmt);
    vsprintf(buf, fmt, args);
    va_end(args);
    write(handle, buf, strlen(buf));
}

#endif /* B_STATS */
